/*

    TransparentApplet.java
    This applet simulates transparency by tiling its background with
    the page's background image.

*/

import java.applet.Applet;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Rectangle;
import java.awt.image.*;
import java.net.URL;

public class TransparentApplet extends Applet implements Runnable, ImageObserver {
    // default values
    static final Color DEFAULT_BGCOLOR = Color.blue;
    static final int DEFAULT_BGX = -10;      // Netscape default left margin is 10
    static final int DEFAULT_BGY = -15;     // Netscape default top margin is 15
    static final int DEFAULT_PAUSE = 250;   // animation pause in ms.
    // heartbeat for applet     
    Thread animator;

    // suspend flag, toggled by mouse click
	boolean threadSuspended = false;

    // background image (null if none)
    Image bgImage;

    // information about the background image
    int bgX, bgY;
    int bgWidth, bgHeight;

    // background color
    Color bgColor;

    // animation pause between steps
    int pause;

    // current animation step
    int step;

    // for coordinating the paint thread
    int stepdrawn;

    // for downloading the background image
    MediaTracker tracker;

    // offscreen buffer for flicker-free animation
    Image offScreenImage = null;
    int offScreenWidth, offScreenHeight;

    /* The rest of the instance variables are specific to the zooming text */
    // cached fonts for the example below
    Font font[];

    // size of each floater
    Dimension floaterSize[];

    // string to display
    String floaterText;

    // color to display text in
    Color floaterColor;

    /* The following overloaded variations of getParameter allow you to
    // specify the default value, so you can guarantee a default value.
    // There is a version for each return type that must be handled.  A
    // fringe benefit is that these methods include the conversion logic
    // from the parameter string to the desired return type.
    */

    /** get a string from the parameter list */
    String getParameter(String label, String defaultValue) {
        String param = getParameter(label);
        if (param == null) {
            return defaultValue;
        } else {
            return param;
        }
    }

    /** get an integer from the parameter list */
    int getParameter(String label, int defaultValue) {
        String param = getParameter(label);
        if (param == null) {
            return defaultValue;
        } else {
       		return Integer.parseInt(param);
        }
    }

    /** convert a pair of hexadecimal characters into a number, -1 if invalid */
    final static int hexPairToInt(char c1, char c2) {
        int n1, n2;
        if (c1 >= '0' && c1 <= '9') {
            n1 = c1 - '0';
        } else if (c1 >= 'A' && c1 <= 'F') {
            n1 = c1 - 'A' + 10;
        } else if (c1 >= 'a' && c1 <= 'f') {
            n1 = c1 - 'a' + 10;
        } else {
            return -1;
        }

        if (c2 >= '0' && c2 <= '9') {
            n2 = c2 - '0';
        } else if (c2 >= 'A' && c2 <= 'F') {
            n2 = c2 - 'A' + 10;
        } else if (c2 >= 'a' && c2 <= 'f') {
            n2 = c2 - 'a' + 10;
        } else {
            return -1;
        }

        return (n1 * 16) + n2;
    }

    /** get a color representation from the parameter list */
    Color getParameter(String label, Color defaultValue) {
        String param = getParameter(label);
        if (param == null) {
            return defaultValue;
        } else {
       		int r, g, b;
            char c[] = new char[7];
            param.getChars(0, 7, c, 0);
            if (c[0] == '#') {       		
                char c1, c2;
                r = hexPairToInt(c[1], c[2]);
                g = hexPairToInt(c[3], c[4]);
                b = hexPairToInt(c[5], c[6]);
                if (r >= 0 && g >= 0 && b >= 0) {
                    return new Color(r, g, b);
                } else {
                    return defaultValue;
                }
            } else {
                return defaultValue;
            }
        }
    }

    /** initialize the applet */
    public void init() {
        tracker = new MediaTracker(this);

        // Get the background color
        bgColor = getParameter("BGCOLOR", DEFAULT_BGCOLOR);
        bgX = getParameter("BGX", DEFAULT_BGX);
        bgY = getParameter("BGY", DEFAULT_BGY);

        pause = getParameter("PAUSE", DEFAULT_PAUSE);

        // Get the image for the background and add it to the MediaTracker
        String bgname = getParameter("BACKGROUND");
        if (bgname == null) {
            TRACE("background tag not present");
            bgImage = null;
        } else {
            try {
                bgImage = getImage(getDocumentBase(), bgname);
                tracker.addImage(bgImage, 0);
            } catch (Exception e) {
                TRACE("Error getting background");
                bgImage = null;
            }
        }

        // Initialize the stuff on top of the background
        initFloater();
    }

    /** Start the thread that drives the applet. */
    public void start() {
        step = 0;
        stepdrawn = 0;
        animator = new Thread(this);
        animator.start();
    }

    /** Stop the thread that drives the applet. */
    public void stop() {
        animator.stop();
        animator = null;
    }

    /** Run the animation thread.
        First wait for the background image (if any) to fully load and paint.
        Then drop into animation loop.
    */

    public void run() {
        if (bgImage != null) {
            try {
                showStatus("Waiting for background image");
                tracker.waitForID(0);
            } catch (InterruptedException e) {
                TRACE("Background image download interrupted");
                return;
            }
            bgWidth = bgImage.getWidth(this);
            bgHeight = bgImage.getHeight(this);

            if (bgWidth <= 0 || bgHeight <= 0) {
                TRACE("Problem with background image");
                bgImage = null;
                return;
            }
        }

        repaint();
        showStatus("");
        Thread me = Thread.currentThread();
        while (animator == me) {
            try {
                Thread.sleep(pause);
            } catch (InterruptedException e) {
                break;
            } 

            if (stepdrawn < step) {
                continue;
            }

            step++;
            Rectangle redrawRect = floaterClipRect();
            repaint(0, redrawRect.x, redrawRect.y, redrawRect.width, redrawRect.height);
        }
    }

    public boolean mouseDown(Event evt, int x, int y) {
        if (animator != null) {
            if (!threadSuspended) {
                animator.suspend();
                threadSuspended = true;
            } else {
                animator.resume();
                threadSuspended = false;
            }
        }        

        return true;
    }

    public boolean imageUpdate(Image img, int infoflags, int x, int y, int width, int height) {
	    if ((infoflags & (FRAMEBITS|ALLBITS)) != 0) {
            repaint();
	    } else if ((infoflags & SOMEBITS) != 0) {
	        // don't paint too often, or the download will take forever
            repaint(100, 0, 0, width, height);
	    }
       
        return (infoflags & (ALLBITS|ERROR)) == 0;
    }

    /** Handle updates */
    public final synchronized void update (Graphics g) {
        // this seems to be necessary for this thread (the painting thread) to
        // stay even with the animation loop
        if (stepdrawn == 0) {    // only need to do it once
            Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
        }

        // paint handles the offscreen buffer
        paint(g);
    }

    public void paint(Graphics g) {
        Dimension d = size();
        if ((offScreenImage == null) || (d.width != offScreenWidth) || (d.height != offScreenHeight)) {
            offScreenImage = createImage(d.width, d.height);
            offScreenWidth = d.width;
            offScreenHeight = d.height;
        }

        // paint the offscreen buffer
        Graphics offScreen = offScreenImage.getGraphics();
             
        //drawOffScreen(offScreen, g.getClipRect());
        Rectangle clip = g.getClipRect();
        offScreen.clipRect(clip.x, clip.y, clip.width, clip.height);
        
        // tile the background image
        drawBackground(offScreen);

        // draw the floater, unless we're waiting for the background
        if (bgWidth > 0 || bgImage == null) {
            drawFloater(offScreen);
        }

        // blast the offscreen buffer onto the screen
        g.drawImage(offScreenImage, 0, 0, null);
    
        if (stepdrawn != step) {
            stepdrawn++;
        }
    }
        

    void drawBackground(Graphics g) {

        /* If there is no background image, or it hasn't started downloading,
        // fill the applet with the background color.  Otherwise fill the
        // applet by tiling the background image.
        */

        if (bgImage == null || bgWidth <= 0 || bgHeight <= 0) {
            g.setColor(bgColor);
            g.fillRect(0, 0, bgWidth, bgHeight);
        } else {
            // Tile the background image.
            //
            // bgX, bgY are the offset of the background from the top left
            // corner of the applet.  For an applet at the top left of the
            // page viewed on Netscape, bgX, bgY should nornally be -10, -15
            // which compensates for Netscape's margins.
            //
            // bgImage is the background image.
            //
            // bgWidth, bgHeight are the dimensions of the background image.
            //
            // offScreenWidth, offScreenHeight are the dimensions of the off
            // screen buffer, which will normally be the same as the applet's
            // dimensions.
            //
            // g is the graphics context for the off screen buffer, which is
            // used to avoid flicker.
            //
            // The logic:  First calculate the starting offset, which should
            // be zero or negative for complete coverage of the exposed 
            // background.  Then draw the image row by row, column by column
            // until the background is filled.

            // starting x coordinate for the tiling
            int startx = bgX % bgWidth;

            if (startx > 0) {
                startx -= bgWidth;
            }

            // starting y coordinate for the tiling
            int y = bgY % bgHeight;
            if (y > 0) {
                y -= bgHeight;
            }

            // loop by row until out of view
            while (y < offScreenHeight) {
                // loop by column
                for (int x = startx; x < offScreenWidth; x += bgWidth)  {
                    // draw the image
                    g.drawImage(bgImage, x, y, this);
                }
                y += bgHeight;
            }
        }
    }

    /** Draw the image that rides on top of the background.  This can be a
        transparent image (a floater), or it can be anything else you can do
        to a Graphics object.

        To do animation, use the instance variable step as a frame counter.
    */

    void drawFloater(Graphics g) {
        // in this non-optimized, hardcoded example, we simply draw the same
        // text over and over again, but at a font size determined by the
        // current step number.

        // get the index for the current step
        int index = stepToIndex(step);

        // center the text in the applet
        Dimension size = size();
        FontMetrics fm = getFontMetrics(font[index]);
        int x = (size.width - floaterSize[index].width) / 2;
        int y = (size.height - floaterSize[index].height) / 2 + fm.getAscent() + (fm.getLeading()/2);

        g.setFont(font[index]);
        g.setColor(floaterColor);
        g.drawString(floaterText, x, y);
    }

    /** initialize the stuff on top of the background */
    void initFloater() {
        // get the color as a parameter
        floaterColor = getParameter("TEXTCOLOR", Color.black);
        // get the text as a parameter
        floaterText = getParameter("TEXT", "Hello!");
        // initialize the font list: 6 fonts, ranging from 14 through 24 (even sizes only) 
        font = new Font[6];
        floaterSize = new Dimension[6];
        for (int i = 0; i < 6; i++) {
            font[i] = new Font("TimesRoman", Font.BOLD, 2 * i + 14);
            FontMetrics fm = getFontMetrics(font[i]);
            floaterSize[i] = new Dimension(fm.stringWidth(floaterText), fm.getHeight());
        }
    }

    /** return the clipping rectangle for redrawing only what is required 
        by the current animation
     */

    Rectangle floaterClipRect() {
        Dimension size = size();
        // if this is the beginning, clip nothing
        if (step == 0) {
            return new Rectangle(0, 0, size.width, size.height);
        }

        // calculate the indexes for both the current step and the last step to ensure
        // erasure of the last image as well as the drawing of the new image
        int oldindex = stepToIndex(step - 1);
        int newindex = stepToIndex(step);

        int maxwidth = Math.max(floaterSize[oldindex].width, floaterSize[newindex].width);
        int maxheight = Math.max(floaterSize[oldindex].height, floaterSize[newindex].height);

        return new Rectangle((size.width - maxwidth) / 2, (size.height - maxheight) / 2, maxwidth, maxheight); 
    }

    /** calculate an index based on the current animation step; this returns
        an index between 0 and 5 inclusive, in a triangular wave function
     */
    final static int stepToIndex(int step) {
        return Math.abs((step % 10) - 5);
    }

    // turn debugging trace on or off by setting the initial value 
    // of this flag
    static final boolean show_trace = true;
    
    // debugging trace
    void TRACE(String str) {
        if (show_trace) {
            System.out.println(str);
        }
    }
}
