/* module pdfg.cpp                                           */
/* Erzeugt ein sehr einfaches PDF Dokument                   */
/*   erzeugt zusaetzlich Document Info                       */
/*   erzeugt eine Annotation                                 */
/* Copyright Dr.E.Huckert 1-96                               */

#include <stdio.h>
#include <fcntl.h>
#include <string.h>
/* Eventuell ntig - je nach Compiler */
/* #include <unistd.h> */

#ifdef LINUX
#include <linux/iso_fs.h>
/* strange ... */
#define O_BINARY ISOFS_FILE_BINARY
/* wo ist unter LINUX die richtige Bibliothek?... */
#define setmode(x,y)
#define CREAT_MASK  0755
#else
#include <io.h>
#define CREAT_MASK 0xff        /* nur fuer MSDOS */
#endif

#define MAXOBJS     100

extern "C"
{
  int exit(int);
};

class PDF
{
private:
  int ofd;
  int no_objs;
  int act_no;
  int root_objno;
  int info_objno;
  long objpos[MAXOBJS];
  long xrefpos;
  long writepos;
public:
  PDF(char *filename);
  ~PDF();
  int StartObject(int no);
  int StartObject(int no, char *type);
  int CloseObject();
  int AddLine(char *line);
  int AddDict(char *line);
  int StartDict();
  int CloseDict();
  int MakeAnnotation(int objno, int x1, int y1,
                     int x2, int y2, char *txt);
  int MakeInfo(int objno, char *author, char *creator, char *date);
  int MakeTextContent(int objno, char *txt);
  int MakeFont(int objno, int fontno, char *fontname);
};

// constructor
PDF::PDF(char *filename)
{
int n;
  no_objs    = 0;
  ofd        = 0;
  act_no     = 0;
  info_objno = 0;
  root_objno = 0;
  xrefpos    = 0L;
  writepos   = 0L;
  for (n=0; n < MAXOBJS; n++)
    objpos[n] = 0L;
  ofd = creat(filename, CREAT_MASK);
  if (ofd < 0)
  {
    fprintf(stderr, "Error:cannot create file\n");
    exit(2);
  }
  setmode(ofd, O_BINARY);
  this->AddLine("%PDF-1.0\n");
}  // end PDF::PDF()

// Destructor
// Erzeugt Cross Reference und Trailer
PDF::~PDF()
{
int n;
char hbuf[128];
  if (ofd != 0)
  {
    /* cross reference  erzeugen */
    this->xrefpos = this->writepos;
    sprintf(hbuf, "xref\n0 %d\n", this->no_objs + 1);
    this->AddLine(hbuf);
    /* a line must have exactly 20 chars incl. the linefeed */
    this->AddLine("0000000000 65535 f \n");
    for (n=1; n <= no_objs; n++)
    {
      sprintf(hbuf, "%010ld 00000 n \n", objpos[n]);
      this->AddLine(hbuf);
    }
    /* trailer erzeugen */
    this->AddLine("trailer\n<<\n");
    sprintf(hbuf, "/Size %d\n", this->no_objs + 1);
    this->AddLine(hbuf);
    if (this->root_objno != 0)
    {
      sprintf(hbuf, "/Root %d 0 R\n", root_objno);
      this->AddLine(hbuf);
    }
    if (this->info_objno != 0)
    {
      sprintf(hbuf, "/Info %d 0 R\n", info_objno);
      this->AddLine(hbuf);
    }
    this->CloseDict();
    sprintf(hbuf,"startxref\n%ld\n", this->xrefpos);
    this->AddLine(hbuf);
    write(this->ofd,"%%EOF\n", 6);
    close(ofd);
  }
}   // end PDF::~PDF()

int PDF::StartObject(int no)
{
char hbuf[40];
int  nb;
  this->act_no = no;
  if (no < MAXOBJS)
    this->objpos[this->act_no] = this->writepos;
  sprintf(hbuf, "%d 0 obj\n", no);
  nb = this->AddLine(hbuf);
  if (no > this->no_objs)
    this->no_objs = no;
  return nb;
}   // end PDF::StartObject()

// fuehrt Buch ueber einige wicgtige Objekttypen
int PDF::StartObject(int no, char *type)
{
int nb;
  nb = StartObject(no);
  if (strcmp(type, "/Root") == 0)
    root_objno = no;
  if (strcmp(type, "/Info") == 0)
    info_objno = no;
  return nb;
}   // end PDF::StartObject()

/* Schreibt eine Folge von Bytes ind ein Objekt */
int PDF::AddLine(char *str)
{
int nb;
  nb = write(this->ofd, str, strlen(str));
  this->writepos += nb;
  return nb;
}

int PDF::StartDict()
{
int nb;
  nb = this->AddLine("<<\n");
  return nb;
}

int PDF::CloseDict()
{
int nb;
  nb = this->AddLine(">>\n");
  return nb;
}

// Erzeugt ein PDF Dictionary (zwischen "<<" und ">>")
int PDF::AddDict(char *str)
{
int nb;
  this->StartDict();
  this->AddLine(str);
  this->CloseDict();
  return nb;
}

int PDF::CloseObject()
{
int nb;
  nb = this->AddLine("endobj\n");
  return nb;
}

/* Erzeugt einen Notiz-Objekt */
int PDF::MakeAnnotation(int objno,
                        int x1, int y1, int x2, int y2,
                        char *txt)
{
char hbuf[256];
int nb;
  this->StartObject(objno);
  this->StartDict();
  this->AddLine("/Type /Annot /Subtype /Text\n");
  sprintf(hbuf, "/Rect [%d %d %d %d]\n", x1, y1, x2, y2);
  this->AddLine(hbuf);
  this->AddLine("/Contents ");
  if (txt != NULL)
  {
    sprintf(hbuf, "(%s)\n", txt);
    this->AddLine(hbuf);
  }
  this->CloseDict();
  nb = this->CloseObject();
  return nb;
}

/* Erzeugt ein Info Objekt */
int PDF::MakeInfo(int objno, char *author, char *creator, char *date)
{
int  nb;
char hbuf[256];
  this->StartObject(objno, "/Info");
  this->StartDict();
  if (creator != NULL)
  {
    sprintf(hbuf, "/Creator (%s)\n", creator);
    this->AddLine(hbuf);
  }
  if (author != NULL)
  {
    sprintf(hbuf, "/Author (%s)\n", author);
    this->AddLine(hbuf);
  }
  if (date != NULL)
  {
    sprintf(hbuf, "/CreationDate (%s)\n", date);
    this->AddLine(hbuf);
  }
  this->CloseDict();
  nb = this->CloseObject();
  return nb;
}

/* Erzeugt ein Text Objekt (/Content) */
int PDF::MakeTextContent(int objno, char *txt)
{
int  nb;
char hbuf[128];
  this->StartObject(objno);
  sprintf(hbuf, "<< /Length %d >>\nstream\n", strlen(txt));
  this->AddLine(hbuf);
  this->AddLine(txt);
  this->AddLine("endstream\n");
  nb = this->CloseObject();
  return nb;
}

/* Erzeugt ein Font Objekt */
int PDF::MakeFont(int objno, int fontno, char *fontname)
{
int nb;
char hbuf[128];
  this->StartObject(objno);
  this->StartDict();
  this->AddLine("/Type /Font\n");
  this->AddLine("/Subtype /Type1\n");
  sprintf(hbuf, "/Name /F%d\n", fontno);
  this->AddLine(hbuf);
  sprintf(hbuf, "/BaseFont /%s\n", fontname);
  this->AddLine(hbuf);
  this->AddLine("/Encoding /WinAnsiEncoding\n");
  this->CloseDict();
  this->CloseObject();
  return nb;
}


char text1[256] = "BT\n/F1 24 Tf\n100 700 Td (Hier ist Text)Tj\nET\n\
6 w 100 600 m\n400 600 l 400 400 l 100 400 l 100 600 l S\n";
char text2[256] = "BT\n/F1 24 Tf\n100 700 Td (Text fuer Seite 2)Tj\nET\n\
6 w 100 600 m\n400 600 l 400 400 l 100 400 l 100 600 l S\n";

main(argc, argv)
int argc;
char *argv[];
{
int  ofd, n;
long ilength;
char hbuf[256];
PDF  *pdf;
  if (argc <= 1)
  {
     fprintf(stderr,"usage:pdfgen outfile\n");
     exit(1);
  }
  pdf = new PDF(argv[1]);

  pdf->StartObject(1, "/Root");
  pdf->AddDict("/Type /Catalog\n/Pages 2 0 R\n/Outlines 7 0 R\n");
  pdf->CloseObject();

  /* Schreibt ein /Pages Objekt, Nr. 2 */
  pdf->StartObject(2);
  pdf->AddDict("/Type /Pages\n/Count 2\n/Kids [3 0 R 12 0 R]\n");
  pdf->CloseObject();

  /* Schreibt ein /Page Objekt, Nr. 3 */
  pdf->StartObject(3);
  pdf->StartDict();
  pdf->AddLine("/Type /Page\n/Parent 2 0 R\n");
  pdf->AddLine("/MediaBox [0 0 612 792]\n");
  pdf->AddLine("/Contents 4 0 R\n");
  pdf->AddLine("/Annots [9 0 R]\n");
  pdf->AddLine("/Resources << /Font << /F1 5 0 R >> /ProcSet 6 0 R >>\n");
  pdf->CloseDict();
  pdf->CloseObject();

  /* Schreibt ein text content Objekt, Nr. 4 */
  pdf->MakeTextContent(4, text1);

  /* Schreibt ein /Font Objekt, Nr. 5 */
  pdf->MakeFont(5, 1, "Helvetica");

  /* Schreibt ein /ProcSet Objekt, Nr. 6 */
  pdf->StartObject(6);
  pdf->AddLine("[/PDF /Text]\n");
  pdf->CloseObject();

  /* Schreibt ein Gliederungsobjekt(/Outlines), Nr. 7 */
  pdf->StartObject(7);
  pdf->StartDict();
  pdf->AddLine("/Type /Outlines\n/Count 0\n>>\n");
  pdf->CloseDict();
  pdf->CloseObject();

  /* Schreibt ein Notiz Objekt, Nr. 9 */
  pdf->MakeAnnotation(9, 500, 400, 700, 500,
                      "Dies ist eine Annotation");

  /* schreibt ein /Page Objekt, Nr.12 */
  pdf->StartObject(12);
  pdf->StartDict();
  pdf->AddLine("\n/Type /Page\n");
  pdf->AddLine("/Parent 2 0 R\n");
  pdf->AddLine("/MediaBox [0 0 612 792]\n");
  pdf->AddLine("/Contents 10 0 R\n");
  pdf->AddLine("/Resources << /Font << /F1 5 0 R >> /ProcSet 6 0 R >>\n");
  pdf->CloseDict();
  pdf->CloseObject();

  /* Screibt ein Text Inhaltsobjekt Nr. 10 */
  pdf->MakeTextContent(10, text2);

  /* schreibt ein /Info Object, Nr. 8 */
  pdf->MakeInfo(8, "E.Huckert", "Program pdfg", "1.3.1996");

  delete pdf;
  return 0;
}   /* end main */


