// AniMov (c) Klaus Zerbe / c't 1997
// ActiveMovie - Player mit integriertem Sprite - Overlay - Filter
// Beispielprogramm fr AM Filterprogrammierung
//
#include <stdio.h>		// ANSI-C File Streams
#include <string.h>		// Stringfunktionen
#include <math.h>		// fmod(), sin(), cos()
#include <windows.h>	// Win32 SDK
#include <streams.h>	// ActiveMovie SDK
#include <initguid.h>	// DEFINE_GUID zur GUID- Generierung

// Maximum / Minimum Defaults fr CSequencedInt
const int maxInt= 0x7fffffff;
const int minInt= -maxInt;

// global synchronisierte Funktionen der Zeit 
//
class CSequencedInt
{
public:
	// Funktionen
	enum Repeat { steady, sawtooth, triangle, sinus, cosinus };
	// setze Zeitpunkt (Frame) t
	static void SetTime(unsigned t);
	// setze Zeitpunkt (Frame) 0
	static void ResetTimer() { SetTime(0); }
	// nchster Frame
	static void Next() { SetTime(++uTime); }
	CSequencedInt();
	~CSequencedInt();
	// initialisiere Startwert, Minimum, Maximum und Schrittweite
	void Init(int start,int min=minInt,int max=maxInt,double step=0);
	// setze Minimum und Maximum
	void SetRange(int min= minInt, int max= maxInt) {
		m_iMin= minInt;	m_iMax= maxInt; }
	// setze Startwert
	void SetStart(int val=0) { m_dValue= m_dStart= double(val); }
	// setze Schrittweite
	void SetStep(double step=0) { m_dDelta= step; }
	void SetRate(int rate=1) { m_dDelta= 1.0/rate; }
	// setze Funktion
	void SetWaveform(Repeat wave= sawtooth) { m_repeat= wave; }
	// lies aktuellen Wert
	double GetValue() { return m_dValue; }
	int Get() { return int(m_dValue); }
private:
	void SetTime(); // aktualisiere Wert
	static CSequencedInt *pFirst, *pLast; // Listenstart/Ende
	static unsigned uTime; // Framenummer
	double m_dStart, m_dValue; // Wert
	double m_dDelta;  // Schrittweite
	int m_iMin, m_iMax;  // Minimal- und Maximalwert
	Repeat m_repeat;  // Funktion 
	CSequencedInt *m_pNext; // nchste Instanz
};

// Bitmapinfo wie in BMP- Dateien
template <int n> struct BITMAPINFO_VAR
{
	BITMAPINFOHEADER  bmiHeader;
	union {
		RGBQUAD  bmiColors[n];
		DWORD bmiColorMask[3];
	};
};

// gerteunabhngige Bitmap 
// (interne Darstellung im RGB555 Pixelformat
//
class CImage {
public:
	CImage() { 
		m_pBMI = NULL; m_pPixels = NULL; Create(16, 16); }
	~CImage() { Release(); }
	// erzeuge Dummy- Bitmap 
	BOOL Create(int iWidth, int iHeight);
	// liefert Bitmap- Pixels (RGB555)
	PWORD GetImage() { return m_pPixels; }
	// Bitmap Dimensionen
    ULONG GetWidth() { return m_pBMI->bmiHeader.biWidth; }
    ULONG GetStorageWidth() { return (GetWidth() + 3) & ~3;}
	ULONG GetHeight() { return m_pBMI->bmiHeader.biHeight; }
	ULONG GetPixels() { return GetWidth() * GetHeight(); }
	// Bitmap Wortbreite
	WORD  GetPixelBits() { return m_pBMI->bmiHeader.biBitCount; }
	// lies Bitmap von Windows DIB- (BMP-) Datei
	virtual BOOL Load(LPSTR pstrFileName);
	// wandle Bitmap in RGB Bitmap mit Bitaufteilung colormask um
	BOOL ConvertToRGB555();
protected:
	// liefert Windows Bitmap Info
	BITMAPINFO* GetInfo() { return (BITMAPINFO*)m_pBMI; }
	RGBQUAD* GetPaletteColors() { 
		return (RGBQUAD*)(((BYTE*)m_pBMI)+sizeof(BITMAPINFOHEADER));}
	void Release(void);
	PWORD m_pPixels;
private:
	// bestimme Farbenzahl von 8-Bit Bitmap
	int CountColors(BITMAPINFO* pBmpInfo);
	// Umwandlung 8->16 Bit Bitmap
	BOOL ConvertPalettedToRGB(WORD colorMask);
	BITMAPINFO_VAR<256> *m_pBMI; 
};

// animierte Bitmap (Sprite)
//
class CSprite : public CImage {
public:
	CSprite() : CImage() { m_pstrName=NULL; Init(); }
	~CSprite() { if (m_pstrName) delete m_pstrName; Release(); }
	// lies Bitmap von Windows DIB- (BMP-) Datei
	BOOL Load(LPSTR pstrFileName);
	// liefert Position und Grsse von Bitmap Ausschnitt
	// bzw. Position des Sprites in Szene
	ULONG GetTopRow() { return m_siCelTop.Get(); }
	ULONG GetLeftCol() { return m_siCelLeft.Get(); }
	ULONG GetRows() { return m_siRows.Get(); }
	ULONG GetCols() { return m_siCols.Get(); }
	ULONG GetPosX() { return m_siPosLeft.Get(); }
	ULONG GetPosY() { return m_siPosTop.Get(); }
	CSequencedInt& TopRow() { return m_siCelTop; }
	CSequencedInt& LeftCol() { return m_siCelLeft; }
	CSequencedInt& Rows() { return m_siRows; }
	CSequencedInt& Cols() { return m_siCols; }
	CSequencedInt& PosX() { return m_siPosLeft; }
	CSequencedInt& PosY() { return m_siPosTop; }
	void SetKeycolor(WORD wKeycolor) { m_wKeycolor= wKeycolor; }
	WORD GetKeycolor() { return m_wKeycolor; }
	void SetFrameInterval(int iStartFrame, int iEndFrame) {
		m_iStartFrame= iStartFrame;	m_iEndFrame= iEndFrame; }
	int GetStartFrame() { return m_iStartFrame; }
	int GetEndFrame() { return m_iEndFrame; }
	void SetNext(CSprite* pNext) { m_pNext= pNext; }
	CSprite* GetNext() { return m_pNext; }
	LPSTR GetName() { return m_pstrName; }
private:
	void Init();
	LPSTR m_pstrName; // Sprite- Name
	CSequencedInt m_siCelTop, m_siCelLeft; // Position Ausschnitt
	CSequencedInt m_siRows, m_siCols;  // Grsse
	CSequencedInt m_siPosTop, m_siPosLeft;  // Position in Szene
	WORD m_wKeycolor;  // Keycolor (transparente Farbe)
	int m_iStartFrame, m_iEndFrame; // Frame- Interval
	CSprite* m_pNext;  // nchstes Sprite in Liste
};


// Sprite - Liste
//
class CSpriteList {
public:
	CSpriteList() { m_pFirst= m_pLast= NULL; }
	~CSpriteList() { Clear(); }
	void Clear();  // lsche Liste
	// liefert neues Sprite am Listenende
	CSprite* Add(); 
	// entfernt pSprite aus Liste
	void Remove(CSprite* pSprite);
	// Listenanfang
	CSprite* GetFirst() { return m_pFirst; }
	// baue Spriteliste aus INI- Datei auf
	void Parse(LPSTR pstrFileName);
private:
	// Hilfsfunktionen fr Parser
	void SetSpriteAttribute(LPSTR pstrFileName, CSprite *pSprite,
         LPSTR pKey, CSequencedInt& si);
	void SetTimeInterval(LPSTR pstrFileName, CSprite* pSprite);
	void SetKeycolor(LPSTR pstrFileName, CSprite* pSprite);
	CSprite *m_pFirst, *m_pLast; // Listenanfang und Ende
};


// generischer ActiveMovie Player
//
class CMoviePlayer {
public:
	CMoviePlayer();
	~CMoviePlayer() { Release(); }
	// prft hr und triggert Exception bei Fehler
	void Check(HRESULT hr, LPSTR pMessage) {
		if (FAILED(hr)) throw pMessage; }
	// baut Filtergraph auf
	virtual void RenderFile(LPSTR szFileName);
	void Play();  // spielt Datei ab
protected:
	virtual void Release(); // gib Interfaces frei
	// notwendige Interfaces
	IGraphBuilder *pGraph;
	IMediaEvent *pEvent;
	IMediaControl *pControl;
	IMediaSeeking *pSeeking;
};

class CVideoOverlayFilter;

// ActiveMovie Player mit Sprite- Stanze
//
class COverlaidPlayer : public CMoviePlayer {
public:
	COverlaidPlayer() { m_pVideoRenderer= NULL; }
	~COverlaidPlayer() { Release(); }
	// baut Filtergraph mit Sprite- Stanze auf
	void RenderFile(LPSTR szFileName);
	// liest Spriteliste aus INI-Datei szFileName
	void LoadSprites(LPSTR szFileName) 
         { m_spriteList.Parse(szFileName); }
	CSpriteList* GetSpriteList() { return &m_spriteList; }
private:
	void Release();
	CVideoOverlayFilter *m_pOvlFilter; // Filter Instanz
	IFilter *m_pVideoRenderer;  // Video Renderer Filter-Interface
	CSpriteList m_spriteList; // Sprite- Liste
};

// Sprite- Stanze
// zeichnet Sprites der Spriteliste auf Videobilder
//
class CVideoOverlayFilter : public CTransInPlaceFilter
{
public: 
	CVideoOverlayFilter(TCHAR *tszName, COverlaidPlayer *pPlayer,
                        LPUNKNOWN punk, HRESULT *phr);
    DECLARE_IUNKNOWN;
private:
	PWORD pRenderSurface;  // Videobild Pixeldaten
	DWORD bmColorMasks[3];  // Videobild RGB Bitaufteilung
	BITMAPINFOHEADER bmiHeader; // Videobild Bitmap- Info 
	REFERENCE_TIME avgTimePerFrame; // Videobild- Rate
	// Sprite Renderer
    HRESULT Transform(IMediaSample *pSample);
	// Bestimmung Videobild- Format
	HRESULT CheckInputType(const CMediaType* mtIn);
	COverlaidPlayer *m_pPlayer;
};

// Klassenvariablen (statisch)
CSequencedInt* CSequencedInt::pFirst= NULL;
CSequencedInt* CSequencedInt::pLast= NULL;
unsigned CSequencedInt::uTime= 0;

// setze alle Variablen auf Wert fr Zeitpunkt t
void CSequencedInt::SetTime(unsigned t)
{
	CSequencedInt*p;
	uTime= t;
	for (p= pFirst; p; p=p->m_pNext) p->SetTime();
}

// initialisiere auf Standardwerte und 
// trage neue Variable in Liste ein
CSequencedInt::CSequencedInt()
{
	m_iMin= minInt;
	m_iMax= maxInt;
	m_dValue= m_dStart= 0.0;
	m_dDelta= 0.0;
	m_repeat= steady;
	m_pNext= NULL;
	if (!pFirst) pFirst= this;
	if (pLast) pLast->m_pNext= this;
	pLast= this;
}

// Initialisierung mit Startwert, Minimum, Maximum und Schrittweite
void CSequencedInt::Init(int val, int min, int max, double step)
{
	m_iMin= min;
	m_iMax= max;
	m_dValue= m_dStart= double(val);
	m_dDelta= step;
	m_repeat= steady;
}

// entferne Variable aus Liste
CSequencedInt::~CSequencedInt()
{
	CSequencedInt *p;
	if (!pFirst) return;
	for (p=pFirst; p && (p->m_pNext != this); p=p->m_pNext);
	if (pFirst == this) pFirst= this->m_pNext;
	if (pLast == this) pLast= p;
	if (!p) return;
	p->m_pNext= this->m_pNext;
}

// setze Instanz entsprechend aktuellem Zeitpunkt
void CSequencedInt::SetTime()
{
	if (!m_dDelta) return;
	m_dValue= m_dStart+ (m_dDelta * uTime);
	switch (m_repeat) {
		case steady:
			if (m_dValue < m_iMin) m_dValue= m_iMin;
			if (m_dValue > m_iMax) m_dValue= m_iMax;
			break;
		case sawtooth:
			m_dValue= fmod(m_dValue-m_iMin,m_iMax-m_iMin)+m_iMin;
			break;
		case triangle: { 
			double dAmp= m_iMax - m_iMin;
			double	dMod= fmod(m_dValue - m_iMin, dAmp);
			m_dValue= (2*dMod > dAmp) ? 
				(dAmp - dMod) + m_iMin : dMod +m_iMin;
			} break; 
		case sinus: {
			double dAmp= (m_iMax - m_iMin) / 2.0;
			m_dValue= dAmp * sin(uTime * m_dDelta) + dAmp; }
			break; 
		case cosinus: {
			double dAmp= (m_iMax - m_iMin) / 2.0;
				m_dValue= dAmp * cos(uTime * m_dDelta) + dAmp; }
			break; 
	}
}

// erzeuge "Dummy"-Bitmap
BOOL CImage::Create(int iWidth, int iHeight)
{
	int iBitsSize;
	Release();
	m_pBMI = new BITMAPINFO_VAR<256>;
	if (!m_pBMI) return FALSE;
	iBitsSize = ((iWidth + 3) & ~3) * iHeight;
	m_pPixels = new WORD[iBitsSize];
	if (!m_pPixels) {
		Release(); return FALSE;
	}
    BITMAPINFOHEADER* pBI = (BITMAPINFOHEADER*) m_pBMI;
    pBI->biSize = sizeof(BITMAPINFOHEADER);
    pBI->biWidth = iWidth;
    pBI->biHeight = iHeight;
    pBI->biPlanes = 1;
    pBI->biBitCount = 16;
    pBI->biCompression = BI_BITFIELDS;
    pBI->biSizeImage = 0;
    pBI->biXPelsPerMeter = 0;
    pBI->biYPelsPerMeter = 0;
    pBI->biClrUsed = 0;
    pBI->biClrImportant = 0;
	m_pBMI->bmiColorMask[0]= 0x001f;
	m_pBMI->bmiColorMask[1]= 0x03e0;
	m_pBMI->bmiColorMask[2]= 0x7c00;
    memset(m_pPixels, 0, iBitsSize);
    return TRUE;
}

// Bitmap freigeben
void CImage::Release(void)
{
    if (m_pBMI) delete m_pBMI;	
    if (m_pPixels) delete m_pPixels;
	m_pBMI= NULL;
	m_pPixels= NULL;
}

// bestimme Farbenzahl fr 8-Bit Bitmap
int CImage::CountColors(BITMAPINFO* pBmpInfo)
{
	int iMax, iColors;
	BITMAPINFOHEADER *pBIH = &(pBmpInfo->bmiHeader);
	iMax= iColors= 1 << pBIH->biBitCount;
	if (pBIH->biClrUsed) iColors = pBIH->biClrUsed;
	if (iColors > iMax) iColors = iMax;
	return iColors;
}

// lies Windows BMP- bzw. DIB- Datei aus pstrFileName
BOOL CImage::Load(LPSTR pstrFileName)
{
	FILE *fp;
    BITMAPINFO_VAR<256>* pBmpInfo = NULL;
	BITMAPINFOHEADER bmpInfoHdr;
	BITMAPFILEHEADER bmpFileHdr;
    BYTE* pBits = NULL;
	int iBytes, iColors, iColorTableSize, iBitsSize;

	if (!pstrFileName) return FALSE;
	// ffne Bitmap Datei
	fp= fopen(pstrFileName,"rb");
	if (!fp) return FALSE;
	fpos_t dwFileStart;
	if(fgetpos(fp, &dwFileStart)) return FALSE;
	// lies Datei- Header
	iBytes = fread(&bmpFileHdr, 1, sizeof(bmpFileHdr), fp);
	if (iBytes != sizeof(bmpFileHdr)) return FALSE;
	// ist es eine Windows Bitmap- Datei ?
	if (bmpFileHdr.bfType != 'MB') return FALSE;
	iBytes = fread(&bmpInfoHdr, 1, sizeof(bmpInfoHdr), fp); 
	if (iBytes != sizeof(bmpInfoHdr)) return FALSE;
	if (bmpInfoHdr.biSize != sizeof(BITMAPINFOHEADER)) return FALSE;
	// Palette oder RGB- Werte?
	iColors = CountColors((BITMAPINFO*) &bmpInfoHdr);
	if (iColors > 256) iColorTableSize = 3 * sizeof(DWORD); // RGB 
	else iColorTableSize = iColors * sizeof(RGBQUAD); // Palette
	iBitsSize = bmpFileHdr.bfSize - bmpFileHdr.bfOffBits;
	// lies Palette
	pBmpInfo = new BITMAPINFO_VAR<256>;
	if (!pBmpInfo) return FALSE;
	memcpy(pBmpInfo, &bmpInfoHdr, sizeof(BITMAPINFOHEADER));
	iBytes = fread(((BYTE*) pBmpInfo) + sizeof(BITMAPINFOHEADER),
		1, iColorTableSize, fp);
	if (iBytes != iColorTableSize) return FALSE;
	// lies Pixeldaten
	pBits = new BYTE[iBitsSize];
	if (!pBits) return FALSE; 
	if (fseek(fp, long(dwFileStart+bmpFileHdr.bfOffBits),SEEK_SET)) 
		return FALSE;
	iBytes = fread(pBits, 1, iBitsSize, fp);
	if (iBytes != iBitsSize) return FALSE;
	// aktualisiere Instanz
	Release();
	m_pBMI = pBmpInfo; 
	m_pPixels = (PWORD)pBits;
	fclose(fp);
	// Umwandlung in RGB555
	return ConvertToRGB555();
}

// 8-Bit Bitmap in 16-Bit RGB umwandeln
BOOL CImage::ConvertToRGB555()
{
	int pixelBytes= GetPixelBits()/8;
	int width= GetStorageWidth();
	if (pixelBytes == 2) return TRUE;
	PWORD pNewRow, pOldBits= m_pPixels;
	PBYTE pOldRow;
	RGBQUAD *palColor;
	// alloziere Speicher fr RGB Bitmap
	PWORD pNewBits= new WORD[width * GetHeight()];
	if (!pNewBits) return FALSE;
	pOldRow=(PBYTE)pOldBits;
	pNewRow=pNewBits;
	// kopiere Pixels
	m_pPixels= pNewBits;
	for (ULONG row=0; row < GetHeight(); row++) {
		for (ULONG col=0; col < GetWidth(); col++) {
			if (pixelBytes == 1) {
				palColor= GetPaletteColors() + pOldRow[col];
			} else {
				palColor= (RGBQUAD*)(pOldRow+(3 * col));
			}
			pNewRow[col] = (palColor->rgbBlue >> 3)
				+ ((palColor->rgbGreen >> 3) << 5)
				+ ((palColor->rgbRed >>3) << 10);
		}
		pNewRow+=width;
		pOldRow+= width * pixelBytes;
	}
	delete pOldBits;
	// setze Bitmap- Info
	m_pBMI->bmiHeader.biBitCount= 16;
	m_pBMI->bmiHeader.biCompression= BI_BITFIELDS;
	m_pBMI->bmiColorMask[0]= 0x001f;
	m_pBMI->bmiColorMask[1]= 0x03e0;
	m_pBMI->bmiColorMask[2]= 0x7c00;
	return TRUE;
}

// initialisiere Vorgabewerte
void CSprite::Init()
{
	m_pNext= NULL;
	if (m_pstrName) delete m_pstrName;
	m_pstrName= NULL;
	m_siCelTop.Init(0);
	m_siCelLeft.Init(0);
	m_siPosTop.Init(0);
	m_siPosLeft.Init(0);;
	m_siRows.Init(GetHeight()); 
	m_siCols.Init(GetWidth()); 
	m_wKeycolor= *m_pPixels;
	m_iStartFrame= m_iEndFrame= 0;
}

// lies Windows Bitmapdatei (BMP/DIB) fr Sprite
// und intialisiere Vorgabewerte
BOOL CSprite::Load(LPSTR pstrFileName)
{
	if (!CImage::Load(pstrFileName))
		return FALSE;
	Init();
	m_pstrName= strdup(pstrFileName);
	return TRUE;
}

const WORD BUFFER_SIZE= 5000;

static char szBuffer[BUFFER_SIZE]; // Parser Zeilenpuffer

// lsche Liste
void CSpriteList::Clear()
{
	CSprite *pSprite, *q;
	for (pSprite=m_pFirst; pSprite; pSprite= q) {
		q= pSprite->GetNext();
		delete pSprite;
	}
	m_pFirst= m_pLast= NULL;
}

// liefert Zeiger auf neues Sprite an Listenende
CSprite* CSpriteList::Add()
{
	CSprite *pSprite= new CSprite;
	if (!pSprite) return NULL;
	if (!m_pFirst) m_pFirst= pSprite;
	if (m_pLast) m_pLast->SetNext(pSprite);
	m_pLast= pSprite;
	return pSprite;
}

// entfernt pSprite aus Liste
void CSpriteList::Remove(CSprite* pSprite)
{
	CSprite *p;
	if(!m_pFirst) return;
	for (p=m_pFirst; p && (p->GetNext()!=pSprite);p=p->GetNext());
	if (m_pFirst == pSprite) m_pFirst= pSprite->GetNext();
	if (m_pLast == pSprite) m_pLast= p;
	if (!p) return;
	p->SetNext(pSprite->GetNext());
}


// baue SpriteListe aus Informationen in INI-Datei pstrFileName auf
void CSpriteList::Parse(LPSTR pstrFileName)
{
	char *p,*pFiles;
	CSprite *pSprite;
	int n;
	// trage alle Sections in szBuffer ein
	n= GetPrivateProfileString(NULL,NULL,"",szBuffer,
                               BUFFER_SIZE,pstrFileName);
	if (n > BUFFER_SIZE-2) return;
	pFiles= new char[n];
	if (!pFiles) return;
	memcpy(pFiles, szBuffer, n);
	// iteriere alle Sections (Sprites)
	for (p=pFiles; strlen(p); p+= strlen(p)+1) {
		pSprite= Add(); // neues Sprite
		if (!pSprite) return;
		if (!pSprite->Load(p)) return; // lies Bitmap- Datei
		SetTimeInterval(pstrFileName, pSprite);
		SetKeycolor(pstrFileName, pSprite);
		SetSpriteAttribute(pstrFileName, pSprite, "posx",
                           pSprite->PosX());
		SetSpriteAttribute(pstrFileName, pSprite, "posy",
                           pSprite->PosY());
		SetSpriteAttribute(pstrFileName, pSprite, "top",
                           pSprite->TopRow());
		SetSpriteAttribute(pstrFileName, pSprite, "left",
                           pSprite->LeftCol());
		SetSpriteAttribute(pstrFileName, pSprite, "rows",
                           pSprite->Rows());
		SetSpriteAttribute(pstrFileName, pSprite, "cols",
                           pSprite->Cols());
	}
	delete pFiles;
}

// setze Sprite-Attribut si (key pKey) fr Sprite pSprite 
void CSpriteList::SetSpriteAttribute(LPSTR pstrFileName, 
                   CSprite *pSprite, LPSTR pKey, CSequencedInt& si)
{
	double pdArgs[5];
	DWORD i,n;
	char *q;
	// lies Werte (start, min, max, step, funktion)
	n=GetPrivateProfileString(pSprite->GetName(), pKey,"",szBuffer,
                              BUFFER_SIZE, pstrFileName);
	if (*szBuffer) { // Attribut definiert?
		memset(pdArgs, 0, sizeof(pdArgs));
		// parse Werte
		q= strtok(szBuffer,",");
		for (i=0; q && (i<5); q= strtok(NULL,",")) {
			pdArgs[i++]= atof(q);
		}
		// bernimm Werte in pSprite Instanz
		si.Init((int)pdArgs[0],(int)pdArgs[1],(int)pdArgs[2],
                pdArgs[3]);
		si.SetWaveform((CSequencedInt::Repeat)(int)pdArgs[4]);
	}
}

// setze Frame-Interval, fr das Sprite erscheinen soll, wenn "time" 
// in INI-Datei angegeben
void CSpriteList::SetTimeInterval(LPSTR pstrFileName,
                                  CSprite* pSprite)
{
	int start,end;
	DWORD n;
	char *q;
	// lies Werte (startframe, endframe)
	n=GetPrivateProfileString(pSprite->GetName(), "time", "",
                              szBuffer, BUFFER_SIZE, pstrFileName);
	if (*szBuffer) { // Attribut definiert?
		q= strtok(szBuffer,",");
		if (!q) return;
		start= atol(q);
		q= strtok(NULL, ",");
		if (!q) return;
		end= atol(q);
		pSprite->SetFrameInterval(start, end);
	}
}

// setze Keycolor, wenn "keycolor" in INI-Datei angegeben
void CSpriteList::SetKeycolor(LPSTR pstrFileName, CSprite* pSprite)
{
	DWORD n;
	WORD color;
	n=GetPrivateProfileString(pSprite->GetName(), "keycolor", "",
                              szBuffer, BUFFER_SIZE, pstrFileName);
	if (*szBuffer) { // Attribut definiert?
		color= atoi(szBuffer);
		pSprite->SetKeycolor(color);
	}
}

// beschaffe alle zum Abspielen notwendigen Interfaces
// und setze Framenummer als Zeitformat fr Seek
CMoviePlayer::CMoviePlayer()
{
	Check(CoInitialize(NULL),"COM Subsystem korrupt");
    Check(CoCreateInstance(CLSID_FilterGraph, NULL, 
          CLSCTX_INPROC_SERVER,IID_IGraphBuilder, (void **) &pGraph),
		"kann Filtergraph-Objekt nicht anlegen");
    Check(pGraph->QueryInterface(IID_IMediaEvent, (void **) &pEvent),
		"kein MediaEvent-Interface vorhanden");
    Check(pGraph->QueryInterface(IID_IMediaControl, (void **)
          &pControl),"kein MediaControl-Interface vorhanden");
    Check(pGraph->QueryInterface(IID_IMediaSeeking, (void **)
        &pSeeking),"kein MediaSeeking-Interface vorhanden");
	// Zeitformat Frame-Nummern setzen
	pSeeking->SetTimeFormat(&TIME_FORMAT_FRAME);
}

// gib Interfaces und COM frei
void CMoviePlayer::Release()
{
	if (pGraph) pGraph->Release();
	pGraph= NULL;
	if (pEvent) pEvent->Release();
	pEvent= NULL;
	if (pControl) pControl->Release();
	pControl= NULL;
	if (pSeeking) pSeeking->Release();
	pSeeking= NULL;
	CoUninitialize();
}

// baue generischen Filtergraph fr Video-Datei szFileName
void CMoviePlayer::RenderFile(LPSTR szFileName)
{
    WCHAR wPath[MAX_PATH];
    MultiByteToWideChar( CP_ACP, 0, szFileName, -1,wPath,MAX_PATH );
    Check(pGraph->RenderFile(wPath, NULL), "Datei nicht vorhanden \
                                            oder ungeeignet");
}

// spiele Videoclip ab
void CMoviePlayer::Play()
{
	HRESULT hr;
	long code,lp1,lp2;
    hr = pControl->Run();
	Check(hr,"kann Datei nicht abspielen");
	do {
		hr= pEvent->GetEvent(&code,&lp1,&lp2,INFINITE);
	} while (!(code==EC_COMPLETE || FAILED(hr)));
	hr= pControl->Stop();
	Check(hr,"keine Stop mglich");
}

// Filtername des Video-Renderers
const LPCWSTR pstrVideoRenderer= L"Video Renderer";


// Freigabe Ressourcen
void COverlaidPlayer::Release()
{
	CMoviePlayer::Release();
	if (m_pOvlFilter) delete m_pOvlFilter;
	m_pVideoRenderer= NULL;
	m_pOvlFilter= NULL;
	m_spriteList.Clear();
}

// Filtergraph mit Sprite-Stanze fr File szFileName aufbauen
void COverlaidPlayer::RenderFile(LPSTR szFileName)
{
	HRESULT hr;
	IFilterGraph *piFG;
	IEnumPins *piEnumPins;
	IPin *piPinIn, *piPinOut; 
	ULONG fetched;
	// generischen Filtergraph aufbauen
	CMoviePlayer::RenderFile(szFileName);
	// Sprite-Overlay Filter in Graphen einbauen
	Check(pGraph->QueryInterface(IID_IFilterGraph, (void**)&piFG),
		"Filtergraph Interface nich gefunden");
	m_pOvlFilter= new CVideoOverlayFilter(NAME("Video Overlay"), 
                                          this, NULL, &hr);
	Check(hr, "kann Overlayfilter nicht erzeugen");
	m_pOvlFilter->AddRef();
	Check(piFG->AddFilter(m_pOvlFilter, NULL),
		"kann Overlayfilter nicht einsetzen");
	// trenne Verbindung zu Video-Renderer
	Check(pGraph->FindFilterByName(pstrVideoRenderer, 
                                   &m_pVideoRenderer),
		"Datei enthlt kein Video");
	Check(m_pVideoRenderer->EnumPins(&piEnumPins),
		"kann Renderer-Pins nicht iterieren");
	Check(piEnumPins->Next(1,&piPinIn,&fetched),
		"kann Renderer-Input-Pin nicht finden");
	Check(piPinIn->ConnectedTo(&piPinOut),
		"kann mit Renderer-Input verbunden Pin nicht finden");
	Check(pGraph->Disconnect(piPinIn),
		"kann VideoRenderer nicht abhngen");
	Check(pGraph->Disconnect(piPinOut),
		"kann VideoRenderer nicht abhngen");
	// Sprite Overlay -Filter verbinden
	Check(pGraph->Render(piPinOut),
		"kann Filtergraph nicht neu rendern");
	piPinOut->Release();
	piPinIn->Release();
	piEnumPins->Release();
	piFG->Release();
}

// Diese GUID nicht einfach abtippen
// sondern fr neues Projekt mit GUIDGEN neu erzeugen

// Video Overlay Filter CLSID= {ABB06811-C603-11d0-A8C8-E1A1DEBE205F}
DEFINE_GUID(CLSID_VideoOverlayFilter, 0xabb06811, 0xc603, 0x11d0,
            0xa8, 0xc8, 0xe1, 0xa1, 0xde, 0xbe, 0x20, 0x5f);

// Konstruktor bernimmt Player-Referenz
// fr Zugriff auf Spriteliste
CVideoOverlayFilter::CVideoOverlayFilter(TCHAR *tszName, 
    COverlaidPlayer *pPlayer, LPUNKNOWN punk, HRESULT *phr) :
	CTransInPlaceFilter (tszName, punk,CLSID_VideoOverlayFilter,phr)
{ 
	m_pPlayer= pPlayer;
}

// zeichne Sprites in Videobild
HRESULT CVideoOverlayFilter::Transform(IMediaSample *pSample)
{ 
	HRESULT hr;
	REFERENCE_TIME timeStart, timeEnd;
	int iFrameNr;
	// bestimmt dargestellte Framenummer
	hr= pSample->GetTime(&timeStart, &timeEnd);
	if (FAILED(hr)) return hr;
	iFrameNr= int(timeStart / avgTimePerFrame);
	// holt Pointer auf Pixeldaten
	hr= pSample->GetPointer((PBYTE*)&pRenderSurface);
	if (FAILED(hr)) return hr;
	// holt Spriteliste
	CSpriteList *pSpriteList= m_pPlayer->GetSpriteList();
	CSprite* pSprite;
	// stellt alle Sprites dar
	for (pSprite= pSpriteList->GetFirst(); pSprite; pSprite= 
         pSprite->GetNext()) {
		int iStart, iEnd;
		// wenn Abspiel-Interval angegeben (time), nur darstellen,
        // wenn im Interval
		if ((iStart= pSprite->GetStartFrame()) &&
            (iEnd= pSprite->GetEndFrame()) && (iEnd > iStart)) {
			if (iFrameNr < iStart || iFrameNr > iEnd) continue;
		}
		// aktualisiert Werte fr Frame
		CSequencedInt::SetTime(iFrameNr);
		// holt alle Werte fr Sprite
		PWORD pBits= pSprite->GetImage();
		LONG bmpHeight= pSprite->GetHeight();
        LONG bmpWidth= pSprite->GetWidth();
		LONG srcTop= pSprite->GetTopRow();
		LONG srcLeft= pSprite->GetLeftCol();
		LONG y, rows= pSprite->GetRows();
		LONG x, cols= pSprite->GetCols();
		LONG destTop= pSprite->GetPosY();
		LONG destLeft= pSprite->GetPosX();
		WORD colorKey= pSprite->GetKeycolor();
		PWORD pSrcRow= pBits + (bmpHeight - srcTop -1) * bmpWidth;
		LONG destHeight= bmiHeader.biHeight;
        LONG destWidth= bmiHeader.biWidth;
		PWORD pDestRow= pRenderSurface + (destHeight - destTop -1)
                        * destWidth;
		PWORD pSrc, pDest;
		// Render- Loop
		for (y=0; y < rows; y++) {
			pDest= pDestRow + destLeft;
			pSrc= pSrcRow + srcLeft;
			for (x=0; x < cols; x++) {
				if (*pSrc == colorKey) { 
					pSrc++; pDest++;
				} else *pDest++ = *pSrc++;
			}
			pDestRow-= destWidth;
			pSrcRow-= bmpWidth;
        }
	}
	return NOERROR;  
}

// akzeptiert Format mtIn, wenn RGB555 Videobild
// sichert Bitmap-Info und gemittelte Framerate
HRESULT CVideoOverlayFilter::CheckInputType(const CMediaType* mtIn)
{ 
	static AM_MEDIA_TYPE amtMust;
	VIDEOINFO *vi;
	amtMust.majortype= MEDIATYPE_Video;
	amtMust.subtype= MEDIASUBTYPE_RGB555;
	amtMust.bFixedSizeSamples= TRUE;
	amtMust.bTemporalCompression= FALSE;
	CMediaType mtMust(amtMust);
	if (!mtIn->MatchesPartial(&mtMust)) return S_FALSE; 
	vi=(VIDEOINFO*)mtIn->pbFormat;
	bmiHeader= vi->bmiHeader;
	if (bmiHeader.biCompression==BI_RGB) {
		bmColorMasks[0]= 0x1fL; 
		bmColorMasks[1]= 0x3e0L;
		bmColorMasks[2]= 0x7c00L;
	} else { 
		for (int i=0; i<3; i++) bmColorMasks[i]= vi->dwBitMasks[i];
	}
	avgTimePerFrame= vi->AvgTimePerFrame;
	return S_OK; 
}

int _CRTAPI1 main(short argc, char *argv[])
{
	try {
		if (argc != 3) throw "Dateiname erwartet";
		COverlaidPlayer player;
		player.RenderFile(argv[1]);
		player.LoadSprites(argv[2]);
		player.Play();
	}
	catch (LPSTR pStr) {
		fprintf(stderr, pStr);
		return -1;
	}
	return 0;
}
