unit DBMem;  // 05-AUG-97 Frank Gamerdinger/as
{ Hngt sich in die Delphi-eigene Speicherverwaltung ein und
  zeichnet Belegungen/Freigaben auf. Ausgaben laufen ber das
  Fenster von DBWin (Unit DBW32 + 16-Bit-Programm DBWin32) }
interface
uses Windows, SysUtils, TypInfo, DBMemTypes, DBW32;

  procedure DBMemInit;  // Einhngen in die Speicherverwaltung
  // Weitere Belegungsaktionen in einem neuen Bereich
  function DBMemNewRange: TDBMemRange;
  procedure DBMemShowRange(Range: TDBMemRange);  // Anzeige
  procedure DBMemShowStats;  // Statistik
  // Heap-Prfung
  procedure DBMemCheckHeap(Callback: TDBCheckCallback);

implementation
  { Zusatzinformationen fr das Heap-Tracking, werden den
    Speicherbereichen vorangestellt (negative Offsets) und
    sind untereinander ber eine doppelt verkettete Liste
    verbunden }
type
  PMemBlockHdr = ^TMemBlockHdr;
  TMemBlockHdr = record
    mbhMagNumber: Longint; // fr Sanity checks
    mbhPrev,               // vorangehender, nachfolgender Block
    mbhNext: PMemBlockHdr;
    mbhBlockSize: Integer; // inklusive des Kopfs
    mbhClass: TClass;      // nil: kein Klassenname verfgbar
    mbhRange: TDBMemRange;
    mbhRealData: record  // der vom Programm angeforderte Platz
                 end;
  end;

const
  MemBlockHeaderMagic = $73616766;  // Eitelkeit mu sein
  MemBlockOverhead = Sizeof(TMemBlockHdr);

var
  MemStats:  record  // Speicherstatistik
     NumAllocs: Integer;    // Belegungen
     NumFrees: Integer;     // Freigaben
     NumReallocs: Integer;  // Umschaufeleien
     LargestAlloc: Integer; // grter Block
     CurrentAlloc: Integer;  // aktueller Platzbedarf
     MaxAlloc: Integer;  // maximaler Platzbedarf
  end;

  Initialized: Boolean;  // True: DBMemInit aufgerufen
  HeapLock: TRTLCriticalSection;  // fr Listenmanipulationen
  MemBlockHead: PMemBlockHdr;  // Anfang der Liste
  RangeCurr: TDBMemRange;  // aktueller Bereich fr Belegungen
  OrgMemManager: TMemoryManager;  // der Delphi-eigene Manager
  // Von DebugGetMem bei jedem Aufruf neu gesetzt
  GetMemCaller, CallingClass: Pointer;
  // Rcksprungadressen zu TObject.NewInstance und NewAnsiString
  NewInstanceCallingAddr, NewAnsiStringAddr: Pointer;

// ------- Bereichsfestlegung und Ausgaberoutinen -------------

{ Neuer Bereich: Weitere Belegungsaktionen im Heap finden
  mit der hier zurckgelieferten Kennziffer statt }
function DBMemNewRange: TDBMemRange;
begin
  Inc(RangeCurr);
  Result := RangeCurr;
end;

procedure DBMemShowStats; // Statistiken
var AvgBlockSize, Overhead: Integer;
begin
  with MemStats do
    if NumAllocs = NumFrees then Overhead := 0
    else
    begin  // Blockgre im Verhltnis zum Overhead
      AvgBlockSize := CurrentAlloc div (NumAllocs-NumFrees);
      Overhead := ((AvgBlockSize+MemBlockOverhead)*100)
          div AvgBlockSize-100;
    end;
  outputDebugString('<<<<Memory Statistics>>>>'#10);
  with MemStats do outputDebugFormat
  ('Num Allocations: %d'#10'Num Frees: %d'#10+
   'Num Reallocs: %d'#10'Largest Alloc: %d'#10+
   'Max Heap Usage: %d'#10'Current Heap Usage: %d'#10+
   'DBMem Overhead: %d percent'#10,
   [NumAllocs,NumFrees,NumReallocs,LargestAlloc,
    MaxAlloc,CurrentAlloc,Overhead]);
end;

// Ist das ein ordentlich belegter Speicherblock mit Info?
function IsDebugMemBlock(Block: PMemBlockHdr): Boolean;
begin
  Result := not IsBadWritePtr(Block, SizeOf(TMemBlockHdr))
    and (Block^.mbhMagNumber = MemBlockHeaderMagic)
    and not IsBadWritePtr(Block, Block^.mbhBlockSize);
end;

// Exception, wenn IsDebugMemBlock False liefert
function AssertDBMemBlock(Block: PMemBlockHdr): PMemBlockHdr;
begin
  if (Block = nil) or IsDebugMemBlock(Block)
   then Result := Block
   else raise Exception.Create(Format(
         '%p - Heap corrupted!',[Block]));
end;

{ Prfung der Zeigerliste. Bricht mit Exception ab, wenn die
  Liste durcheinandergekommen ist (d.h. Schreibaktionen "neben"
  ordnungsgem belegten Bereichen stattgefunden haben) }
procedure DBMemCheckHeap(Callback: TDBCheckCallback);
var P, LastP: PMemBlockHdr;
begin
  P := MemBlockHead; LastP := nil;
  while (P <> nil) do
    if not IsDebugMemBlock(P) or (P^.mbhPrev <> LastP)
      then raise Exception.Create(Format('%p - Heap corrupted!',[P]))
      else
      begin // Rckruf mit Abbruchmglichkeit
        if Assigned(Callback) then
          with P^ do if not Callback(@mbhRealData,
            mbhBlockSize-MemBlockOverhead,mbhClass) then Break;
        LastP := P; P := P^.mbhNext;
      end;
end;

// Informationen ber einen Speicherblock ausgeben
procedure ShowMemBlock(Header: PMemBlockHdr);
const HexBytes: Array[0..15] of Char = '0123456789ABCDEF';
type TStrRec = record RefCnt, Len: Integer; end; PStrRec = ^TStrRec;
var UnitName: String; PData: PByte; x, ByteCount: Integer;
begin
  with Header^ do
  begin
    outputDebugFormat('%5d @%p : (%7d Bytes) : ',
     [mbhRange, @mbhRealData, mbhBlockSize-MemBlockOverhead]);
    if mbhClass = DBMemStringClass then
    begin  // String
      outputDebugFormat('String (%d Chars): %32s'#10,
        [PStrRec(@mbhRealData)^.Len,
         PChar(LongInt(@mbhRealData)+SizeOf(TStrRec))]);
    end else if mbhClass <> nil then
    begin
      if mbhClass.ClassInfo = nil then UnitName := ''
       else UnitName :=
         GetTypeData(mbhClass.ClassInfo).UnitName+'.';
       outputDebugFormat('Class %s%s'#10,
         [UnitName, mbhClass.ClassName]);
    end else  // keine Klasseninformationen:
    begin     // Hexdump der ersten max. 32 Bytes
      ByteCount := mbhBlockSize-MemBlockOverhead;
      if ByteCount > 32 then ByteCount := 32;
      if IsBadReadPtr(@mbhRealData,ByteCount)
       then UnitName := '#INVALID#'
      else
      begin
        SetLength(UnitName,ByteCount*4+5);
        FillChar(UnitName[1],ByteCount*4+5,' ');
        PData := @mbhRealData;
        for x := 0 to ByteCount-1 do
        begin
          UnitName[x*3+1] := HexBytes[PData^ shr 4];
          UnitName[x*3+2] := HexBytes[PData^ and $0F];
          // ASCII-Reprsentation
          if PData^ > 32  // "druckbares" Zeichen
             then UnitName[ByteCount*3+5+x] := Chr(PData^);
          Inc(PData);
        end;
      end;
      UnitName := UnitName+#10; // Zeilenvorschub
      outputDebugString(PChar(UnitName));
    end;
  end;
end;

procedure DBMemShowRange(Range: TDBMemRange);  // Anzeige
var P: PMemBlockHdr; RMsg: ShortString;
begin
  // Aktuellen Stand vorher festhalten wg. Format
  P := MemBlockHead;
  if Range = DBMemAllRanges then RMsg := 'all Ranges'
    else RMsg := Format('Range %d',[Range]);
  outputDebugFormat('<<<<MemState for %s>>>>'#10, [RMsg]);

  while AssertDBMemBlock(P) <> nil do
  begin
    if (P^.mbhRange >= Range) then ShowMemBlock(P);
    P := P^.mbhNext;
  end;
  outputDebugString('<<<<---------------------->>>>'#10);
end;

// -------- Die Heap-Verwaltung -------------------------

function GetDebugHeader(P: Pointer): PMemBlockHdr;
begin  // prft, ob dem Block Debug-Informationen vorangehen
  Result := Pointer(Integer(P)-MemBlockOverhead);
  if not IsDebugMemBlock(Result) then
     Result := nil;
end;

function DebugGetMem(Size: Integer): Pointer;
begin
// Irgendwann wird Borland vielleicht mal AND und OR einfhren...
{$IFDEF VER90} {$DEFINE VERSIONOK} {$ENDIF}
{$IFDEF VER100} {$DEFINE VERSIONOK}{$ENDIF}
{$IFNDEF VERSIONOK}
{$ERROR Nur fr Delphi 2/3 getestet!}
{$ENDIF}
  asm // Obacht: Versionsspezifisch bis zum Abwinken
     // ebp+0*4 = esp; ebp+1*4: Rcksprung zu @GetMem
     // Strings: Adresse in @NewAnsiString,
     // Objekte: Adresse in TObject.NewInstance
     mov edx,[ebp+2*4]
     mov GetMemCaller, edx
     // Wenn GetMemCaller = TObject.NewInstance, dann ist
     // das hier der Zeiger auf die Klasse
     mov edx,[ebp+3*4]
     mov CallingClass,edx
  end;
  Result := OrgMemManager.GetMem(Size+MemBlockOverhead);
  if Result <> nil then
  begin // Verwaltungsdaten und Info eintragen
     if IsMultiThread then EnterCriticalSection(HeapLock);
     try
       with PMemBlockHdr(Result)^ do
       begin
         mbhMagNumber := MemBlockHeaderMagic;
         mbhRange := RangeCurr;
         mbhBlockSize := Size + MemBlockOverhead;
         if GetMemCaller = NewInstanceCallingAddr
             then mbhClass := CallingClass
          else if GetMemCaller = NewAnsiStringAddr
             then mbhClass := DBMemStringClass
          else mbhClass := nil;
         // Block in die Liste eintragen
         mbhNext := AssertDBMemBlock(MemBlockHead);
         if MemBlockHead <> nil then memBlockHead^.mbhPrev := Result;
         mbhPrev := nil;  // Block steht an 1. Position: kein Vorgnger
       end;
       MemBlockHead := Result;  // neue Blocks jeweils an 1. Position
       // Zeiger auf mbhRealData als Ergebnis zurckliefern
       Inc(LongInt(Result), MemBlockOverhead);
       // Statistik
       with MemStats do
       begin
         Inc(NumAllocs); Inc(CurrentAlloc,Size);
         if CurrentAlloc > MaxAlloc then MaxAlloc := CurrentAlloc;
         if Size > LargestAlloc then LargestAlloc := Size;
       end;
     finally
       if IsMultiThread then LeaveCriticalSection(HeapLock);
     end;
  end; // if Result <> nil
end;

function DebugFreeMem(P: Pointer): Integer;
var Header, NextBlock: PMemBlockHdr;
begin
  Header := GetDebugHeader(P);  // Debug-Vorspann
  if Header <> nil then
  begin
    if IsMultiThread then EnterCriticalSection(HeapLock);
    try
      with Header^ do
      begin  // raus aus der Liste
         NextBlock := AssertDBMemBlock(mbhNext);
         if AssertDBMemBlock(mbhPrev) <> nil
           then mbhPrev^.mbhNext := NextBlock
           else MemBlockHead := NextBlock;  // war 1. Block
        if NextBlock <> nil then NextBlock^.mbhPrev := mbhPrev;
        Dec(MemStats.CurrentAlloc,mbhBlockSize-MemBlockOverhead);
      end;
      Inc(MemStats.NumFrees);
      P := Header;  // Debug-Header wird mit freigegeben
    finally
      if IsMultiThread then LeaveCriticalSection(HeapLock);
    end;
  end;
  // Freigabe des Bereichs (+Debug-Header) ber Delphis Verwaltung
  Result := OrgMemManager.FreeMem(P);
end;

function DebugReallocMem(P: Pointer; Size: Integer): Pointer;
var Header: PMemBlockHdr;
begin
  if IsMultiThread then EnterCriticalSection(HeapLock);
  try
    Header := GetDebugHeader(P);  // Debug-Vorspann
    if Header <> nil then
    begin
      Result := OrgMemManager.ReallocMem(Header, Size + MemBlockOverHead);
      if Result <> nil then
      begin
        with PMemBlockHdr(Result)^ do
        begin
          // Next-Zeiger des Vorgngers auf den umgesetzten Bereich
          if AssertDBMemBlock(mbhPrev) <> nil
            then mbhPrev^.mbhNext := Result
            else MemBlockHead := Result;  // war 1. Block
          // Prev-Zeiger des Nachfolgers auf den umgesetzten Bereich
          if AssertDBMemBlock(mbhNext) <> nil
            then mbhNext^.mbhPrev := Result;
          // Neue Gre eintragen
          Dec(MemStats.CurrentAlloc,mbhBlockSize);  // alte Gre
          mbhBlockSize := Size+MemBlockOverhead;
          Inc(MemStats.CurrentAlloc,mbhBlockSize);  // neue Gre
          with MemStats do if CurrentAlloc > MaxAlloc
             then MaxAlloc := CurrentAlloc;
        end;
        // zurckgeliefert wird auch hier der Zeiger auf mbhRealData
        Inc(Integer(Result), MemBlockOverhead);
      end;
    end
      else Result := OrgMemManager.ReallocMem(P, Size);

    if Result <> nil then
    with MemStats do
    begin  // Statistik auch fr "Debug-freie" Blocks
      Inc(NumReallocs);
      if Size > LargestAlloc then LargestAlloc := Size;
    end;
  finally
    if IsMultiThread then LeaveCriticalSection(HeapLock);
  end;
end;

procedure DBMemInit;
const  // So will SetMemoryManager die Einsprungpunkte
  DebugMemManager: TMemoryManager =
  ( GetMem: DebugGetMem; FreeMem: DebugFreeMem;
    ReallocMem: DebugReallocMem;
   );
{$O-}  // Optimierung aus wg. der beiden Dummy-Variablen
var S: String; O: TObject;
begin
  if not Initialized then
  begin
    outputDebugString('<<------- DBMem-Init ------->>'#10);
    // Kritischer Abschnitt fr Listenmanipulationen
    InitializeCriticalSection(HeapLock);
    GetMemoryManager(OrgMemManager);  // Delphi-Speicherverwaltung
    SetMemoryManager(DebugMemManager);  // DBMem-Routinen einsetzen
    // String zur Ermittlung der Belegungsadresse anlegen
    S := 'DBMem'; S := S + '-Test';
    NewAnsiStringAddr := GetMemCaller;
    // Objekt anlegen -> NewInstanceCallingAddr ermitteln
    GetMemCaller := nil;
    O := TObject.Create;
    if CallingClass = TObject
       then NewInstanceCallingAddr := GetMemCaller;
    O.Free;
    Initialized := True;
  end;
end;

initialization
  // falls nicht bereits ber das Projekt geschehen
  DBMemInit;

finalization
 if Initialized then
 begin
   // Sonst geht TDW32 baden...
   SetMemoryManager(OrgMemManager);
   DBMemShowStats;
   if MemBlockHead <> nil then DBMemShowRange(DBMemAllRanges)
     else outputDebugString('Heap is empty'#10);
   DeleteCriticalSection(HeapLock);
 end;
end.
