unit DXControls;

{$INCLUDE COMSWITCH.INC}

interface

uses
  Windows, Messages, SysUtils, Classes,
  Graphics, Controls, Forms, Dialogs,
  OLE2,
  DXTools,
  ddraw,
  d3dtypes,
  d3d,
  d3dcaps,
  d3drm,
  d3drmdef,
  d3drmobj,
  d3drmwin;

type
  TFrontlightType = (None,Pointlight,Spotlight,Directional);
  TShadeMode = (Wireframe,UnlitFlat,Flat,Gouraud,Phong);
  TColormodel = (Ramp,RGB);

  TPixelFormat = (
    FormatUnknown,
    Format1Bit,
    Format2Bit,
    Format4Bit,
    Format8Bit,
    Format15Bit,
    Format16Bit,
    Format24Bit,
    Format32Bit
  );

const
  DefaultColormodel = TColormodel(Ramp);
  DefaultShadeMode = TShadeMode(Gouraud);
  DefaultFOV = 0.5;
  DefaultFrontClipping = 1.0;
  DefaultBackClipping = 10000.0;
  DefaultFrontlightUmbra = 0.4;
  DefaultFrontlightPenumbra = 0.4;
  DefaultHeight = 64;
  DefaultWidth  = 64;
  DefaultAmbientBrightness = 0.1;
  DefaultFrontlightBrightness = 0.8;
  DefaultFrontlightType = TFrontlightType(Pointlight);
  DefaultSunlightBrightness = 0.5;
  DefaultZBufferBits = 16;

type
  TSetColorRGBCallback = procedure(r,g,b: TD3DValue) of object;
  TColorRGB = class(TPersistent)
  private
    SetColorRGB : TSetColorRGBCallback;
    FColor : TD3DColorValue;
    procedure SetColorValue(Index: integer; Value: TD3DValue); virtual;
    procedure SetColor(Value: TD3DColorValue); virtual;
  published
    Property Red : TD3DValue index 0 read FColor.R write SetColorValue;
    Property Green : TD3DValue index 1 read FColor.G write SetColorValue;
    Property Blue : TD3DValue index 2 read FColor.B write SetColorValue;
  public
    constructor Create(Callback: TSetColorRGBCallback);
    Property Color : TD3DColorValue read FColor write SetColor;
  end;

  TRenderValues = class(TPersistent)
  private
    FShades : integer;
    FTextureShades : integer;
    FTextureColors : integer;
    FDither : boolean;
    Bits : integer;
    D3DBrowser : TObject;
    procedure SetColorShades(index: integer; Value: integer); virtual;
    procedure SetDither(Value: boolean); virtual;
  published
    property Shades : integer index 0 read FShades write SetColorShades;
    property TextureShades : integer index 1 read FTextureShades write SetColorShades;
    property TextureColors : integer index 2 read FTextureColors write SetColorShades;
    property Dither : boolean read FDither write SetDither;
  public
    constructor Create(Browser: TObject; ColorBits: integer);
  end;

  TRenderQuality = class(TPersistent)
  private
    FPalette8Bit : TRenderValues;
    FHighColor16Bit : TRenderValues;
    FTrueColor24Bit : TRenderValues;
  published
    property Palette8Bit : TRenderValues read FPalette8Bit write FPalette8Bit;
    property HighColor16Bit : TRenderValues read FHighColor16Bit write FHighColor16Bit;
    property TrueColor24Bit : TRenderValues read FTrueColor24Bit write FTrueColor24Bit;
  public
    constructor Create(Browser: TObject);
    destructor Destroy; override;
  end;

  TDisplaymode = class (TObject)
  private
    FRefreshRate : integer;
    FName : string;
    FDirectDraw2 : IDirectDraw2;
    FEmpty : boolean;
    FBitsPerPixel : integer;
    FHeight, FWidth : integer;
    procedure SetWidth(Value: integer); virtual;
    procedure SetHeight(Value: integer); virtual;
    procedure SetPixelFormat(PixelFormat: TPixelformat); virtual;
    function GetPixelFormat : TPixelFormat; virtual;
    procedure SetBitsPerPixel(BitsPerPixel: integer); virtual;
  published
    property PixelFormat : TPixelFormat read GetPixelFormat write SetPixelFormat;
    property Width : integer read FWidth write SetWidth;
    property Height : integer read FHeight write SetHeight;
  public
    property RefreshRate : integer read FRefreshrate;
    property Name : string read FName;
    property Empty : boolean read FEmpty;
    property BitsPerpixel : integer read FBitsPerPixel write SetBitsPerPixel;
    constructor Create(CWidth, CHeight, CBitsPerPixel, CRefreshrate: integer; DD: IDirectDraw2);
    destructor Destroy; override;
    function SetMode : boolean; virtual;
    function CreateOffscreenSurface(var Surf: IDirectDrawSurface; Caps: DWord) : boolean; virtual;
    function SizeOfPixel : integer; virtual;
  end;

  TDisplaymodeList = class(TList)
  private
    FDirectDraw2 : IDirectDraw2;
    function GetItems(index: integer) : TDisplaymode;
  public
    property DirectDraw2 : IDirectDraw2 read FDirectDraw2;
    property Items[index: integer] : TDisplaymode read GetItems;
    constructor CreateFromDirectDraw2(DD: IDirectDraw2);
    function Fill : boolean;
    destructor Destroy; override;
  end;

  TD3DDevice = class(TObject)
  public
    GUID : TGUID;
    Direct3D : IDirect3D;
    Name,Description : string;
    Colormodel : TColormodel;
    SoftwareDevice : boolean;
    PixelFormats : set of TPixelFormat;
    ZBufferFormats : set of TPixelFormat;
 end;

  TD3DDeviceList = class(TList)
  private
    FDirect3D : IDirect3D;
    function GetItems(index: integer) : TD3DDevice;
  public
    property Items[index: integer] : TD3DDevice read GetItems;
    constructor Create;
    constructor CreateFromDirect3D(D3D: IDirect3D);
    function Fill : boolean;
    destructor Destroy; override;
  end;

type
  TD3DBrowser = class(TWinControl)
  private
    // Basisschnitstellen
    FDirectDraw    : IDirectDraw;
    FDirectDraw2   : IDirectDraw2;
    FDirect3D      : IDirect3D;
    FDirect3DRM    : IDirect3DRM;
    FRootframe     : IDirect3DRMFrame;
    FCameraframe   : IDirect3DRMFrame;
    FMeshframe     : IDirect3DRMFrame;
    FSunlightframe : IDirect3DRMFrame;
    FAmbientLight  : IDirect3DRMLight;
    FSunlight      : IDirect3DRMLight;
    FFrontlight    : IDirect3DRMLight;
    // Abhngige Schnittstellen
    FDevice      : IDirect3DRMDevice;
    FWinDevice   : IDirect3DRMWinDevice;
    FViewport    : IDirect3DRMViewport;
    FPrimeSurface  : IDirectDrawSurface;
    FBackSurface   : IDirectDrawSurface;
    FZBufferSurface: IDirectDrawSurface;
    FPalette     : IDirectDrawPalette;
    // Felder der Eigenschaften welche initialisiert werden mssen
    FDisplaymodeList : TDisplaymodeList;
    FDisplaymodeIndex : integer;
    FD3DDeviceList : TD3DDeviceList;
    FD3DDeviceIndex : integer;
    FColormodel    : TColormodel;
    FFullscreen    : boolean;
    FFOV           : TD3DValue;
    FShadeMode : TShadeMode;
    FSunlightColor : TColorRGB;
    FAmbientColor : TColorRGB;
    FFrontlightColor : TColorRGB;
    FFrontlightUmbra : TD3DValue;
    FFrontlightPenumbra : TD3DValue;
    FFrontClipping : TD3DValue;
    FBackClipping  : TD3DValue;
    FRenderQuality : TRenderQuality;
    // Felder der Eigenschaften die optionell gesetzt werden knnen
    FXFile         : string;
    // Fr die Verwaltung
    FOnLoaded : TNotifyEvent;
    FDeviceInitialized : boolean;
    FDisplaying : boolean; // nach Initialisierung auf true setzen
    FShades         : integer;
    FDither         : boolean;
    FTextureColors  : integer;
    FTextureShades  : integer;
    FMeshsize       : TD3DValue;
    // Die Direct3D Interface-Ohjektfelder
    FFrontlightType : TFrontlightType;
    FResizeToParentWindow : boolean;
  protected
    TD3DColorModel    : TD3DColorModel;
    D3DRenderquality : TD3DRMRenderQuality;
    GUID                  : TGUID;
    NotLoaded : boolean;
    // Methoden von vorzuinitialisierenden Eigenschaften
    procedure SetHeight(Value: integer); virtual;
    procedure SetWidth(Value: integer); virtual;
    function GetHeight : integer; virtual;
    function GetWidth : integer; virtual;
    procedure SetFullscreen(Fullscreen: boolean); virtual;
    function GetDisplaymode : TDisplaymode; virtual;
    procedure SetDisplaymode(Mode: TDisplaymode); virtual;
    procedure SetDisplaymodeIndex(index : integer); virtual;
    function GetD3DDevice : TD3DDevice; virtual;
    procedure SetD3DDevice(Device: TD3DDevice); virtual;
    procedure SetD3DDeviceIndex(index : integer); virtual;
    procedure SetShadeMode(ShadeMode: TShadeMode); virtual;
    procedure SetFov(FOV: TD3DValue); virtual;
    procedure SetFrontClipping(Front: TD3DValue); virtual;
    procedure SetBackClipping(Back: TD3DValue); virtual;
    procedure SetAmbientBrightness(Brightness: TD3DValue); virtual;
    function GetAmbientBrightness : TD3DValue; virtual;
    procedure SetFrontlightType(FrontlightType: TFrontlightType); virtual;
    procedure SetFrontlightUmbra(Umbra: TD3DValue); virtual;
    procedure SetFrontlightPenumbra(Penumbra: TD3DValue); virtual;
    procedure SetFrontlightBrightness(Brightness: TD3DValue); virtual;
    function GetFrontlightBrightness : TD3DValue; virtual;
    procedure SetSunlightBrightness(Brightness: TD3DValue); virtual;
    function GetSunlightBrightness : TD3DValue; virtual;
    procedure SetShades(Value: integer);
    procedure SetTextureShades(Value: integer);
    procedure SetTextureColors(Value: integer);
    procedure SetDither(Value: boolean);
    // Methoden von optionellen Eigenschaften
    procedure SetXFile(XFile: string); virtual;
    function GetCurrentBitsPerPixel : integer; virtual;
    procedure SetResizeToParentWindow(Resize: boolean); virtual;
    procedure SetDisplaying(Displaying: boolean); virtual;
    function DestroyDevice : boolean; virtual;
    procedure SetMeshsize(Meshsize: TD3DValue); virtual;
    procedure AppMessage(var Msg: TMsg; var Handled: Boolean);
  published
    // Ereignisse
    property OnLoaded : TNotifyEvent read FOnLoaded write FOnLoaded;
    // Eigenschaften welche initialisiert werden mssen
    property Height : integer read GetHeight write SetHeight default DefaultHeight;
    property Width : integer read GetWidth write SetWidth default DefaultWidth;
    property Fullscreen : boolean read FFullscreen write SetFullscreen;
//    property Colormodel : TColormodel read FColormodel write SetColormodel default DefaultColormodel;
    property ShadeMode : TShadeMode read FShadeMode  write SetShadeMode default DefaultShadeMode;
    property FOV : TD3DValue read FFOV write SetFOV;
    property FrontClipping : TD3DValue read FFrontClipping write SetFrontClipping;
    property BackClipping : TD3DValue read FBackClipping write SetBackClipping;
    property AmbientBrightness : TD3DValue read GetAmbientBrightness write SetAmbientBrightness;
    property AmbientColor : TColorRGB read FAmbientColor write FAmbientColor;
    property FrontlightType : TFrontlightType read FFrontlightType write SetFrontlightType default DefaultFrontlightType;
    property FrontlightUmbra : TD3DValue read FFrontlightUmbra write SetFrontlightUmbra;
    property FrontlightPenumbra : TD3DValue read FFrontlightPenumbra write SetFrontlightPenumbra;
    property FrontlightBrightness : TD3DValue read GetFrontlightBrightness write SetFrontlightBrightness;
    property FrontlightColor : TColorRGB read FFrontlightColor write FFrontlightColor;
    property SunlightBrightness : TD3DValue read GetSunlightBrightness write SetSunlightBrightness;
    property SunlightColor : TColorRGB read FSunlightColor write FSunlightColor;
    property RenderQuality : TRenderQuality read FRenderQuality write FRenderQuality;
    // Eigenschaften die optionell gesetzt werden knnen
    property XFile : string read FXFile write SetXFile;
    property ResizeToParentWindow : boolean read FResizeToParentWindow write SetResizeToParentWindow;
  public
    FrameConstraint : TD3DRMFrameConstraint;
    PaletteFilename : string;
    property DisplaymodeList : TDisplaymodeList read FDisplaymodeList;
    property Displaymode : TDisplaymode read GetDisplaymode write SetDisplaymode;
    property DisplaymodeIndex : integer read FDisplaymodeIndex write SetDisplaymodeIndex;
    property D3DDeviceList : TD3DDeviceList read FD3DDeviceList;
    property D3DDevice : TD3DDevice read GetD3DDevice write SetD3DDevice;
    property D3DDeviceIndex : integer read FD3DDeviceIndex write SetD3DDeviceIndex;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Loaded; override;
    procedure IdleProc(Sender: TObject; var Done: Boolean);
    procedure WMActivate(var Msg: TMessage); message WM_ACTIVATE;
    procedure WMPaint(var Msg: TWMPaint); message WM_PAINT;
    function SetWindowSize(x,y: integer) : boolean; virtual;
    function UpdateMeshsize : TD3DValue;
    property Displaying : boolean read FDisplaying write SetDisplaying;
    property Meshsize : TD3DValue read FMeshsize write SetMeshsize;
    property DeviceInitialized : boolean read FDeviceInitialized;
    property DirectDraw  : IDirectDraw read FDirectDraw;
    property DirectDraw2 : IDirectDraw2 read FDirectDraw2;
    property Direct3D  : IDirect3D read FDirect3D;
    property Direct3DRM  : IDirect3DRM read FDirect3DRM;
    property Palette     : IDirectDrawPalette read FPalette;
    property Device      : IDirect3DRMDevice read FDevice;
    property WinDevice   : IDirect3DRMWinDevice read FWinDevice;
    property Viewport    : IDirect3DRMViewport read FViewport;
    property Rootframe   : IDirect3DRMFrame read FRootframe;
    property Meshframe   : IDirect3DRMFrame read FMeshframe;
    property Cameraframe : IDirect3DRMFrame read FCameraframe;
    property Sunlightframe  : IDirect3DRMFrame read FSunlightframe;
    property AmbientLight : IDirect3DRMLight read FAmbientLight;
    property FrontLight   : IDirect3DRMLight read FFrontLight;
    property Sunlight : IDirect3DRMLight read FSunlight;
    // Informationen
    property Shades : integer read FShades write SetShades;
    property TextureShades : integer read FTextureShades write SetTextureShades;
    property TextureColors : integer read FTextureColors write SetTextureColors;
    property Dither : boolean read FDither write SetDither;
    procedure UpdateRenderQuality; virtual;
    function UpdateDevice : boolean; virtual;
    procedure SetRenderQuality(Bits: integer); virtual;
    function ChooseD3DDevice : integer; virtual;
    function ChooseDisplaymode : integer; virtual;
    // Methoden fr die Bewegung im Raum
    procedure LookAtCenter; virtual;
    procedure AdjustDistance(Factor: TD3DValue); virtual;
    procedure AdjustMeshsize(Factor: TD3DValue); virtual;
    procedure Tick(Move: TD3DValue); virtual;
    procedure CenterMesh; virtual;
    procedure RotateLeft(Angle: TD3DValue); virtual;
    procedure RotateRight(Angle: TD3DValue); virtual;
    procedure RotateUp(Angle: TD3DValue); virtual;
    procedure RotateDown(Angle: TD3DValue); virtual;
    procedure TurnLeft(Angle: TD3DValue); virtual;
    procedure TurnRight(Angle: TD3DValue); virtual;
    procedure TurnUp(Angle: TD3DValue); virtual;
    procedure TurnDown(Angle: TD3DValue); virtual;
    procedure RollLeft(Angle: TD3DValue); virtual;
    procedure RollRight(Angle: TD3DValue); virtual;
    procedure MoveLeft(Space: TD3DValue); virtual;
    procedure MoveRight(Space: TD3DValue); virtual;
    procedure MoveUp(Space: TD3DValue); virtual;
    procedure MoveDown(Space: TD3DValue); virtual;
    procedure MoveForward(Space: TD3DValue); virtual;
    procedure MoveBackward(Space: TD3DValue); virtual;
    procedure SetAmbientColor(r,g,b: TD3DValue); virtual;
    procedure SetFrontlightColor(r,g,b: TD3DValue); virtual;
    procedure SetSunlightColor(r,g,b: TD3DValue); virtual;
    // Eigenschaften von TWinControl die nicht im Objektinspektor erscheinen sollen
    property HelpContext;
    property Hint;
    property Tag;
  end;

const
  Debugg : boolean = true;

procedure Register;

implementation

constructor TD3DBrowser.Create(AOwner: TComponent);
begin
  NotLoaded := true;
  FDeviceInitialized := false;
  FDisplaying := false;
  FrameConstraint := D3DRMCONSTRAIN_Z;
  inherited Create(AOwner);
  // Die Unterobjekte erzeugen
  FAmbientColor := TColorRGB.Create(SetAmbientColor);
  FFrontlightColor := TColorRGB.Create(SetFrontlightColor);
  FSunlightColor := TColorRGB.Create(SetSunlightColor);
  FRenderQuality := TRenderQuality.Create(self);
  // Startwerte der Eigenschaften festlegen
  // Sind nur wirksam wenn die Komponente zum erten mal dem Formular
  // hinzugefgt wird; Spter werden sie durch die Objektinspectorwerte berschrieben.
  Height := DefaultHeight;
  Width := DefaultWidth;
  AmbientBrightness := DefaultAmbientBrightness;
  FrontlightBrightness := DefaultFrontlightBrightness;
  SunlightBrightness := DefaultSunlightBrightness;

  FColorModel := DefaultColormodel;
  FShadeMode := DefaultShadeMode;
  FFOV := DefaultFOV;
  FFrontClipping := DefaultFrontClipping;
  FBackClipping := DefaultBackClipping;
  FFrontlightUmbra := DefaultFrontlightUmbra;
  FFrontlightPenumbra := DefaultFrontlightPenumbra;
  FFrontlightType := DefaultFrontlightType;
end;

destructor TD3DBrowser.Destroy;
begin
  FDisplaymodeList.Free;
  FD3DDeviceList.Free;
  FRenderQuality.Free;
  if not NotLoaded then begin
    if DeviceInitialized then DestroyDevice;
    ReleaseCOM ( FPalette );
    ReleaseCOMe( FFrontlight );
    ReleaseCOMe( FSunlight );
    ReleaseCOMe( FAmbientlight );
    ReleaseCOMe( FSunlightframe );
    ReleaseCOMe( FMeshframe );
    ReleaseCOMe( FCameraframe );
    ReleaseCOMe( FRootframe );
    ReleaseCOMe( FDirect3DRM );
    ReleaseCOMe( FDirect3D );
    ReleaseCOMe( FDirectDraw );
    NotLoaded := true;
  end;
  inherited Destroy;
end;

procedure TD3DBrowser.AppMessage(var Msg: TMsg; var Handled: Boolean);
begin
  Handled := false;
  with Msg do // SC_KEYMENU SC_DEFAULT SC_MOUSEMENU
    if (Message = WM_SYSCOMMAND)// and (wParam = SC_DEFAULT) //and Fullscreen
      then asm nop end; //Handled := true;
end;

procedure TD3DBrowser.Loaded;
// Hier wird die eigendliche Initialsierung vorgenommen, da jetzt alle
// Eigenschaftswerte aus der Formular/Exe-Datei geladen sind
begin
  inherited Loaded;
  if csDesigning in ComponentState then exit;

  case FColormodel of
     Ramp : TD3DColorModel := D3DCOLOR_MONO;
     RGB  : TD3DColorModel := D3DCOLOR_RGB;
  end;
  case FShadeMode of
    Wireframe : D3DRenderquality := D3DRMRENDER_WIREFRAME;
    Flat      : D3DRenderquality := D3DRMRENDER_FLAT;
    Gouraud   : D3DRenderquality := D3DRMRENDER_GOURAUD;
    Phong     : D3DRenderquality := D3DRMRENDER_PHONG;
  end;

  try
    DXStat := 'Initialisieren der DX-Objekte'+#13+'Methode Loaded';
    DXCheck( DirectDrawCreate(nil,FDirectDraw,nil) );
    DXCheck( FDirectDraw.QueryInterface(IID_IDirectDraw2,FDirectDraw2) );

    FDisplaymodeList := TDisplaymodeList.CreateFromDirectDraw2(FDirectDraw2);
    FDisplaymodeList.Fill;

    DXCheck( FDirectDraw2.QueryInterface(IID_IDirect3D,FDirect3D) );

    FD3DDeviceList := TD3DDeviceList.CreateFromDirect3D(FDirect3D);
    FD3DDeviceList.Fill;

    FD3DDeviceIndex := ChooseD3DDevice;
    FDisplaymodeIndex := ChooseDisplaymode;

    DXCheck( Direct3DRMCreate(FDirect3DRM) );

    DXCheck( FDirect3DRM.CreateFrame(nil,FRootframe) );
    DXCheck( FDirect3DRM.CreateFrame(FRootframe,FCameraframe) );
    DXCheck( FDirect3DRM.CreateFrame(FRootframe,FMeshframe) );
    DXCheck( FDirect3DRM.CreateFrame(FRootframe,FSunlightframe) );
  {  DXCheck( FCameraframe.AddMoveCallback(CameraMoveCallback,nil) );}
    DXCheck( FCameraframe.SetOrientation(FRootframe,0,0,1,0,1,0) );
    DXCheck( FCameraframe.SetPosition(FRootframe,0,0,-50) );
    DXCheck( FSunlightframe.SetOrientation(FRootframe,0,-1,0,0,1,0) );
    DXCheck( FSunlightframe.SetPosition(FRootFrame,0,50,0) );

    DXCheck( FDirect3DRM.CreateLightRGB(D3DRMLIGHT_AMBIENT,FAmbientColor.Red,
             FAmbientColor.Green,FAmbientColor.Blue,FAmbientlight) );
    DXCheck( FRootFrame.AddLight(FAmbientlight) );

    DXCheck( FDirect3DRM.CreateLightRGB(D3DRMLIGHT_DIRECTIONAL,FSunlightColor.Red,
             FSunlightColor.Green,FSunlightColor.Blue,FSunlight) );
    DXCheck( FSunlightFrame.AddLight(FSunlight) );

    case FFrontlightType of
        Pointlight : DXCheck( FDirect3DRM.CreateLightRGB(D3DRMLIGHT_POINT,
                              FFrontlightColor.Red,FFrontlightColor.Green,
                              FFrontlightColor.Blue,FFrontlight) );
        Spotlight  : begin
                       DXCheck( FDirect3DRM.CreateLightRGB(D3DRMLIGHT_SPOT,
                              FFrontlightColor.Red,FFrontlightColor.Green,
                              FFrontlightColor.Blue,FFrontlight) );
                     end;
        Directional: DXCheck( FDirect3DRM.CreateLightRGB(D3DRMLIGHT_DIRECTIONAL,
                              FFrontlightColor.Red,FFrontlightColor.Green,
                              FFrontlightColor.Blue,FFrontlight) );
        None       : DXCheck( FDirect3DRM.CreateLightRGB(D3DRMLIGHT_AMBIENT,0,0,0,FFrontlight) );
    end;
    if FrontlightType <> None then DXCheck( FCameraFrame.AddLight(FFrontlight) );
  except
    on Exception do begin
    ReleaseCOM( FFrontlight );
    ReleaseCOM( FSunlight );
    ReleaseCOM( FAmbientlight );
    ReleaseCOM( FSunlightframe );
    ReleaseCOM( FMeshframe );
    ReleaseCOM( FCameraframe );
    ReleaseCOM( FRootframe );
    ReleaseCOM( FDirect3DRM );
    ReleaseCOM( FDirect3D );
    ReleaseCOM( FDirectDraw2 );
    ReleaseCOM( FDirectDraw );
    raise;
    end;
  end;
  NotLoaded := false;
  SetFrontlightUmbra(FFrontlightUmbra);
  SetFrontlightPenumbra(FFrontlightPenumbra);
  SetResizeToParentWindow(FResizeToParentWindow);
  SetXfile(FXFile);
//Application.OnMessage := AppMessage;
  if assigned(FOnLoaded) then FOnLoaded(self);
  Application.OnIdle := IdleProc;
end;

procedure TD3DBrowser.SetDisplaying(Displaying: boolean);
begin
  if Displaying <> FDisplaying then begin
    FDisplaying := Displaying;
    if not FDeviceInitialized then UpdateDevice;
  end;
end;

procedure TD3DBrowser.UpdateRenderQuality;
begin
  DXCheck( FDevice.SetQuality(D3DRenderquality) );
  DXCheck( FDevice.SetShades(Shades) );
  DXCheck( FDirect3DRM.SetDefaultTextureShades(TextureShades) );
  DXCheck( FDirect3DRM.SetDefaultTextureColors(TextureColors) );
  DXCheck( FDevice.SetDither(Dither) );
end;

function TD3DBrowser.UpdateDevice : boolean;

{function FindGUID : PGUID;  // Shows how to use FindDevice
var
  searchdata : D3DFINDDEVICESEARCH;
  Resultdata : D3DFINDDEVICEResult;
begin
  fillchar(searchdata,sizeof(searchdata),0);
  with searchdata do begin
    dwSize := sizeof(searchdata);
    dwFlags := D3DFDS_COLORMODEL;
    dcmColorModel := TD3DColorModel;
  end;
  fillchar(Resultdata,sizeof(Resultdata),0);
  Resultdata.dwSize := sizeof(Resultdata);
  DXCheck( FDirect3D.FindDevice(searchdata,Resultdata) );
  GUID := Resultdata.guid;
  Result := @GUID;
end;}

var
  faults : integer;

function CreateFullscreenDevice : boolean;
var
  Desc : TDDSurfaceDesc;
  Caps : TDDSCaps;
  pal : TColorTable;
begin
  Result := false;
  DXStat := 'CreateFullscreenDevice';
  if (not assigned(FPalette)) and (Displaymode.BitsPerPixel = 8) then
    if not LoadPaletteFromJASCFile(PaletteFilename,pal) then exit;

  DXCheck( FDirectDraw2.SetCooperativelevel(Application.Handle,
      DDSCL_FULLSCREEN + DDSCL_ALLOWREBOOT + DDSCL_NOWINDOWCHANGES
      + DDSCL_EXCLUSIVE) );
                                      
  if not Displaymode.SetMode then exit;

  InitRecord(Desc,sizeof(Desc));
  with Desc do begin
    dwFlags := DDSD_CAPS + DDSD_BACKBUFFERCOUNT;
    dwBackBufferCount := 1;
    ddsCaps.dwCaps := DDSCAPS_PRIMARYSURFACE + DDSCAPS_3DDEVICE
                       + DDSCAPS_FLIP + DDSCAPS_COMPLEX;
  end;
  try
    DXCheck( FDirectDraw2.CreateSurface(Desc,FPrimeSurface,nil) );

    Caps.dwCaps := DDSCAPS_BACKBUFFER;
    DXCheck( FPrimeSurface.GetAttachedSurface(Caps,FBackSurface) );

    InitRecord(Desc,sizeof(Desc));
    with Desc do begin
      dwFlags := DDSD_WIDTH + DDSD_HEIGHT + DDSD_CAPS + DDSD_ZBUFFERBITDEPTH;
      dwWidth := Displaymode.Width;
      dwHeight := Displaymode.Height;
      dwZBufferBitDepth := DefaultZBufferBits;
      ddsCaps.dwCaps := DDSCAPS_ZBUFFER;
      if D3DDevice.SoftwareDevice then inc(ddsCaps.dwCaps,DDSCAPS_SYSTEMMEMORY)
        else inc(ddsCaps.dwCaps,DDSCAPS_VIDEOMEMORY)
    end;
    DXCheck( FDirectDraw2.CreateSurface(Desc,FZBufferSurface,nil) );
    DXCheck( FBackSurface.AddAttachedSurface(FZBufferSurface) );

    if Displaymode.BitsPerPixel = 8 then begin
      if not assigned(FPalette) then
        DXCheck( FDirectDraw2.CreatePalette(DDPCAPS_8BIT,@pal,FPalette,nil) );
      DXCheck( FPrimeSurface.SetPalette(FPalette) );
      DXCheck( FBackSurface.SetPalette(FPalette) );
    end;

    try
      DXCheck( FDirect3DRM.CreateDeviceFromSurface(@D3DDevice.GUID,FDirectDraw,FBacksurface,FDevice) );
    except // wenn Fehler, dann mitanderer Device-GUID probieren
      ReleaseCOM( FDevice );
      ReleaseCOM( FZBufferSurface );
      ReleaseCOM( FPrimeSurface );
      inc(faults);
      if faults < D3DDeviceList.Count then // solange bis alle durch sind
      begin
        FD3DDeviceIndex := (FD3DDeviceIndex + 1) mod FD3DDeviceList.Count;
        FDirectDraw2.RestoreDisplayMode;
        DXCheck( FDirectDraw2.SetCooperativelevel(Application.Handle,DDSCL_NORMAL) );
        if faults = 1 then SM('Der Gewnschter Treiber konnte nicht initialisiert werden!'+
                          #13+'Es wird versucht einen Anderen zu verwenden.');
        Result := CreateFullscreenDevice;  // oder bis ein Device funktioniert
      end;
      exit;
    end;

    SetRenderQuality(0);
    UpdateRenderQuality;

    DXCheck( FDirect3DRM.CreateViewport(FDevice,FCameraframe,0,0,
           FDevice.GetWidth,FDevice.GetHeight,FViewport) );
    DXCheck( FViewport.SetField(FFOV) );
    DXCheck( FViewport.SetFront(FFrontClipping) );
    DXCheck( FViewport.SetBack(FBackClipping) );
    Result := true;
  except
    on DirectXException do begin
      ReleaseCOM( FViewport );
      ReleaseCOM( FDevice );
      ReleaseCOM( FPrimeSurface );
      ReleaseCOM( FZBufferSurface );
      FDirectDraw2.RestoreDisplayMode;
      DXCheck( FDirectDraw2.SetCooperativelevel(Application.Handle,DDSCL_NORMAL) );
    end else raise;
  end;
end;

function CreateWindowedDevice : boolean;
const
  LClipper : IDirectDrawClipper = nil;
begin
  Result := false;
  DXStat := 'CreateWindowedDevice';
  DXCheck( FDirectDraw2.SetCooperativelevel(Application.Handle,DDSCL_NORMAL) );

  try
    DXCheck( DirectDrawCreateClipper(0,LClipper,nil) );
    DXCheck( LClipper.SetHWnd(0,Handle) );
    try
      DXCheck( FDirect3DRM.CreateDeviceFromClipper(LClipper,@D3DDevice.GUID,Width,Height,FDevice) );
    except
      ReleaseCOM( FDevice );
      ReleaseCOM( LClipper );
      if faults = 0 then SM('Der Gewnschter Treiber konnte nicht initialisiert werden!'+
                        #13+'Es wird versucht einen Anderen zu verwenden.');
      inc(faults);
      if faults < D3DDeviceList.Count then
      begin
        FD3DDeviceIndex := (FD3DDeviceIndex + 1) mod FD3DDeviceList.Count;
        Result := CreateWindowedDevice;
      end;
      exit;
    end;
    ReleaseCOM( IUnknown(LClipper) );
  except
    ReleaseCOM( IUnknown(LClipper) );
    exit;
  end;

  try
    DXCheck( FDevice.QueryInterface(IID_IDirect3DRMWinDevice,FWinDevice) );

    SetRenderQuality(0);
    UpdateRenderQuality;

    DXCheck( FDirect3DRM.CreateViewport(FDevice,FCameraframe,0,0,
             FDevice.GetWidth,FDevice.GetHeight,FViewport) );
    DXCheck( FViewport.SetField(FFOV) );
    DXCheck( FViewport.SetFront(FFrontClipping) );
    DXCheck( FViewport.SetBack(FBackClipping) );
    Result := true;
  except
    on DirectXException do begin
      ReleaseCOM( FViewport );
      ReleaseCOM( FWinDevice );
      ReleaseCOM( FDevice );
    end else raise;
  end;
end;

begin
  Result := false;
  faults := 0;
  if NotLoaded then exit;
  DestroyDevice;
  if FDisplaying then begin
    if FFullscreen then FDeviceInitialized := CreateFullscreenDevice
      else FDeviceInitialized := CreateWindowedDevice;
  end;
  Result := FDeviceInitialized;
end;

function TD3DBrowser.DestroyDevice : boolean;
begin
  Result := false;
  DXStat := 'DestroyDevice';
  if NotLoaded then exit;
  if DeviceInitialized then begin
    if FFullscreen then begin
      ReleaseCOMe( FViewport );
      ReleaseCOMe( FDevice );
      ReleaseCOMe( FZBufferSurface );
      ReleaseCOMe( FPrimeSurface );
      // FBacksurface wird mit FPimeSurface entlassen
      FBackSurface := nil;
      FDirectDraw2.RestoreDisplaymode;
    end else begin
      ReleaseCOMe( FViewport );
      ReleaseCOMe( FWinDevice );
      ReleaseCOMe( FDevice );
    end;
    FDeviceInitialized := false;
    Result := true;
  end;
end;

procedure TD3DBrowser.SetXFile(XFile: string);
const
  Meshbuilder : IDirect3DRMMeshbuilder = nil;
  Visuals : IDirect3DRMVisualArray = nil;
  Visual  : IDirect3DRMVisual = nil;
  Filename : PChar = nil;
var
  i : integer;
  err : integer;
label
  error;
begin
  FXFile := XFile;
  if NotLoaded then exit;
  if XFile = '' then exit;
  try
    DXStat := 'DirectX-Datei laden';
    DXCheck( FDirect3DRM.CreateMeshBuilder(Meshbuilder) );
    Filename := StrAlloc(Length(XFile)+1);
    StrPCopy(Filename,XFile);
    err := Meshbuilder.Load(Filename,nil,D3DRMLOAD_FROMFILE,nil,nil);
    if (err = D3DRMERR_FILENOTFOUND) or (err = D3DRMERR_BADFILE) then goto error;
    DXCheck(err);
    DXCheck( FMeshframe.GetVisuals(Visuals) );
    for i := 0 to Visuals.GetSize-1 do begin
      DXCheck(Visuals.GetElement(i,Visual) );
      DXCheck( FMeshframe.DeleteVisual(Visual) );
      ReleaseCOMe( IUnknown(Visual) );
    end;
    DXCheck( FMeshframe.AddVisual(Meshbuilder) );
    UpdateMeshsize;
error:
  finally
    ReleaseCOM( IUnknown(Visual) );
    ReleaseCOM( IUnknown(Visuals) );
    ReleaseCOM( IUnknown(Meshbuilder) );
    if assigned(Filename) then StrDispose(Filename);
  end;
end;

procedure TD3DBrowser.IdleProc(Sender: TObject; var Done: Boolean);
begin
  if csDesigning in ComponentState then exit;
  if (not Application.Terminated) and FDeviceInitialized and FDisplaying then begin
    Done := False;
    if FFullscreen then begin
      with FPrimeSurface do if IsLost = DDERR_SURFACELOST then Restore;
      with FBackSurface do if IsLost = DDERR_SURFACELOST then Restore;
      FRootframe.Move(1.0);
      FViewport.Clear;
      ClearSurface(FBackSurface,0);
      FViewport.Render(FRootframe);
      FDevice.Update;
      FPrimeSurface.Flip(nil,DDFLIP_WAIT);
    end else begin
      FDirect3DRM.Tick(1.0);
      {FRootframe.Move(1.0);
      FViewport.Clear;
      FViewport.Render(FRootframe);
      FDevice.Update;}
    end;
  end else Done := True;
end;

procedure TD3DBrowser.SetRenderQuality(Bits: integer);
begin
  if Bits = 0 then case GetCurrentBitsPerPixel of // Wenn 0 dann aktuelle
     8 : begin                                    // Renderwerte einstellen
           FShades := RenderQuality.Palette8Bit.Shades;
           FTextureShades := RenderQuality.Palette8Bit.TextureShades;
           FTextureColors := RenderQuality.Palette8Bit.TextureColors;
           FDither := RenderQuality.Palette8Bit.Dither;
         end;
    15 : begin
           FShades := RenderQuality.Highcolor16Bit.Shades;
           FTextureShades := RenderQuality.Highcolor16Bit.TextureShades;
           FTextureColors := RenderQuality.Highcolor16Bit.TextureColors;
           FDither := RenderQuality.Highcolor16Bit.Dither;
         end;
    16 : begin
           FShades := RenderQuality.Highcolor16Bit.Shades;
           FTextureShades := RenderQuality.Highcolor16Bit.TextureShades;
           FTextureColors := RenderQuality.Highcolor16Bit.TextureColors;
           FDither := RenderQuality.Highcolor16Bit.Dither;
         end;
    24 : begin
           FShades := RenderQuality.Truecolor24Bit.Shades;
           FTextureShades := RenderQuality.Truecolor24Bit.TextureShades;
           FTextureColors := RenderQuality.Truecolor24Bit.TextureColors;
           FDither := RenderQuality.Truecolor24Bit.Dither;
         end;
    32 : begin
           FShades := RenderQuality.Truecolor24Bit.Shades;
           FTextureShades := RenderQuality.Truecolor24Bit.TextureShades;
           FTextureColors := RenderQuality.Truecolor24Bit.TextureColors;
           FDither := RenderQuality.Truecolor24Bit.Dither;
         end;
  end;
  // Wenn Parameter aktueller Farbtiefe entspricht dann Update auslsen
  if GetCurrentBitsPerPixel = Bits then UpdateDevice;
end;

procedure TD3DBrowser.SetShades(Value: integer);
begin
  FShades := Value;
  UpdateRenderQuality;
end;

procedure TD3DBrowser.SetTextureShades(Value: integer);
begin
  FTextureShades := Value;
  UpdateRenderQuality;
end;

procedure TD3DBrowser.SetTextureColors(Value: integer);
begin
  FTextureColors := Value;
  UpdateRenderQuality;
end;

procedure TD3DBrowser.SetDither(Value: boolean);
begin
  FDither := Value;
  UpdateRenderQuality;
end;

procedure TD3DBrowser.WMActivate(var Msg: TMessage);
begin
  DXStat := 'WMActivate';
  inherited;
  if DeviceInitialized and (not FFullscreen) then DXCheck( FWinDevice.HandleActivate(Msg.WParam) );
end;

procedure TD3DBrowser.WMPaint(var Msg: TWMPaint);
begin
  DXStat := 'WMPaint';
  inherited;
  if DeviceInitialized and (not FFullscreen) then DXCheck( FWinDevice.HandlePaint(Msg.DC) );
end;

function TD3DBrowser.SetWindowSize(x,y: integer) : boolean;
var
  oldDisplaying : boolean;
begin
  if (x <> Width) or (y <> Height) then begin
    oldDisplaying := FDisplaying;
    Displaying := false;
    Width := x;
    Height := y;
    if oldDisplaying then Displaying := true;
    Result := true;
  end else Result := false;
end;

procedure TD3DBrowser.SetMeshsize(Meshsize: TD3DValue);
var
  factor : TD3DValue;
begin
  DXStat := 'SetMeshsize';
  if FMeshsize = 0.0 then exit;
  factor := Meshsize / FMeshsize;
  DXCheck( FMeshframe.AddScale(D3DRMCOMBINE_REPLACE,factor,factor,factor) );
  UpdateMeshsize;
end;

function TD3DBrowser.UpdateMeshsize : TD3DValue;
var
  box : TD3DRMBox;
  x,y,z : TD3DValue;
begin
  DXStat := 'UpdateMeshsize';
  FMeshsize := 0;
  GetFrameBox(FMeshframe,box);
  x := box.max.x - box.min.x;
  y := box.max.y - box.min.y;
  z := box.max.z - box.min.z;
  FMeshsize := x;
  if y > FMeshsize then FMeshsize := y;
  if z > FMeshsize then FMeshsize := z;
  Result := FMeshSize;
end;

procedure TD3DBrowser.LookAtCenter;
begin
  DXStat := 'LookAtCenter';
  if not NotLoaded then
    DXCheck( FCameraframe.LookAt(FMeshframe,FRootframe,FrameConstraint) );
end;

procedure TD3DBrowser.AdjustDistance(Factor: TD3DValue);
var
  pos : TD3DVector;
begin
  DXStat := 'AdjustDistance';
  DXCheck( FCameraframe.GetPosition(FRootframe,pos) );
  Factor := ( ((Meshsize / 2) / sin(FFOV / 2)) / D3DRMVectorModulus(pos)
              ) / Factor;
  {Factor := (Meshsize / 2) / FFov / D3DRMVectorModulus(pos) / Factor;}
  pos.x := pos.x * Factor;
  pos.y := pos.y * Factor;
  pos.z := pos.z * Factor;
  DXCheck( FCameraframe.SetPosition(Frootframe,pos.x,pos.y,pos.z) );
end;

procedure TD3DBrowser.AdjustMeshsize(Factor: TD3DValue);
var
  pos : TD3DVector;
begin
  DXStat := 'AdjustMeshsize';
  DXCheck( FCameraframe.GetPosition(FRootframe,pos) );
  Factor := Factor * (
        ( D3DRMVectorModulus(pos) * sin(FFOV / 2)) / (Meshsize / 2) );
  DXCheck( FMeshframe.AddScale(D3DRMCOMBINE_REPLACE,factor,factor,factor) );
end;

procedure TD3DBrowser.CenterMesh;
var
  Box: TD3DRMBox;
  x,y,z : TD3DValue;
begin
  GetFrameBox(FMeshframe,Box);
  x := (Box.max.x - Box.min.x) / 2 + Box.min.x;
  y := (Box.max.y - Box.min.y) / 2 + Box.min.y;
  z := (Box.max.z - Box.min.z) / 2 + Box.min.z;
  DXStat := 'CenterMesh';
  DXCheck( FMeshframe.SetPosition(FRootframe,-x,-y,-z) );
end;

procedure TD3DBrowser.RotateLeft(Angle: TD3DValue);
var
  p : TD3DVector;
begin
  DXCheck( FRootframe.GetPosition(FCameraframe,p) );
  DXCheck( FCameraframe.SetPosition(FRootframe,0,0,0) );
  TurnRight(Angle);
  DXCheck( FCameraframe.SetPosition(FCameraframe,-p.x,-p.y,-p.z) );
end;

procedure TD3DBrowser.RotateRight(Angle: TD3DValue);
var
  p : TD3DVector;
begin
  DXCheck( FRootframe.GetPosition(FCameraframe,p) );
  DXCheck( FCameraframe.SetPosition(FRootframe,0,0,0) );
  TurnLeft(Angle);
  DXCheck( FCameraframe.SetPosition(FCameraframe,-p.x,-p.y,-p.z) );
end;

procedure TD3DBrowser.RotateUp(Angle: TD3DValue);
var
  p : TD3DVector;
begin
  DXCheck( FRootframe.GetPosition(FCameraframe,p) );
  DXCheck( FCameraframe.SetPosition(FRootframe,0,0,0) );
  TurnDown(Angle);
  DXCheck( FCameraframe.SetPosition(FCameraframe,-p.x,-p.y,-p.z) );
end;

procedure TD3DBrowser.RotateDown(Angle: TD3DValue);
var
  p : TD3DVector;
begin
  DXCheck( FRootframe.GetPosition(FCameraframe,p) );
  DXCheck( FCameraframe.SetPosition(FRootframe,0,0,0) );
  TurnUp(Angle);
  DXCheck( FCameraframe.SetPosition(FCameraframe,-p.x,-p.y,-p.z) );
end;

procedure TD3DBrowser.TurnLeft(Angle: TD3DValue);
begin
  Angle := -Angle;
  DXCheck( FCameraframe.SetOrientation(FCameraframe,sin(Angle),0,cos(Angle),0,1,0) );
end;

procedure TD3DBrowser.TurnRight(Angle: TD3DValue);
begin
  DXCheck( FCameraframe.SetOrientation(FCameraframe,sin(Angle),0,cos(Angle),0,1,0) );
end;

procedure TD3DBrowser.TurnUp(Angle: TD3DValue);
begin
  DXCheck( FCameraframe.SetOrientation(FCameraframe,0,sin(Angle),cos(Angle),0,cos(Angle),sin(Angle)) );
end;

procedure TD3DBrowser.TurnDown(Angle: TD3DValue);
begin
  Angle := -Angle;
  DXCheck( FCameraframe.SetOrientation(FCameraframe,0,sin(Angle),cos(Angle),0,cos(Angle),sin(Angle)) );
end;

procedure TD3DBrowser.RollLeft(Angle: TD3DValue);
begin
  DXCheck( FCameraframe.SetOrientation(FCameraframe,0,0,1,sin(Angle),cos(Angle),0) );
end;

procedure TD3DBrowser.RollRight(Angle: TD3DValue);
begin
  Angle := -Angle;
  DXCheck( FCameraframe.SetOrientation(FCameraframe,0,0,1,sin(Angle),cos(Angle),0) );
end;

procedure TD3DBrowser.MoveLeft(Space: TD3DValue);
begin
  DXCheck( FCameraframe.SetPosition(FCameraframe,-Space,0,0) );
end;

procedure TD3DBrowser.MoveRight(Space: TD3DValue);
begin
  DXCheck( FCameraframe.SetPosition(FCameraframe,Space,0,0) );
end;

procedure TD3DBrowser.MoveUp(Space: TD3DValue);
begin
  DXCheck( FCameraframe.SetPosition(FCameraframe,0,Space,0) );
end;

procedure TD3DBrowser.MoveDown(Space: TD3DValue);
begin
  DXCheck( FCameraframe.SetPosition(FCameraframe,0,-Space,0) );
end;

procedure TD3DBrowser.MoveForward(Space: TD3DValue);
begin
  DXCheck( FCameraframe.SetPosition(FCameraframe,0,0,Space) );
end;

procedure TD3DBrowser.MoveBackward(Space: TD3DValue);
begin
  DXCheck( FCameraframe.SetPosition(FCameraframe,0,0,-Space) );
end;

procedure TD3DBrowser.Tick(Move: TD3DValue);
begin
  DXCheck( Direct3DRM.Tick(Move) );
end;

function TD3DBrowser.GetCurrentBitsPerPixel : integer;
var
  dc : HDC;
begin
  if FFullscreen then
  begin
    if assigned(Displaymode) then
      Result := Displaymode.BitsPerPixel
    else
      Result := 0;
  end
  else
  begin
    dc := GetDC(Handle);
    Result := GetDeviceCaps(dc, BITSPIXEL);
    ReleaseDC(Handle, dc);
  end;
end;

procedure TD3DBrowser.SetResizeToParentWindow(Resize: boolean);
var
  oldDisplaying : boolean;
begin
  FResizeToParentWindow := Resize;
  if Resize and ( (Left <> 0) or (Top <> 0)
       or (Height <> Parent.ClientHeight) or (Width <> Parent.ClientWidth) )
  then begin
    oldDisplaying := FDisplaying;
    Displaying := false;
    Left := 0;
    Top := 0;
    SetWindowSize(Parent.ClientWidth,Parent.ClientHeight);
    if oldDisplaying then Displaying := true;
  end;
end;

// ****************************************************************************
// Methoden der Eigenschaften die vom Anwender initialisiert werden mssen
// ****************************************************************************

procedure TD3DBrowser.SetHeight(Value: integer);
begin
  if Value = 0 then begin
    Top := 0;
    Value := parent.ClientWidth;
  end;
  if Value = 0 then Value := DefaultHeight;
  if inherited Height <> Value then begin
    inherited Height := Value;
    UpdateDevice;
  end;
end;

procedure TD3DBrowser.SetWidth(Value: integer);
begin
  if Value = 0 then begin
    Left := 0;
    Value := parent.ClientWidth;
  end;
  if Value = 0 then Value := DefaultWidth;
  if inherited Width <> Value then begin
    inherited Width := Value;
    UpdateDevice;
  end;
end;

function TD3DBrowser.GetHeight : integer;
begin
  Result := inherited Height;
end;

function TD3DBrowser.GetWidth : integer;
begin
  Result := inherited Width;
end;

procedure TD3DBrowser.SetFullscreen(Fullscreen: boolean);
begin
  if (csDesigning in ComponentState) or NotLoaded then FFullscreen := Fullscreen;
  if Fullscreen <> FFullscreen then begin
    DestroyDevice;
    FFullscreen := Fullscreen;
    // wenn Fehler dann wieder alten Zustand herstellen
    if not UpdateDevice then begin
      DestroyDevice;
      FFullscreen := not Fullscreen;
      UpdateDevice;
      Application.Messagebox('Vollbildwechsel konnte nicht durchgefhrt werden.','Fehler:',MB_APPLMODAL);
    end;
  end;
end;

function TD3DBrowser.GetDisplaymode : TDisplaymode;
begin
  if (not assigned(FDisplaymodeList)) or (FDisplaymodeList.Count = 0)
    then Result := nil
  else
    Result := FDisplaymodeList.Items[DisplaymodeIndex];
end;

procedure TD3DBrowser.SetDisplaymode(Mode: TDisplaymode);
begin
  SetDisplaymodeIndex(FDisplaymodeList.IndexOF(Mode));
end;

procedure TD3DBrowser.SetDisplaymodeIndex(index : integer);
var
  old : integer;
begin
  if (index >= 0) and (index < FDisplaymodeList.Count) then
  begin
    old := FDisplaymodeIndex;
    FDisplaymodeIndex := index;
    if FFullScreen then
    begin
      DestroyDevice;
      if not UpdateDevice then
      begin
        FDisplaymodeIndex := old;
        UpdateDevice;
      end;
    end;
  end;
end;

function TD3DBrowser.GetD3DDevice : TD3DDevice;
begin
  if (not assigned(FD3DDeviceList)) or (FD3DDeviceList.Count = 0)
    then Result := nil
  else
    Result := FD3DDeviceList.Items[FD3DDeviceIndex];
end;

procedure TD3DBrowser.SetD3DDevice(Device: TD3DDevice);
begin
  SetD3DDeviceIndex(FD3DDeviceList.IndexOF(Device));
end;

procedure TD3DBrowser.SetD3DDeviceIndex(index : integer);
var
  old : integer;
begin
  if (index >= 0) and (index < FD3DDeviceList.Count) then
  begin
    old := FD3DDeviceIndex;
    DestroyDevice;
    FD3DDeviceIndex := index;
    if not UpdateDevice then
    begin
      FD3DDeviceIndex := old;
      UpdateDevice;
    end;
  end;
end;

procedure TD3DBrowser.SetShadeMode(ShadeMode: TShadeMode);
begin
  case ShadeMode of
    Wireframe : D3DRenderquality := D3DRMRENDER_WIREFRAME;
    UnlitFlat : D3DRenderquality := D3DRMRENDER_UNLITFLAT;
    Flat      : D3DRenderquality := D3DRMRENDER_FLAT;
    Gouraud   : D3DRenderquality := D3DRMRENDER_GOURAUD;
    Phong     : D3DRenderquality := D3DRMRENDER_PHONG;
  end;
  if ShadeMode <> FShadeMode then begin
    FShadeMode := ShadeMode;
    UpdateDevice;
  end;
end;

procedure TD3DBrowser.SetFOV(FOV: TD3DValue);
begin
  DXStat := 'SetFOV';
  if (FOV <= 0.0) or (FOV > 3.0) then FOV := DefaultFOV;
  FFOV := FOV;
  if FDeviceInitialized then DXCheck( FViewport.SetField(FOV) );
end;

procedure TD3DBrowser.SetFrontClipping(Front: TD3DValue);
begin
  DXStat := 'SetFrontClipping';
  if Front <= 0.0 then Front := DefaultFrontClipping;
  if Front >= FBackClipping then Front := FBackClipping - 1;
  FFrontClipping := Front;
  if FDeviceInitialized then DXCheck( FViewport.SetFront(Front) );
end;

procedure TD3DBrowser.SetBackClipping(Back: TD3DValue);
begin
  DXStat := 'SetBackClipping';
  if Back <= 0.0 then Back := DefaultBackClipping;
  if Back <= FFrontClipping then Back := FFrontClipping + 1;
  FBackClipping := Back;
  if FDeviceInitialized then DXCheck( FViewport.SetBack(Back) );
end;

procedure TD3DBrowser.SetAmbientBrightness(Brightness: TD3DValue);
var
  Color : TD3DColorValue;
begin
  if Brightness < 0.0 then Brightness := 0.0;
  Color.R := FAmbientColor.Red;
  Color.G := FAmbientColor.Green;
  Color.B := FAmbientColor.Blue;
  SetBrightness(Color.R,Color.G,Color.B,Brightness);
  FAmbientColor.Color := Color;
end;

function TD3DBrowser.GetAmbientBrightness : TD3DValue;
begin
  with FAmbientColor do Result := GetBrightness(Red,Green,Blue);
end;

procedure TD3DBrowser.SetAmbientColor(r,g,b: TD3DValue);
begin
  DXStat := 'SetAmbientColor';
  if not NotLoaded then DXCheck( FAmbientLight.SetColorRGB(r,g,b) );
end;

procedure TD3DBrowser.SetFrontlightType(FrontlightType: TFrontlightType);
const
  OldLight : IDirect3DRMLight = nil;
begin
  DXStat := 'SetFrontlightType';
  if FrontlightType <> FFrontlightType then begin
    if (FFrontlightType <> None) and (not NotLoaded) then DXCheck( FCameraFrame.DeleteLight(FFrontlight) );
    FFrontlightType := FrontlightType;
    if NotLoaded then exit;
    OldLight := FFrontlight;
    try case FrontlightType of
      Pointlight : DXCheck( FDirect3DRM.CreateLightRGB(D3DRMLIGHT_POINT,
                            FFrontlightColor.Red,FFrontlightColor.Green,
                            FFrontlightColor.Blue,FFrontlight) );
      Spotlight  : begin
                     DXCheck( FDirect3DRM.CreateLightRGB(D3DRMLIGHT_SPOT,
                            FFrontlightColor.Red,FFrontlightColor.Green,
                            FFrontlightColor.Blue,FFrontlight) );
                     SetFrontlightUmbra(FFrontlightUmbra);
                     SetFrontlightPenumbra(FFrontlightPenumbra);
                   end;
      Directional: DXCheck( FDirect3DRM.CreateLightRGB(D3DRMLIGHT_DIRECTIONAL,
                            FFrontlightColor.Red,FFrontlightColor.Green,
                            FFrontlightColor.Blue,FFrontlight) );
      None       : DXCheck( FDirect3DRM.CreateLightRGB(D3DRMLIGHT_AMBIENT,0,0,0,FFrontlight) );
    end
    except
      on DirectXException do begin
        ReleaseCOM( FFrontlight );
        FFrontlight := OldLight;
        raise;
      end else raise;
    end;
    ReleaseCOMe( IUnknown(OldLight) );
    if FrontlightType <> None then DXCheck( FCameraFrame.AddLight(FFrontlight) );
  end;
end;

procedure TD3DBrowser.SetFrontlightUmbra(Umbra: TD3DValue);
begin
  DXStat := 'SetFrontlightUmbra';
  if Umbra <= 0.0 then Umbra := 0.01;
  if (Umbra > FFrontlightPenumbra) or (Umbra > 3.0) then Umbra := FFrontlightPenumbra;
  FFrontlightUmbra := Umbra;
  if not NotLoaded then DXCheck( FFrontlight.SetUmbra(Umbra) );
end;

procedure TD3DBrowser.SetFrontlightPenumbra(Penumbra: TD3DValue);
begin
  DXStat := 'SetFrontlightPenumbra';
  if (Penumbra < FFrontlightUmbra) then Penumbra := FFrontlightUmbra;
  if (Penumbra <= 0.0) or (Penumbra > 3.0) then Penumbra := DefaultFrontlightPenumbra;
  FFrontlightPenumbra := Penumbra;
  if not NotLoaded then DXCheck( FFrontlight.SetPenumbra(Penumbra) );
end;

procedure TD3DBrowser.SetFrontlightBrightness(Brightness: TD3DValue);
var
  Color : TD3DColorValue;
begin
  if Brightness < 0.0 then Brightness := 0.0;
  Color.R := FFrontlightColor.Red;
  Color.G := FFrontlightColor.Green;
  Color.B := FFrontlightColor.Blue;
  SetBrightness(Color.R,Color.G,Color.B,Brightness);
  FFrontlightColor.Color := Color;
end;

function TD3DBrowser.GetFrontlightBrightness : TD3DValue;
begin
  with FFrontlightColor do Result := GetBrightness(Red,Green,Blue);
end;

procedure TD3DBrowser.SetFrontlightColor(r,g,b: TD3DValue);
begin
  DXStat := 'SetFrontlightColor';
  if not NotLoaded then DXCheck( FFrontlight.SetColorRGB(r,g,b) );
end;

procedure TD3DBrowser.SetSunlightBrightness(Brightness: TD3DValue);
var
  Color : TD3DColorValue;
begin
  if Brightness < 0.0 then Brightness := 0.0;
  Color.R := FSunlightColor.Red;
  Color.G := FSunlightColor.Green;
  Color.B := FSunlightColor.Blue;
  SetBrightness(Color.R,Color.G,Color.B,Brightness);
  FSunlightColor.Color := Color;
end;

function TD3DBrowser.GetSunlightBrightness : TD3DValue;
begin
  with FSunlightColor do Result := GetBrightness(Red,Green,Blue);
end;

procedure TD3DBrowser.SetSunlightColor(r,g,b: TD3DValue);
begin
  DXStat := 'SetSunlightColor';
  if not NotLoaded then DXCheck( FSunlight.SetColorRGB(r,g,b) );
end;

function TD3DBrowser.ChooseD3DDevice : integer;
var
  i : integer;
begin
  Result := 0;
  if (not assigned(D3DDevice)) or (D3DDeviceList.Count = 0) then exit;
  i := 0;
  while D3DDeviceList.Items[i].SoftwareDevice do
  begin
    inc(i);
    if i = D3DDeviceList.Count then exit;
  end;
  Result := i;
end;

function TD3DBrowser.ChooseDisplaymode : integer;
var
  i : integer;
begin
  Result := 0;
  if (not assigned(Displaymode)) or (DisplaymodeList.Count = 0) then exit;
  if (not assigned(D3DDevice)) or (D3DDeviceList.Count = 0) then exit;
  i := 0;
  if D3DDevice.SoftwareDevice then
    while ((DisplaymodeList.Items[i].Width <> 640)
      or (DisplaymodeList.Items[i].Height <> 480)
      or (DisplaymodeList.Items[i].BitsPerPixel <> 8)) do
    begin
      inc(i);
      if i = DisplaymodeList.Count then exit;
    end
  else
    while ((DisplaymodeList.Items[i].Width <> 640)
      or (DisplaymodeList.Items[i].Height <> 480)
      or (DisplaymodeList.Items[i].BitsPerPixel <> 16)) do
    begin
      inc(i);
      if i = DisplaymodeList.Count then exit;
    end;
  Result := i;
end;

constructor TColorRGB.Create(Callback: TSetColorRGBCallback);
begin
  inherited Create;
  SetColorRGB := Callback;
  FColor.A := 1.0;
end;

procedure TColorRGB.SetColorValue(Index: integer; Value: TD3DValue);
begin
  if Value < 0.0 then Value := 0.0;
  case Index of
    0 : FColor.R := Value;
    1 : FColor.G := Value;
    2 : FColor.B := Value;
  end;
  if assigned(SetColorRGB) then SetColorRGB(Red,Green,Blue);
end;


procedure TColorRGB.SetColor(Value: TD3DColorValue);
begin
  FColor := Value;
  if assigned(SetColorRGB) then SetColorRGB(Red,Green,Blue);
end;

procedure TRenderValues.SetColorShades(index: integer; Value: integer);
begin
  case index of
    0 : FShades := Value;
    1 : FTextureShades := Value;
    2 : FTextureColors := Value;
  end;
  with D3DBrowser as TD3DBrowser do SetRenderQuality(Bits);
end;

procedure TRenderValues.SetDither(Value: boolean);
begin
  FDither := Value;
  with D3DBrowser as TD3DBrowser do SetRenderQuality(Bits);
end;

constructor TRenderValues.Create(Browser: TObject; ColorBits: integer);
begin
  inherited Create;
  Bits := ColorBits;
  D3DBrowser := Browser;
end;

constructor TRenderQuality.Create(Browser: TObject);
begin
  inherited Create;
  FPalette8Bit := TRenderValues.Create(Browser,8);
  FHighColor16Bit := TRenderValues.Create(Browser,16);
  FTrueColor24Bit := TRenderValues.Create(Browser,24);
  with Palette8Bit do begin
    FShades := 16;
    FTextureShades := 16;
    FTextureColors := 32;
    FDither := false;
  end;
  with HighColor16Bit do begin
    FShades := 32;
    FTextureShades := 32;
    FTextureColors := 64;
    FDither := false;
  end;
  with TrueColor24Bit do begin
    FShades := 256;
    FTextureShades := 256;
    FTextureColors := 64;
    FDither := false;
  end;
end;

destructor TRenderQuality.Destroy;
begin
  FPalette8Bit.Free;
  FHighColor16Bit.Free;
  FTrueColor24Bit.Free;
  inherited Destroy;
end;

////////////////////////////////////////////////////////////////////////////////
// TDisplaymode
////////////////////////////////////////////////////////////////////////////////

procedure TDisplaymode.SetBitsPerPixel(BitsPerPixel: integer);
begin
  FBitsPerPixel := BitsPerPixel;
end;

procedure TDisplaymode.SetPixelFormat(PixelFormat: TPixelformat);
begin
  case PixelFormat of
    Format1Bit : SetBitsPerPixel(1);
    Format2Bit : SetBitsPerPixel(2);
    Format4Bit : SetBitsPerPixel(4);
    Format8Bit : SetBitsPerPixel(8);
    Format15Bit : SetBitsPerPixel(15);
    Format16Bit : SetBitsPerPixel(16);
    Format24Bit : SetBitsPerPixel(24);
    Format32Bit : SetBitsPerPixel(32);
    else SetBitsPerPixel(0);
  end;
end;

function TDisplaymode.GetPixelFormat : TPixelFormat;
begin
  case FBitsPerPixel of
    1 : Result := Format1Bit;
    2 : Result := Format2Bit;
    4 : Result := Format4Bit;
    8 : Result := Format8Bit;
    15 : Result := Format15Bit;
    16 : Result := Format16Bit;
    24 : Result := Format24Bit;
    32 : Result := Format32Bit;
    else Result := FormatUnknown;
  end;
end;

function TDisplaymode.SizeOfPixel : integer;
begin
  Result := (FBitsPerPixel + 1) div 8;
end;

procedure TDisplaymode.SetHeight(Value: integer);
begin
  FHeight := Value;
end;

procedure TDisplaymode.SetWidth(Value: integer);
begin
  FWidth := Value;
end;

function TD3DDeviceList_EnumDevicesCallback(const lpGuid: TGUID;
      lpDeviceDescription: LPSTR; lpDeviceName: LPSTR;
      const lpD3DHWDeviceDesc: TD3DDeviceDesc;
      const lpD3DHELDeviceDesc: TD3DDeviceDesc;
      lpUserArg: Pointer) : HResult; stdcall;

var
  thisDevice : TD3DDevice;
  desc : PD3DDeviceDesc;
begin
  with TObject(lpUserArg) as TD3DDeviceList do
  begin
    desc := @lpD3DHELDeviceDesc;
    if desc^.dcmColorModel <> D3DCOLOR_INVALID_0 then
    begin
      thisDevice := TD3DDevice.Create;
      thisDevice.Direct3D := FDirect3D;
      thisDevice.GUID := lpGUID;
      thisDevice.Name := lpDeviceName;
      thisDevice.Description := lpDeviceDescription;
      with thisDevice do
      begin
        case desc^.dcmColorModel of
           D3DCOLOR_MONO : Colormodel := Ramp;
           D3DCOLOR_RGB  : Colormodel := RGB;
        end;
        SoftwareDevice := true;
        Pixelformats := [];
        if longbool(desc^.dwDeviceRenderBitDepth and DDBD_8)
          then Include(Pixelformats,Format8Bit);
        if longbool(desc^.dwDeviceRenderBitDepth and DDBD_16)
          then Include(Pixelformats,Format16Bit);
        if longbool(desc^.dwDeviceRenderBitDepth and DDBD_24)
          then Include(Pixelformats,Format24Bit);
        if longbool(desc^.dwDeviceRenderBitDepth and DDBD_32)
          then Include(Pixelformats,Format32Bit);
        ZBufferformats := [];
        if longbool(desc^.dwDeviceZBufferBitDepth and DDBD_16)
          then Include(ZBufferformats,Format16Bit);
        if longbool(desc^.dwDeviceZBufferBitDepth and DDBD_24)
          then Include(ZBufferformats,Format24Bit);
        if longbool(desc^.dwDeviceZBufferBitDepth and DDBD_32)
          then Include(ZBufferformats,Format32Bit);
      end;
      Add(thisDevice);
    end;

    desc := @lpD3DHWDeviceDesc;
    if desc^.dcmColorModel <> D3DCOLOR_INVALID_0 then
    begin
      thisDevice := TD3DDevice.Create;
      thisDevice.Direct3D := FDirect3D;
      thisDevice.GUID := lpGUID;
      thisDevice.Name := lpDeviceName;
      thisDevice.Description := lpDeviceDescription;
      with thisDevice do
      begin
        case desc^.dcmColorModel of
           D3DCOLOR_MONO : Colormodel := Ramp;
           D3DCOLOR_RGB  : Colormodel := RGB;
        end;
        SoftwareDevice := false;
        Pixelformats := [];
        if longbool(desc^.dwDeviceRenderBitDepth and DDBD_8)
          then Include(Pixelformats,Format8Bit);
        if longbool(desc^.dwDeviceRenderBitDepth and DDBD_16)
          then Include(Pixelformats,Format16Bit);
        if longbool(desc^.dwDeviceRenderBitDepth and DDBD_24)
          then Include(Pixelformats,Format24Bit);
        if longbool(desc^.dwDeviceRenderBitDepth and DDBD_32)
          then Include(Pixelformats,Format32Bit);
        ZBufferformats := [];
        if longbool(desc^.dwDeviceZBufferBitDepth and DDBD_16)
          then Include(ZBufferformats,Format16Bit);
        if longbool(desc^.dwDeviceZBufferBitDepth and DDBD_24)
          then Include(ZBufferformats,Format24Bit);
        if longbool(desc^.dwDeviceZBufferBitDepth and DDBD_32)
          then Include(ZBufferformats,Format32Bit);
      end;
      Add(thisDevice);
    end;
  end;
  result := DDENUMRET_OK;
end;

function TD3DDeviceList.GetItems(index: integer) : TD3DDevice;
begin
  Result := TD3DDevice(TList(self).Items[index]);
end;

constructor TD3DDeviceList.Create;
begin
  inherited;
end;

constructor TD3DDeviceList.CreateFromDirect3D(D3D: IDirect3D);
begin
  inherited Create;
  if assigned(D3D) then FDirect3D := D3D
    else FDirect3D := nil; 
end;

function TD3DDeviceList.Fill : boolean;
var
  i : integer;
  err : HResult;
begin
  for i := 0 to Count-1 do Items[i].Free;
  err := FDirect3D.EnumDevices(TD3DDeviceList_EnumDevicesCallback,self);
  Result := (err = D3D_OK);
  if Debugg then DXCheck(err);
end;

destructor TD3DDeviceList.Destroy;
var
  i : integer;
begin
  for i := 0 to Count-1 do Items[i].Free;
  ReleaseCOM(FDirect3D);
  inherited;
end;

////////////////////////////////////////////////////////////////////////////////
// TDisplaymodeList
////////////////////////////////////////////////////////////////////////////////

function TDisplaymodeList_EnumDisplayModesCallback(const Desc: TDDSurfaceDesc; Context: Pointer ): HResult; stdcall;
var
  Displaymode: TDisplaymode;
begin
  Displaymode := TDisplaymode.Create(Desc.dwWidth,Desc.dwHeight
                 ,Desc.ddpfPixelFormat.dwRGBBitCount,Desc.dwRefreshrate,
                 TDisplaymodeList(Context).DirectDraw2);
  TDisplaymodeList(Context).Add(Displaymode);
  Result := DDENUMRET_OK;
end;

function TDisplaymodeList.GetItems(index: integer) : TDisplaymode;
begin
  Result := TDisplaymode(TList(self).Items[index]);
end;

destructor TDisplaymodeList.Destroy;
var
  i : integer;
begin
  for i := 0 to Count-1 do Items[i].Free;
  ReleaseCOM( FDirectDraw2 );
  inherited;
end;

constructor TDisplaymodeList.CreateFromDirectDraw2(DD: IDirectDraw2);
begin
  inherited Create;
  if assigned(DD) then FDirectDraw2 := DD
    else FDirectDraw2 := nil;
  FDirectDraw2.AddRef;
end;

function TDisplaymodeList.Fill : boolean;
var
  err : HResult;
  i : integer;
begin
  for i := 0 to Count-1 do Items[i].Free;
  err := FDirectDraw2.EnumDisplayModes(DDEDM_REFRESHRATES,nil,self,
                  TDisplaymodeList_EnumDisplaymodesCallback);
  Result := (err = DD_OK);
  if Debugg then DXCheck(err);
end;


////////////////////////////////////////////////////////////////////////////////
// TDisplaymode
////////////////////////////////////////////////////////////////////////////////

constructor TDisplaymode.Create(CWidth, CHeight, CBitsPerPixel, CRefreshrate: integer; DD: IDirectDraw2);
begin
  inherited Create;
  Width := CWidth;
  Height := CHeight;
  BitsPerPixel := CBitsPerPixel;
  FRefreshrate := CRefreshrate;
  if assigned(DD) then FDirectDraw2 := DD
    else FDirectDraw2 := nil;
  FDirectDraw2.AddRef;
  FName := IntToStr(Width) + 'x' + IntToStr(Height) + 'x' + IntToStr(BitsPerPixel);
end;

destructor TDisplaymode.Destroy;
begin
  ReleaseCOM( FDirectDraw2 );
  inherited;
end;

function TDisplaymode.SetMode : boolean;
var
  err : HResult;
begin
  err := FDirectDraw2.SetDisplayMode(Width,Height,BitsPerPixel,RefreshRate,0);
  if err <> DD_OK then
  begin
    Result := false;
    FDirectDraw2.RestoreDisplayMode;
    if Debugg then DXCheck(err);
  end
    else Result := true;
end;

function TDisplaymode.CreateOffscreenSurface(var Surf: IDirectDrawSurface; Caps: DWord) : boolean;
var
  SurfDesc : TDDSurfaceDesc;
begin
  Result := false;
  InitRecord(SurfDesc,sizeof(SurfDesc));
  with SurfDesc do begin
    dwFlags := DDSD_CAPS + DDSD_HEIGHT + DDSD_WIDTH + DDSD_PIXELFORMAT ;
    dwHeight := Height;
    dwWidth := Width;
    ddsCaps.dwCaps := Caps + DDSCAPS_OFFSCREENPLAIN;
    InitRecord(ddpfPixelFormat,sizeof(ddpfPixelFormat));
    with ddpfPixelFormat do begin
      dwFlags := DDPF_RGB;
      dwRGBBitCount := BitsPerPixel;
      case BitsPerPixel of
        1 : dwFlags := dwFlags + DDPF_PALETTEINDEXED1;
        2 : dwFlags := dwFlags + DDPF_PALETTEINDEXED2;
        4 : dwFlags := dwFlags + DDPF_PALETTEINDEXED4;
        8 : dwFlags := dwFlags + DDPF_PALETTEINDEXED8;
      end;
    end;
  end;
  if FDirectDraw2.CreateSurface(SurfDesc,Surf,nil) = DD_OK
    then Result := true;
end;

procedure Register;
begin
  RegisterComponents('Beispiele', [TD3DBrowser]);
end;

end.

