unit DXTools;

interface

{$INCLUDE COMSWITCH.INC}

uses
{$IFDEF D2COM}
  OLE2,
{$ENDIF}
  Windows,
  SysUtils,
  Graphics,
  DDraw,
  D3D,
  D3DRM,
  D3DRMDef,
  D3DRMObj,
  D3DTypes;

const
  Exceptions : boolean = True;
  DXStat : string = '';

type
  PTrueColor = ^TTrueColor;
  TTrueColor = record
    case integer of
      1 : (Data : DWORD);
      2 : (R,G,B,A : byte);
  end;

  PColorTable = ^TColorTable;
  TColorTable = array [0..255] of TTrueColor;

  TSingleQuadruppel = array[0..4-1] of single;

  DirectXException = class(Exception)
  public
    constructor Create(Error: integer);
  end;

const
  IdentityMatrix : TD3DMatrix = (
    _11: 1; _12: 0; _13: 0; _14: 0;
    _21: 0; _22: 1; _23: 0; _24: 0;
    _31: 0; _32: 0; _33: 1; _34: 0;
    _41: 0; _42: 0; _43: 0; _44: 1 );


  ZeroMatrix : TD3DMatrix = (
    _11: 0; _12: 0; _13: 0; _14: 0;
    _21: 0; _22: 0; _23: 0; _24: 0;
    _31: 0; _32: 0; _33: 0; _34: 0;
    _41: 0; _42: 0; _43: 0; _44: 0 );

type
  PMatrix1D = ^TMatrix1D;
  TMatrix1D = record
    case integer of
      0 : (D3DVector: TD3DVector);
      1 : (D3DColorValue: TD3DColorValue);
      2 : (a: array [0..4-1] of TD3DValue);
  end;

  PMatrix4D = ^TMatrix4D;
  TMatrix4D = record
    case integer of
      0 : (D3DMatrix: TD3DMatrix);
      1 : (_: TD3DMatrix_);
      2 : (a: array [0..4*4-1] of TD3DValue);
  end;

  TStack = class (TObject)
  private
    FItemSize : integer;
    FCount : integer;
    FTop : pointer;
    FBase : pointer;
    FStackEnd : pointer;
    FStackSize : integer;
    FIncAmount : integer;
  protected
    procedure SetCount(Value: integer);
    procedure SetTop(Value: pointer);
    procedure SetIncAmount(Value: integer);
    procedure SetStackSize(Value: integer);
    function GetStackSize : integer;
  public
    property ItemSize : integer read FItemSize;
    property Count : integer read FCount write SetCount;
    property Top : pointer read FTop write SetTop;
    property Base : pointer read FBase;
    property IncAmount : integer read FIncAmount write SetIncAmount;
    property StackSize : integer read GetStackSize write SetStackSize;
    constructor Create(ItemSize: integer);
    constructor CreateForByte;
    constructor CreateForInteger;
    constructor CreateForPointer;
    constructor CreateForSingle;
    constructor CreateForDouble;
    constructor CreateForMatrix1D;
    constructor CreateForMatrix4D;
    procedure Push(const Item);
    function Pop : pointer;
    procedure PushByte(Item: byte);
    function PopByte : byte;
    procedure PushInteger(Item: integer);
    function PopInteger : integer;
    procedure PushPointer(Item: pointer);
    function PopPointer : pointer;
    procedure PushSingle(Item: single);
    function PopSingle : single;
    procedure PushMatrix1D(Item: TMatrix1D);
    function PopMatrix1D : TMatrix1D;
    procedure PushMatrix4D(Item: TMatrix4D);
    function PopMatrix4D : TMatrix4D;
    procedure Increase;
    procedure Decrease;
    procedure IncreaseMulti(ItemCount: integer);
    procedure DecreaseMulti(ItemCount: integer);
    procedure IncStackSize;
    procedure Reset;
    destructor Destroy; override;
  end;

function ProjectionMatrix(near_plane,     // distance to near clipping plane
                          far_plane,      // distance to far clipping plane
                          fov: TD3DValue) : TD3DMatrix; // field of view angle,
                                                      // in radians
function ViewMatrix(from,                  // camera location
                    at,                    // camera look-at target
                    world_up: TD3DVector;  // world's up, usually 0, 1, 0
                    roll: TD3DValue) : TD3DMatrix; // clockwise roll around
                                                 //    viewing direction,
                                                 //    in radians

function TransformationYZ(y, z: TD3DVector) : TD3DMatrix;

function TranslateMatrix(dx, dy, dz: TD3DValue) : TD3DMatrix;

function RotateXMatrix(rads: TD3DValue) : TD3DMatrix;

function RotateYMatrix(rads: TD3DValue) : TD3DMatrix;

function RotateZMatrix(rads: TD3DValue) : TD3DMatrix;

function ScaleMatrix(size: TD3DValue) : TD3DMatrix;

function MatrixMul(a, b: TD3DMatrix) : TD3DMatrix;

procedure InitRecord(var DXRecord; Size: integer);

function GetBrightness(Red,Green,Blue: TD3DValue) : TD3DValue;

procedure SetBrightness(var Red,Green,Blue: TD3DValue; Brightness: TD3DValue);

function ReleaseObj(var Obj) : boolean;

function ReleaseCOM(var COM) : boolean;

procedure ReleaseCOMe(var COM);

function AddCOM(const COM) : pointer;

function DXErrorstring(Value: HResult) : string;

procedure DXCheck ( Value: HResult );

function GetFrameBox(Frame: IDirect3DRMFrame; var FrameBox: TD3DRMBox) : boolean;

procedure SM(Message: string);

function LoadPaletteFromJASCFile(Filename: string; var Palette: TColorTable) : boolean;

procedure ClearSurface(Surface: IDirectDrawSurface; Color: integer);

function GetBitsPerPixelFromBitmap(Bitmap: Graphics.TBitmap) : integer;

procedure ReadOnlyProperty;

procedure NotReady;




implementation



////////////////////////////////////////////////////////////////////////////////
// TStack
////////////////////////////////////////////////////////////////////////////////


procedure TStack.SetCount(Value: integer);
begin
  if (Value <> FCount) and (Value >= 0) and
     (Value < (FStackSize div FItemSize -1)) then
  begin
    FCount := Value;
    integer(FTop) := FCount * FItemSize;
  end;
end;

procedure TStack.SetTop(Value: pointer);
begin
  if (integer(Value) >= integer(FBase)) and
     (integer(Value) < integer(FStackEnd)) then FTop := Value;
end;

procedure TStack.SetIncAmount(Value: integer);
begin
  if (Value > 0) and (Value >= FIncAmount) and (Value <= 1024) then
    FIncAmount := Value
  else
    FIncAmount := FItemSize * 4;
end;

procedure TStack.SetStackSize(Value: integer);
begin
  if (integer(FBase) + Value) < integer(FStackEnd) then exit;
  integer(FStackEnd) := integer(FBase) + Value - FItemSize;
  FStackSize := Value;
  ReallocMem(FBase,FStackSize);
end;

function TStack.GetStackSize : integer;
begin
  Result := integer(FStackEnd) + FItemSize - integer(FBase);
end;

procedure TStack.Reset;
begin
  FStackSize := FIncAmount;
  ReallocMem(FBase,FStackSize);
  integer(FStackEnd) := integer(FBase) + FStackSize - FItemSize;
  FTop := FBase;
  FCount := 0;
end;

constructor TStack.Create(ItemSize: integer);
begin
  inherited Create;
  FItemSize := ItemSize;
  SetIncAmount(-1);
  FBase := nil;
  FStackEnd := nil;
  Reset;
end;

constructor TStack.CreateForByte;
begin
  Create(SizeOf(Byte));
end;

constructor TStack.CreateForInteger;
begin
  Create(SizeOf(Integer));
end;

constructor TStack.CreateForPointer;
begin
  Create(SizeOf(Pointer));
end;

constructor TStack.CreateForSingle;
begin
  Create(SizeOf(Single));
end;

constructor TStack.CreateForDouble;
begin
  Create(SizeOf(Double));
end;

constructor TStack.CreateForMatrix1D;
begin
  Create(SizeOf(TMatrix1D));
end;

constructor TStack.CreateForMatrix4D;
begin
  Create(SizeOf(TMatrix4D));
end;

procedure TStack.Push(const Item);
begin
  Move(Item,Top^,FItemSize);
  Inc(FCount);
  Inc(integer(FTop),FItemSize);
  if integer(Top) > integer(FStackEnd) then IncStackSize;
end;

procedure TStack.IncStackSize;
begin
  Inc(FStackSize,FIncAmount);
  Inc(integer(FStackEnd),FIncAmount);
  ReAllocMem(FBase,FStackSize);
end;

function TStack.Pop : pointer;
begin
  if FCount > 0 then
  begin
    Dec(FCount);
    Dec(integer(FTop),FItemSize);
  end;
  Result := FTop;
end;

procedure TStack.Increase;
begin
  Inc(FCount);
  Inc(integer(FTop),FItemSize);
  if integer(Top) > integer(FStackEnd) then IncStackSize;
end;

procedure TStack.Decrease;
begin
  Pop;
end;

procedure TStack.IncreaseMulti(ItemCount: integer);
begin
  Inc(FCount,ItemCount);
  Inc(integer(FTop),ItemCount*FItemSize);
  if integer(Top) > integer(FStackEnd) then IncStackSize;
end;

procedure TStack.DecreaseMulti(ItemCount: integer);
begin
  Dec(FCount,ItemCount);
  Dec(integer(FTop),ItemCount*FItemSize);
  if (FCount < 0) or ((integer(FTop)-integer(FBase)) < 0) then Reset;
end;

destructor TStack.Destroy;
begin
  ReAllocMem(FBase,0);
  inherited;
end;

procedure TStack.PushByte(Item: byte);
begin
  byte(Top^) := Item;
  Inc(FCount);
  Inc(integer(FTop),SizeOf(Item));
  if integer(Top) > integer(FStackEnd) then IncStackSize;
end;

function TStack.PopByte : byte;
begin
  if FCount > 0 then
  begin
    Dec(FCount);
    Dec(integer(FTop),SizeOf(Result));
  end;
  Result := byte(FTop^);
end;

procedure TStack.PushInteger(Item: integer);
begin
  integer(Top^) := Item;
  Inc(FCount);
  Inc(integer(FTop),SizeOf(Item));
  if integer(Top) > integer(FStackEnd) then IncStackSize;
end;

function TStack.PopInteger : integer;
begin
  if FCount > 0 then
  begin
    Dec(FCount);
    Dec(integer(FTop),SizeOf(Result));
  end;
  Result := integer(FTop^);
end;

procedure TStack.PushPointer(Item: pointer);
begin
  pointer(Top^) := Item;
  Inc(FCount);
  Inc(integer(FTop),SizeOf(Item));
  if integer(Top) > integer(FStackEnd) then IncStackSize;
end;

function TStack.PopPointer : pointer;
begin
  if FCount > 0 then
  begin
    Dec(FCount);
    Dec(integer(FTop),SizeOf(Result));
  end;
  Result := pointer(FTop^);
end;

procedure TStack.PushSingle(Item: single);
begin
  single(Top^) := Item;
  Inc(FCount);
  Inc(integer(FTop),SizeOf(Item));
  if integer(Top) > integer(FStackEnd) then IncStackSize;
end;

function TStack.PopSingle : single;
begin
  if FCount > 0 then
  begin
    Dec(FCount);
    Dec(integer(FTop),SizeOf(Result));
  end;
  Result := single(FTop^);
end;

procedure TStack.PushMatrix1D(Item: TMatrix1D);
begin
  TMatrix1D(Top^) := Item;
  Inc(FCount);
  Inc(integer(FTop),SizeOf(Item));
  if integer(Top) > integer(FStackEnd) then IncStackSize;
end;

function TStack.PopMatrix1D : TMatrix1D;
begin
  if FCount > 0 then
  begin
    Dec(FCount);
    Dec(integer(FTop),SizeOf(Result));
  end;
  Result := TMatrix1D(FTop^);
end;

procedure TStack.PushMatrix4D(Item: TMatrix4D);
begin
  TMatrix4D(Top^) := Item;
  Inc(FCount);
  Inc(integer(FTop),SizeOf(Item));
  if integer(Top) > integer(FStackEnd) then IncStackSize;
end;

function TStack.PopMatrix4D : TMatrix4D;
begin
  if FCount > 0 then
  begin
    Dec(FCount);
    Dec(integer(FTop),SizeOf(Result));
  end;
  Result := TMatrix4D(FTop^);
end;

////////////////////////////////////////////////////////////////////////////////
// DXTools
////////////////////////////////////////////////////////////////////////////////

function ArcTan2(Y, X: Extended): Extended;
asm
        FLD     Y
        FLD     X
        FPATAN
        FWAIT
end;

function TransformationYZ(y, z: TD3DVector) : TD3DMatrix;
var
  ret : TD3DMatrix;
begin
  with y do
    if X <> 0.0 then
      ret := RotateZMatrix( ArcTan2(Y,X) )
    else
      ret := IdentityMatrix;

  with z do
    if Z <> 0.0 then
      ret := MatrixMul(ret, RotateYMatrix( ArcTan2(X,Z) ));

  with y do
    if Z <> 0.0 then
      ret := MatrixMul(ret, RotateXMatrix( ArcTan2(Y,Z) ));

  Result := ret;
end;

function ProjectionMatrix(near_plane,     // distance to near clipping plane
                          far_plane,      // distance to far clipping plane
           fov: TD3DValue) : TD3DMatrix;    // field of view angle, in radians
var
  c, s, Q : TD3DValue;
  ret : TD3DMatrix;
begin
    c := cos(fov*0.5);
    s := sin(fov*0.5);
    Q := s/(1.0 - near_plane/far_plane);

    ret := ZeroMatrix;
    ret._11 := c;
    ret._22 := c;
    ret._33 := Q;

    ret._43 := -Q*near_plane;
    ret._34 := s;
    result := ret;
end;

function TranslateMatrix(dx, dy, dz: TD3DValue) : TD3DMatrix;
var
  ret : TD3DMatrix;
begin
    ret := IdentityMatrix;
    ret._41 := dx;
    ret._42 := dy;
    ret._43 := dz;
    result := ret;
end;

function RotateXMatrix(rads: TD3DValue) : TD3DMatrix;
var
  ret : TD3DMatrix;
  cosine, sine : TD3DValue;
begin
    cosine := cos(rads);
    sine := sin(rads);
    ret := IdentityMatrix;
    ret._22 := cosine;
    ret._33 := cosine;
    ret._23 := -sine;
    ret._32 := sine;
    result := ret;
end;

function RotateYMatrix(rads: TD3DValue) : TD3DMatrix;
var
  ret : TD3DMatrix;
  cosine, sine : TD3DValue;
begin
    cosine := cos(rads);
    sine := sin(rads);
    ret := IdentityMatrix;
    ret._11 := cosine;
    ret._33 := cosine;
    ret._13 := sine;
    ret._31 := -sine;
    result := ret;
end;

function RotateZMatrix(rads: TD3DValue) : TD3DMatrix;
var
  ret : TD3DMatrix;
  cosine, sine : TD3DValue;
begin
    cosine := cos(rads);
    sine := sin(rads);
    ret := IdentityMatrix;
    ret._11 := cosine;
    ret._22 := cosine;
    ret._12 := -sine;
    ret._21 := sine;
    result := ret;
end;

function ScaleMatrix(size: TD3DValue) : TD3DMatrix;
var
  ret : TD3DMatrix;
begin
    ret := IdentityMatrix;
    ret._11 := size;
    ret._22 := size;
    ret._33 := size;
    result := ret;
end;

function ViewMatrix(from,                  // camera location
                    at,                    // camera look-at target
                    world_up: TD3DVector;  // world's up, usually 0, 1, 0
                    roll: TD3DValue) : TD3DMatrix; // clockwise roll around
                                                 //    viewing direction,
                                                 //    in radians
var
  view : TD3DMatrix;
  up, right, view_dir : TD3DVector;
begin
    view := IdentityMatrix;

    view_dir := VectorNormalize(VectorSub(at,from));
    right := VectorCrossProduct(world_up, view_dir);
    up := VectorCrossProduct(view_dir, right);

    right := VectorNormalize(right);
    up := VectorNormalize(up);

    view._11 := right.x;
    view._21 := right.y;
    view._31 := right.z;
    view._12 := up.x;
    view._22 := up.y;
    view._32 := up.z;
    view._13 := view_dir.x;
    view._23 := view_dir.y;
    view._33 := view_dir.z;

    view._41 := -VectorDotProduct(right, from);

    view._42 := -VectorDotProduct(up, from);
    view._43 := -VectorDotProduct(view_dir, from);

    if roll <> 0.0 then
        // MatrixMult function shown below
        view := MatrixMul(RotateZMatrix(-roll), TD3DMatrix(view));

    result := view;
end;

// Multiplies two matrices.
function MatrixMul(a, b: TD3DMatrix) : TD3DMatrix;
var
  ret : TMatrix4D;
  i,j,k : integer;
begin
  ret.D3DMatrix := ZeroMatrix;
  for i := 0 to 3 do
    for j := 0 to 3 do
      for k := 0 to 3 do
        ret._[i,j] := ret._[i,j] + (TD3DMatrix_(a)[k,j] * TD3DMatrix_(b)[i,k]);
  result := ret.D3DMatrix;
end;


function GetBitsPerPixelFromBitmap(Bitmap: Graphics.TBitmap) : integer;
var
  bm : Windows.TBitmap;
begin
  if GetObject(Bitmap.Handle, sizeof(bm), @bm) = 0 then Result := 0
    else Result := bm.bmBitsPixel;
end;

procedure InitRecord(var DXRecord; Size: integer);
begin
  ZeroMemory(@DXRecord,Size);
  DWORD(DXRecord) := Size;
end;

function GetDDFromDevice2(Device2: IDirect3DDevice2) : IDirectDraw;
const
  DirectDraw : IDirectDraw = nil;
  Target : IDirectDrawSurface = nil;
  Target2 : IDirectDrawSurface2 = nil;
begin
  try
    // get the render target (we need it to get the IDirectDraw)
    DxCheck( Device2.GetRenderTarget(Target) );
    // get the DirectDraw object, but first we need a IDirectDrawSurface2
    DxCheck( Target.QueryInterface(IID_IDirectDrawSurface2,Target2) );
    DxCheck( Target2.GetDDInterface(DirectDraw) );
  finally
    ReleaseCOM( Target );
    ReleaseCOM( Target2 );
    Result := DirectDraw;
  end
end;

procedure ReadOnlyProperty;
begin
  if Exceptions then Exception.Create('Property is Read-Only !');
end;

procedure NotReady;
begin
  if Exceptions then Exception.Create('Not implemented, yet !');
end;

procedure ClearSurface(Surface: IDirectDrawSurface; Color: integer);
var
  bltfx : TDDBltFX;
begin
  InitRecord(bltfx,sizeof(bltfx));
  bltfx.dwFillColor := Color;
  dxCheck( Surface.Blt(nil,nil,nil,DDBLT_COLORFILL + DDBLT_WAIT,bltfx) );
end;

constructor DirectXException.Create(Error: integer);
begin
  inherited Create(#13+#13+DXErrorstring(Error)+#13+#13+'At: '+DXStat+#13+#13);
end;

function LoadPaletteFromJASCFile(Filename: string; var Palette: TColorTable) : boolean;
var
  f : text;
  i : integer;
  s : string;
  b : byte;
  Code : integer;

procedure ReadWd;
var
  c : AnsiChar;
begin
  s := '';
  repeat
    read(f,c);
    if c <> ' ' then s := s + c;
  until c = ' ';
end;

label
  ende;
begin
  Result := false;
  assign(f,Filename);
  {$i-}  reset(f);
  if ioResult <> 0 then goto ende;
  readln(f,s);
  readln(f,s);
  readln(f,s);
  for i := 0 to 255 do begin
    ReadWd;
    Val(s,b,Code);
    if Code <> 0 then goto ende;
    Palette[i].R := b;
    ReadWd;
    Val(s,b,Code);
    if Code <> 0 then goto ende;
    Palette[i].G := b;
    ReadLn(f,s);
    Val(s,b,Code);
    if Code <> 0 then goto ende;
    Palette[i].B := b;
    Palette[i].A := PC_EXPLICIT;
  end;
  Result := true;
ende:
  close(f); {$I+}
end;

function GetBrightness(Red,Green,Blue: TD3DValue) : TD3DValue;
begin
  Result := (Red * 0.3) + (Green * 0.59) + (Blue * 0.11);
end;

procedure SetBrightness(var Red,Green,Blue: TD3DValue; Brightness: TD3DValue);
// var  factor : TD3DValue;
begin
// Setzt entsprechenden Grauton:
  Red := Brightness;
  Green := Brightness;
  Blue := Brightness;
//Behlt Farbe bei Helligkeitsnderung bei:
{  if GetBrightness(Red,Green,Blue) = 0.0 then begin
    Red := 0.0;
    Green := 0.0;
    Blue := 0.0;
  end else begin
    factor := Brightness / GetBrightness(Red,Green,Blue);
    Red := Red * factor;
    if Red > 1.0 then Red := 1.0;
    Green := Green * factor;
    if Green > 1.0 then Green := 1.0;
    Blue := Blue * factor;
    if Blue > 1.0 then Blue := 1.0;
  end;}
end;

procedure SM(Message: string);
begin
  MessageBox(0,PChar(Message),'DirectX-Application:',MB_APPLMODAL);
end;

function AddCOM(const COM) : pointer;
begin
{$IFDEF D2COM}
  if Assigned( IUnknown(COM) ) then IUnknown(COM).AddRef;
{$ELSE}
  if Assigned( IUnknown(COM) ) then IUnknown(COM)._AddRef;
{$ENDIF}
  Result := pointer(COM);
end;

function ReleaseObj(var Obj) : boolean;
begin
  if assigned( TObject(Obj) ) then
    begin
      TObject(Obj).Destroy;
      TObject(Obj) := nil;
      Result := True;
    end
  else
    Result := False;
end;

function ReleaseCOM(var COM) : boolean;  // Interfaceobjekt freigeben
begin
  if Assigned( IUnknown(COM) ) then // wenn Zeigerwert nicht nil dann:
    begin
{$IFDEF D2COM}
      IUnknown(COM).Release;        // Referenzzhler um eins erniedrigen
{$ELSE}
      IUnknown(COM)._Release;       // Referenzzhler um eins erniedrigen
{$ENDIF}
      IUnknown(COM) := nil;         // Zeiger auf null setzt,
      Result := True;
    end     // um weitere versehentlicher Zugriffe zu vermeiden
  else
    Result := false;
end;

procedure ReleaseCOMe(var COM);
begin
  if Assigned( IUnknown(COM) ) then
    begin
{$IFDEF D2COM}
       IUnknown(COM).Release;
{$ELSE}
       IUnknown(COM)._Release;
{$ENDIF}
       IUnknown(COM) := nil;
    end
  else
    raise Exception.Create(DXStat+#13+'ReleaseCOM of NULL object');
end;

function GetFrameBox(Frame: IDirect3DRMFrame; var FrameBox: TD3DRMBox) : boolean;
const
  Visuals : IDirect3DRMVisualArray = nil;
  Visual  : IDirect3DRMVisual = nil;
  Mesh    : IDirect3DRMMesh = nil;
  Meshbuilder : IDirect3DRMMeshbuilder = nil;
var
  Box : TD3DRMBox;
  i,n : integer;
begin
  with FrameBox do begin
    with min do begin
      x := 0;
      y := 0;
      z := 0;
    end;
    with max do begin
      x := 0;
      y := 0;
      z := 0;
    end;
  end;
  Result := false;
  if not assigned(Frame) then exit;
  dxCheck( Frame.GetVisuals(Visuals) );
  n := Visuals.GetSize;
  if n = 0 then exit;
  for i := 0 to n-1 do begin
    Result := false;
    dxCheck( Visuals.GetElement(i,Visual) );
    if Visual.QueryInterface(IID_IDirect3DRMMesh,Mesh) = D3D_OK then begin
      dxCheck( Mesh.GetBox(Box) );
      ReleaseCOMe( Mesh );
      Result := true;
    end
    else if Visual.QueryInterface(IID_IDirect3DRMMeshbuilder,Meshbuilder) = D3D_OK then begin
      dxCheck( Meshbuilder.GetBox(Box) );
      ReleaseCOMe( Meshbuilder );
      Result := true;
    end;
    ReleaseCOMe( Visual );
    if Box.min.x < FrameBox.min.x then FrameBox.min.x := Box.min.x;
    if Box.min.y < FrameBox.min.y then FrameBox.min.y := Box.min.y;
    if Box.min.z < FrameBox.min.z then FrameBox.min.z := Box.min.z;
    if Box.max.x > FrameBox.max.x then FrameBox.max.x := Box.max.x;
    if Box.max.y > FrameBox.max.y then FrameBox.max.y := Box.max.y;
    if Box.max.z > FrameBox.max.z then FrameBox.max.z := Box.max.z;
  end;
  ReleaseCOM( Visuals );
end;

function DXErrorstring(Value: HResult) : string;
begin
    Case Value of
        0: Result :=  'No error';
        DDERR_ALREADYINITIALIZED: Result := 'This object is already initialized.';
        DDERR_BLTFASTCANTCLIP: Result := ' if a clipper object is attached to the source surface passed into a BltFast call.';
        DDERR_CANNOTATTACHSURFACE: Result := 'This surface can not be attached to the requested surface.';
        DDERR_CANNOTDETACHSURFACE: Result := 'This surface can not be detached from the requested surface.';
        DDERR_CANTCREATEDC: Result := 'Windows can not create any more DCs.';
        DDERR_CANTDUPLICATE: Result := 'Cannot duplicate primary & 3D surfaces, or surfaces that are implicitly created.';
        DDERR_CLIPPERISUSINGHWND: Result := 'An attempt was made to set a cliplist for a clipper object that is already monitoring an hwnd.';
        DDERR_COLORKEYNOTSET: Result := 'No src color key specified for this operation.';
        DDERR_CURRENTLYNOTAVAIL: Result := 'Support is currently not available.';
        DDERR_DIRECTDRAWALREADYCREATED: Result := 'A DirectDraw object representing this driver has already been created for this process.';
        DDERR_EXCEPTION: Result := 'An exception was encountered while performing the requested operation.';
        DDERR_EXCLUSIVEMODEALREADYSET: Result := 'An attempt was made to set the cooperative level when it was already set to exclusive.';
        DDERR_GENERIC: Result := 'Generic failure.';
        DDERR_HEIGHTALIGN: Result := 'Height of rectangle provided is not a multiple of reqd alignment.';
        DDERR_HWNDALREADYSET: Result := 'The CooperativeLevel HWND has already been set. It can not be reset while the process has surfaces or palettes created.';
        DDERR_HWNDSUBCLASSED: Result := 'HWND used by DirectDraw CooperativeLevel has been subclassed, this prevents DirectDraw from restoring state.';
        DDERR_IMPLICITLYCREATED: Result := 'This surface can not be restored because it is an implicitly created surface.';
        DDERR_INCOMPATIBLEPRIMARY: Result := 'Unable to match primary surface creation request with existing primary surface.';
        DDERR_INVALIDCAPS: Result := 'One or more of the caps bits passed to the callback are incorrect.';
        DDERR_INVALIDCLIPLIST: Result := 'DirectDraw does not support the provided cliplist.';
        DDERR_INVALIDDIRECTDRAWGUID: Result := 'The GUID passed to DirectDrawCreate is not a valid DirectDraw driver identifier.';
        DDERR_INVALIDMODE: Result := 'DirectDraw does not support the requested mode.';
        DDERR_INVALIDOBJECT: Result := 'DirectDraw received a pointer that was an invalid DIRECTDRAW object.';
        DDERR_INVALIDPARAMS: Result := 'One or more of the parameters passed to the function are incorrect.';
        DDERR_INVALIDPIXELFORMAT: Result := 'The pixel format was invalid as specified.';
        DDERR_INVALIDPOSITION: Result := 'Returned when the position of the overlay on the destination is no longer legal for that destination.';
        DDERR_INVALIDRECT: Result := 'Rectangle provided was invalid.';
        DDERR_LOCKEDSURFACES: Result := 'Operation could not be carried out because one or more surfaces are locked.';
        DDERR_NO3D: Result := 'There is no 3D present.';
        DDERR_NOALPHAHW: Result := 'Operation could not be carried out because there is no alpha accleration hardware present or available.';
        DDERR_NOBLTHW: Result := 'No blitter hardware present.';
        DDERR_NOCLIPLIST: Result := 'No cliplist available.';
        DDERR_NOCLIPPERATTACHED: Result := 'No clipper object attached to surface object.';
        DDERR_NOCOLORCONVHW: Result := 'Operation could not be carried out because there is no color conversion hardware present or available.';
        DDERR_NOCOLORKEY: Result := 'Surface does not currently have a color key';
        DDERR_NOCOLORKEYHW: Result := 'Operation could not be carried out because there is no hardware support of the destination color key.';
        DDERR_NOCOOPERATIVELEVELSET: Result := 'Create function called without DirectDraw object method SetCooperativeLevel being called.';
        DDERR_NODC: Result := 'No DC was ever created for this surface.';
        DDERR_NODDROPSHW: Result := 'No DirectDraw ROP hardware.';
        DDERR_NODIRECTDRAWHW: Result := 'A hardware-only DirectDraw object creation was attempted but the driver did not support any hardware.';
        DDERR_NOEMULATION: Result := 'Software emulation not available.';
        DDERR_NOEXCLUSIVEMODE: Result := 'Operation requires the application to have exclusive mode but the application does not have exclusive mode.';
        DDERR_NOFLIPHW: Result := 'Flipping visible surfaces is not supported.';
        DDERR_NOGDI: Result := 'There is no GDI present.';
        DDERR_NOHWND: Result := 'Clipper notification requires an HWND or no HWND has previously been set as the CooperativeLevel HWND.';
        DDERR_NOMIRRORHW: Result := 'Operation could not be carried out because there is no hardware present or available.';
        DDERR_NOOVERLAYDEST: Result := 'Returned when GetOverlayPosition is called on an overlay that UpdateOverlay has never been called on to establish a destination.';
        DDERR_NOOVERLAYHW: Result := 'Operation could not be carried out because there is no overlay hardware present or available.';
        DDERR_NOPALETTEATTACHED: Result := 'No palette object attached to this surface.';
        DDERR_NOPALETTEHW: Result := 'No hardware support for 16 or 256 color palettes.';
        DDERR_NORASTEROPHW: Result := 'Operation could not be carried out because there is no appropriate raster op hardware present or available.';
        DDERR_NOROTATIONHW: Result := 'Operation could not be carried out because there is no rotation hardware present or available.';
        DDERR_NOSTRETCHHW: Result := 'Operation could not be carried out because there is no hardware support for stretching.';
        DDERR_NOT4BITCOLOR: Result := 'DirectDrawSurface is not in 4 bit color palette and the requested operation requires 4 bit color palette.';
        DDERR_NOT4BITCOLORINDEX: Result := 'DirectDrawSurface is not in 4 bit color index palette and the requested operation requires 4 bit color index palette.';
        DDERR_NOT8BITCOLOR: Result := 'DirectDrawSurface is not in 8 bit color mode and the requested operation requires 8 bit color.';
        DDERR_NOTAOVERLAYSURFACE: Result := 'Returned when an overlay member is called for a non-overlay surface.';
        DDERR_NOTEXTUREHW: Result := 'Operation could not be carried out because there is no texture mapping hardware present or available.';
        DDERR_NOTFLIPPABLE: Result := 'An attempt has been made to flip a surface that is not flippable.';
        DDERR_NOTFOUND: Result := 'Requested item was not found.';
        DDERR_NOTLOCKED: Result := 'Surface was not locked.  An attempt to unlock a surface that was not locked at all, or by this process, has been attempted.';
        DDERR_NOTPALETTIZED: Result := 'The surface being used is not a palette-based surface.';
        DDERR_NOVSYNCHW: Result := 'Operation could not be carried out because there is no hardware support for vertical blank synchronized operations.';
        DDERR_NOZBUFFERHW: Result := 'Operation could not be carried out because there is no hardware support for zbuffer blitting.';
        DDERR_NOZOVERLAYHW: Result := 'Overlay surfaces could not be z layered based on their BltOrder because the hardware does not support z layering of overlays.';
        DDERR_OUTOFCAPS: Result := 'The hardware needed for the requested operation has already been allocated.';
        DDERR_OUTOFMEMORY: Result := 'DirectDraw does not have enough memory to perform the operation.';
        DDERR_OUTOFVIDEOMEMORY: Result := 'DirectDraw does not have enough memory to perform the operation.';
        DDERR_OVERLAYCANTCLIP: Result := 'The hardware does not support clipped overlays.';
        DDERR_OVERLAYCOLORKEYONLYONEACTIVE: Result := 'Can only have ony color key active at one time for overlays.';
        DDERR_OVERLAYNOTVISIBLE: Result := 'Returned when GetOverlayPosition is called on a hidden overlay.';
        DDERR_PALETTEBUSY: Result := 'Access to this palette is being refused because the palette is already locked by another thread.';
        DDERR_PRIMARYSURFACEALREADYEXISTS: Result := 'This process already has created a primary surface.';
        DDERR_REGIONTOOSMALL: Result := 'Region passed to Clipper::GetClipList is too small.';
        DDERR_SURFACEALREADYATTACHED: Result := 'This surface is already attached to the surface it is being attached to.';
        DDERR_SURFACEALREADYDEPENDENT: Result := 'This surface is already a dependency of the surface it is being made a dependency of.';
        DDERR_SURFACEBUSY: Result := 'Access to this surface is being refused because the surface is already locked by another thread.';
        DDERR_SURFACEISOBSCURED: Result := 'Access to surface refused because the surface is obscured.';
        DDERR_SURFACELOST: Result := 'Access to this surface is being refused because the surface memory is gone. The DirectDrawSurface object representing this surface should have Restore called on it.';
        DDERR_SURFACENOTATTACHED: Result := 'The requested surface is not attached.';
        DDERR_TOOBIGHEIGHT: Result := 'Height requested by DirectDraw is too large.';
        DDERR_TOOBIGSIZE: Result := 'Size requested by DirectDraw is too large, but the individual height and width are OK.';
        DDERR_TOOBIGWIDTH: Result := 'Width requested by DirectDraw is too large.';
        DDERR_UNSUPPORTED: Result := 'Action not supported.';
        DDERR_UNSUPPORTEDFORMAT: Result := 'FOURCC format requested is unsupported by DirectDraw.';
        DDERR_UNSUPPORTEDMASK: Result := 'Bitmask in the pixel format requested is unsupported by DirectDraw.';
        DDERR_VERTICALBLANKINPROGRESS: Result := 'Vertical blank is in progress.';
        DDERR_WASSTILLDRAWING: Result := 'Informs DirectDraw that the previous Blt which is transfering information to or from this Surface is incomplete.';
        DDERR_WRONGMODE: Result := 'This surface can not be restored because it was created in a different mode.';
        DDERR_XALIGN: Result := 'Rectangle provided was not horizontally aligned on required boundary.';

        D3DERR_BADMAJORVERSION: Result := 'D3DERR_BADMAJORVERSION';
        D3DERR_BADMINORVERSION: Result := 'D3DERR_BADMINORVERSION';
(*
 * An invalid device was requested by the application.
 *)
        D3DERR_INVALID_DEVICE: Result := 'D3DERR_INITFAILED';
        D3DERR_INITFAILED: Result := 'D3DERR_INITFAILED';
(*
 * SetRenderTarget attempted on a device that was
 * QI'd off the render target.
 *)
        D3DERR_DEVICEAGGREGATED: Result := 'D3DERR_DEVICEAGGREGATED';

        D3DERR_EXECUTE_CREATE_FAILED: Result := 'D3DERR_EXECUTE_CREATE_FAILED';
        D3DERR_EXECUTE_DESTROY_FAILED: Result := 'D3DERR_EXECUTE_DESTROY_FAILED';
        D3DERR_EXECUTE_LOCK_FAILED: Result := 'D3DERR_EXECUTE_LOCK_FAILED';
        D3DERR_EXECUTE_UNLOCK_FAILED: Result := 'D3DERR_EXECUTE_UNLOCK_FAILED';
        D3DERR_EXECUTE_LOCKED: Result := 'D3DERR_EXECUTE_LOCKED';
        D3DERR_EXECUTE_NOT_LOCKED: Result := 'D3DERR_EXECUTE_NOT_LOCKED';

        D3DERR_EXECUTE_FAILED: Result := 'D3DERR_EXECUTE_FAILED';
        D3DERR_EXECUTE_CLIPPED_FAILED: Result := 'D3DERR_EXECUTE_CLIPPED_FAILED';

        D3DERR_TEXTURE_NO_SUPPORT: Result := 'D3DERR_TEXTURE_NO_SUPPORT';
        D3DERR_TEXTURE_CREATE_FAILED: Result := 'D3DERR_TEXTURE_CREATE_FAILED';
        D3DERR_TEXTURE_DESTROY_FAILED: Result := 'D3DERR_TEXTURE_DESTROY_FAILED';
        D3DERR_TEXTURE_LOCK_FAILED: Result := 'D3DERR_TEXTURE_LOCK_FAILED';
        D3DERR_TEXTURE_UNLOCK_FAILED: Result := 'D3DERR_TEXTURE_UNLOCK_FAILED';
        D3DERR_TEXTURE_LOAD_FAILED: Result := 'D3DERR_TEXTURE_LOAD_FAILED';
        D3DERR_TEXTURE_SWAP_FAILED: Result := 'D3DERR_TEXTURE_SWAP_FAILED';
        D3DERR_TEXTURE_LOCKED: Result := 'D3DERR_TEXTURELOCKED';
        D3DERR_TEXTURE_NOT_LOCKED: Result := 'D3DERR_TEXTURE_NOT_LOCKED';
        D3DERR_TEXTURE_GETSURF_FAILED: Result := 'D3DERR_TEXTURE_GETSURF_FAILED';

        D3DERR_MATRIX_CREATE_FAILED: Result := 'D3DERR_MATRIX_CREATE_FAILED';
        D3DERR_MATRIX_DESTROY_FAILED: Result := 'D3DERR_MATRIX_DESTROY_FAILED';
        D3DERR_MATRIX_SETDATA_FAILED: Result := 'D3DERR_MATRIX_SETDATA_FAILED';
        D3DERR_MATRIX_GETDATA_FAILED: Result := 'D3DERR_MATRIX_GETDATA_FAILED';
        D3DERR_SETVIEWPORTDATA_FAILED: Result := 'D3DERR_SETVIEWPORTDATA_FAILED';

        D3DERR_INVALIDCURRENTVIEWPORT: Result := 'D3DERR_INVALIDCURRENTVIEWPORT';
        D3DERR_INVALIDPRIMITIVETYPE: Result := 'D3DERR_INVALIDPRIMITIVETYPE';
        D3DERR_INVALIDVERTEXTYPE: Result := 'D3DERR_INVALIDVERTEXTYPE';
        D3DERR_TEXTURE_BADSIZE: Result := 'D3DERR_TEXTURE_BADSIZE';
        D3DERR_INVALIDRAMPTEXTURE: Result := 'D3DERR_INVALIDRAMPTEXTURE';

        D3DERR_MATERIAL_CREATE_FAILED: Result := 'D3DERR_MATERIAL_CREATE_FAILED';
        D3DERR_MATERIAL_DESTROY_FAILED: Result := 'D3DERR_MATERIAL_DESTROY_FAILED';
        D3DERR_MATERIAL_SETDATA_FAILED: Result := 'D3DERR_MATERIAL_SETDATA_FAILED';
        D3DERR_MATERIAL_GETDATA_FAILED: Result := 'D3DERR_MATERIAL_GETDATA_FAILED';
        D3DERR_INVALIDPALETTE: Result := 'D3DERR_INVALIDPALETTE';

        D3DERR_ZBUFF_NEEDS_SYSTEMMEMORY: Result := 'D3DERR_ZBUFF_NEEDS_SYSTEMMEMORY';
        D3DERR_ZBUFF_NEEDS_VIDEOMEMORY: Result := 'D3DERR_ZBUFF_NEEDS_VIDEOMEMORY';
        D3DERR_SURFACENOTINVIDMEM: Result := 'D3DERR_SURFACENOTINVIDMEM';

        D3DERR_LIGHT_SET_FAILED: Result := 'D3DERR_LIGHT_SET_FAILED';
        D3DERR_LIGHTHASVIEWPORT: Result := 'D3DERR_LIGHTHASVIEWPORT';
        D3DERR_LIGHTNOTINTHISVIEWPORT: Result := 'D3DERR_LIGHTNOTINTHISVIEWPORT';

        D3DERR_SCENE_IN_SCENE: Result := 'D3DERR_SCENE_IN_SCENE';
        D3DERR_SCENE_NOT_IN_SCENE: Result := 'D3DERR_SCENE_NOT_IN_SCENE';
        D3DERR_SCENE_BEGIN_FAILED: Result := 'D3DERR_SCENE_BEGIN_FAILED';
        D3DERR_SCENE_END_FAILED: Result := 'D3DERR_SCENE_END_FAILED';

        D3DERR_INBEGIN: Result := 'D3DERR_INBEGIN';
        D3DERR_NOTINBEGIN: Result := 'D3DERR_NOTINBEGIN';
        D3DERR_NOVIEWPORTS: Result := 'D3DERR_NOVIEWPORTS';
        D3DERR_VIEWPORTDATANOTSET: Result := 'D3DERR_VIEWPORTDATANOTSET';
        D3DERR_VIEWPORTHASNODEVICE: Result := 'D3DERR_VIEWPORTHASNODEVICE';
        D3DERR_NOCURRENTVIEWPORT: Result := 'D3DERR_NOCURRENTVIEWPORT';


        D3DRMERR_BADOBJECT: Result := 'D3DRMERR_BADOBJECT';
        D3DRMERR_BADTYPE: Result := 'D3DRMERR_BADTYPE';
        D3DRMERR_BADALLOC: Result := 'D3DRMERR_BADALLOC';
        D3DRMERR_FACEUSED: Result := 'D3DRMERR_FACEUSED';
        D3DRMERR_NOTFOUND: Result := 'D3DRMERR_NOTFOUND';
        D3DRMERR_NOTDONEYET: Result := 'D3DRMERR_NOTDONEYET';
        D3DRMERR_FILENOTFOUND: Result := 'The file was not found.';
        D3DRMERR_BADFILE: Result := 'D3DRMERR_BADFILE';
        D3DRMERR_BADDEVICE: Result := 'D3DRMERR_BADDEVICE';
        D3DRMERR_BADVALUE: Result := 'D3DRMERR_BADVALUE';
        D3DRMERR_BADMAJORVERSION: Result := 'D3DRMERR_BADMAJORVERSION';
        D3DRMERR_BADMINORVERSION: Result := 'D3DRMERR_BADMINORVERSION';
        D3DRMERR_UNABLETOEXECUTE: Result := 'D3DRMERR_UNABLETOEXECUTE';
        else Result := 'Unrecognized error value.';
    end;
end;

procedure DXCheck_(Value: HResult); { Check the Result of a COM operation }
var
  s : string;
begin
  if FAILED(Value) then
  begin
    s := IntToHex(Value,8);  // for debugging
    raise DirectXException.Create(Value);
  end;
end;

procedure DXCheck(Value: HResult); { Check the Result of a COM operation }
begin
  if FAILED(Value) then raise DirectXException.Create(Value);
end;

end.
