/* Puzzle Assembler
 * Copyright (C) 2003  Andreas Rver
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <sys/types.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

/* string length for the puzzle piece shape description string
 * you need to have at least xsize*ysize*zsize+1 characters
 */
#define PIECESIZE 300

/* the ct puzzle requires a bit more than 18000 nodes */
#define NODELIMIT 600000

/* the order of pieces is mostly unimportant, except for the
 * very first piece. this is added with removed symmetry of the
 * target shape. for this the first piece must not have andy symmetry
 * at all, or the program will also calc some or all rotatet solutions
 */
#define PIECELIMIT 50


typedef struct {

  /* size of piece */
  int sx, sy, sz;

  /* each character represents one cell, everything except space
   * is considered a filled block
   */
  char desc[PIECESIZE];

} t_piece;

/* the definition of the assembled block
 */
t_piece result;

/* two values that belong to the assembled block: the maximum number 
 * of cubes that may be filled and the number of cubes that may or may
 * not be filled
 *
 * so the minimum number of cubes that need to be filled is
 * res_filled - res_vari
 */
int res_filled;
int res_vari;

int piecenumber;
t_piece pieces[PIECELIMIT];

/* this are the members of the node. One array for each member. This
 * accellerates access.
 *
 * colCount is a shared member. it's column for normal nodes and count for
 * column header nodes
 */
int left[NODELIMIT];
int right[NODELIMIT];
int up[NODELIMIT];
int down[NODELIMIT];
int colCount[NODELIMIT];

/* number of solutions found so far */
int solutions = 0;

/* mask used to only print every n-th solution */
int solutionsmask;

/* time we started searching, used for speed estimation
 */
clock_t start;

int rows[PIECELIMIT];

/* print out one piece, so that it's relatively easy to
 * check if the shape is correct
 */
void printpiece(t_piece p) {
  int x, y, z;

  if (p.desc[0]) {

    for (z = 0; z < p.sz; z++) {
      printf(" +");
      for (x = 0; x < p.sx; x++)
        printf("-");
      printf("+");
    }
    printf("\n");
  
    for (y = 0; y < p.sy; y++) {
      for (z = 0; z < p.sz; z++) {
        printf(" +");
        for (x = 0; x < p.sx; x++)
          if (p.desc[x+p.sx*(y+p.sy*z)] != ' ')
            if (p.desc[x+p.sx*(y+p.sy*z)] == '+')
              printf("+");
            else
              printf("#");
          else
            printf(" ");
        printf("+");
      }
      printf("\n");
    }
  
    for (z = 0; z < p.sz; z++) {
      printf(" +");
      for (x = 0; x < p.sx; x++)
        printf("-");
      printf("+");
    }
    printf("\n");

  } else {
    printf("sx: %i, sy: %i, sz: %i\n", p.sx, p.sy, p.sz);
  }
}

/* load the puzzle from the file
 *
 * the format is quite simple:
 * first line is x y z size of the assembled block separated by space
 * all other lines stand one line for one piece
 * the first 3 number are the dimensions followed by a string in ""
 * in this string everything else esxept for space is asumed to be filled
 * the string is a linear list of all blocks inside the piece block
 * with x the lowest and z the highest dimension. So the formula to get
 * the position of one block is x+xs*(y+ys*z)
 */
bool load(char * fname) {
  FILE * f = fopen(fname, "rb");
  int i;

  piecenumber = 0;

  fscanf(f, "%i %i %i ",
         &result.sx,
         &result.sy,
         &result.sz);

  while (getc(f) != '"');

  for (i = 0; i < result.sx * result.sy * result.sz; i++) {
    char c = getc(f);
    if (c == '"')
      if (i == 0) {
        result.desc[0] = 0;
        break;
      } else {
        printf("  not enough cubes for result shape\n");
        return false;
      }
    else
      result.desc[i] = c;
  }

  getc(f);
  fscanf(f, "\n");

  printpiece(result);

  if (result.desc[0]) {
    for (i = 0; i < result.sx*result.sy*result.sz; i++) {
      if (result.desc[i] != ' ')
        res_filled++;
      if (result.desc[i] == '+')
        res_vari++;
    }
  } else {
    res_filled = result.sx*result.sy*result.sz;
    res_vari = 0;
  }

  while (!feof(f)) {

    if (piecenumber >= PIECELIMIT) {
      printf("  too many pieces\n");
      return false;
    }

    fscanf(f, "%i %i %i ",
           &pieces[piecenumber].sx,
           &pieces[piecenumber].sy,
           &pieces[piecenumber].sz);

    while (getc(f) != '"');

    for (i = 0; i < pieces[piecenumber].sx * pieces[piecenumber].sy * pieces[piecenumber].sz; i++)
      pieces[piecenumber].desc[i] = getc(f);

    getc(f);
    fscanf(f, "\n");

    printpiece(pieces[piecenumber]);

    piecenumber++;
  }

  return true;
}

/* this function compares 2 pieces and returns true, if they
 * are the same shape and same orientation */
bool same_shape (t_piece p1, t_piece p2) {
  int i;

  if (p1.sx != p2.sx) return false;
  if (p1.sy != p2.sy) return false;
  if (p1.sz != p2.sz) return false;

  for (i = 0; i < p1.sx*p1.sy*p1.sz; i++)
    if ((p1.desc[i]==' ') ^ (p2.desc[i]==' ')) return false;

  return true;
}

/* these 3 functions rotate a piece around one axis by 90 degree */
void rotatex(t_piece *p) {
  int x, y, z;

  int tmp = p->sy;
  p->sy = p->sz;
  p->sz = tmp;

  char s[PIECESIZE];

  for (x = 0; x < p->sx; x++)
    for (y = 0; y < p->sy; y++)
      for (z = 0; z < p->sz; z++)
        s[x+p->sx*(y+p->sy*z)] = p->desc[x+p->sx*(z+p->sz*(p->sy-y-1))];

  s[p->sx*p->sy*p->sz] = 0;
  memmove(p->desc, s, PIECESIZE);
}

void rotatey(t_piece *p) {
  int x, y, z;

  int tmp = p->sx;
  p->sx = p->sz;
  p->sz = tmp;

  char s[PIECESIZE];

  for (x = 0; x < p->sx; x++)
    for (y = 0; y < p->sy; y++)
      for (z = 0; z < p->sz; z++)
        s[x+p->sx*(y+p->sy*z)] = p->desc[z+p->sz*(y+p->sy*(p->sx-x-1))];

  s[p->sx*p->sy*p->sz] = 0;
  memmove(p->desc, s, PIECESIZE);
}

void rotatez(t_piece *p) {
  int x, y, z;

  int tmp = p->sy;
  p->sy = p->sx;
  p->sx = tmp;

  char s[PIECESIZE];

  for (x = 0; x < p->sx; x++)
    for (y = 0; y < p->sy; y++)
      for (z = 0; z < p->sz; z++)
        s[x+p->sx*(y+p->sy*z)] = p->desc[y+p->sy*((p->sx-x-1)+p->sx*z)];

  s[p->sx*p->sy*p->sz] = 0;
  memmove(p->desc, s, PIECESIZE);
}

/* all possible rotations of one piece can be generated
 * using this function by iterating nr from 0 to 63 including
 * probably these are too many possibilities, but better be sure
 * double shapes are sorted out automatically
 */
t_piece transform(t_piece p, int nr) {

  int rotx = (nr >> 0) & 0x3;
  int roty = (nr >> 2) & 0x3;
  int rotz = (nr >> 4) & 0x3;

  t_piece erg = p;
  int i;

  for (i = 0; i < rotz; i++) rotatez(&erg);
  for (i = 0; i < roty; i++) rotatey(&erg);
  for (i = 0; i < rotx; i++) rotatex(&erg);

  return erg;
}

/* we have to change the order of the columns for the case
 * that there are variable cubes in th assembled puzzle. We swap
 * the columns so that we first have the piece columns followed by
 * the columns for cubes that need to be filled followed
 * by columns that may be empty
 *
 * this function calcs the index to the column given the
 * x, y and z position for the cube you are looking for
 */
int column(int x, int y, int z) {
  if (result.desc[0]) {
    int c, v;
    c = v = 0;
    for (int i = 0; i < x + result.sx * (y + result.sy * z); i++)
      if (result.desc[i] == '+')
        v++;
      else if (result.desc[i] != ' ')
        c++;

    if (result.desc[x + result.sx * (y + result.sy * z)] == ' ')
      return -1;
    else if (result.desc[x + result.sx * (y + result.sy * z)] == '+')
      return 1 + piecenumber + res_filled - res_vari + v;
    else
      return 1 + piecenumber + c;

  } else
    return piecenumber + 1 + x + result.sx * (y + result.sy * z);
}

/* this function creates the neccessary data structures for the
 * dancing link algorithm */
int prepare(void) {

  int i, p, t, placements;

  /* node 0 is the start node for everything */

  /* nodes 1..n are the columns nodes */
  for (i = 0; i < 1 + piecenumber + res_filled - res_vari; i++) {
    right[i] = i+1;
    left[i+1] = i;
    up[i] = i;
    down[i] = i;
  }

  /* make the linked list cyclic */
  left[0] = result.sx*result.sy*result.sz+piecenumber;
  right[piecenumber + res_filled - res_vari] = 0;

  /* create column nodes for vari columns, these
   * are not inside the column header list and so
   * are not forced to be filled but otherwise behave
   * like normal columns
   */
  for (i = 0; i < res_vari; i++) {
    left[i + piecenumber + 1 + res_filled - res_vari] = i + piecenumber + 1 + res_filled - res_vari;
    right[i + piecenumber + 1 + res_filled - res_vari] = i + piecenumber + 1 + res_filled - res_vari;
    up[i + piecenumber + 1 + res_filled - res_vari] = i + piecenumber + 1 + res_filled - res_vari;
    down[i + piecenumber + 1 + res_filled - res_vari] = i + piecenumber + 1 + res_filled - res_vari;
  }

  /* first free node, we'll start to fill up starting from there */
  int firstfree = 1 + piecenumber + res_filled;

  /* now we insert one shape after another */
  for (p = 0; p < piecenumber; p++) {

    /* this array contains all the pieces found so far, this will help us
     * to not add two times the same piece to the structur */
    t_piece cache[64];
    int cachefill = 0;

    placements = 0;

    /* go through all possible rotations of the piece */
    for (t = 0; t < 64; t++) {
      cache[cachefill] = transform(pieces[p], t);

      /* look into our cache, if we have this transformation already inserted
       * because it was found using a differen kind of transpormation */
      int found = 0;
      for (i = 0; i < cachefill; i++)
        if (same_shape(cache[i], cache[cachefill])) {
          found = 1;
          break;
        }

      /* didn't find piece, so it's new shape, add to cache and add to
       * node structure */
      if (!found) {

        int x, y, z;

        /* find all possible translations of piece and add them */
        for (x = 0; x < result.sx - cache[cachefill].sx + 1; x++)
          for (y = 0; y < result.sy - cache[cachefill].sy + 1; y++)
            for (z = 0; z < result.sz - cache[cachefill].sz + 1; z++) {

              int px, py, pz;

              /* check if the piece fits with this rotation and translation */
              {
                bool dont_fit = false;

                for (pz = 0; (pz < cache[cachefill].sz) && (!dont_fit); pz++)
                  for (py = 0; (py < cache[cachefill].sy) && (!dont_fit); py++)
                    for (px = 0; (px < cache[cachefill].sx) && (!dont_fit); px++)
                      if ((cache[cachefill].desc[px + cache[cachefill].sx * (py + cache[cachefill].sy * pz)] != ' ') &&
                          (result.desc[(x+px) + result.sx * ((y+py) + result.sy * (z+pz))] == ' '))
                        dont_fit = true;

                if (dont_fit) continue;
              }

              placements++;

              /* add one line to structure */
              int piecenode = firstfree++;
              if (firstfree >= NODELIMIT) {
                printf("please increase node limit\n");
                return 0;
              }

              /* first the node in the pieces column */
              left[piecenode] = piecenode;
              right[piecenode] = piecenode;
              down[piecenode] = p+1;
              up[piecenode] = up[p+1];
              down[up[p+1]] = piecenode;
              up[p+1] = piecenode;

              colCount[piecenode] = p+1;
              colCount[p+1]++;

              /* now add the used cubes of the piece */
              for (pz = 0; pz < cache[cachefill].sz; pz++)
                for (py = 0; py < cache[cachefill].sy; py++)
                  for (px = 0; px < cache[cachefill].sx; px++)
                    if (cache[cachefill].desc[px + cache[cachefill].sx * (py + cache[cachefill].sy * pz)] != ' ') {

                      /* lets calc the number of the column the node needs to be added to */
                      int col = column(x+px, y+py, z+pz);

                      /* add node to this column */
                      int newnode = firstfree++;
                      if (firstfree >= NODELIMIT) {
                        printf("please increase node limit\n");
                        return 0;
                      }

                      right[newnode] = piecenode;
                      left[newnode] = left[piecenode];
                      right[left[piecenode]] = newnode;
                      left[piecenode] = newnode;

                      down[newnode] = col;
                      up[newnode] = up[col];
                      down[up[col]] = newnode;
                      up[col] = newnode;

                      colCount[newnode] = col;
                      colCount[col]++;
                    }
            }
        if (p == 0) {
          /* for the first piece we also add a few more transformations to the cache
           * so that we do not add them to the structure, this will remove rotations
           * from the solutions list this list is dependent on the symmetry of the
           * reult shape, for a cube it would contain every possible rotation, but
           * for our brick shaped result we do have only 4 symmetries so we add the
           * missing 3 beseides the one that is already there */

          /* this is code specially for ct puzzle, in other puzzles we need
           * more general symmetry handling by checking the symmetry of the
           * solution and then remove these symmetries from the 1st placed piece
           *
           * it handles two cases cube where all dimensions are equal
           * and brick where no 2 dimensions are equeal
           */
          if (result.sx == result.sy)
            if (result.sx == result.sz)
              break;


          int c = cachefill++;

          cache[cachefill++] = transform(cache[c], 2 );
          cache[cachefill++] = transform(cache[c], 8 );
          cache[cachefill++] = transform(cache[c], 2 + 8 );
        } else
          cachefill++;
      }
    }
    printf("%i placements for piece %i\n", placements, p+1);
  }
  return 1;
}

/* cover one column:
 * - remove the column from the column header node list,
 * - remove all rows where the given column is 1
 */
inline void cover(register int col) {

  {
    register int l, r;

    l = left[col];
    r = right[col];

    left[r] = l;
    right[l] = r;
  }

  register int i = down[col];
  while (i != col) {

    register int j = right[i];
    while (j != i) {

      {
        register int u, d;

        u = up[j];
        d = down[j];

        up[d] = u;
        down[u] = d;
      }

      colCount[colCount[j]]--;

      j = right[j];
    }

    i = down[i];
  }
  
}

/* uncover the given column
 * this is the exact inverse operation of cover. It requires that the
 * maxtrix has the same state as after the corresponding cover.
 * so
 *
 * cover(i); uncover(i);
 *
 * will leave the matrix unchanged
 */
inline void uncover(register int col) {

  register int i = up[col];
  while (i != col) {

    register int j = left[i];
    while (j != i) {

      colCount[colCount[j]]++;

      up[down[j]] = j;
      down[up[j]] = j;

      j = left[j];
    }

    i = up[i];
  }

  left[right[col]] = col;
  right[left[col]] = col;
}

/* 2 helper functions that cover and uncoverone
 * selected row
 */
inline void cover_row(register int r) {
  register int j = right[r];
  while (j != r) {
    cover(colCount[j]);
    j = right[j];
  }
}

inline void uncover_row(register int r) {
  register int j = left[r];
  while (j != r) {
    uncover(colCount[j]);
    j = left[j];
  }
}

/* not useful with the ctPuzzle, but may be very useful with other puzzles
 *
 * the function tires to remove possible piece placementy by checking if, after
 * the piece has peen placed somewhere, that all the other pieces still can be
 * placed and all holes can still be filled.
 * if this is not the case then this placement can be removed
 */
bool checkmatrix(int rec) {

  int col = right[0];

  while (col) {
    /* if there is a column that can not be covered
     * any longer, the whole problem can't be solved
     */
    if (colCount[col] == 0)
      return true;

    /* if one column must be covered in a fixed
     * way cover it and check if the result is solvable
     */
    if (rec && (colCount[col] == 1)) {

      bool ret = false;

      cover(col);
      cover_row(down[col]);

      ret = checkmatrix(rec - 1);

      uncover_row(down[col]);
      uncover(col);

      if (ret) return true;
    }

    col = right[col];
  }

  return false;
}

/* try to reduce possible placements by removing the ones that
 * will inevitable lead to dead ends. this will gain nothing for
 * the ct puzzle but may be helpful with others
 */
void reduce(void) {
  bool remove;
  int removed;
  bool rem_sth;
  int iteration = 1;

  do {

    printf(" iteration %i\n", iteration);
    rem_sth = false;

    for (int p = 0; p < piecenumber; p++) {
  
      cover(p+1);

      removed = 0;
      int r = down[p+1];

      for (int row = 0; row < colCount[p+1]; row++) {
  
        cover_row(r);
        remove = checkmatrix(1);
        uncover_row(r);

        int rr = down[r];
  
        if (remove) {
  
          uncover(p+1);

          int s = r;
  
          do {
            up[down[s]] = up[s];
            down[up[s]] = down[s];
            colCount[colCount[s]]--;
            s = right[s];
          } while (s != r);
  
          row--;
          removed++;
          rem_sth = true;

          cover(p+1);
        }
  
        r = rr;
  
      }
      printf("  removed %i placements of piece %i, %i placements left\n", removed, p+1, colCount[p+1]);
      uncover(p+1);
    }
    iteration++;
  } while (rem_sth);
}

void printstatus(void) {

  printf(" %i solutions found with a rate of %f solutions per second\n",
         solutions,
         1.0*CLOCKS_PER_SEC*solutions / (1.0*clock() - start + 1) );

  int i, x, y, z;
  int r[PIECELIMIT];
  char name[PIECELIMIT];

  for (i = 0; i < piecenumber; i++) {
    r[i] = rows[i];
    while (colCount[r[i]] > piecenumber) r[i] = right[r[i]];
    name[i] = 'a' + colCount[r[i]] - 1;
    r[i] = right[r[i]];
  }

  for (z = 0; z < result.sz; z++) {
    for (y = 0; y < result.sy; y++) {
      for (x = 0; x < result.sx; x++) {
        int c = column(x, y, z);
        if (c == -1)
          printf(" ");
        else {
          int found = 0;

          for (i = 0; i < piecenumber; i++)
            if (colCount[r[i]] == c) {
              printf("%c", name[i]);
              r[i] = right[r[i]];
              found = 1;
              break;
            }
          if (!found)
            printf(" ");
        }
      }
      printf("   ");
    }
    printf("\n");
  }

  printf("\n");
}

/* the recursive search function.
 *
 */
void search(int pos) {

  if (!right[0]) {

   /* one solution found. We could print here
    * information about the solution but we only need
    * to cound so, jus print a dot after 256 solutions */
    solutions++;
    if (!(solutions % solutionsmask))
      printstatus();
  } else {

    /* select a column by using the one with the smallest number of rows set */
    int c = right[0];
    int s = colCount[c];

    register int j = right[c];

    while (j) {

      if (colCount[j] < s) {

        if (!colCount[j]) return;

        c = j;
        s = colCount[c];

      }
      j = right[j];
    }

    /* remove this column and all rows where this column is set from matrix */
    cover(c);

    /* go check each row that our selected column is set */
    int r = down[c];
    while (r != c) {

      rows[pos] = r;

      cover_row(r);
      search(pos + 1);
      uncover_row(r);

      r = down[r];
    }

    uncover(c);
  }
}

int main(int argn, char* args[]) {

  printf("loading...\n");

  if (!load(args[1])) {
    printf("  could not load puzzle\n");
    return 1;
  }

  printf("preparing...\n");

  if (!prepare()) {
    printf("  could not prepare puzzle\n");
    return 1;
  }

  printf("reduce...\n");
  reduce();

  start = clock();

  if (argn == 3)
    solutionsmask = atoi(args[2]);
  else
    solutionsmask = 0x3FF;

  printf("print every %i-th solution\n", solutionsmask);

  printf("searching...\n");
  search(0);

  printf(" %i solutions found with a rate of %f solutions per second\n",
         solutions,
         1.0*CLOCKS_PER_SEC*solutions / (1.0*clock() - start + 1) );

  return 0;
}
