/* neut2slp.c
 * by Arnold Wendl, Frth, anno Domini 2003
 * Das neutrale Lsungsbergabeformat in eine SLP-Datei wandeln
 */
#include <stdio.h>
#include <mem.h>
#include <malloc.h>
#include <windows.h> /* Codepageumschaltung */

char SETUP_FILE[MAX_PATH] = "neut2slp.map";
char PROGRAMM[MAX_PATH] = "rtd";
int delete_slp = 0;

typedef struct TP_MATRIX {
  float MATRIX[12];
} MATRIX;

struct TP_BausteinReferenz {
  int Nr;
  MATRIX Mat;
  float scale;
  char Datei[MAX_PATH];
  struct TP_BausteinReferenz *next;
} *BausteinReferenz = NULL;

void AddBausteinReferenz(int Nr, MATRIX Mat, float scale, char *Datei)
{ struct TP_BausteinReferenz *neu;
  if ((neu = malloc(sizeof(struct TP_BausteinReferenz))) == NULL) return;
  neu->Nr = Nr;
  neu->Mat = Mat;
  neu->scale = scale;
  strncpy(neu->Datei, Datei, MAX_PATH);
  neu->Datei[MAX_PATH - 1] = '\0';
  neu->next = BausteinReferenz;
  BausteinReferenz = neu;
}

int SetUpBausteinReferenz(void)
{ FILE *dat;
  char Dat[2 * MAX_PATH], Zeile[2 * MAX_PATH];
  MATRIX Mat; float Scale; int BS;
  if ((dat = fopen(SETUP_FILE, "r")) == NULL) {
    fprintf(stderr, "Konfiguration '%s' nicht lesbar!\n", SETUP_FILE);
    return -1;
  }
  while (fgets(Zeile, 2 * MAX_PATH, dat)) {
    Dat[0] = '\0';
    sscanf(Zeile, "%d%f%f%f%f%f%f%f%f%f%f%f%f%f%s", &BS,
      &(Mat.MATRIX[ 0]), &(Mat.MATRIX[ 1]), &(Mat.MATRIX[ 2]), 
      &(Mat.MATRIX[ 3]), &(Mat.MATRIX[ 4]), &(Mat.MATRIX[ 5]), 
      &(Mat.MATRIX[ 6]), &(Mat.MATRIX[ 7]), &(Mat.MATRIX[ 8]), 
      &(Mat.MATRIX[ 9]), &(Mat.MATRIX[10]), &(Mat.MATRIX[11]), 
      &Scale, Dat);
    if (Dat[0]) AddBausteinReferenz(BS, Mat, Scale, Dat);
  }
  fclose(dat);
  return 0;
}

struct TP_BausteinReferenz *BausteinRefernezByNumber(int Nr)
{ struct TP_BausteinReferenz *br;
  for (br = BausteinReferenz; br != NULL; br = br->next)
   if (br->Nr == Nr) return br;
  return NULL;
}

MATRIX MatMul(MATRIX m1, MATRIX m2)
{ MATRIX erg;
  memset(&erg, 0, sizeof(MATRIX));
  erg.MATRIX[0] = m1.MATRIX[0] * m2.MATRIX[0] + m1.MATRIX[3] * m2.MATRIX[1] + m1.MATRIX[6] * m2.MATRIX[2];
  erg.MATRIX[1] = m1.MATRIX[1] * m2.MATRIX[0] + m1.MATRIX[4] * m2.MATRIX[1] + m1.MATRIX[7] * m2.MATRIX[2];
  erg.MATRIX[2] = m1.MATRIX[2] * m2.MATRIX[0] + m1.MATRIX[5] * m2.MATRIX[1] + m1.MATRIX[8] * m2.MATRIX[2];
  erg.MATRIX[3] = m1.MATRIX[0] * m2.MATRIX[3] + m1.MATRIX[3] * m2.MATRIX[4] + m1.MATRIX[6] * m2.MATRIX[5];
  erg.MATRIX[4] = m1.MATRIX[1] * m2.MATRIX[3] + m1.MATRIX[4] * m2.MATRIX[4] + m1.MATRIX[7] * m2.MATRIX[5];
  erg.MATRIX[5] = m1.MATRIX[2] * m2.MATRIX[3] + m1.MATRIX[5] * m2.MATRIX[4] + m1.MATRIX[8] * m2.MATRIX[5];
  erg.MATRIX[6] = m1.MATRIX[0] * m2.MATRIX[6] + m1.MATRIX[3] * m2.MATRIX[7] + m1.MATRIX[6] * m2.MATRIX[8];
  erg.MATRIX[7] = m1.MATRIX[1] * m2.MATRIX[6] + m1.MATRIX[4] * m2.MATRIX[7] + m1.MATRIX[7] * m2.MATRIX[8];
  erg.MATRIX[8] = m1.MATRIX[2] * m2.MATRIX[6] + m1.MATRIX[5] * m2.MATRIX[7] + m1.MATRIX[8] * m2.MATRIX[8];
  erg.MATRIX[ 9] = m1.MATRIX[0] * m2.MATRIX[9] + m1.MATRIX[3] * m2.MATRIX[10] + m1.MATRIX[6] * m2.MATRIX[11] + m1.MATRIX[ 9];
  erg.MATRIX[10] = m1.MATRIX[1] * m2.MATRIX[9] + m1.MATRIX[4] * m2.MATRIX[10] + m1.MATRIX[7] * m2.MATRIX[11] + m1.MATRIX[10];
  erg.MATRIX[11] = m1.MATRIX[2] * m2.MATRIX[9] + m1.MATRIX[5] * m2.MATRIX[10] + m1.MATRIX[8] * m2.MATRIX[11] + m1.MATRIX[11];
  return erg;
}

void SchreibeBausteinSLP(FILE *aus, float size, char *von, MATRIX m)
{ char Zeile[100];
  FILE *ein;
  float fx, fy, fz;
  if ((ein = fopen(von, "rb")) == NULL) {
    fprintf(stderr, "Kann Datei '%s' nicht lesen!\n", von);
    return;
  }
  while (fgets(Zeile, 100, ein) != NULL) {
    if (!strncmp(Zeile, "      normal ", 13)) {
      sscanf(&(Zeile[13]), "%e%e%e", &fx, &fy, &fz);
      fprintf(aus, "      normal %.6e %.6e %.6e\r\n",
        m.MATRIX[0] * fx + m.MATRIX[3] * fy + m.MATRIX[6] * fz,
        m.MATRIX[1] * fx + m.MATRIX[4] * fy + m.MATRIX[7] * fz,
        m.MATRIX[2] * fx + m.MATRIX[5] * fy + m.MATRIX[8] * fz);
    } else
    if (!strncmp(Zeile, "      vertex ", 13)) {
      sscanf(&(Zeile[13]), "%e%e%e", &fx, &fy, &fz);
      fprintf(aus, "      vertex %.6e %.6e %.6e\r\n",
        m.MATRIX[0] * fx + m.MATRIX[3] * fy + m.MATRIX[6] * fz + size * m.MATRIX[ 9],
        m.MATRIX[1] * fx + m.MATRIX[4] * fy + m.MATRIX[7] * fz + size * m.MATRIX[10],
        m.MATRIX[2] * fx + m.MATRIX[5] * fy + m.MATRIX[8] * fz + size * m.MATRIX[11]);
    } else
      fprintf(aus, "%s", Zeile);
  }
  fclose(ein);
}

void Verarbeite(char *D)
{ char Pfad[MAX_PATH], CMD[2 * MAX_PATH + 5];
  int ct;
  struct TP_BausteinReferenz *br;
  FILE *Aus, *Ein;
  MATRIX Mat;
  strcpy(Pfad, D);
  ct = strlen(Pfad) - 4;
  if ((ct > 0) && !stricmp(&(Pfad[ct]), ".pos")) strcpy(&(Pfad[ct]), ".slp");
  fprintf(stderr, "%s -> %s ...\n", D, Pfad);
  if ((Ein = fopen(D, "r")) == NULL) {
    fprintf(stderr, "Datei '%s' nicht lesbar!\n", D);
    return;
  }
  if ((Aus = fopen(Pfad, "wb")) == NULL) {
    fprintf(stderr, "Datei '%s' kann nicht erzeugt werden!\n", Pfad);
    fclose(Ein);
    return;
  }
  while (fgets(CMD, 2 * MAX_PATH, Ein)) {
    ct = -1;
    sscanf(CMD, "%d%f%f%f%f%f%f%f%f%f%f%f%f", &ct,
      &(Mat.MATRIX[ 0]), &(Mat.MATRIX[ 1]), &(Mat.MATRIX[ 2]), 
      &(Mat.MATRIX[ 3]), &(Mat.MATRIX[ 4]), &(Mat.MATRIX[ 5]), 
      &(Mat.MATRIX[ 6]), &(Mat.MATRIX[ 7]), &(Mat.MATRIX[ 8]), 
      &(Mat.MATRIX[ 9]), &(Mat.MATRIX[10]), &(Mat.MATRIX[11]));
    if (ct >= 0) {
      if ((br = BausteinRefernezByNumber(ct)) == NULL) {
        fprintf(stderr, "Baustein Nr. %d ist aber nicht definiert!\n", ct);
      } else {
        SchreibeBausteinSLP(Aus, br->scale, br->Datei, MatMul(Mat, br->Mat));
      }
    }
  }
  fclose(Aus); fclose(Ein);
  if (PROGRAMM[0]) {
    sprintf(CMD, "%s %s", PROGRAMM, Pfad);
    system(CMD);
    if (delete_slp) unlink(Pfad);
  }
}

void show_syntax(void)
{ printf("SYNTAX:\n"
         "neut2slp [Optionen] <POS-Dateien>\n\n"
	 "Erzeugt aus den Teilenummern und Tranformationsmatrizen der\n"
	 "'XXX.pos'-Datei eine 'XXX.slp'-Datei und startet danach ein\n"
	 "Programm zum Ansehen der SLP-Datei.\n\n"
	 "  <POS-Dateien> auch mehrfach\n"
	 " Optionen:\n"
	 "  -M<MAP-Datei>        alternative Einrichtdatei angeben\n"
	 "      Standard: neut2slp.map\n"
	 "  -L<Viewer-Programm>  Programm, welches die SLP-Datei darstellen soll\n"
	 "      Standard: rtd\n"
	 "  -L-                  Kein Viewerprogramm starten\n"
	 "  -d                   SLP-Datei danach lschen - unsinnig,\n"
	 "                       mit Option -L- zusammen benutzen zu wollen.\n"
	 " Alle Optionsparameter mssen direkt auf den Optionsbuchstaben folgen,\n"
	 " es sind keine Leerzeichen dazwischen erlaubt.\n\n"
	 "Arnold Wendl, Frth, anno Domini 2003\n");
}

int main(int argc, char **argv)
{ int ct, initialized = 0;
  UINT CodePage;
  CodePage = GetConsoleOutputCP();
  SetConsoleOutputCP(1252);
  if (argc <= 1) { show_syntax(); return 1; }
  for (ct = 1; ct < argc; ct++) if (argv[ct][0] == '-') {
    switch (argv[ct][1]) {
      case 'M':
	strncpy(SETUP_FILE, &(argv[ct][2]), MAX_PATH);
	SETUP_FILE[MAX_PATH - 1] = '\0';
      break;
      case 'd': delete_slp = 1;
      break;
      case 'L':
        if (argv[ct][2] == '-') PROGRAMM[0] = '\0';
	else {
	  strncpy(PROGRAMM, &(argv[ct][2]), MAX_PATH);
	  PROGRAMM[MAX_PATH - 1] = '\0';
	}
      break;
      default:
        show_syntax();
    }
  } else {
    if (!initialized) {
      if (SetUpBausteinReferenz()) return 1;
      initialized = 1;
    }
    Verarbeite(argv[ct]);
  }
  SetConsoleOutputCP(CodePage);
  return 0;
}

