/* Loesung fuer das ct-Puzzle
   Daniel Pirch (dpirch@web.de), April 2003 */
 
#include <stdio.h>
#include <stdlib.h>

/* auszufuelender raum (rechtshaendiges koordinatensystem) */
#define SPACEX 5 
#define SPACEY 4
#define SPACEZ 3
#define SPACEXYZ 60 /* SPACEX*SPACEY*SPACEZ */

#define PARTMAXCUBES 6 /* aus wie vielen wuerfeln kann ein teil bestehen (max)? */
#define NPARTS 12 /* anzahl der teile */
#define MAXPARTPOSITIONS 1440 /* 24*SPACEX*SPACEY*SPACEZ, maximale anzahl der
                        moeglichen positionen/orientierungen eines teils*/


/****************************************************************/
/* DATENSTRUKTUREN */
/****************************************************************/

/* datentypen zur darstellung des raumes .
   PLATTFORMABHAENGIG fuer 32 bit */
#define INTBITS 32
#define BFSIZE 2 /* zur darstellung des bitfields noetige ints */
#define BF6SIZE 12 /* zur darstellung des bitfield6s noetige ints 
                      (es passen 5 voxel in einen integer)*/

/* es muss also SPACEX*SPACEY*SPACEZ <= INTBITS*BFSIZE sein. */
/* speichert ein bit fuer jedes voxel */
typedef struct
{ unsigned int d[BFSIZE]; } Bitfield;

/* speichert fuer jedes voxel 6 bit. die 6 bit fuer ein voxel liegen
   hintereinander und werden nicht zwischen zwei ints aufgespalten.
   dieser datentyp wird verwendet, um zu speichern, ob die benachbarten
   voxel einer bestimmten koordinate belegt sind.  */
typedef struct
{ unsigned int d[BF6SIZE]; } Bitfield6;


/* datentyp fuer ein teil in vektordarstellung
   (nicht als bitfield, sondern durch die koordinaten der einzelnen wurfel)
   in diesen datentyp werden die teile definiert, gedreht und verschoben,
   bevor sie dann in ein bitfield umgewandelt werden */
typedef struct
{ /*anzahl der wuerfel, aus denen das teil besteht (hier 4 bis 6)*/
  int ncubes;
  /*maximale koordinatenwerte*/
  int maxx, maxy,maxz;
  /*wuerfelkoordinaten, jede der drei koordinaten muss bei mindestens einem
    wuerfel 0 und bei mindestens einem wuerfel max[xyz] sein*/
  int x[PARTMAXCUBES]; int y[PARTMAXCUBES];  int z[PARTMAXCUBES];
} Part;

/****************************************************************/
/* GLOBALE VARIABLEN */
/****************************************************************/

/* anzahl der gesetzten bits in einem feld aus 6 bits (64 kombinationen) */
int bit6count[64];

/*arrays mit einem element fuer jede raumkoordinate */
Bitfield posfield[SPACEXYZ]; /* enthalt nur einen wurfel */
Bitfield6 neighborfield[SPACEXYZ]; /* das neighborfield einer bestimmten
   koordinate ist an dieser koordinate selbst leer, ist aber an den
   umliegenden koordinaten entsprechend gesetzt */

/* werte fuer die voxel in einem bitfield6 (werden dann oder-kombiniert) */
#define B6NX 1 /*nachbar in negativer x-richting */
#define B6PX 2 /*nachbar in positiver x-richting */ 
#define B6NY 4 /*nachbar in negativer y-richting */
#define B6PY 8 /*nachbar in positiver y-richting */
#define B6NZ 16 /*nachbar in negativer z-richting */
#define B6PZ 32 /*nachbar in positiver z-richting */

/* ordnet einer 6-bit richtungsfeld (veroderte richtungsbits) ein array
   aus richtungsfeldern zu, das das komplement sowie die felder, bei denen
   nicht alle der bits des komplementents gesetzt sind, enthaelt. auf jeden
   fall sind alle bits des urspruenglichen feldes in den zugeordneten feldern
   null.
   felder mit weniger als 3 bits werden weggelassen, da wir sie beim loesen
   nicht brauchen (es gibt immer ecken mit 3 nachbarn) */
int emptyneighbors[64]; /* arraygroesse... */
int* emptyneighborindex[64]; /* ...und das array selbst */
	
Part partdef[NPARTS]={ /* zur verfuegung stehende teile*/
  {5,2,2,0, {2,1,1,1,0,0}, {2,2,1,0,1,0}, {0,0,0,0,0,0}},
  {5,1,2,0, {1,0,0,0,1,0}, {2,2,1,0,0,0}, {0,0,0,0,0,0}},
  {6,1,3,0, {0,0,0,0,1,1}, {3,2,1,0,0,2}, {0,0,0,0,0,0}},
  {5,2,2,0, {0,1,2,1,1,0}, {2,2,2,1,0,0}, {0,0,0,0,0,0}},
  {5,2,2,0, {2,1,1,1,0,0}, {2,2,1,0,0,0}, {0,0,0,0,0,0}},
  {5,1,2,0, {0,1,0,1,0,0}, {2,2,1,1,0,0}, {0,0,0,0,0,0}},
  {5,1,1,1, {0,0,0,0,1,0}, {0,1,0,1,0,0}, {1,1,0,0,0,0}},
  {5,2,2,0, {0,1,2,1,1,0}, {1,1,1,0,2,0}, {0,0,0,0,0,0}},
  {4,1,1,1, {0,0,0,1,0,0}, {0,1,1,1,0,0}, {1,1,0,0,0,0}},
  {5,3,1,0, {0,0,1,2,3,0}, {1,0,0,0,0,0}, {0,0,0,0,0,0}},
  {5,2,2,0, {0,0,1,1,2,0}, {0,1,1,2,2,0}, {0,0,0,0,0,0}},
  {5,3,1,0, {0,1,2,3,1,0}, {0,0,0,0,1,0}, {0,0,0,0,0,0}}  };

/* bezeichner fuer die teile, die wir schoen oder-kombinieren
   koennen */
const int partbit[NPARTS]={1,2,4,8,16,32,64,128,256,512,1024,2048};

int partpositions[NPARTS]; /* anzahl der moeglichen positionen eines teils*/
Bitfield* placedpartfield[NPARTS]; /* fuer jedes teil ein array aus
  (partpositions) bitfields, die die vom teil in den einzelnen positionen
  besetzen einheitswuerfeln enthaelt. dadurch muessen wir die teile nicht
  erst waehrend der eigentlichen suche drehen und verschieben*/
Bitfield6* placedneighborfield[NPARTS]; /* informationen fuer die dem teil
                                           benachbarten voxel */
Bitfield6* placedpartneighbormask[NPARTS]; /* in alle richtungen 0 bei voxeln,
  an denen sich das teil befindet, ansonsten ueberall 1 */
int voxelpartpositions[NPARTS][SPACEXYZ][64]; /* Anzahl der Positionen eines
  Teils (erste Dimension), die einen bestimten Einheitswuerfel (zweite
  Dimension) belegen, und sich von diesem aus in bestimmte Richtungen
  (dritte Dimension (6bits->64 kombinationen)) NICHT erstrecken. */
int *voxelpartpositionindex[NPARTS][SPACEXYZ][64]; /* arrays den indizes dieser
                                                      positionen*/

/* variablen waerend des loesens*/
Bitfield solvefillfield[NPARTS+1];
Bitfield6 solveneighborfield[NPARTS+1];

int solutionsfound=0; /* gefundene loesungen zaehlen */
/****************************************************************/
/* GRUNDLEGENDE FUNKTIONEN */
/****************************************************************/

/* meckern und programm beenden */
void OOM()
{ fprintf(stderr,"out of memory\n"); exit(1); }

#if (BFSIZE==2) 
const Bitfield bfnull={{0,0}};

/* prueft zwei bitfields auf gleichheit */
int bfequal(Bitfield* a, Bitfield* b)
{ return a->d[0]==b->d[0] && a->d[1]==b->d[1]; }

/* prueft, ob alle bits null sind */
int bfzero(Bitfield* bf) { return !bf->d[0] && !bf->d[1]; }

/* boolesche operationen: ergebnis wird wieder in a gespeichert*/
void bfor(Bitfield* a, Bitfield* b) { a->d[0]|=b->d[0]; a->d[1]|=b->d[1]; }
void bfxor(Bitfield* a, Bitfield* b) { a->d[0]^=b->d[0]; a->d[1]^=b->d[1]; }
void bfand(Bitfield* a, Bitfield* b) { a->d[0]&=b->d[0]; a->d[1]&=b->d[1]; }

/* boolesche operationen: ergebnis wird in c gespeichert*/
void bfand2(Bitfield* a, Bitfield* b, Bitfield* c)
{ c->d[0] = a->d[0] & b->d[0];  c->d[1] = a->d[1] & b->d[1]; }
void bfor2(Bitfield* a, Bitfield* b, Bitfield* c)
{ c->d[0] = a->d[0] | b->d[0];  c->d[1] = a->d[1] | b->d[1]; }
void bfxor2(Bitfield* a, Bitfield* b, Bitfield* c)
{ c->d[0] = a->d[0] ^ b->d[0];  c->d[1] = a->d[1] ^ b->d[1]; }

/* prueft, ob sich zwei bitfields ueberschneiden. */
int bfintersect(Bitfield* a, Bitfield* b)
{ return (a->d[0]&b->d[0]) || (a->d[1]&b->d[1]); }

#else
#error "Bitfield Funktionen fuer BFSIZE!=2 nicht implementiert."
#endif

/* ein einzelnes bit in einem bitfield setzen */
void bfsetbit(Bitfield* bf, int n)
{ bf->d[n/INTBITS] |= 1<<(n%INTBITS); }

/* wert eines einzelnen bit ermitteln */
int bfgetbit(Bitfield* bf, int n)
{ return (bf->d[n/INTBITS]) & (1<<(n%INTBITS)); }

#if (BF6SIZE==12) 

const Bitfield6 bf6null={{0,0,0,0,0,0,0,0,0,0,0,0}};
const Bitfield6 bf6set={{
  0xFFFFFFFF,0xFFFFFFFF,0xFFFFFFFF,0xFFFFFFFF,
  0xFFFFFFFF,0xFFFFFFFF,0xFFFFFFFF,0xFFFFFFFF,
  0xFFFFFFFF,0xFFFFFFFF,0xFFFFFFFF,0xFFFFFFFF}};

/* a |= b */
void bf6or(Bitfield6* a, Bitfield6* b)
{ a->d[0]|=b->d[0]; a->d[1]|=b->d[1]; a->d[2]|=b->d[2];
  a->d[3]|=b->d[3]; a->d[4]|=b->d[4]; a->d[5]|=b->d[5];
  a->d[6]|=b->d[6]; a->d[7]|=b->d[7]; a->d[8]|=b->d[8];
  a->d[9]|=b->d[9]; a->d[10]|=b->d[10]; a->d[11]|=b->d[11];
} 

/* a &= b */
void bf6and(Bitfield6* a, Bitfield6* b)
{ a->d[0]&=b->d[0]; a->d[1]&=b->d[1]; a->d[2]&=b->d[2];
  a->d[3]&=b->d[3]; a->d[4]&=b->d[4]; a->d[5]&=b->d[5];
  a->d[6]&=b->d[6]; a->d[7]&=b->d[7]; a->d[8]&=b->d[8];
  a->d[9]&=b->d[9]; a->d[10]&=b->d[10]; a->d[11]&=b->d[11];
}

/* c = a | b */
void bf6or2(Bitfield6* a, Bitfield6* b, Bitfield6* c)
{ c->d[0]=a->d[0]|b->d[0]; c->d[1]=a->d[1]|b->d[1];
  c->d[2]=a->d[2]|b->d[2]; c->d[3]=a->d[3]|b->d[3];
  c->d[4]=a->d[4]|b->d[4]; c->d[5]=a->d[5]|b->d[5];
  c->d[6]=a->d[6]|b->d[6]; c->d[7]=a->d[7]|b->d[7];
  c->d[8]=a->d[8]|b->d[8]; c->d[9]=a->d[9]|b->d[9];
  c->d[10]=a->d[10]|b->d[10]; c->d[11]=a->d[11]|b->d[11];
}

/* masken fuer die fuenf 6bit-werte, die in einen integer passen */
const unsigned int bit6mask[5]={63,4032,258048,16515072,1056964608};
const unsigned int bit6kompmask[5]={~63,~4032,~258048,~16515072,~1056964608};

/* setzt ein voxel in einem Bitfield6 auf einen bestimmten wert,
   der in 6 bit passen muss (wird nicht geprueft)*/
void bf6setvoxel(Bitfield6* b, int voxel, int val)
{ unsigned int* d=&b->d[voxel/5]; voxel%=5; val<<=(voxel*6);
  *d &= bit6kompmask[voxel]; *d |= val;
}

/* liefert den wert eines voxels in einem Bitfield6 */
int bf6getvoxel(Bitfield6* b, int voxel)
{ int i=voxel%5;
  return (b->d[voxel/5] & bit6mask[i]) >> (i*6);
}

/* sucht ein voxels, bei dem die meisten bits
   gesetzt sind (max. 6). da bei der loesungssuche immer ecken mit 3 nachbarn
   existieren, wird davon ausgegangen, dass mindestens voxel mit 3 bits
   existieren.
   zurueckgegeben wird die maximal gefundene anzahl an gesetzten bits eines
   voxels, der index und wert dieses voxels wird in *index und *voxel
   gespeichert.
   Wenn zwischendurch voxel gefunden werden, bei denen das Bitfield f
   gesetzt ist, wird b an dieser stelle auf null gesetzt. dadurch wird
   verhindert, dass voxel, die bereits belegt sind, gefunden werden
   (kommt trotz placedpartneighbormask gelegentlich vor)*/

int findmaxneighborvoxel(Bitfield6* b, Bitfield *f, int* index, int* voxel)
{ int max=2,i,j; unsigned int v;
  for(i=0;i<BF6SIZE;i++) {
    v=b->d[i]; if(bit6count[v&63]>max) {
      j=5*i; if(bfgetbit(f,j)) bf6setvoxel(b,j,0);
      else {*index=j; if((max=bit6count[*voxel=v&63])==6) return 6; }}
    v>>=6; if(bit6count[v&63]>max) {
      j=5*i+1; if(bfgetbit(f,j)) bf6setvoxel(b,j,0);
      else {*index=j; if((max=bit6count[*voxel=v&63])==6) return 6; }}
    v>>=6; if(bit6count[v&63]>max) {
      j=5*i+2; if(bfgetbit(f,j)) bf6setvoxel(b,j,0);
      else {*index=j; if((max=bit6count[*voxel=v&63])==6) return 6; }}
    v>>=6; if(bit6count[v&63]>max) {
      j=5*i+3; if(bfgetbit(f,j)) bf6setvoxel(b,j,0);
      else {*index=j; if((max=bit6count[*voxel=v&63])==6) return 6; }}
    v>>=6; if(bit6count[v]>max) {
      j=5*i+4; if(bfgetbit(f,j)) bf6setvoxel(b,j,0);
      else {*index=j; if((max=bit6count[*voxel=v])==6) return 6; }} }
  return max;
}

#else
#error "Bitfield6 Funktionen fuer BF6SIZE!=12 nicht implementiert."
#endif

/* bitnummer in einem bitfield, die eine bestimmte raumkoordinate repraesentiert*/
int posindex(int x, int y, int z)
{ return z*SPACEX*SPACEY + y*SPACEX + x; }

/* umgekehrt */
void indexpos(int i, int* x, int* y, int* z)
{ *z=i/(SPACEX*SPACEY); i%=SPACEX*SPACEY;
  *y=i/SPACEX; *x=i%SPACEX; }

/* position zu einem index drucken (ohne \n)*/
void printindexpos(int i)
{ int x,y,z; indexpos(i,&x,&y,&z);
  printf("(%d,%d,%d)",x,y,z);
}

/* zwei bitfields zusammen drucken */
void printbitfields(Bitfield *a, Bitfield* b, int mp)
{ int x,y,z,p,ap,bp=0;
  for(y=SPACEY-1; y>=0; y--) {
    for(z=0; z<SPACEZ; z++) {
      for(x=0; x<SPACEX; x++) {
	p=posindex(x,y,z); ap=bfgetbit(a,p);
        if(b) bp=bfgetbit(b,p);
	printf("%c%c",(p==mp)?'[':(p==mp+1&&x>0)?']':' ',
                      ap?(bp?'O':'x'):(bp?'o':'.')); }
      printf("%c   ",(p==mp)?']':' '); }
    printf("\n"); }
}



/*rotieren um 90 grad um die x-achse, also z:=y und y:=-z */
void partrotatex(Part* p)
{ int i; int tmp;
  for(i=0; i<p->ncubes;i++){
    tmp=p->z[i]; p->z[i] = p->y[i]; p->y[i] = p->maxz-tmp; }
  tmp=p->maxz; p->maxz=p->maxy; p->maxy=tmp;
}

/*rotieren um 90 grad um die y-achse, also x:=z und z:=-x */
void partrotatey(Part* p)
{ int i; int tmp;
  for(i=0; i<p->ncubes;i++){
    tmp=p->x[i]; p->x[i] = p->z[i]; p->z[i] = p->maxx-tmp; }
  tmp=p->maxx; p->maxx=p->maxz; p->maxz=tmp;
}

/*rotieren um 90 grad um die z-achse, also x:=-y und y:=x */
void partrotatez(Part* p)
{ int i; int tmp;
  for(i=0; i<p->ncubes;i++){
    tmp=p->y[i]; p->y[i] = p->x[i]; p->x[i] = p->maxy-tmp; }
  tmp=p->maxy; p->maxy=p->maxx; p->maxx=tmp;
}

/****************************************************************/
/* INITIALISIERUNG */
/****************************************************************/

/* initialisiert posfield und neighborfield */
void initposfields()
{ int x,y,z,i;
  for(x=0; x<SPACEX; x++)
    for(y=0; y<SPACEY; y++)
      for(z=0; z<SPACEZ; z++) {
        i=posindex(x,y,z);
	posfield[i]=bfnull; neighborfield[i]=bf6null;
	bfsetbit(&posfield[i],i);
	if(x<SPACEX-1) bf6setvoxel(&neighborfield[i],posindex(x+1,y,z),B6NX);
	if(x>0) bf6setvoxel(&neighborfield[i],posindex(x-1,y,z),B6PX);
	if(y<SPACEY-1) bf6setvoxel(&neighborfield[i],posindex(x,y+1,z),B6NY);
	if(y>0) bf6setvoxel(&neighborfield[i],posindex(x,y-1,z),B6PY);
	if(z<SPACEZ-1) bf6setvoxel(&neighborfield[i],posindex(x,y,z+1),B6NZ);
	if(z>0) bf6setvoxel(&neighborfield[i],posindex(x,y,z-1),B6PZ);
      }
}

void initemptyneighbors()
{ int i,j;
  for(i=0; i<64; i++) { j=0;
    if(i&1)j++; if(i&2)j++; if(i&4)j++;
    if(i&8)j++; if(i&16)j++; if(i&32)j++;
    bit6count[i]=j; }
  for(i=0; i<64; i++) {
    emptyneighborindex[i]=malloc(sizeof(int)*64);
    if(!emptyneighborindex[i]) OOM();
    emptyneighbors[i]=0;
    for(j=0;j<64;j++) {
      if(bit6count[j]>=3 && !(i&j)) {
        emptyneighborindex[i][emptyneighbors[i]++]=j; } } 
    emptyneighborindex[i]=realloc(
      emptyneighborindex[i],sizeof(int)*emptyneighbors[i]);
  }
}

/* fuegt ein teil mit einer bestimmten. orientierung den placedpartsfields[i]
   hinzu.
   prueft vorher, ob das teil in dieser orentierung nicht identisch zu einem
   der *ncf bitfields in cf ist; wenn das teil wirklich hinzugefuegt wird,
   wird ncf und cf erweitert (es muss genuegend platz in cf sein)*/
void placeorientedpart(int i, int* ncf, Bitfield* cf , Part*p)
{ Bitfield bf=bfnull; int j,k,l,m,x,y,z;
  /* abbrechen, wenn es ein identisches bitfield gibt*/
  for(j=0; j<p->ncubes; j++)
    bfor(&bf,&posfield[posindex(p->x[j],p->y[j],p->z[j])]);
  for(j=0;j<*ncf;j++) if(bfequal(&(cf[j]),&bf)) return;
  /* ansonsten hinzufuegen */
  cf[*ncf]=bf; (*ncf)++;
  
  for(x=0; x<SPACEX-p->maxx; x++)
    for(y=0; y<SPACEY-p->maxy; y++)
      for(z=0; z<SPACEZ-p->maxz; z++) {
	/* partpositions[i] ist der index der position, die wir setzen
	   wollen. zunaechst reset... */
	placedpartfield[i][partpositions[i]]=bfnull;
	placedneighborfield[i][partpositions[i]]=bf6null;
	placedpartneighbormask[i][partpositions[i]]=bf6set;
	/* fuer jeden wuerfel des teil setzen wir das entsprechende
	   (verschobene) voxel in placedpartfield und placedneighborfield */
	for(j=0; j<p->ncubes; j++) {
	  k=posindex(x+p->x[j], y+p->y[j], z+p->z[j]);
	  bfor(&placedpartfield[i][partpositions[i]],&posfield[k]);
	  bf6or(&placedneighborfield[i][partpositions[i]],&neighborfield[k]); 
	  bf6setvoxel(&placedpartneighbormask[i][partpositions[i]],k,0); }
	/* wir gehen die wuerfel nochmal durch, da die placedneighborfield
           informationen eines voxels durch andere wurfel eines teils
	   beeinflusst werden. damit erweitern wir die voxelpartpositionindex
	   tabelle */
	
	for(j=0; j<p->ncubes; j++) {
	  k=posindex(x+p->x[j], y+p->y[j], z+p->z[j]);
	  l=bf6getvoxel(&placedneighborfield[i][partpositions[i]],k);
	  for(m=0; m<emptyneighbors[l]; m++) {
	    voxelpartpositionindex[i][k][emptyneighborindex[l][m]]
	      [voxelpartpositions[i][k][emptyneighborindex[l][m]]]
	      =partpositions[i];
	    voxelpartpositions[i][k][emptyneighborindex[l][m]]++; } }
	partpositions[i]++;  }
}

/* berechnet placedparts aus partdef */
void initplacedparts()
{ int i,j,k; Part p;
  Bitfield orientedfield[24]; int ncf;
  for(i=0;i<NPARTS;i++) {
    p=partdef[i];
    partpositions[i]=0;
    ncf=0; /* anzahl der bisher gefundenen unterschdl. orientierungen*/
    /* wir reservieren genug platz fuer alle moeglichen positionen*/
    placedpartfield[i]=malloc(sizeof(Bitfield)*MAXPARTPOSITIONS);
      if(!(placedpartfield[i])) OOM();
    placedneighborfield[i]=malloc(sizeof(Bitfield6)*MAXPARTPOSITIONS);
      if(!(placedneighborfield[i])) OOM(); 
    placedpartneighbormask[i]=malloc(sizeof(Bitfield6)*MAXPARTPOSITIONS);
      if(!(placedpartneighbormask[i])) OOM(); 
    for(j=0;j<SPACEXYZ;j++) {
      for(k=0; k<64; k++) {
        voxelpartpositions[i][j][k]=0;
        voxelpartpositionindex[i][j][k]=malloc(sizeof(int)*24*p.ncubes); } }

    /* wir speichern das teil in allen moeglichen positionen. das erste Teil
       wird nicht so oft gespeichert, da sich die gleiche konfiguration
       durch drehen des gesamten raumes erreichen liesse. es ist wichtig, das
       dieses erste teil tatsaechlich ein teil mit 24 _unterschiedlichen_
       moeglichen orientierungen ist */
    placeorientedpart(i,&ncf,orientedfield,&p);
    partrotatez(&p); placeorientedpart(i,&ncf,orientedfield,&p);
    if(i) {
      partrotatez(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatez(&p); placeorientedpart(i,&ncf,orientedfield,&p); }
    partrotatex(&p); placeorientedpart(i,&ncf,orientedfield,&p);
    partrotatey(&p); placeorientedpart(i,&ncf,orientedfield,&p);
    if(i) {
      partrotatey(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatey(&p); placeorientedpart(i,&ncf,orientedfield,&p); }
    partrotatez(&p); placeorientedpart(i,&ncf,orientedfield,&p);
    partrotatex(&p); placeorientedpart(i,&ncf,orientedfield,&p);
    if(i) {
      partrotatex(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatex(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatey(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatez(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatez(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatez(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatex(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatey(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatey(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatey(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatez(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatex(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatex(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      partrotatex(&p); placeorientedpart(i,&ncf,orientedfield,&p);
      /* partrotatey wuerde das teil wieder in die urspruengliche orientierung
         zurueckbringen */
    }    
    /* wir haben oben wahrscheinlich zu viel speicher reserviert und geben
       den rest wieder frei*/
    placedpartfield[i]=realloc(placedpartfield[i],
      sizeof(Bitfield)*partpositions[i]);
    placedneighborfield[i]=realloc(placedneighborfield[i],
      sizeof(Bitfield6)*partpositions[i]); 
    placedpartneighbormask[i]=realloc(placedpartneighbormask[i],
      sizeof(Bitfield6)*partpositions[i]); 
    for(j=0;j<SPACEXYZ;j++)
      for(k=0; k<64; k++) 
        voxelpartpositionindex[i][j][k]=realloc(voxelpartpositionindex[i][j][k],
          sizeof(int)*voxelpartpositions[i][j][k]);
	
    printf("Teil %2d: %2d Orientierungen, insgesamt %3d Lagemoeglichkeiten.\n",
            i,ncf,partpositions[i]);
  }
}
/* Setzt den raum auf den anfangszustand. */
void initspace()
{ int x,y,z,s;
  solvefillfield[0]=bfnull; solveneighborfield[0]=bf6null;
  for(x=0; x<SPACEX; x++)
    for(y=0; y<SPACEY; y++)
      for(z=0; z<SPACEZ; z++) { s=0;
        if(x==0) s |= B6NX; if(x==SPACEX-1) s |=B6PX; 
        if(y==0) s |= B6NY; if(y==SPACEY-1) s |=B6PY; 
        if(z==0) s |= B6NZ; if(z==SPACEZ-1) s |=B6PZ; 
        bf6setvoxel(solveneighborfield,posindex(x,y,z),s);
      }
}

/****************************************************************/
/* LOESEN DES PUZZLES */
/****************************************************************/

/* aufgerufen, wenn eine loesung gefunden wurde */
void solutionfound()
{ solutionsfound++;
/*  if(solutionsfound>=1000) {
    printf("%d Loesungen gefunden, breche ab...\n",solutionsfound);
    exit(0);}*/
}

/*#define IA*/

/* findet alle loesungen des puzzles */
void solve(int depth,int usedparts)
{ int i,j,vppn,fillindex,fillneighbors,pos;
  int *vppa;
  int nextdepth=depth+1;
  /* wir suchen ein moeglichst zugebautes loch. */
  i=findmaxneighborvoxel(&solveneighborfield[depth],&solvefillfield[depth],
                         &fillindex,&fillneighbors);
#ifdef IA
  printf("Beginn Solve, Tiefe %d; Feld und fillindex ",depth);
  printindexpos(fillindex); printf(" %d Nachbarn\n",i);
  printbitfields(&solvefillfield[depth],0,fillindex);
  for(j=0;j<60;j++) printf("%2d ",bf6getvoxel(&solveneighborfield[depth],j));
  printf("\nTRA\n");
#endif
  /* ein loch mit 6 ausgefuellten nachbarn koennen wir mit unseren teilen
     nicht mehr fuellen */
  if(i==6) return;
  /* ansonsten probieren wir jetzt alle teile in allen in das loch  passenden
     positionen aus */
  for(i=0; i<NPARTS; i++) {
#ifdef IA
    printf("TRB\n");
#endif
    if(usedparts & partbit[i]) {
#ifdef IA
      printf("TRC\n");
#endif
      continue; /* teil bereits verwendet */
    }
    vppn=voxelpartpositions[i][fillindex][fillneighbors];
    vppa=voxelpartpositionindex[i][fillindex][fillneighbors];
#ifdef IA
    printf("Teste Teil %d: %d in frage kommende Positionen.\n",i,vppn);
#endif
    for(j=0; j<vppn; j++) {
#ifdef IA
      printf("TRD\n");
#endif
      pos=vppa[j];
      /* wir haben eine position eines teils, die das loch ausfuellen wuerde,
         aber vielleicht passt es nicht in den leeren raum? */
      if(!bfintersect(&solvefillfield[depth],&placedpartfield[i][pos])) {
#ifdef IA
        printf("TRE\n");
        printf("Teil passt in vppa %d:\n",j);
	printbitfields(&solvefillfield[depth],&placedpartfield[i][pos],fillindex);
#endif
        /* keine ueberschneidung, das teil passt.  */
	if(depth==11 /*NPARTS-1*/) {
          /* wir haben alle teile platziert und damit eine loesung gefunden.
             VORSICHT: die letzten solvefillfield und solveneighborfield
             variablen sind nicht richtig gesetzt (falls solutionfound
             die braucht) */
#ifdef IA
          printf("Loesung gefunden,\n");
#endif
          solutionfound(); 
        }
        /* alles fuer den naechsten rekursionsschritt vorbereiten */
	bfor2(&solvefillfield[depth],&placedpartfield[i][pos],
              &solvefillfield[nextdepth]);
	bf6or2(&solveneighborfield[depth],&placedneighborfield[i][pos],
               &solveneighborfield[nextdepth]);
	/* an den stellen, an denen sich das neue teil befindet, muss das
	   neighborfield auf null gesetzt werden */
        bf6and(&solveneighborfield[nextdepth],&placedpartneighbormask[i][pos]);
	solve(nextdepth,usedparts|partbit[i]);
#ifdef IA
        printf("%d Kehre zurueck...\n",depth);
#endif
      }
    }
  }  
}

/***********************************************************/


int main()
{ int i;
  initposfields();
  initemptyneighbors();
  initplacedparts();
  initspace();
  fprintf(stdout,"Beginn der Suche...\n");
  solve(0,0);
  printf("Fertig. Es wurden %d Loesungen gefunden.\n", solutionsfound);

  return 0;
}
