/* ctpz.cc -- C't journal 2003/3/24 puzzle.
**
** Copyright (C) 2003 Eric Laroche.  All rights reserved.
**
** @author Eric Laroche <laroche@lrdev.com>
** @version @(#)$Id: ctpz.cc,v 1.1 2003/04/30 21:34:12 laroche Exp $
** @url http://www.lrdev.com/lr/c/ctpz.html
**
** @reference c't 7/2003, p. 234 [c't puzzle]
** @reference http://www.lrdev.com/lr/c/sqfig.html [pentomino tilings]
**
** This program is free software;
** you can redistribute it and/or modify it.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


// implemented interface
#include "ctpz.hh"


#include "ctpzut.hh"

#include <string.h>
#include <stdlib.h>


// [read-only data]
static char const dot = '.';


Cube::Cube(char const* s)
{
	init(s);
}


Cube::Cube(StringRange const& sr)
{
	int size = sr.m_end - sr.m_begin; // exclusive terminating nul
	char buf[128]; // must hold three (small) ints (with dots between)
	if (size >= sizeof(buf)) {
		size = sizeof(buf) - 1;
	}
	memcpy(buf, &sr.m_s[sr.m_begin], size);
	buf[size] = '\0';

	init(buf);
}


Cube::Cube(Cube const& that) :
	x(that.x),
	y(that.y),
	z(that.z)
{
}


Cube::Cube() :
	x(0),
	y(0),
	z(0)
{
}


Cube& Cube::operator=(Cube const& that)
{
	if (this != &that) {
		x = that.x;
		y = that.y;
		z = that.z;
	}

	return *this;
}


void Cube::init(char const* s)
{
	x = y = z = 0;

	// assume no leading and trailing blanks

	char const* p = s;
	char* q = 0; // [constness missing in strtol interface]

	int xx = strtol(p, &q, 10);
	if (q == p) {
		return;
	}

	p = q;
	if (*p != dot) {
		return;
	}
	p++;

	int yy = strtol(p, &q, 10);
	if (q == p) {
		return;
	}

	p = q;
	if (*p != dot) {
		return;
	}
	p++;

	int zz = strtol(p, &q, 10);
	if (q == p) {
		return;
	}

	// string representation starts at 1,
	// internal representation starts at 0
	x = xx - 1;
	y = yy - 1;
	z = zz - 1;
}


int Cube::compare(Cube const& a, Cube const& b)
{
	if (a.z != b.z) {
		return (a.z >= b.z ? 1 : -1);
	}
	if (a.y != b.y) {
		return (a.y >= b.y ? 1 : -1);
	}
	if (a.x != b.x) {
		return (a.x >= b.x ? 1 : -1);
	}
	return 0;
}

void Cube::move(int deltaX, int deltaY, int deltaZ)
{
	x += deltaX;
	y += deltaY;
	z += deltaZ;
}


Figure::Figure(char const* s)
{
	init(SetParser(s));
}


Figure::Figure(StringRange const& sr)
{
	init(SetParser(sr));
}


Figure::Figure(Figure const& that)
{
	m_size = that.m_size;
	m_cubes = new Cube*[m_size];

	int i;
	for (i = 0; i < m_size; i++) {
		m_cubes[i] = new Cube(*that.m_cubes[i]);
	}
}


Figure::Figure(int size)
{
	m_size = size;
	m_cubes = new Cube*[m_size];

	int i;
	for (i = 0; i < m_size; i++) {
		m_cubes[i] = new Cube();
	}
}


Figure& Figure::operator=(Figure const& that)
{
	if (this != &that) {
		deinit();

		m_size = that.m_size;
		m_cubes = new Cube*[m_size];

		int i;
		for (i = 0; i < m_size; i++) {
			m_cubes[i] = new Cube(*that.m_cubes[i]);
		}
	}

	return *this;
}


Figure::~Figure()
{
	deinit();
}


static void figureParserCallback(StringRange const& sr, void* userData)
{
	Cube**& cubes = *(Cube***)userData;

	*cubes = new Cube(sr);
	cubes++;
}


void Figure::init(SetParser const& p)
{
	m_size = p.count();
	m_cubes = new Cube*[m_size];

	Cube** cubes = m_cubes;
	p.enumerate(figureParserCallback, &cubes);
}


void Figure::deinit()
{
	int i;
	for (i = 0; i < m_size; i++) {
		delete m_cubes[i];
	}
	delete[] m_cubes;
}


// helper for qsort [adaptor]
static int cmp1(void const* a, void const* b)
{
	return Cube::compare(**(Cube const* const*)a, **(Cube const* const*)b);
}


void Figure::normalize()
{
	// get min/max values
	int minX, minY, minZ, maxX, maxY, maxZ;
	getMinMax(minX, minY, minZ, maxX, maxY, maxZ);

	// normalize to min=={0,0,0}
	if (minX != 0 || minY != 0 || minZ != 0) {
		move(-minX, -minY, -minZ);
	}

	// sort the position-normalized cubes
	qsort(m_cubes, m_size, sizeof(*m_cubes), cmp1);
}


void Figure::getMinMax(
	int& minX,
	int& minY,
	int& minZ,
	int& maxX,
	int& maxY,
	int& maxZ) const
{
	minX = minY = minZ = maxX = maxY = maxZ = 0;

	int i;
	for (i = 0; i < m_size; i++) {
		if (i == 0 || m_cubes[i]->x < minX) {
			minX = m_cubes[i]->x;
		}
		if (i == 0 || m_cubes[i]->y < minY) {
			minY = m_cubes[i]->y;
		}
		if (i == 0 || m_cubes[i]->z < minZ) {
			minZ = m_cubes[i]->z;
		}
		if (i == 0 || m_cubes[i]->x > maxX) {
			maxX = m_cubes[i]->x;
		}
		if (i == 0 || m_cubes[i]->y > maxY) {
			maxY = m_cubes[i]->y;
		}
		if (i == 0 || m_cubes[i]->z > maxZ) {
			maxZ = m_cubes[i]->z;
		}
	}
}


void Figure::doCubes(
	void (*what)(Cube& cube, void* data),
	void* data,
	bool doNormalize)
{
	int i;
	for (i = 0; i < m_size; i++) {
		(*what)(*m_cubes[i], data);
	}
	if (doNormalize) {
		normalize();
	}
}


static void mover(Cube& cube, void* data)
{
	int* deltas = (int*)data;
	cube.move(deltas[0], deltas[1], deltas[2]);
}


void Figure::move(int deltaX, int deltaY, int deltaZ)
{
	int deltas[3];
	deltas[0] = deltaX, deltas[1] = deltaY, deltas[2] = deltaZ;
	doCubes(mover, deltas, false);
}


static void xRotator(Cube& cube, void*)
{
	// rotate left: z' := y ; y' := -z
	int y = cube.y, z = cube.z;
	cube.y = -z, cube.z = y;
}


void Figure::xRotate()
{
	doCubes(xRotator);
}


static void yRotator(Cube& cube, void*)
{
	// rotate left: x' := z ; z' := -x
	int z = cube.z, x = cube.x;
	cube.z = -x, cube.x = z;
}


void Figure::yRotate()
{
	doCubes(yRotator);
}


static void zRotator(Cube& cube, void*)
{
	// rotate left: y' := x ; x' := -y
	int x = cube.x, y = cube.y;
	cube.x = -y, cube.y = x;
}


void Figure::zRotate()
{
	doCubes(zRotator);
}


static void xMirrorer(Cube& cube, void*)
{
	// x' := -x
	int x = cube.x;
	cube.x = -x;
}


void Figure::xMirror()
{
	doCubes(xMirrorer);
}


static void yMirrorer(Cube& cube, void*)
{
	// y' := -y
	int y = cube.y;
	cube.y = -y;
}


void Figure::yMirror()
{
	doCubes(yMirrorer);
}


static void zMirrorer(Cube& cube, void*)
{
	// z' := -z
	int z = cube.z;
	cube.z = -z;
}


void Figure::zMirror()
{
	doCubes(zMirrorer);
}


int Figure::compareNormalized(Figure const& a, Figure const& b)
{
	int size = (a.m_size <= b.m_size ? a.m_size : b.m_size);

	int i;
	for (i = 0; i < size; i++) {
		int cmp = Cube::compare(*a.m_cubes[i], *b.m_cubes[i]);
		if (cmp != 0) {
			return cmp;
		}
	}

	if (a.m_size != b.m_size) {
		return (a.m_size >= b.m_size ? 1 : -1);
	}

	return 0;
}


static void figureSetParseCallback(StringRange const& sr, void* userData)
{
	Figure**& f = *(Figure***)userData;

	*f = new Figure(sr);
	f++;
}


Figure** Figure::figureSetParse(char const* s)
{
	SetParser p = SetParser(s);
	int size = p.count();
	Figure** f = new Figure*[size + 1]; // null terminated

	Figure** f2 = f;
	p.enumerate(figureSetParseCallback, &f2);
	f[size] = 0;

	return f;
}


void Figure::release(Figure** figures)
{
	if (figures != 0) {
		int i;
		for (i = 0; figures[i] != 0; i++) {
			delete figures[i];
		}
		delete[] figures;
	}
}


int Figure::count(Figure const* const* figures)
{
	int n = 0;
	while (figures[n] != 0) {
		n++;
	}
	return n;
}


static void generateZRotameres(Figure const& figure, Figure** rotameres, int& position)
{
	Figure f = figure;
	f.normalize();

	Figure g = f;
	do {

		// check if it's already there (through other combinations of
		// rotations); this is a (slow) linear search, however we do not
		// need to be specifically fast at this time; neither are we
		// confining the position value to its initial value
		int i;
		for (i = 0; i < position; i++) {
			if (Figure::compareNormalized(g, *rotameres[i]) == 0) {
				break;
			}
		}
		if (i == position) {
			// add rotamere
			rotameres[position++] = new Figure(g);
		}

		// rotate until back to identity; this will take 1, 2, or 4 steps,
		// depending on the figure's symmetry
		g.zRotate();

	} while (Figure::compareNormalized(g, f) != 0);
}


// naming: compiler problem encountered resolving local function vs.
// overloaded member function
static Figure** generateRotameresX(Figure const& figure)
{
	// there will be at most 24 rotameres; since most of them are not
	// that symmetrical, no big loss occurs if the maximal size is
	// allocated for each figure
	int size = 24;
	Figure** rotameres = new Figure*[size + 1]; // null terminated

	int i = 0; // over-all rotamere counter

	Figure f = figure;
	f.normalize();

	// rotate until back to identity; this will take 1, 2, or 4 steps,
	// depending on the figure's symmetry
	Figure g = f;
	do {
		// note: this _may_ produce duplicates [which will be purged]
		generateZRotameres(g, rotameres, i);
		g.xRotate();
	} while (Figure::compareNormalized(g, f) != 0);

	g = f; // this case is already handled, so instantly rotate
	g.yRotate();
	// check some more symmetry
	if (Figure::compareNormalized(g, f) != 0) {
		generateZRotameres(g, rotameres, i);
		g.yRotate(); // this case is already handled [ry*ry==rx*rx*rz*rz]
		g.yRotate();
		generateZRotameres(g, rotameres, i);
	}

	// note: mirrors (reflections) are not considered, since the
	// [hardware] pieces can't be reflected [due to the typical lack of
	// a fourth dimension]

	rotameres[i] = 0; // null terminated

	return rotameres;
}


Figure*** Figure::generateRotameres(Figure const* const* figures)
{
	int size = count(figures);
	Figure*** rotameres = new Figure**[size + 1]; // null terminated

	int i;
	for (i = 0; i < size; i++) {
		rotameres[i] = generateRotameresX(*figures[i]);
	}
	rotameres[size] = 0;

	return rotameres;
}


void Figure::release(Figure*** figureLists)
{
	if (figureLists != 0) {
		int i;
		for (i = 0; figureLists[i] != 0; i++) {
			release(figureLists[i]);
		}
		delete[] figureLists;
	}
}


int Figure::count(Figure const* const* const* figureLists)
{
	int n = 0;
	while (figureLists[n] != 0) {
		n++;
	}
	return n;
}

