/* ctpz.hh -- C't journal 2003/3/24 puzzle.
**
** Copyright (C) 2003 Eric Laroche.  All rights reserved.
**
** @author Eric Laroche <laroche@lrdev.com>
** @version @(#)$Id: ctpz.hh,v 1.1 2003/04/30 21:35:04 laroche Exp $
** @url http://www.lrdev.com/lr/c/ctpz.html
**
** @reference c't 7/2003, p. 234 [c't puzzle]
** @reference http://www.lrdev.com/lr/c/sqfig.html [pentomino tilings]
**
** This program is free software;
** you can redistribute it and/or modify it.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


#ifndef CTPZ_HH
#define CTPZ_HH


// type declarations
class SetParser;
class StringRange;


/** A cube, the figure's building block.  X, Y and Z coordinates.
* Coordinates are signed rather than unsigned to allow the
* implementation of simpler rotate functions, etc.  Coordinates use the
* native CPU register type (int), for that is assumed to be handled
* fastest [proper alignment, no conversion steps before/after operations
* needed].  Data members are public to allow fast inline operations
* [diminuishes encapsulation however].
*/
class Cube
{

public :

	/** Construct a Cube from string specifications.
	*/
	Cube(char const* s);
	Cube(StringRange const& sr);

	/** [copy constructor]
	*/
	Cube(Cube const& that);

	Cube();

	Cube& operator=(Cube const& that);

	/** Move a cube (translation).
	*/
	void move(int deltaX, int deltaY, int deltaZ);

	/** Compare two cubes.
	*/
	static int compare(Cube const& a, Cube const& b);

	/** X, Y and Z coordinates.  These data members are public to allow
	* fast inline operations.
	*/
	int x;
	int y;
	int z;

private :

	// common initializer
	void init(char const* s);
};


/** A figure.  An array of cubes.  The cubes are ordered (precedence is:
* Z, Y, X).  No special Figure list type is implemented (to cut some
* syntax overhead).
*/
class Figure
{

public :

	/** Construct a Figure from string specifications.
	*/
	Figure(char const* s);
	Figure(StringRange const& sr);

	/** [copy constructor]
	*/
	Figure(Figure const& that);

	Figure(int size);

	Figure& operator=(Figure const& that);

	~Figure();

	/** Normalize a Figure: normalize cube offsets (minimal coordinates
	* are {0,0,0}; normalize cube order (ordered, as defined by
	* Cube::compare).
	*/
	void normalize();

	/** Get minimum and maximum coordinate values.
	*/
	void getMinMax(
		int& minX,
		int& minY,
		int& minZ,
		int& maxX,
		int& maxY,
		int& maxZ) const;

	/** Move a figure (i.e. its cubes), translation.
	*/
	void move(int deltaX, int deltaY, int deltaZ);

	/** Rotate a figure 90 degrees left about the X axis and normalize.
	*/
	void xRotate();

	/** Rotate a figure 90 degrees left about the Y axis and normalize.
	*/
	void yRotate();

	/** Rotate a figure 90 degrees left about the Z axis and normalize.
	*/
	void zRotate();

	/** Mirror (reflect) normal to X axis and normalize.
	*/
	void xMirror();

	/** Mirror (reflect) normal to Y axis and normalize.
	*/
	void yMirror();

	/** Mirror (reflect) normal to Z axis and normalize.
	*/
	void zMirror();

	/** Compare two _normalized_ figures (normalized as precondition for
	* speed reason).  Normalization is _not_ checked.
	*/
	static int compareNormalized(Figure const& a, Figure const& b);

	/** Construct a null-terminated list of pointers to Figures from a
	* string specification.  [factory method]
	*/
	static Figure** figureSetParse(char const* s);

	/** Release (delete) Figure list.
	*/
	static void release(Figure** figures);

	/** Count Figure list members.
	*/
	static int count(Figure const* const* figures);

	/** Construct a null-terminated list of pointers to null-terminated
	* lists of pointers [rotameres] to Figures.  [factory method]
	*/
	static Figure*** generateRotameres(Figure const* const* figures);

	/** Release (delete) Figure list list.
	*/
	static void release(Figure*** figureLists);

	/** Count Figure list list members.
	*/
	static int count(Figure const* const* const* figureLists);

	/** Number of cubes and array of cubes.  These data members are
	* public to allow fast inline operations.
	*/
	int m_size;
	Cube** m_cubes;

private :

	// common initializer
	void init(SetParser const& p);

	// common de-initializer
	void deinit();

	// cube iterator
	void doCubes(
		void (*what)(Cube& cube, void* data),
		void* data = 0,
		bool doNormalize = true);
};


#endif

