/* ctpz.cc -- C't journal 2003/3/24 puzzle, driver program.
**
** Copyright (C) 2003 Eric Laroche.  All rights reserved.
**
** @author Eric Laroche <laroche@lrdev.com>
** @version @(#)$Id: ctpzdr.cc,v 1.1 2003/04/30 21:34:12 laroche Exp $
** @url http://www.lrdev.com/lr/c/ctpz.html
**
** This program is free software;
** you can redistribute it and/or modify it.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


#include <iostream.h>
#include <fstream.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>


#include "ctpz.hh"
#include "ctpzsp.hh"
#include "ctpzt.hh"
#include "ctpzut.hh"
#include "ctpzvr.hh"


/** Usage message.
*/
static void usage(void)
{
	cout <<
		"usage: ctpz"
		" [-count]"
		" [-displayinput]"
		" [-threads #]"
		" [-verbose]"
		" [-vrml]"
		"\n";
	cout << "e.g. 'ctpz' (or 'ctpz -count') for solution count\n";
	cout << "e.g. 'ctpz -threads 4' for solution count on a 4 CPU machine\n";
	cout << "e.g. 'ctpz -vrml' for VRML representations\n";
	cout << "e.g. 'ctpz -displayinput -vrml' for VRML representation of input\n";
	exit(1);
}


/** Get a numerical argument.
*/
static int getNumericalArgument(char const* s)
{
	char* p = 0;
	int n = strtol(s, &p, 10);
	if (p == s) {
		usage();
	}
	return n;
}


/** Called back from solution generator, outputs VRML representations of
* the solutions.
*/
static void vrmlCallback(Figure** figures, void* data)
{
	int& count = *(int*)data;
	count++;

	char buf[64];
	sprintf(buf, "%s.%d.wrl", "ctpz", count);

	ofstream out(buf);
	VrmlPrinter::print(out, figures);
}


/** Application starting point.  Note: not cleaning up memory.
*/
int main(int argc, char** argv)
{
	// options
	bool count = false, displayinput = false, verbose = false, vrml = false;
	int threads = 0; // 0 means: don't use multiple threads

	// options parsing
	int i;
	for (i = 1; i < argc; i++) {
		if (strcmp(argv[i], "-count") == 0) {
			count = true;
		} else if (strcmp(argv[i], "-displayinput") == 0) {
			displayinput = true;
		} else if (strcmp(argv[i], "-threads") == 0) {
			if (++i >= argc) {
				usage();
			}
			threads = getNumericalArgument(argv[i]);
		} else if (strcmp(argv[i], "-verbose") == 0) {
			verbose = true;
		} else if (strcmp(argv[i], "-vrml") == 0) {
			vrml = true;
		} else if (strcmp(argv[i], "-?") == 0) {
			usage();
		} else if (argv[i][0] == '-') {
			usage();
		} else {
			usage();
		}
	}

	// sanity checks
	if (count && (displayinput || vrml)) {
		usage();
	}

	// default, the count, as requested by the puzzle
	if (!count && !displayinput && !vrml) {
		count = true;
	}

	if (count) {
		// print solution count
		if (threads == 0) {
			cout << Tile::count(
				Figure::generateRotameres(
					Figure::figureSetParse(ctpz)),
				(verbose ? &cout : 0)) << endl;
		} else {
			cout << Tile::countMt(
				Figure::generateRotameres(
					Figure::figureSetParse(ctpz)),
				threads,
				(verbose ? &cout : 0)) << endl;
		}
	} else if (displayinput) {
		if (vrml) {
			// print VRML representation of the input
			VrmlPrinter::printSeparated(
				cout,
				Figure::figureSetParse(ctpz));
		} else {
			// print string representation of the input
			cout << ctpz << endl;
		}
	} else if (vrml) {
		int n = 0; // counter
		Tile::generateSolutions(
			Figure::generateRotameres(Figure::figureSetParse(ctpz)),
			vrmlCallback,
			&n);
	}

	return 0;
}

