/* ctpz.cc -- C't journal 2003/3/24 puzzle, tiling (packing).
**
** Copyright (C) 2003 Eric Laroche.  All rights reserved.
**
** @author Eric Laroche <laroche@lrdev.com>
** @version @(#)$Id: ctpzt.cc,v 1.1 2003/04/30 21:34:12 laroche Exp $
** @url http://www.lrdev.com/lr/c/ctpz.html
**
** @reference c't 7/2003, p. 234 [c't puzzle]
** @reference http://www.lrdev.com/lr/c/sqfig.html [pentomino tilings]
**
** This program is free software;
** you can redistribute it and/or modify it.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


// implemented interface
#include "ctpzt.hh"


#include "ctpz.hh"
#include "ctpzmt.hh"

#include <iostream.h>
#include <string.h>
#include <time.h>


// [asserts commented out for speed reasons]
//#define assert(expression) {if (!(expression)) {cerr << "problem @" << __LINE__ << endl << flush;}}


// the tiling functions are optimized for speed, so there's less
// generality [i.e. much more hard coded stuff] and more constants

/** Number of pieces in the puzzle, number of pieces to place.
*/
#define PIECES 12

/** Maximal number of cubes in any input piece.  This is needed for fast
* helper data (piece position cache).
*/
#define MAXFIGURESIZE 6

/** 3D-board sizes.
*/
#define XDIMENSION 3
#define YDIMENSION 4
#define ZDIMENSION 5


/** Search the puzzle solutions.  This function is optimized for speed,
* so there's less generality, i.e. much hard coded stuff.  [recursive]
*/
static void tilingEngine(
	Figure const* const* const* figures, // [size: 'PIECES'; figure sizes variable]
	int board[XDIMENSION][YDIMENSION][ZDIMENSION], // layed out board (packing)
	int nextX, // next cube's X coordinate to use [first free board cell]
	int nextY, // next cube's Y coordinate to use [first free board cell]
	int nextZ, // next cube's Z coordinate to use [first free board cell]
	bool taken[PIECES],
	int order[2 * PIECES],
	int ndone,
	void (*found)( // 'found' callback
		int board[XDIMENSION][YDIMENSION][ZDIMENSION],
		int order[2 * PIECES],
		void* data),
	void* data, // callback data
	int nfrom = 0, // search range start, inclusive, for partial solutions / multithreading
	int nto = PIECES) // search range end, exclusive, for partial solutions / multithreading
{
	// recursion termination I [solution found]
	if (ndone == PIECES) {
		(*found)(board, order, data);
		return;
	}

	//assert(nfrom >= 0);
	//assert(nto <= PIECES);
	//assert(nfrom <= nto);
	// note: the above imply nfrom<=PIECES and nto>=0

	// iterate over the pieces
	int piece;
	for (piece = nfrom; piece < nto; piece++) {

		// check if it's been taken; skip if so
		if (taken[piece]) {
			continue;
		}

		// iterate over the rotameres
		int rotamere;
		Figure const* f;
		for (rotamere = 0; (f = figures[piece][rotamere]) != 0; rotamere++) {

			// [sanity check; could be earlier and once only (doesn't matter)]
			//assert(f->m_size <= MAXFIGURESIZE);

			// piece position cache
			int xc[MAXFIGURESIZE], yc[MAXFIGURESIZE], zc[MAXFIGURESIZE];

			//assert(f->m_cubes[0]->x == 0);
			//assert(f->m_cubes[0]->y == 0);
			//assert(f->m_cubes[0]->z == 0);

			// start at 1 (instead of 0) since first cube is at {0,0,0} and free
			xc[0] = nextX, yc[0] = nextY, zc[0] = nextZ;
			int i;
			for (i = 1; i < f->m_size; i++) {

				Cube const& c = *f->m_cubes[i];

				// calculate and check ranges
				// note: cubes' coordinates are always >=0,
				// so there's no check needed for <0

				int x = nextX + c.x;
				if (x < 0 || x >= XDIMENSION) {
					goto next;
				}

				int y = nextY + c.y;
				if (y < 0 || y >= YDIMENSION) {
					goto next;
				}

				int z = nextZ + c.z;
				if (z < 0 || z >= ZDIMENSION) {
					goto next;
				}

				// check for availability
				if (board[x][y][z] != -1) {
					goto next;
				}

				// cache [for speedup]
				xc[i] = x, yc[i] = y, zc[i] = z;
			}

			taken[piece] = true;

			// update board [using cached data]
			for (i = 0; i < f->m_size; i++) {
				//assert(xc[i] >= 0 && xc[i] < XDIMENSION);
				//assert(yc[i] >= 0 && yc[i] < YDIMENSION);
				//assert(zc[i] >= 0 && zc[i] < ZDIMENSION);

				board[xc[i]][yc[i]][zc[i]] = piece;
			}

			// next free board cell; linear search (only takes a few [up
			// to f->m_size] steps)
			// [declaration/initialization separation due to another
			// compiler bug encountered with the label below]
			int nx, ny, nz;
			nx = nextX, ny = nextY, nz = nextZ;
			while (nz < ZDIMENSION && board[nx][ny][nz] != -1) {
				nx++;
				if (nx == XDIMENSION) {
					nx = 0;
					ny++;
					if (ny == YDIMENSION) {
						ny = 0;
						nz++;
					}
				}
			}

			// solution representation; ususally used for status output
			order[2 * ndone] = piece;
			order[2 * ndone + 1] = rotamere;

			// recursion step
			tilingEngine(figures, board, nx, ny, nz, taken, order, ndone + 1, found, data);

			// reset board
			for (i = 0; i < f->m_size; i++) {
				board[xc[i]][yc[i]][zc[i]] = -1;
			}

			taken[piece] = false;

next :;
		}
	}

	// recursion termination II [search space exhausted, backtracking]
}


static void tile(
	Figure const* const* const* figures, // [size: 'PIECES'; variable figure sizes]
	void (*found)( // 'found' callback
		int board[XDIMENSION][YDIMENSION][ZDIMENSION],
		int order[2 * PIECES],
		void* data),
	void* data, // callback data
	int nfrom = 0,
	int nto = PIECES)
{
	// [sanity check]
	if (Figure::count(figures) != PIECES) {
		return;
	}

	// setup to find first solution

	int board[XDIMENSION][YDIMENSION][ZDIMENSION]; // layed out board
	int x, y, z;
	for (z = 0; z < ZDIMENSION; z++) {
		for (y = 0; y < YDIMENSION; y++) {
			for (x = 0; x < XDIMENSION; x++) {
				board[x][y][z] = -1;
			}
		}
	}

	bool taken[PIECES];
	int i;
	for (i = 0; i < PIECES; i++) {
		taken[i] = false;
	}

	int order[2 * PIECES];
	for (i = 0; i < PIECES; i++) {
		order[2 * i] = order[2 * i + 1] = -1;
	}

	// let the tiling engine produce solutions
	tilingEngine(figures, board, 0, 0, 0, taken, order, 0, found, data, nfrom, nto);
}


/** Normalize cube offsets suitable for the tiling engine.  Confine the
* rotameres of one unsymmetric piece to exclude even rotations too.
* This cuts the number of solutions by factor four and makes the search
* faster.
*/
static Figure*** generateNormalizedForTiling(Figure const* const* const* figureLists)
{
	int size = Figure::count(figureLists);
	Figure*** nft = new Figure**[size + 1]; // null terminated

	// copy
	int i, j, fsize;
	for (i = 0; i < size; i++) {
		fsize = Figure::count(figureLists[i]);
		nft[i] = new Figure*[fsize + 1]; // null terminated
		for (j = 0; j < fsize; j++) {
			nft[i][j] = new Figure(*figureLists[i][j]);
		}
		nft[i][fsize] = 0;
	}
	nft[size] = 0;

	// find first unsymmetric piece (if there's any)
	// this requires the figures to be still originally normalized
	// (for the compare operation), so this step must be done early
	int unsymmetric = -1, quiteUnsymmetric = -1;
	for (i = 0; i < size; i++) {
		fsize = Figure::count(nft[i]);

		// must have the 24 rotameres
		if (fsize < 24) {
			continue;
		}

		// at least a quite unsymmetric found
		if (quiteUnsymmetric == -1) {
			quiteUnsymmetric = i;
		}

		// mirror the first rotamere and search for it
		// (in the whole array [the first rotamere may be x-symmetrical])
		Figure m = Figure(*nft[i][0]);
		m.xMirror();
		for (j = 0; j < fsize; j++) {
			if (Figure::compareNormalized(m, *nft[i][j]) == 0) {
				break;
			}
		}
		if (j < fsize) {
			continue;
		}

		// found
		unsymmetric = i;
		break;
	}

	// see if there really is an unsymmetric piece, else use the quite
	// unsymmetric piece (which is sufficient too)
	if (unsymmetric == -1) {
		unsymmetric = quiteUnsymmetric;
	}
	//assert(unsymmetric != -1);

	// reduce the (possibly quite) unsymmetric piece's rotameres
	// to exclude rotameres obtained by rotating 180 degrees;
	// this cuts symmertic solutions by factor 4
	fsize = Figure::count(nft[unsymmetric]);
	for (i = 0; i < fsize; i++) {
		int rm;
		for (rm = 0; rm < 3; rm++) {
			Figure rr = Figure(*nft[unsymmetric][i]);
			if (rm == 0) {
				rr.xRotate();
				rr.xRotate();
			} else if (rm == 1) {
				rr.yRotate();
				rr.yRotate();
			} else if (rm == 2) {
				rr.zRotate();
				rr.zRotate();
			}
			j = i;
			while (j < fsize) {
				if (Figure::compareNormalized(rr, *nft[unsymmetric][j]) == 0) {
					delete nft[unsymmetric][j];
					memmove(
						&nft[unsymmetric][j],
						&nft[unsymmetric][j + 1],
						((fsize - (j + 1)) + 1) * sizeof(**nft));
					fsize--;
				} else {
					j++;
				}
			}
		}
	}
	//assert(fsize * 4 == 24);

	// normalize the first cubes to be @{0,0,0}
	for (i = 0; i < size; i++) {
		fsize = Figure::count(nft[i]);
		for (j = 0; j < fsize; j++) {
			nft[i][j]->move(
				-nft[i][j]->m_cubes[0]->x,
				-nft[i][j]->m_cubes[0]->y,
				-nft[i][j]->m_cubes[0]->z);
		}
	}

	return nft;
}


/** Count-callback.  Count and possibly output some status about the
* progress in the solution search.  Status printing slows things down of
* course (expensive tty operations, thread synchronizing).
*/
static void countCallback(
	int /*board*/[XDIMENSION][YDIMENSION][ZDIMENSION],
	int order[2 * PIECES],
	void* data)
{
	int& count = *(int*)((void**)data)[0];
	count++;

	// optional status part
	ostream* out = (ostream*)((void**)data)[1];
	if (out != 0) {

		// lock to not mix status outputs and to protect possibly
		// problematic ctime; [Lock is an auto-releasing thread
		// synchronization lock]
		Lock lock;

		time_t now = time(0);
		*out << "#" << count << " ";
		int i;
		for (i = 0; i < PIECES; i++) {
			if (i > 0) {
				*out << ".";
			}
			*out << (order[2 * i] + 1) << (char)('a' + order[2 * i + 1]);
		}
		*out << " ";
		*out << "@" << ctime(&now); // ctime already contains a newline
		*out << flush; // _flush_ status
	}
}


int Tile::count(Figure const* const* const* figures, ostream* status)
{
	Figure*** f = generateNormalizedForTiling(figures);

	int count = 0;

	void* data[2];
	data[0] = &count, data[1] = status;

	tile(f, countCallback, data);

	Figure::release(f);

	return count;
}


/** This function runs in its own thread.
*/
static int tileWrapper(void* data)
{
	Figure*** f = (Figure***)((void**)data)[2];
	int i = (int)((void**)data)[3];

	// confine the number of running threads
	Semaphore semaphore;

	tile(f, countCallback, data, i, i + 1);

	return 0;
}


int Tile::countMt(
	Figure const* const* const* figures,
	int threads,
	ostream* status)
{
	if (threads == -1) {
		threads = PIECES;
	}

	Figure*** f = generateNormalizedForTiling(figures);

	// dispatch each first figure to a thread but run only 'threads'
	// threads at the same time

	Semaphore::setCount(threads);

	int counts[PIECES];
	void* datas[PIECES][4]; // additional elements compared to 'count()'
	int threadIds[PIECES];
	int i;
	for (i = 0; i < PIECES; i++) {

		counts[i] = 0;

		datas[i][0] = &counts[i], datas[i][1] = status;
		datas[i][2] = f, datas[i][3] = (void*)i; // additionals

		threadIds[i] = Thread::startThread(tileWrapper, datas[i]);
	}

	// join threads
	for (i = 0; i < PIECES; i++) {
		Thread::waitForThread(threadIds[i]);
	}

	int count = 0;
	for (i = 0; i < PIECES; i++) {
		count += counts[i];
	}

	Figure::release(f);

	return count;
}


static Figure** boardToFigureList(int board[XDIMENSION][YDIMENSION][ZDIMENSION])
{
	// allocate
	Figure** f = new Figure*[PIECES + 1]; // null terminated
	int i;
	for (i = 0; i < PIECES; i++) {
		f[i] = new Figure(MAXFIGURESIZE);
		f[i]->m_size = 0; // reset size
	}
	f[i] = 0;

	// translate/fill
	// use the original figures order;
	// this usually won't produce a color gradent but somewhat reflect
	// the solution permutation
	int x, y, z;
	for (z = 0; z < ZDIMENSION; z++) {
		for (y = 0; y < YDIMENSION; y++) {
			for (x = 0; x < XDIMENSION; x++) {
				int p = board[x][y][z];

				//assert(p >= 0);
				//assert(p < PIECES);
				//assert(f[p]->m_size < MAXFIGURESIZE);

				Cube& c = *f[p]->m_cubes[f[p]->m_size];
				c.x = x, c.y = y, c.z = z;
				f[p]->m_size++;
			}
		}
	}

	return f;
}


static void callbackTranslator(
	int board[XDIMENSION][YDIMENSION][ZDIMENSION],
	int /*order*/[2 * PIECES],
	void* data)
{
	Figure** f = boardToFigureList(board);

	void (*callback)(Figure** figures, void* data) =
		*(void (**)(Figure**, void*))((void**)data)[0];
	void* d = *(void**)((void**)data)[1];

	(*callback)(f, d);

	Figure::release(f);
}


void Tile::generateSolutions(
	Figure const* const* const* figures,
	void (*callback)(Figure** figures, void* data),
	void* data)
{
	Figure*** f = generateNormalizedForTiling(figures);

	void* d[2];
	// note: yet another compiler problem with casting 'callback' to void*
	d[0] = &callback, d[1] = &data;

	tile(f, callbackTranslator, d);

	Figure::release(f);
}

