/* ctpzut.cc -- C't journal 2003/3/24 puzzle, utilities.
**
** Copyright (C) 2003 Eric Laroche.  All rights reserved.
**
** @author Eric Laroche <laroche@lrdev.com>
** @version @(#)$Id: ctpzut.cc,v 1.1 2003/04/30 21:34:12 laroche Exp $
** @url http://www.lrdev.com/lr/c/ctpz.html
**
** This program is free software;
** you can redistribute it and/or modify it.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


// implemented interface
#include "ctpzut.hh"


#include <string.h>


StringRange::StringRange(char const* s, int begin, int end)
{
	init(s, begin, end);
}


StringRange::StringRange(char const* s)
{
	// [catch null string]
	init(s, 0, (s != 0 ? strlen(s) : 0));
}


StringRange::StringRange(StringRange const& that)
{
	init(that.m_s, that.m_begin, that.m_end);
}


StringRange& StringRange::operator=(StringRange const& that)
{
	if (this != &that) {
		init(that.m_s, that.m_begin, that.m_end);
	}

	return *this;
}


void StringRange::init(char const* s, int begin, int end)
{
	// pre-init to unset
	m_s = 0;
	m_begin = m_end = 0;

	if (s == 0 || end < begin) {
		return;
	}
	// [note: this implicitly allows negative offsets
	// and end past nul byte]

	m_s = s;
	m_begin = begin;
	m_end = end;
}


// note: read-only data
static char const openingBrace = '{';
static char const closingBrace = '}';
static char const comma = ',';
static char const blank = ' ';


SetParser::SetParser(StringRange const& sr) :
	m_sr(0, 0, 0) // [init to unset]
{
	init(sr);
}


SetParser::SetParser(char const* s) :
	m_sr(0, 0, 0) // [init to unset]
{
	// init via temporary object
	init(StringRange(s));
}


// skip blanks
static void skipBlanks(StringRange& sr, bool fromEnd = false)
{
	if (sr.m_s == 0) {
		return;
	}

	// which to test/move
	int& which = (fromEnd ? sr.m_end : sr.m_begin);
	int offset = (fromEnd ? -1 : 0);
	int direction = (fromEnd ? -1 : 1);

	// [don't run over 'end']
	while (sr.m_begin < sr.m_end && sr.m_s[which + offset] == blank) {
		which += direction;
	}
}


void SetParser::init(StringRange const& sr)
{
	if (sr.m_s == 0) {
		return;
	}

	// need (yet another) temporary to let const 'sr' intact and
	// preserve initial state
	StringRange srt = sr;

	// skip initial blanks
	skipBlanks(srt);

	// check for opening brace (required), skip it
	if (srt.m_begin >= srt.m_end || srt.m_s[srt.m_begin] != openingBrace) {
		return;
	}
	srt.m_begin++;

	// skip more blanks
	skipBlanks(srt);

	// skip terminating blanks
	skipBlanks(srt, true);

	// check for closing brace (required), skip it
	if (srt.m_begin >= srt.m_end || srt.m_s[srt.m_end - 1] != closingBrace) {
		return;
	}
	srt.m_end--;

	// skip more blanks
	skipBlanks(srt, true);

	// skip trailing comma, skip it
	if (srt.m_begin < srt.m_end && srt.m_s[srt.m_end - 1] == comma) {
		srt.m_end--;
	}

	// skip more blanks
	skipBlanks(srt, true);

	if (srt.m_begin >= srt.m_end) {
		return;
	}

	m_sr = srt;
}


static void countCallback(StringRange const& /*sr*/, void* userData)
{
	int& count = *(int*)userData;
	count++;
}


int SetParser::count() const
{
	int count = 0;
	enumerate(countCallback, &count);
	return count;
}


void SetParser::enumerate(
	void (*callback)(StringRange const& sr, void* userData),
	void* userData
) const
{
	if (m_sr.m_s == 0) {
		return;
	}

	StringRange sri = m_sr; // iterator range

	// preconditions: leading and trailing blanks, commas, etc
	// have been excluded

	while (sri.m_begin < sri.m_end) {

		int end2 = sri.m_begin;

		// check if the set element is a set itself
		if (sri.m_s[end2] == openingBrace) {

			// find the _corresponding_ closing brace
			end2++;
			int braceCount = 1;
			while (end2 < sri.m_end) {

				if (sri.m_s[end2] == openingBrace) {
					braceCount++;
				} else if (sri.m_s[end2] == closingBrace) {
					braceCount--;
				}

				if (braceCount == 0) {
					break;
				}

				end2++;
			}

			// check for mismatch ["unmatched opening brace"]
			if (end2 == sri.m_end) {
				return;
			}

			end2++;

		} else { // element is not a set

			// find a comma or the end of the range
			while (end2 < sri.m_end) {

				if (sri.m_s[end2] == comma) {
					break;
				}

				end2++;
			}
		}

		// trim blanks on temporary
		StringRange srt = StringRange(sri.m_s, sri.m_begin, end2);
		skipBlanks(srt);
		skipBlanks(srt, true);

		// issue the callback
		(*callback)(srt, userData);

		sri.m_begin = end2;

		// skip blanks, comma, and blanks
		skipBlanks(sri);
		if (sri.m_begin < sri.m_end && sri.m_s[sri.m_begin] == comma) {
			sri.m_begin++;
			skipBlanks(sri);
		}
	}
}

