/* ctpzvr.cc -- C't journal 2003/3/24 puzzle, virtual reality modeling.
**
** Copyright (C) 1998,2003 Eric Laroche.  All rights reserved.
**
** @author Eric Laroche <laroche@lrdev.com>
** @version @(#)$Id: ctpzvr.cc,v 1.1 2003/04/30 21:34:12 laroche Exp $
** @url http://www.lrdev.com/lr/c/ctpz.html
**
** @reference http://www.lrdev.com/lr/math/glass.html [vrml glass]
**
** This program is free software;
** you can redistribute it and/or modify it.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


// implemented interface
#include "ctpzvr.hh"


#include "ctpz.hh"

#include <iostream.h>
#include <string.h>


/** Fraction denominator used in 'putFraction', etc.  Must be a power of
* ten.
*/
static int const fractionDenominator = 1000;


/** Do fractions normalized to 'fractionDenominator'.  The output is
* minimized, i.e. trailing zeroes are truncated.
*/
static char* fractionOnly(char* buf, int size, int n)
{
	// catch sick input
	if (size <= 0 || buf == 0 || n < 0 || n >= fractionDenominator) {
		return 0;
	}

	int i = 0; // size overflow checked too (truncated, typically)

	if (n != 0 && i < size - 1) { // zero does not produce output
		buf[i++] = '.';

		int d = fractionDenominator / 10;
		while (n != 0 && i < size - 1) { // trailing zeroes cut
			buf[i++] = "0123456789"[n / d];
			n %= d;
			d /= 10;
		}
	}

	buf[i] = '\0';

	return buf;
}


/** Do fractions normalized to 'fractionDenominator'.  The output is
* minimized, i.e. trailing zeroes are truncated and dot omitted if
* possible.
*/
static char* fraction(char* buf, int size, int n)
{
	// catch sick input
	if (size <= 0 || buf == 0) {
		return 0;
	}

	int i = 0; // size overflow checked too (truncated, typically)

	if (n < 0 && i < size - 1) {
		buf[i++] = '-';
		n = -n; // [note: may not cover 'INT_MIN']
	}

	int m = n / fractionDenominator;

	int d = 1;
	while (d <= m) {
		// [note: may overflow near 'INT_MAX'
		// (which will however not be the case
		// since divided by 'fractionDenominator' above)]
		d *= 10;
	}

	// catch 0 [0 is represented with the same number of digits
	// as 1, 2, etc are (i.e.: there's no 'empty' representation)]
	if (d == 1) {
		d *= 10;
	}

	d /= 10;
	while (d != 0 && i < size - 1) {
		buf[i++] = "0123456789"[m / d];
		m %= d;
		d /= 10;
	}

	buf[i] = '\0';

	n %= fractionDenominator;

	if (n != 0) {
		fractionOnly(&buf[i], size - i, n);
	}

	return buf;
}


/** Print fractions normalized to 'fractionDenominator'.  The output is
* minimized, i.e. trailing zeroes are truncated and dot omitted if
* possible.
*/
static void putFraction(ostream& out, int n)
{
	char buf[128];
	out << fraction(buf, sizeof(buf), n);
}


/** Print a color component normalized to 255 as fraction.
*/
static void putColorComponent(ostream& out, int c)
{
	// cut values outside range
	if (c > 255) {
		c = 255;
	} else if (c < 0) {
		c = 0;
	}

	// normalize to 'fractionDenominator'th
	putFraction(out, (c * fractionDenominator) / 255);
}


/** Print an RGB (red/green/blue component) color value.
*/
static void putRgb(ostream& out, int r, int g, int b)
{
	putColorComponent(out, r);
	out << " ";
	putColorComponent(out, g);
	out << " ";
	putColorComponent(out, b);
}


/** Convert HSB (hue/saturation/brightness component) color value to RGB
* (red/green/blue component) color value.
*/
static void rgbFromHsb(int h, int s, int b, int& rr, int& gg, int& bb)
{
	// normalize values
	// [hue modulus, rest clipped]
	h %= 256;
	if (h < 0) {
		h += 256;
	}
	if (s < 0) {
		s = 0;
	} else if (s > 255) {
		s = 255;
	}
	if (b < 0) {
		b = 0;
	} else if (b > 255) {
		b = 255;
	}

	// chose from 6 sectors
	int m = 6 * h;
	int i = m / 256; // choser

	int f = m - 256 * i; // hue relative to chosen sector
	int p = (b * (255 - s)) / 255; // 'greyness'
	int q = (b * (255 - (s * f) / 255)) / 255; // additional component
	int t = (b * (255 - (s * (255 - f)) / 255)) / 255; // additional component

	switch (i) {
	case 0 :
		rr = b, gg = t, bb = p;
		break;
	case 1 :
		rr = q, gg = b, bb = p;
		break;
	case 2 :
		rr = p, gg = b, bb = t;
		break;
	case 3 :
		rr = p, gg = q, bb = b;
		break;
	case 4 :
		rr = t, gg = p, bb = b;
		break;
	case 5 :
		rr = b, gg = p, bb = q;
		break;
	}
}


/** Print an HSB (hue/saturation/brightness component) color value.
*/
static void putHsb(ostream& out, int h, int s, int b)
{
	int rr = 0, gg = 0, bb = 0;
	rgbFromHsb(h, s, b, rr, gg, bb);
	putRgb(out, rr, gg, bb);
}


static void doIndexDeltas(
	ostream& out,
	char const* indent,
	int origin,
	int delta1,
	int delta2)
{
	out << indent;
	out << origin << ", ";
	out << (origin + delta1) << ", ";
	out << (origin + delta1 + delta2) << ", ";
	out << (origin + delta2) << ", ";
	out << "-1,";
	out << "\n";
}


static void printCube(
	ostream& out,
	char const* indent,
	Cube const& cube,
	int tunnels = 309) // 0.309, half of golden ratio 0.618
{
	// cube entitity; contains coordinates and indices
	out << indent << "" << "Separator" << " " << "{" << "\n";

	// points [8 vertices each cube]
	out << indent << " " << "Coordinate3" << " " << "{" << "\n";
	out << indent << "  " << "point" << " " << "[" << "\n";

	// for each of 3 dimensions: points at 0 and 1
	int sx, sy, sz;
	out << indent << "   ";
	for (sz = 0; sz <= 1; sz++) {
		for (sy = 0; sy <= 1; sy++) {
			for (sx = 0; sx <= 1; sx++) {
				if (sx != 0 || sy != 0 || sz != 0) {
					out << " ";
				}
				out << (cube.x + sx) << " ";
				out << (cube.y + sy) << " ";
				out << (cube.z + sz) << ",";
			}
		}
	}
	out << "\n";

	// additional optional tunnel coordinates
	if (tunnels != 0) {

		int t[2];
		t[0] = (fractionDenominator - tunnels) / 2;
		t[1] = fractionDenominator - t[0];

		// pre-fetch fraction the representations
		char tt[2][64];
		int i;
		for (i = 0; i < 2; i++) {
			tt[i][0] = '\0';
			fractionOnly(tt[i], sizeof(tt[i]), t[i]);
		}

		// a smaller cube inside
		out << indent << "   ";
		for (sz = 0; sz <= 1; sz++) {
			for (sy = 0; sy <= 1; sy++) {
				for (sx = 0; sx <= 1; sx++) {
					if (sx != 0 || sy != 0 || sz != 0) {
						out << " ";
					}
					out << cube.x << tt[sx] << " ";
					out << cube.y << tt[sy] << " ";
					out << cube.z << tt[sz] << ",";
				}
			}
		}
		out << "\n";
	}

	// points end
	out << indent << "  " << "]" << "\n";
	out << indent << " " << "}" << "\n";

	// polygons [6 squares each cube]
	out << indent << " " << "IndexedFaceSet" << " " << "{" << "\n";
	out << indent << "  " << "coordIndex" << " " << "[" << "\n";

	char nextIndent[64];
	nextIndent[0] = '\0';
	if (strlen(indent) < sizeof(nextIndent) - 16) {
		strcpy(nextIndent, indent);
	}
	strcat(nextIndent, "   ");

	// for each of 3 dimensions: do the two sides (6 sides total)
	int dimension;
	for (dimension = 0; dimension < 3; dimension++) {
		int deltaA = 1 << dimension;
		int deltaB = 1 << ((dimension + 1) % 3);
		int deltaC = 1 << ((dimension + 2) % 3);

		// 'tunnel dimension'
		int delta4 = 1 << 3;

		int origin;
		for (origin = 0; origin <= deltaA; origin += deltaA) {

			if (tunnels == 0) {

				// span a polygon [a square] from origin [either zero or
				// deltaA] to origin+deltaB to origin+deltaB+deltaC to
				// origin+deltaC [and back to origin]
				doIndexDeltas(out, nextIndent, origin, deltaB, deltaC);

			} else {

				// span four polygons from the outer cube's side to the
				// inner's; avoid duplicate polygons: round #1
				// (dimension 0) is completely done, which will already
				// cover 8 out of 12 polygons, round #2 is only done if
				// round #1's origin delta is in use [additional 4
				// polygons], round #3 is left out completely

				if (dimension == 0 || (dimension == 1 && deltaB == 1)) {
					doIndexDeltas(out, nextIndent, origin, deltaB, delta4);
					doIndexDeltas(out, nextIndent, origin + deltaB + deltaC, -deltaB, delta4);
				}
				if (dimension == 0 || (dimension == 1 && deltaC == 1)) {
					doIndexDeltas(out, nextIndent, origin + deltaB, deltaC, delta4);
					doIndexDeltas(out, nextIndent, origin + deltaC, -deltaC, delta4);
				}
			}
		}
	}

	// polygons end
	out << indent << "  " << "]" << "\n";
	out << indent << " " << "}" << "\n";

	out << indent << "" << "}" << "\n";
}


static void printFigures(ostream& out, char const* indent, Figure const* const* figures)
{
	// get figure count [needed for color choser, etc]
	int nfigures = Figure::count(figures);

	// loop over the figures
	int i, j;
	for (i = 0; i < nfigures; i++) {

		// figure entitity; contains 'Material' property
		out << indent << "" << "Separator" << " " << "{" << "\n";

		out << indent << " " << "#" << " " << "figure" << " " << "#" << (i + 1) << "\n";

		// color, etc
		out << indent << " " << "Material" << " " << "{" << "\n";
		out << indent << "  " << "#" << " " << "hue" << " " << i << "/" << nfigures << "\n";
		out << indent << "  " << "diffuseColor" << " ";
		// hue calculated linearly from i;
		// 'nfigures' used as denominator (instead of 'nfigures-1'
		// since hues 0 and 1 are the same
		putHsb(
			out,
			(255 * i) / nfigures,
			255, // full saturation
			255); // full brightness
		out << "\n";
		out << indent << "  " << "shininess" << " " << "0.5" << "\n";
		out << indent << " " << "}" << "\n";

		// loop over the cubes
		for (j = 0; j < figures[i]->m_size; j++) {

			char nextIndent[64];
			nextIndent[0] = '\0';
			if (strlen(indent) < sizeof(nextIndent) - 16) {
				strcpy(nextIndent, indent);
			}
			strcat(nextIndent, " ");

			printCube(out, nextIndent, *figures[i]->m_cubes[j]);
		}

		out << indent << "" << "}" << "\n";
	}
}


void VrmlPrinter::print(ostream& out, Figure const* const* figures)
{
	// VRML 1.0 header
	out << "" << "#VRML V1.0 ascii" << "\n";
	out << "" << "# generated by 'ctpz' [see http://www.lrdev.com/lr/c/ctpz.html]" << "\n";

	// VRML root object
	out << "" << "Separator" << " " << "{" << "\n";

	// turn 'z coordinate to the vertical
	out << " " << "MatrixTransform" << " " << "{" << "\n";
	out << "  " << "matrix" << "\n";
	out << "   " << "1 0 0 0" << "\n";
	out << "   " << "0 0 -1 0" << "\n";
	out << "   " << "0 1 0 0" << "\n";
	out << "   " << "0 0 0 1" << "\n";
	out << " " << "}" << "\n";

	// white background
	out << " " << "# white background" << "\n";
	out << " " << "DEF BackgroundColor Info {string \"1 1 1\"}" << "\n";

	printFigures(out, " ", figures);

	// end root object
	out << "" << "}" << "\n";
}


void VrmlPrinter::printSeparated(ostream& out, Figure const* const* figures)
{
	// copy
	Figure** f = new Figure*[Figure::count(figures) + 1];

	int x = 0, y = 0, z = 0; // figure offset(s)

	int i;
	for (i = 0; figures[i] != 0; i++) {
		f[i] = new Figure(*figures[i]);

		// get min/max info
		int minX, minY, minZ, maxX, maxY, maxZ;
		f[i]->getMinMax(minX, minY, minZ, maxX, maxY, maxZ);

		// normalize; consider min values and figure offset(s)
		f[i]->move(x - minX, y - minY, z - minZ);

		// update offset(s); x is gapped by 2
		x += (maxX - minX) + 2;
	}
	f[i] = 0;

	print(out, f);

	Figure::release(f);
}

