/*
 *  Lsung des ct'-Puzzles. 
 *
 *  Florian Gagel, f.gagel@onlinehome.de.
 *
 *  $Id: sct2.cpp,v 1.6 2003/04/30 15:51:48 flo Exp flo $
 *
 *  Vorgehen: Es wird der Wrfel successive aufgefllt, wobei
 *            die mglichen Zge nach erstem gesetztem Bit sortiert sind.
 *            Auf die Zge noch freien Teile wird ber eine Lookup-Tabelle
 *            direkt zugegriffen.
 *            Symmetrieeliminierung ber Teil 6 (keine Drehsymmetrien).
 *
 *  bersetzt mit g++-3.2 -pipe -Wall -DNDEBUG -O3 -fomit-frame-pointer 
 *                        -mcpu=pentium4 -o sct2.cpp sct2.cpp
 *
 *  Zeit: 24 min. auf Athlon 2000+, 512 MB SDRAM.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#define DIM_X 5
#define DIM_Y 4
#define DIM_Z 3

#define NTEILE 12

typedef int T_Point[3];

// Teile, wie unter http://www.heise.de/ct/puzzle/teile.shtml abgebildet
T_Point Teil_0 [5] = {{0,0,0},{1,0,0},{2,0,0},{0,1,0},{2,1,0}},
        Teil_1 [5] = {{1,0,0},{0,1,0},{1,1,0},{2,1,0},{0,2,0}}, 
        Teil_2 [6] = {{0,0,0},{1,0,0},{2,0,0},{3,0,0},{1,1,0},{3,1,0}},
        Teil_3 [5] = {{1,0,0},{1,1,0},{0,2,0},{1,2,0},{2,2,0}},
        Teil_4 [5] = {{0,0,0},{0,1,0},{0,1,1},{0,1,2},{0,2,2}},
        Teil_5 [5] = {{0,0,0},{1,0,0},{2,0,0},{0,1,0},{1,1,0}},
        Teil_6 [5] = {{0,0,0},{1,0,0},{0,1,0},{0,0,1},{0,1,1}},
        Teil_7 [5] = {{1,0,0},{0,1,0},{1,1,0},{2,1,0},{1,2,0}},
        Teil_8 [4] = {{0,1,0},{1,1,0},{0,0,1},{0,1,1}},
        Teil_9 [5] = {{0,0,0},{1,0,0},{2,0,0},{3,0,0},{0,1,0}},
        Teil_10[5] = {{0,0,0},{0,1,0},{1,1,0},{1,2,0},{2,2,0}},
        Teil_11[5] = {{0,0,0},{1,0,0},{2,0,0},{3,0,0},{1,1,0}}; // gecheckt

typedef struct
{
    int ind_a, ind_b, ind_c;
    int sign_a, sign_b, sign_c;
} T_Rot;

#define NROT 24 /* Zahl der Raumdrehungen */
class T_RotMap // Drehungsklasse
{
    T_Rot Rotation[NROT]; // Eigentliche Tabelle der Drehungen (Indices u. VZ-Info)

  public:

/* Rotationstabelle generieren: Es werden saemtliche (vorzeichenbehaftete)
 * Permutationen generiert, aber nur die zugelassen, deren zugeordnete
 * Matrix die Determinate 1 hat, also einer Drehmatrix entspricht.
 */
    T_RotMap()
    {
        T_Rot *p = Rotation;
        for (int a=-3; a<=3; a+= ((a==-1) ? 2:1))
        for (int b=-3; b<=3; b+= ((b==-1) ? 2:1)) if (abs(b)!=abs(a))
        for (int c=-3; c<=3; c+= ((c==-1) ? 2:1)) if (abs(c)!=abs(a) && abs(c)!=abs(b))
        {
            int det = (a==1 || a==3 || a==-2) ? 1 : -1;
            det *= (abs(b) < abs(c)) ? 1:-1;
            if (b < 0) det *= -1;
            if (c < 0) det *= -1;
            if (det ==1) 
            {
                p->ind_a = abs(a)-1;
                p->ind_b = abs(b)-1;
                p->ind_c = abs(c)-1;
		p->sign_a = (a>0) ? 1:-1;
		p->sign_b = (b>0) ? 1:-1;
		p->sign_c = (c>0) ? 1:-1;
                ++p;
            }
        }
        assert((p-Rotation)==NROT);
    }
    T_Rot & operator [] (int i) {return Rotation[i];}
}; // T_RotMap

static T_RotMap RotMap;


// Tabelle des ersten Punkts = 60 - max. Blockanzahl pro Teil -1
#define FDIM (DIM_X*DIM_Y*DIM_Z - 3)

#define MAX_MOVE 64

typedef struct 
{
    int nMove;
    long long Move[MAX_MOVE]; // Bitfeld der Positionen
} T_Moves;


// Tabelle der mglichen Zge: Erste Dim = Erstes belegtes Bit von Zug,
// zweite Dim: Teileindex
T_Moves MTab[FDIM][NTEILE];

void InitMTab()
{
    for (int i=0; i<FDIM; i++)
	for (int j=0; j<NTEILE; j++)
	    MTab[i][j].nMove = 0;
}

// Teil in MTab eintragen (Rotation + Translation)
void AddTeil(int iTeil, int nBlock, T_Point *Block)
{
    nBlock -= 1; // Bloecke von 0 bis nBlock jetzt
    for (int nrot = NROT-1; nrot >= 0; --nrot)
    {
        T_Rot &R = RotMap[nrot];

        T_Point Rotated[6];
	
        int minx, maxx, miny, maxy, minz, maxz;

      // Berechnung der transformierten Figur und ihres Ferrets
        for (int j = 0; j<=nBlock; j++)
        {
	    Rotated[j][0] = Block[j][R.ind_a]*R.sign_a; 
	    Rotated[j][1] = Block[j][R.ind_b]*R.sign_b; 
	    Rotated[j][2] = Block[j][R.ind_c]*R.sign_c; 

	    if (j==0)
            {
                minx = maxx = Rotated[0][0];
                miny = maxy = Rotated[0][1];
                minz = maxz = Rotated[0][2];
	    }
            else
	    {
	        if (Rotated[j][0] < minx) minx = Rotated[j][0];
	        else if (Rotated[j][0] > maxx) maxx = Rotated[j][0];

	        if (Rotated[j][1] < miny) miny = Rotated[j][1];
	        else if (Rotated[j][1] > maxy) maxy = Rotated[j][1];

	        if (Rotated[j][2] < minz) minz = Rotated[j][2];
	        else if (Rotated[j][2] > maxz) maxz = Rotated[j][2];
            }
	}
      // Translationen der transformierten Figur in 5x4x3 Box
        for (int i=-minx; i<DIM_X-maxx; i++) 
        for (int j=-miny; j<DIM_Y-maxy; j++) 
        for (int k=-minz; k<DIM_Z-maxz; k++) 
        {
            long long mask = (long long) 0;
	    int firstPos = FDIM;
            for (int l=nBlock; l>=0; --l)
            {
                int a = Rotated[l][0]+i;
	        int b = Rotated[l][1]+j;
	        int c = Rotated[l][2]+k;
	        assert(a >=0 && a < DIM_X);
	        assert(b >=0 && b < DIM_Y);
	        assert(c >=0 && c < DIM_Z);

	        int pos = c + b*DIM_Z +a*(DIM_Z*DIM_Y);
                mask |= (1LL << pos);
	        if (pos < firstPos) firstPos = pos;
            }
	    assert(firstPos < FDIM);
	    assert( mask & ((1LL) << firstPos));

	    T_Moves &Mv = MTab[firstPos][iTeil];
          // Symmetrien eliminieren
	    int q = Mv.nMove;
	    while (--q >= 0)
	    {
	        if (Mv.Move[q] == mask) break;
            }
	    if (q<0) 
	    {
	        Mv.Move[Mv.nMove++] = mask; // uebernehmen
	    }
        } 
    }
} // AddTeil


// Bitdefinitonen fr Koordinaten-Swaps
#define A0 0xfffLL
#define A1 0xfff000LL
#define A2 0xfff000000LL
#define A3 0xfff000000000LL
#define A4 0xfff000000000000LL
inline long long swapx(long long w)
{
    return  ((w&A0)<<48) |
	    ((w&A1)<<24) |
	    ((w&A2)) |
	    ((w&A3) >>24) |
	    ((w&A4) >> 48);
}

#define BM0 0x7007007007007LL
#define BM3 0x38038038038038LL
#define BM6 0x1c01c01c01c01c0LL
#define BM9 0xe00e00e00e00e00LL
inline long long swapy(long long w)
{
    return  ((w&BM0)<<9) |
	    ((w&BM3)<<3) |
	    ((w&BM6)>>3) |
	    ((w&BM9)>>9);
}

#define C0 0x249249249249249LL
#define C1 0x492492492492492LL
#define C2 0x924924924924924LL
inline long long swapz(long long w)
{
    return  (w&C1) |
	    ((w&C0)<<2) |
	    ((w&C2)>>2);
}

inline int firstBit(long long w)
{
    int i=0;
    while (((1LL << i)&w)==0LL) ++i;
    return i;
}

// Symmetrien von Teil Nr. i eliminieren, in MTab
void DelSym(int iTeil)
{
    for (int i=0; i<FDIM; i++)
    {
	 T_Moves &Mv = MTab[i][iTeil];
	 for (int m=0; m < Mv.nMove; m++)
	 {
	    // Symmetrische Positionen
	      long long sym[3];

	      sym[0] = swapy(swapx(Mv.Move[m]));
	      sym[1] = swapz(swapy(Mv.Move[m]));
	      sym[2] = swapx(swapz(Mv.Move[m]));
             
              for (int k=0; k<3; k++)
	      {
	          int fb = firstBit(sym[k]);
		  if (fb <i) break;
		  if (fb==i)
		  {
		      for (int j=m+1; j<Mv.nMove; j++)
		      {
		          if (Mv.Move[j]==sym[k])
			  {
			      Mv.Move[j] = Mv.Move[--Mv.nMove];
			      break;
			  }
			  assert("Huh ?");
		      }
		  }
		  else //fb > i
		  {
                      T_Moves &Mq = MTab[fb][iTeil];
		      for (int j=0; j<Mq.nMove; j++)
		      {
			   if (Mq.Move[j]==sym[k])
			   {
			       Mq.Move[j] = Mq.Move[--Mq.nMove];
			       break;
			   }
		      }
		      assert("Huh ?");
		  }
	      }
	 }
    }
} // DelSym

// Bestimmung des ersten Null-Bits (bsfl-Maschinenbefehl)
inline int first0(long long l)
{
    register int pos; 

    register int a = ~(((unsigned int *) &l)[0]);
    if (a==0)
    {
        register int b = ~(((unsigned int *) &l)[1]);	    
        __asm__ __volatile__
        (
            "bsfl %1,%0 \n\t"
            :
            "=r" (pos)     // Bitindex
            :
            "r"  (b)      // Argument
        );
	return pos+32;
    }
    else
    {
        __asm__ __volatile__
        (
            "bsfl %1,%0 \n\t"
            :
            "=r" (pos)    // Bitindex
            :
            "r"  (a)      // Argument
        );
        return pos;
    }
}

#define GMOVE_MAX 128

typedef struct
{
    long long Move;
    int TeilNeu;
} T_GMove;

typedef struct
{
    int nMove;
    T_GMove GMove[GMOVE_MAX];
} T_Q;

// Zuegetabelle, nach erstem Bit und (Bitfeld der noch freien Teile)
T_Q QTab[FDIM][2048];

int NSolution;

// Lsungsfunktionen nach Level der gesetzten Teile, identisch bis auf 0,10.
void Solve10(int TeileMask, long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][TeileMask];
    int j = Q.nMove;
    while (j-- != 0)
    {
        if (!(Q.GMove[j].Move & Blocks))
        {
	    ++NSolution;
	    if ((NSolution & 0x7ff) == 0) printf("[%i]\n", NSolution);
	    return; // keine weitere Lsung mglich
	}
    }
} // Solve10

void Solve9(int TeileMask, long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][TeileMask];
    int j = Q.nMove;
    while (j-- != 0)
    {
        if (!(Q.GMove[j].Move & Blocks))
        {
	    Solve10( Q.GMove[j].TeilNeu, Blocks | (Q.GMove[j].Move));
	}
    }
} // Solve9

void Solve8(int TeileMask, long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][TeileMask];
    int j = Q.nMove;
    while (j-- != 0)
    {
        if (!(Q.GMove[j].Move & Blocks))
        {
	    Solve9( Q.GMove[j].TeilNeu, Blocks | (Q.GMove[j].Move));
	}
    }
} // Solve8

void Solve7(int TeileMask, long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][TeileMask];
    int j = Q.nMove;
    while (j-- != 0)
    {
        if (!(Q.GMove[j].Move & Blocks))
        {
	    Solve8( Q.GMove[j].TeilNeu, Blocks | (Q.GMove[j].Move));
	}
    }
} // Solve7

void Solve6(int TeileMask, long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][TeileMask];
    int j = Q.nMove;
    while (j-- != 0)
    {
        if (!(Q.GMove[j].Move & Blocks))
        {
	    Solve7( Q.GMove[j].TeilNeu, Blocks | (Q.GMove[j].Move));
	}
    }
} // Solve6

void Solve5(int TeileMask, long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][TeileMask];
    int j = Q.nMove;
    while (j-- != 0)
    {
        if (!(Q.GMove[j].Move & Blocks))
        {
	    Solve6( Q.GMove[j].TeilNeu, Blocks | (Q.GMove[j].Move));
	}
    }
} // Solve5

void Solve4(int TeileMask, long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][TeileMask];
    int j = Q.nMove;
    while (j-- != 0)
    {
        if (!(Q.GMove[j].Move & Blocks))
        {
	    Solve5( Q.GMove[j].TeilNeu, Blocks | (Q.GMove[j].Move));
	}
    }
} // Solve4

void Solve3(int TeileMask, long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][TeileMask];
    int j = Q.nMove;
    while (j-- != 0)
    {
        if (!(Q.GMove[j].Move & Blocks))
        {
	    Solve4( Q.GMove[j].TeilNeu, Blocks | (Q.GMove[j].Move));
	}
    }
} // Solve3

void Solve2(int TeileMask, long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][TeileMask];
    int j = Q.nMove;
    while (j-- != 0)
    {
        if (!(Q.GMove[j].Move & Blocks))
        {
	    Solve3( Q.GMove[j].TeilNeu, Blocks | (Q.GMove[j].Move));
	}
    }
} // Solve2

void Solve1(int TeileMask, long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][TeileMask];
    int j = Q.nMove;
    while (j-- != 0)
    {
        if (!(Q.GMove[j].Move & Blocks))
        {
	    Solve2( Q.GMove[j].TeilNeu, Blocks | (Q.GMove[j].Move));
	}
    }
} // Solve1

void Solve0(long long Blocks)
{
    T_Q & Q = QTab[first0(Blocks)][0x7ff];
    int j = Q.nMove;
    while (j-- != 0)
    {
        Solve1( Q.GMove[j].TeilNeu, Q.GMove[j].Move | Blocks);
    }
} // Solve0

int main() 
{ 
  // Aufstellen der Zugtabelle
    InitMTab();	
    AddTeil(0, 5, Teil_0);
    AddTeil(1, 5, Teil_1);
    AddTeil(2, 6, Teil_2);
    AddTeil(3, 5, Teil_3);
    AddTeil(4, 5, Teil_4);
    AddTeil(5, 5, Teil_5);
    AddTeil(6, 5, Teil_7);
    AddTeil(7, 4, Teil_8);
    AddTeil(8, 5, Teil_9);
    AddTeil(9, 5, Teil_10);
    AddTeil(10, 5, Teil_11);
    AddTeil(11, 5, Teil_6);

    DelSym(11); // Symmetrien von Teil 11 eliminieren

    NSolution = 0;

  // Teil 11 wird direkt gesetzt => kleinere Dimension von MTab
    for (int i=0; i<FDIM; i++) for (int j=0; j<MTab[i][11].nMove; j++)
    {
	long long mask = MTab[i][11].Move[j];
      // Aufstellen von QTab, vertraeglich mit Teil 11
        for (int i2=FDIM-1; i2>=0; --i2)
	{
            for (int iq=0;iq<2048; iq++)
	    {
		T_Q & Q = QTab[i2][iq];
		Q.nMove = 0;
	        for (int it=0; it<11; it++) if ((1<<it) & iq)
		{
		    int NeuMask = (iq ^ (1<<it));
		    T_Moves &Mv = MTab[i2][it];
		    for (int j3=0; j3 < Mv.nMove; j3++) if (!(Mv.Move[j3]&mask))
		    {
			 assert(Q.nMove < GMOVE_MAX);
			 Q.GMove[Q.nMove].Move = Mv.Move[j3];
			 Q.GMove[Q.nMove].TeilNeu = NeuMask;
			 ++Q.nMove;
		    }
		}
	    }
	}
      // Starten von Run 
        Solve0(mask);
    }

    printf("\n=== Solutions: %i.\n", NSolution);
   
    return 0;
} // main


