/*   ct-puzzle-solver.cpp
 *
 * Programm zur Bestimmung der Anzahl der Lsungen eines ct-Puzzles
 * (c't 2003, Heft 7, Seite 234)
 *
 * Copyright (C) 2003 Georgios Papoutsis <gepap@gmx.de>
 *
 *
 * Kompilieren mit:
 *
 *   g++ -O3 -fomit-frame-pointer -o ct-puzzle-solver ct-puzzle-solver.cpp
 *
 * Mit Benutzung von Threads (liefert evtl. Zeitvorteil, falls Hyperthreading
 * vom Kernel untersttzt wird... ich hatte leider keinen Pentium-4 zum
 * ausprobieren):
 *
 *   g++ -O3 -fomit-frame-pointer -DTHREADS -o ct-puzzle-solver-threads ct-puzzle-solver.cpp -lpthread
 *
 * Zum debug (dauert viel lnger, ca. 40 min im Athlon XP1700+, alle
 * gefundenen Lsungen werden in ASCII dargestellt):
 *
 *   g++ -O3 -fomit-frame-pointer -DDEBUG -o ct-puzzle-solver-debug ct-puzzle-solver.cpp
 *
 * (nicht -DDEBUG und -DTHREADS gleichzeitig benutzen!)
 *
 * Algorithmus:
 *
 * Es wird zuerst eins von den zwlf Puzzle-Teilen irgendwo im Quader gesetzt.
 * Um Symmetrien vom Anfang an auszuschliessen, wird dieses Teil nur in
 * jeweils einer von vier symmetrischen Positionen gesetzt. (Falls zwei oder
 * mehr dieser symmetrischen Positionen auf die selbe Konfiguration fhren,
 * wird iterativ ein weiteres Teil dazugenommen, bis alle Symmetrien aufgehoben
 * sind).
 * Nachdem das erste Teil irgendwo liegt, werden die freie Felder in einer
 * festen Reihenfolge untersucht, und es wird jeweils versucht, ein Teil
 * in so zu legen, dass dieses Feld besetzt ist. Das wird iterativ im
 * Depth-First-Search Verfahren fortgesetzt, bis entweder eine Lsung
 * gefunden wird, oder kein Teil mehr gesetzt werden kann, und dann zum
 * nchsten Teil bergegangen.
 *
 * Optimierungen:
 *
 * - Als erstes Teil zum Platzieren wird das gewhlt, das die wenigsten
 *   Setz-Mglichkeiten besitzt (im vorgegebenen Fall vom Quader ist das
 *   das '+'-Teil.
 * - Von den vier mglichen symmetrischen Mglichkeiten zum setzen des
 *   ersten Teils, wird diejenige gewhlt, die mglichst am Anfang der
 *   Suche "strt".
 * - Jede mgliche Setz-Position eines Teils im Quader wird als Bitmap
 *   in einem 'unsigned long long' gespeichert (es werden 60 bits gebraucht).
 * - Es werden zwei Hash-Tabellen erzeugt, die alle mglichen Kombinationen
 *   enthalten, dass drei bzw. vier Teile in den "letzten" Positionen
 *   des Quaders liegen. (Letzte Positionen im Hinblick auf die Suchreihenfolge
 *   der Felder im Quader. Dadurch wird es wahrscheinlicher, dass wenn nur
 *   drei bzw. vier Teile zu setzen sind, die freien Felder in diesem
 *   Bereich liegen, dass die Kombination also im Hash zu finden ist).
 *
 */

#include <vector>
#include <set>
#include <stream.h>
#include <algorithm>
#include <unistd.h>
#include <pthread.h>

using namespace std;

typedef vector <vector <vector <char> > > vvvc;

unsigned long long *bitmaps_ptr[11][64];

#ifndef DEBUG
unsigned int *dreier_ptr[11][11][11][1025];
unsigned int *dreier_num[11][11][11][1025];
unsigned int *vierer_ptr[11][11][11][11][1025];
unsigned int *vierer_num[11][11][11][11][1025];
#endif

int permutation[12];
int solutions;
#ifdef THREADS
int solution1, solution2;
int permutation1[12];
int permutation2[12];
int done1, done2;
struct threadargs {
	unsigned long long filled;
	int n;
};
#endif

char bitnames[]="0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwx";
int bitpos[256];

#ifdef DEBUG
string solution;
#endif

void init_bitpos() {
	int i;
	for (i=0;i<256;i++) bitpos[i]=-1;
	for (i=0;i<60;i++) bitpos[int(bitnames[i])]=59-i;
}

#define BOARD_DIM 5

char *board[5][5]={ { "..012", "..CDE", "..OPQ", "..abc", "..mno" },
                    { "..345", "..FGH", "..RST", "..def", "..pqr" },
                    { "..678", "..IJK", "..UVW", "..ghi", "..stu" },
                    { "..9AB", "..LMN", "..XYZ", "..jkl", "..vwx" },
                    { ".....", ".....", ".....", ".....", "....." } };

// Anzahl und Anordnung der Symmetrien (beim 3x4x5 Quader 4 Symmetrische
// Anordnungen, die man durch Drehungen erhlt).

#define NUM_SYM 4

char *boardsym[]=
	{"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwx",
	 "vwxstupqrmnojklghidefabcXYZUVWRSTOPQLMNIJKFGHCDE9AB678345012",
	 "onmrqputsxwvcbafedihglkjQPOTSRWVUZYXEDCHGFKJINML210543876BA9",
	 "BA9876543210NMLKJIHGFEDCZYXWVUTSRQPOlkjihgfedcbaxwvutsrqponm"};

/* Eine Alternative:
   Anzugeben sind:
     BOARD_DIM     : Die Dimension eines Wrfels, der die Struktur der Lsung
                     zeigt.
     board[][][]   : Ein BOARD_DIM x BOARD_DIM x BOARD_DIM char-Array, das
                     die Struktur der Lsung zeigt. Von Puzzle-Teilen zu
                     besetzte Positionen sind mit '0-9A-Za-x' bezeichnet,
                     freie Felder sind mit '.' definiert. Die Reihenfolge
                     0-x gibt auch die Suchreihenfolge an, also ist es
                     gnstiger mit 'schwierigen Feldern anzufangen.
     NUM_SYM       : Die Anzahl der Symmetrien der ganzen Struktur.
     boardsym[]    : Die Beschreibung der NUM_SYM Symmetrien der Struktur.
                     Wenn man die Felder von boardsym[x] durch die
                     entsprechenden von boardsym[y] in board[][][] ersetzt,
                     erhlt man die gleiche Struktur, die durch einfaches
                     Drehen von der ursprnglichen bekommt.

Z.B. fr die Pyramide aus www.ctpuzzle.de ergibt sich:

#define BOARD_DIM 5

char *board[5][5]={ { ".....", ".....", ".....", "ABCDE", "Zabcd" },
                    { ".....", ".....", ".123.", "FGHIJ", "efghi" },
                    { ".....", "..0..", ".456.", "KLMNO", "jklmn" },
                    { ".....", ".....", ".789.", "PQRST", "opqrs" },
                    { ".....", ".....", ".....", "UVWXY", "tuvwx" } };

#define NUM_SYM 4

char *boardsym[60]=
	{"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwx",
	 "0369258147EJOTYDINSXCHMRWBGLQVAFKPUdinsxchmrwbglqvafkpuZejot",
	 "0987654321YXWVUTSRQPONMLKJIHGFEDCBAxwvutsrqponmlkjihgfedcbaZ",
	 "0741852963UPKFAVQLGBWRMHCXSNIDYTOJEtojeZupkfavqlgbwrmhcxsnid"};

*/

char *pieces[12][4][4]={ { { "##..", "....", "....", "...." },
                           { "#...", "....", "....", "...." },
                           { "##..", "....", "....", "...." },
                           { "....", "....", "....", "...." } },

                         { { ".##.", "....", "....", "...." },
                           { ".#..", "....", "....", "...." },
                           { "##..", "....", "....", "...." },
                           { "....", "....", "....", "...." } },

                         { { ".#..", "....", "....", "...." },
                           { ".#..", "....", "....", "...." },
                           { ".#..", "....", "....", "...." },
                           { "##..", "....", "....", "...." } },

                         { { ".#..", "....", "....", "...." },
                           { ".#..", "....", "....", "...." },
                           { "##..", "....", "....", "...." },
                           { ".#..", "....", "....", "...." } },

                         { { "##..", "#...", "....", "...." },
                           { "##..", "....", "....", "...." },
                           { "....", "....", "....", "...." },
                           { "....", "....", "....", "...." } },

                         { { "###.", "....", "....", "...." },
                           { ".#..", "....", "....", "...." },
                           { ".#..", "....", "....", "...." },
                           { "....", "....", "....", "...." } },

                         { { "#...", "....", "....", "...." },
                           { "##..", "....", "....", "...." },
                           { "##..", "....", "....", "...." },
                           { "....", "....", "....", "...." } },

                         { { "#...", "....", "....", "...." },
                           { "##..", "....", "....", "...." },
                           { ".##.", "....", "....", "...." },
                           { "....", "....", "....", "...." } },

                         { { "#...", "....", "....", "...." },
                           { "#...", "##..", "....", "...." },
                           { "....", "....", "....", "...." },
                           { "....", "....", "....", "...." } },

                         { { ".##.", "....", "....", "...." },
                           { "##..", "....", "....", "...." },
                           { ".#..", "....", "....", "...." },
                           { "....", "....", "....", "...." } },

                         { { ".#..", "....", "....", "...." },
                           { "###.", "....", "....", "...." },
                           { ".#..", "....", "....", "...." },
                           { "....", "....", "....", "...." } },

                         { { "#...", "....", "....", "...." },
                           { "##..", "....", "....", "...." },
                           { "#...", "....", "....", "...." },
                           { "##..", "....", "....", "...." } } };

template <int DIM>
vvvc char2vvvc (char *src[DIM][DIM]) {
	vvvc res;
	vector <vector <char> > temp_vvc;
	vector <char> temp_vc(DIM,' ');
	int i,j,k;
	res.clear();
	for (i=0;i<DIM;i++) {
		temp_vvc.clear();
		for (j=0;j<DIM;j++) {
			for (k=0;k<DIM;k++) {
				temp_vc[k]=src[i][j][k];
			}
			temp_vvc.push_back(temp_vc);
		}
		res.push_back(temp_vvc);
	}
	return res;
}

//
// Wrfeldrehen...
// Der Wrfel src wird r1 mal um die x1-Achse, r2 mal um die x2-Achse
// und r3 mal um die x3-Achse gedreht (in dieser Reihenfolge!).
// Die Parameter:  (r1=0..3, r2=0, r3=0..3) bzw. (r1=0, r2=1,3, r3=0..3)
// liefern alle 24 mglichen Wrfeldrehungen.
//

vvvc rotate(int r1, int r2, int r3, vvvc src) {
	int i,x1,x2,x3;
	int dim=src.size();
	vvvc temp1=src;
	vvvc temp2=temp1;

	// Drehung um die x1-Achse

	for (i=0;i<r1;i++) {
		for (x1=0;x1<dim;x1++)
		for (x2=0;x2<dim;x2++)
		for (x3=0;x3<dim;x3++) {
			temp2[x1][x2][x3]=temp1[x1][x3][dim-x2-1];
		}
		temp1=temp2;
	}
	
	// Drehung um die x2-Achse

	for (i=0;i<r2;i++) {
		for (x1=0;x1<dim;x1++)
		for (x2=0;x2<dim;x2++)
		for (x3=0;x3<dim;x3++) {
			temp2[x1][x2][x3]=temp1[dim-x3-1][x2][x1];
		}
		temp1=temp2;
	}
	
	// Drehung um die x3-Achse

	for (i=0;i<r3;i++) {
		for (x1=0;x1<dim;x1++)
		for (x2=0;x2<dim;x2++)
		for (x3=0;x3<dim;x3++) {
			temp2[x1][x2][x3]=temp1[x2][dim-x1-1][x3];
		}
		temp1=temp2;
	}
	
	return temp1;
}

//
// create_bitmaps(brd,pc): erzeugt aus einer board-Beschreibung (brd)
//   und einer Teile-Beschreibung (pc) einen Liste von Bit-Vectoren
//   (unsigned long long reicht aus, da 60 bits gebraucht werden), die die
//   mglichen Lagen des Teils im board definieren. Gesetzte bits
//   bezeichnen die Felder im bitnames[]-Vektor, die vom Teil besetzt
//   werden. Benutzt wird dazu die Routine:
// check_fit(brd,pc,k1,k2,k3): kontrolliert, ob das Teil pc, verschoben
//   um (k1,k2,k3) relativ zu brd, im brd enthalten ist, und gibt
//   ggf. ein Bitmap als Antwort zurck, das die Position beschreibt.
//

unsigned long long check_fit(vvvc brd, vvvc pc, int k1, int k2, int k3);

set <unsigned long long> create_bitmaps(vvvc brd, vvvc pc) {
	int i,j;
	int k1,k2,k3;
	vvvc pc_rot;
	set <unsigned long long> res;
	unsigned long long r;
	res.clear();
	int pc_dim=pc.size();
	int brd_dim=brd.size();
	for (i=0;i<4;i++) for (j=0;j<4;j++) {
		pc_rot=rotate(i,0,j,pc);
		for (k1=-pc_dim;k1<brd_dim;k1++)
		for (k2=-pc_dim;k2<brd_dim;k2++)
		for (k3=-pc_dim;k3<brd_dim;k3++)
			if ((r=check_fit(brd,pc_rot,k1,k2,k3))) res.insert(r);
	}
	for (i=1;i<4;i+=2) for (j=0;j<4;j++) {
		pc_rot=rotate(0,i,j,pc);
		for (k1=-pc_dim;k1<brd_dim;k1++)
		for (k2=-pc_dim;k2<brd_dim;k2++)
		for (k3=-pc_dim;k3<brd_dim;k3++)
			if ((r=check_fit(brd,pc_rot,k1,k2,k3))) res.insert(r);
	}
	return res;
}

unsigned long long check_fit(vvvc brd, vvvc pc, int k1, int k2, int k3) {
	int i1,i2,i3;
	int x1,x2,x3;
	int p;
	int brd_dim=brd.size();
	int pc_dim=pc.size();
	unsigned long long res=0LL;
	for (i1=0;i1<pc_dim;i1++)
	for (i2=0;i2<pc_dim;i2++)
	for (i3=0;i3<pc_dim;i3++) {
		if (pc[i1][i2][i3]=='#') {
			x1=k1+i1; x2=k2+i2; x3=k3+i3;
			if (x1<0 || x1>=brd_dim) return 0LL;
			if (x2<0 || x2>=brd_dim) return 0LL;
			if (x3<0 || x3>=brd_dim) return 0LL;
			if ((p=bitpos[brd[x1][x2][x3]])<0) return 0LL;
			res|=1LL<<p;
		}
	}
	return res;
}

template <class _T>
bool larger(const set<_T>& a, const set<_T>& b) {
	return a.size()>b.size();
}

unsigned long long get_sym(unsigned long long from, int n) {
	int i;
	unsigned long long res=0LL;
	for (i=0;i<60;i++) {
		if (from & (1LL<<i) )
			res|=1LL<<bitpos[int(boardsym[n][59-i])];
	}
	return res;
}

#ifdef DEBUG
string bm2string(unsigned long long x, char c) {
	string res(60,'.');
	int i;
	for (i=0;i<60;i++) if ((1ULL<<i)&x) res[i]=c;
	return res;
}

void printsolution() {
	int i,j,k;
	for (i=0;i<BOARD_DIM;i++) {
		for (j=0;j<BOARD_DIM;j++) {
			for (k=0;k<BOARD_DIM;k++) {
				if (board[i][j][k]=='.') cout << '.';
				else cout << solution[bitpos[board[i][j][k]]];
			}
			cout << ' ';
		}
		cout << endl;
	}
	cout << endl;
}
#endif

//
// first_pieces(): Erzeugt eine Liste von allen mglichen Anfangspositionen
// zum Platzieren des ersten Teils (Symmetrien nicht mitgerechnet). Falls
// noch eine Drehung des ganzen existiert, die zur selben Konfiguration
// fhrt, wird iterativ ein weiteres Teil hinzugenommen, bis alle
// Symmetrien eliminiert sind. Die Struktur pair <unsigned long long, int>
// enthlt das Bitmap, das die besetzten Felder im board beschreibt, sowie
// die Anzahl der noch zu setzenden Teile.
//

#ifdef DEBUG
set <pair <pair <unsigned long long, int>, string > > first_pieces(
    unsigned long long start, set <unsigned long long> bm[12], int n,
    vector <int> sym)
#else
set <pair <unsigned long long, int> > first_pieces(unsigned long long start,
    set <unsigned long long> bm[12], int n, vector <int> sym)
#endif
{
	int i,done;
	unsigned long long tmp,new_start;
	set <unsigned long long>::iterator sll_it;
#ifdef DEBUG
	set <pair <pair <unsigned long long, int>, string> > res;
	set <pair <pair <unsigned long long, int>, string> > last;
#else
	set <pair <unsigned long long, int> > res;
	set <pair <unsigned long long, int> > last;
#endif
	res.clear();
	vector <int> new_sym;
	for (sll_it=bm[n].begin();sll_it!=bm[n].end();sll_it++) {
		new_sym=sym;
		done=1;
		if (start & *sll_it) continue;
		new_start=start | *sll_it;
		for (i=1;i<sym.size();i++) if (sym[i]) {
			tmp=get_sym(new_start,i);
			if (tmp>new_start) break;
			if (tmp<new_start) new_sym[i]=0;
			if (tmp==new_start) done=0;
		}
		if (i<sym.size()) continue;
#ifdef DEBUG
		string res2,res3;
		int k;
		set <pair <pair <unsigned long long, int>, string> >::iterator it;
		if (done) {
			res2=bm2string(*sll_it,'A'+11-n);
			res.insert(make_pair(make_pair(new_start,n),res2));
		} else {
			last=first_pieces(new_start,bm,n-1,new_sym);
			res2=bm2string(*sll_it,'A'+11-n);
			for (it=last.begin();it!=last.end();it++) {
				res3=(*it).second;
				for (k=0;k<res2.size();k++)
					if (res2[k]=='A'+11-n) res3[k]=res2[k];
				res.insert(make_pair( (*it).first, res3) );
			}
		}
#else
		if (done) res.insert(make_pair(new_start,n));
		else {
			last=first_pieces(new_start,bm,n-1,new_sym);
			res.insert(last.begin(),last.end());
		}
#endif
	}
	return res;
}

// Eine Routine, die das erste gesetzte bit einer unsigned long long
// Zahl liefert (in Assembler, um den Befehl 'bsr' auszunutzen).

int first_bit(unsigned long long x) __asm__("asm_first_bit");

__asm__ (
"	.text\n"
"	.align 4\n"
".globl asm_first_bit\n"
"asm_first_bit:\n"
"	movl	8(%esp), %eax\n"
"	bsrl	%eax, %edx\n"
"	jnz	high\n"
"	movl	4(%esp), %edx\n"
"	bsrl	%edx, %eax\n"
"	ret\n"
"high:\n"
"	movl	%edx, %eax\n"
"	orl	$32, %eax\n"
"	ret\n"
);

// Die eigentliche Funktion, die die Anzahl der Lsungen berechnet.
// N ist die Anzahl der noch zu setzenden Teile.
// filled ist ein Bitmap, das anzeigt, welche Positionen schon besetzt
//   sind.
// permutation[] (ein globaler Vektor) zeigt an, welche Stcke noch
//   zu setzen sind (die Teile permutation[0] bis permutation[N-1]).

// Es wird versucht, eins der noch zu setzenden Stcke so zu setzen,
// dass das grte noch nicht gesetzte bit gesetzt wird. Dann wird
// solve() iterativ fr die restlichen Teile aufgerufen.

// Wenn nur drei Teile zu setzen sind, und die freien Positionen
// nur bits zwischen 0 und 31 betreffen, wird im dreier_ptr
// Vektor gesucht, ob es eine (oder mehrere) Kombinationen existieren,
// die zu einer Lsung fhren.
// Der gleiche Trick wird auch fr vier Teile und freien Positionen
// zwischen 0 und 29 angewendet.

template <int N>
inline void solve(unsigned long long filled) {
	int k=first_bit(~filled);
	const int newN=N-1<0?0:N-1;
	int i,c,t;
	unsigned long long *xptr;
#ifdef DEBUG
	if (N>1)
#else
	if (N>4 || N==2 || (N==3 && k>31) || (N==4 && k>29) )
#endif
	{
		c=permutation[newN];
		i=newN;
		do {
			xptr=bitmaps_ptr[c][k];
			while (~(*xptr)) {
				if (!(*xptr & filled)) {
#ifdef DEBUG
	string temp1=bm2string(*xptr,'A'+12-N);
	for (int k=0;k<60;k++) if ((1ULL<<k)&(*xptr)) solution[k]='A'+12-N;
#endif
					solve<newN>(filled | *xptr);
				}
				xptr++;
			}
			t=c;
			c=permutation[i-1];
			permutation[i-1]=permutation[i];
			permutation[i]=t;
			i--;
		} while (i>=1);
		xptr=bitmaps_ptr[c][k];
		while (~(*xptr)) {
			if (!(*xptr & filled)) {
#ifdef DEBUG
	string temp1=bm2string(*xptr,'A'+12-N);
	for (int k=0;k<60;k++) if ((1ULL<<k)&(*xptr)) solution[k]='A'+12-N;
#endif
				solve<newN>(filled | *xptr);
			}
			xptr++;
		}
		permutation[0]=c;
	} else {
		if (N==1) {
			xptr=bitmaps_ptr[permutation[0]][k];
			while (~(*xptr)) {
				if (!(*xptr & filled)) {
#ifdef DEBUG
	string temp1=bm2string(*xptr,'A'+12-N);
	for (int k=0;k<60;k++) if ((1ULL<<k)&(*xptr)) solution[k]='A'+12-N;
	printsolution();
#endif
					solutions++;
				}
				xptr++;
			}
		}
#ifndef DEBUG
		else if (N==3) {
			int a=permutation[0];
			int b=permutation[1];
			int c=permutation[2];
			int start,stop,mid;
			unsigned int f2=int((~filled) & 0x00000000FFFFFFFFULL);
			unsigned int *x1;
			unsigned int p=(f2&0xFFC00000)>>22;
			x1=dreier_ptr[a][b][c][p];
			stop=(dreier_ptr[a][b][c][p+1]-x1)-1;
			if (stop<0) return;
			start=0;
			while (start!=stop) {
				mid=(start+stop)>>1;
				if (x1[mid]<f2) start=mid+1;
				else stop=mid;
			}
			if (x1[start]==f2)
				solutions+=dreier_num[a][b][c][p][start];
		} else { // N==4
			int a=permutation[0];
			int b=permutation[1];
			int c=permutation[2];
			int d=permutation[3];
			int start,stop,mid;
			unsigned int f2=int((~filled) & 0x000000003FFFFFFFULL);
			unsigned int *x1;
			unsigned int p=(f2&0x3FF00000)>>20;
			x1=vierer_ptr[a][b][c][d][p];
			stop=(vierer_ptr[a][b][c][d][p+1]-x1)-1;
			if (stop<0) return;
			start=0;
			while (start!=stop) {
				mid=(start+stop)>>1;
				if (x1[mid]<f2) start=mid+1;
				else stop=mid;
			}
			if (x1[start]==f2)
				solutions+=vierer_num[a][b][c][d][p][start];
		}
#endif
	}
}

#ifdef THREADS
template <int N>
inline void solve1(unsigned long long filled) {
	int k=first_bit(~filled);
	const int newN=N-1<0?0:N-1;
	int i,c,t;
	unsigned long long *xptr;
	if (N>4 || N==2 || (N==3 && k>31) || (N==4 && k>29) )
	{
		c=permutation1[newN];
		i=newN;
		do {
			xptr=bitmaps_ptr[c][k];
			while (~(*xptr)) {
				if (!(*xptr & filled)) {
					solve1<newN>(filled | *xptr);
				}
				xptr++;
			}
			t=c;
			c=permutation1[i-1];
			permutation1[i-1]=permutation1[i];
			permutation1[i]=t;
			i--;
		} while (i>=1);
		xptr=bitmaps_ptr[c][k];
		while (~(*xptr)) {
			if (!(*xptr & filled)) {
				solve1<newN>(filled | *xptr);
			}
			xptr++;
		}
		permutation1[0]=c;
	} else {
		if (N==1) {
			xptr=bitmaps_ptr[permutation1[0]][k];
			while (~(*xptr)) {
				if (!(*xptr & filled)) {
					solution1++;
				}
				xptr++;
			}
		}
		else if (N==3) {
			int a=permutation1[0];
			int b=permutation1[1];
			int c=permutation1[2];
			int start,stop,mid;
			unsigned int f2=int((~filled) & 0x00000000FFFFFFFFULL);
			unsigned int *x1;
			unsigned int p=(f2&0xFFC00000)>>22;
			x1=dreier_ptr[a][b][c][p];
			stop=(dreier_ptr[a][b][c][p+1]-x1)-1;
			if (stop<0) return;
			start=0;
			while (start!=stop) {
				mid=(start+stop)>>1;
				if (x1[mid]<f2) start=mid+1;
				else stop=mid;
			}
			if (x1[start]==f2)
				solution1+=dreier_num[a][b][c][p][start];
		} else { // N==4
			int a=permutation1[0];
			int b=permutation1[1];
			int c=permutation1[2];
			int d=permutation1[3];
			int start,stop,mid;
			unsigned int f2=int((~filled) & 0x000000003FFFFFFFULL);
			unsigned int *x1;
			unsigned int p=(f2&0x3FF00000)>>20;
			x1=vierer_ptr[a][b][c][d][p];
			stop=(vierer_ptr[a][b][c][d][p+1]-x1)-1;
			if (stop<0) return;
			start=0;
			while (start!=stop) {
				mid=(start+stop)>>1;
				if (x1[mid]<f2) start=mid+1;
				else stop=mid;
			}
			if (x1[start]==f2)
				solution1+=vierer_num[a][b][c][d][p][start];
		}
	}
}

template <int N>
inline void solve2(unsigned long long filled) {
	int k=first_bit(~filled);
	const int newN=N-1<0?0:N-1;
	int i,c,t;
	unsigned long long *xptr;
	if (N>4 || N==2 || (N==3 && k>31) || (N==4 && k>29) )
	{
		c=permutation2[newN];
		i=newN;
		do {
			xptr=bitmaps_ptr[c][k];
			while (~(*xptr)) {
				if (!(*xptr & filled)) {
					solve2<newN>(filled | *xptr);
				}
				xptr++;
			}
			t=c;
			c=permutation2[i-1];
			permutation2[i-1]=permutation2[i];
			permutation2[i]=t;
			i--;
		} while (i>=1);
		xptr=bitmaps_ptr[c][k];
		while (~(*xptr)) {
			if (!(*xptr & filled)) {
				solve2<newN>(filled | *xptr);
			}
			xptr++;
		}
		permutation2[0]=c;
	} else {
		if (N==1) {
			xptr=bitmaps_ptr[permutation2[0]][k];
			while (~(*xptr)) {
				if (!(*xptr & filled)) {
					solution2++;
				}
				xptr++;
			}
		}
		else if (N==3) {
			int a=permutation2[0];
			int b=permutation2[1];
			int c=permutation2[2];
			int start,stop,mid;
			unsigned int f2=int((~filled) & 0x00000000FFFFFFFFULL);
			unsigned int *x1;
			unsigned int p=(f2&0xFFC00000)>>22;
			x1=dreier_ptr[a][b][c][p];
			stop=(dreier_ptr[a][b][c][p+1]-x1)-1;
			if (stop<0) return;
			start=0;
			while (start!=stop) {
				mid=(start+stop)>>1;
				if (x1[mid]<f2) start=mid+1;
				else stop=mid;
			}
			if (x1[start]==f2)
				solution2+=dreier_num[a][b][c][p][start];
		} else { // N==4
			int a=permutation2[0];
			int b=permutation2[1];
			int c=permutation2[2];
			int d=permutation2[3];
			int start,stop,mid;
			unsigned int f2=int((~filled) & 0x000000003FFFFFFFULL);
			unsigned int *x1;
			unsigned int p=(f2&0x3FF00000)>>20;
			x1=vierer_ptr[a][b][c][d][p];
			stop=(vierer_ptr[a][b][c][d][p+1]-x1)-1;
			if (stop<0) return;
			start=0;
			while (start!=stop) {
				mid=(start+stop)>>1;
				if (x1[mid]<f2) start=mid+1;
				else stop=mid;
			}
			if (x1[start]==f2)
				solution2+=vierer_num[a][b][c][d][p][start];
		}
	}
}

void *solve1a(void *args) {
	threadargs a1;
	a1.filled=((threadargs*)(args))->filled;
	a1.n=((threadargs*)(args))->n;
	switch (a1.n) {
		case 11: solve1<11>(a1.filled); break;
		case 10: solve1<10>(a1.filled); break;
		case 9: solve1<9>(a1.filled); break;
		case 8: solve1<8>(a1.filled); break;
		case 7: solve1<7>(a1.filled); break;
		case 6: solve1<6>(a1.filled); break;
		case 5: solve1<5>(a1.filled); break;
		case 4: solve1<4>(a1.filled); break;
		case 3: solve1<3>(a1.filled); break;
		case 2: solve1<2>(a1.filled); break;
		case 1: solve1<1>(a1.filled); break;
	}
	done1=1;
	pthread_exit(NULL);
}

void *solve2a(void *args) {
	threadargs a2;
	a2.filled=((threadargs*)(args))->filled;
	a2.n=((threadargs*)(args))->n;
	switch (a2.n) {
		case 11: solve2<11>(a2.filled); break;
		case 10: solve2<10>(a2.filled); break;
		case 9: solve2<9>(a2.filled); break;
		case 8: solve2<8>(a2.filled); break;
		case 7: solve2<7>(a2.filled); break;
		case 6: solve2<6>(a2.filled); break;
		case 5: solve2<5>(a2.filled); break;
		case 4: solve2<4>(a2.filled); break;
		case 3: solve2<3>(a2.filled); break;
		case 2: solve2<2>(a2.filled); break;
		case 1: solve2<1>(a2.filled); break;
	}
	done2=1;
	pthread_exit(NULL);
}

#endif

int main () {
	init_bitpos();
	int i, j, k, q, p;
	int sum, pos;
	unsigned int last;
	set <unsigned long long> bm[12];
	vector <unsigned long long> bm_vec[12];
	vvvc brd=char2vvvc<5>(board);

// Hier werden die Vektoren bm_vec[i] generiert, die die mglichen
// Positionen des Teils i enthalten (definiert als bitmap). Jeder
// dieser Vektoren ist sortiert, und die Teile werden auch nach
// Schwierigkeitsgrad sortiert (i=0 ist das Teil, das die meisten
// mglichen Positionen im Quader hat, i=11 das Teil mit den
// wenigsten Mglichkeiten.

	for (i=0;i<12;i++) {
		bm[i]=create_bitmaps(brd,char2vvvc<4>(pieces[i]));
	}
	sort(bm,bm+12,larger<unsigned long long>);
	for (i=0;i<12;i++) {
		bm_vec[i]=vector <unsigned long long> (bm[i].begin(),bm[i].end());
	}

// Hier werden die Vektoren bm_vec[i] in Abschnitten mit gemeinsamen
// erstem bit, jeweils abgeschlossen mit einem 0xFFFFFFFFFFFFFFFF.
// bitmaps_ptr[i][k] enthlt dann einen pointer auf den Abschnitt
// des Teils i, in dem das erste gesetzte bit in der Position k liegt.

	vector <unsigned long long>::iterator v_ull_it;
	vector <unsigned long long>::iterator last_int[11];
	vector <unsigned long long>::iterator last2_int[11];
	unsigned long long *bitmaps;
	sum=0;
	for (i=0;i<11;i++) sum+=bm[i].size();
	bitmaps=new unsigned long long[sum+11*64];
	pos=0;
	for (i=0;i<11;i++) {
		k=0;
		bitmaps_ptr[i][0]=bitmaps+pos;
		for (v_ull_it=bm_vec[i].begin();v_ull_it!=bm_vec[i].end();
		     v_ull_it++) {
			if (first_bit(*v_ull_it)<32) last_int[i]=v_ull_it;
			if (first_bit(*v_ull_it)<30) last2_int[i]=v_ull_it;
			while (first_bit(*v_ull_it)>k) {
				bitmaps[pos++]=0xFFFFFFFFFFFFFFFFULL;
				bitmaps_ptr[i][++k]=bitmaps+pos;
			}
			bitmaps[pos++]=(*v_ull_it);
		}
		while (k<63) {
			bitmaps[pos++]=0xFFFFFFFFFFFFFFFFULL;
			bitmaps_ptr[i][++k]=bitmaps+pos;
		}
		bitmaps[pos++]=0xFFFFFFFFFFFFFFFFULL;
		last_int[i]++;
		last2_int[i]++;
	}

// Hier werden je drei Teile in Kombinationsmglichkeiten zusammengefasst.
// dreier_ptr[i][j][k][p] ist am Schluss ein pointer auf einen Bereich,
// der bitmaps der mgliche Kombinationen der Teile i,j,k enthlt, nur
// bits aus den niedrigen 32 bits gesetzt sind, und die bits 31...22
// durch die Zahl p angegeben sind.
// dreier_num[i][j][k][p] ist ein pointer mit gleich vielen Elementen
// wie dreier_ptr[i][j][k][p], das fr jedes bitmap angibt, auf wie
// viele Weisen das aus den Teilen i,j und k erzeugt werden kann.
// Das gleiche wird auch fr Kombinationen von vier Teilen gemacht,
// allerdings aus Platzgrnden nur fr Kombinationen, die die letzten
// 30 bits besetzen.

#ifndef DEBUG
	vector <unsigned long long>::iterator v_ull_it_dreier1;
	vector <unsigned long long>::iterator v_ull_it_dreier2;
	vector <unsigned long long>::iterator v_ull_it_dreier3;
	vector <unsigned int> dreier_temp;
	vector <unsigned int>::iterator v_ui_it;
	unsigned int ui_temp;
	for (i=0;i<11;i++)
	for (j=i+1;j<11;j++)
	for (k=j+1;k<11;k++) {
		dreier_temp.clear();
		for (v_ull_it_dreier1=bm_vec[i].begin();
		     v_ull_it_dreier1!=last_int[i];
		     v_ull_it_dreier1++) {
			for (v_ull_it_dreier2=bm_vec[j].begin();
			     v_ull_it_dreier2!=last_int[j];
			     v_ull_it_dreier2++)
			if (!(*v_ull_it_dreier1 & *v_ull_it_dreier2)) {
				ui_temp=*v_ull_it_dreier1 | *v_ull_it_dreier2;
				for (v_ull_it_dreier3=bm_vec[k].begin();
				     v_ull_it_dreier3!=last_int[k];
				     v_ull_it_dreier3++)
				if (!(ui_temp & *v_ull_it_dreier3)) {
					dreier_temp.push_back(ui_temp | *v_ull_it_dreier3);
				}
			}
		}
		sort(dreier_temp.begin(),dreier_temp.end());
		dreier_temp.push_back(0xFFFFFFFF);
		p=0;
		pos=0;
		dreier_ptr[i][j][k][0]=new unsigned int[dreier_temp.size()+1];
		dreier_num[i][j][k][0]=new unsigned int[dreier_temp.size()+1];
		last=0xFFFFFFFF;
		for (v_ui_it=dreier_temp.begin(); v_ui_it!=dreier_temp.end();
		     v_ui_it++) {
			if (*v_ui_it == last) dreier_num[i][j][k][0][pos-1]++;
			else {
				dreier_ptr[i][j][k][0][pos]=*v_ui_it;
				dreier_num[i][j][k][0][pos]=1;
				while (((*v_ui_it & 0xFFC00000)>>22) > p) {
					p++;
					dreier_ptr[i][j][k][p]=dreier_ptr[i][j][k][0]+pos;
					dreier_num[i][j][k][p]=dreier_num[i][j][k][0]+pos;
				}
				pos++;
				last=*v_ui_it;
			}
		}
		for (p=0;p<=1024;p++) {
			dreier_ptr[i][k][j][p]=dreier_ptr[i][j][k][p];
			dreier_ptr[j][i][k][p]=dreier_ptr[i][j][k][p];
			dreier_ptr[j][k][i][p]=dreier_ptr[i][j][k][p];
			dreier_ptr[k][i][j][p]=dreier_ptr[i][j][k][p];
			dreier_ptr[k][j][i][p]=dreier_ptr[i][j][k][p];
			dreier_num[i][k][j][p]=dreier_num[i][j][k][p];
			dreier_num[j][i][k][p]=dreier_num[i][j][k][p];
			dreier_num[j][k][i][p]=dreier_num[i][j][k][p];
			dreier_num[k][i][j][p]=dreier_num[i][j][k][p];
			dreier_num[k][j][i][p]=dreier_num[i][j][k][p];
		}
	}


	vector <unsigned long long>::iterator v_ull_it_vierer1;
	vector <unsigned long long>::iterator v_ull_it_vierer2;
	vector <unsigned long long>::iterator v_ull_it_vierer3;
	vector <unsigned long long>::iterator v_ull_it_vierer4;
	vector <unsigned int> vierer_temp;
	unsigned int ui_temp2;
	vector <int> perm4(4,0);
	for (i=0;i<11;i++)
	for (j=i+1;j<11;j++)
	for (k=j+1;k<11;k++)
	for (q=k+1;q<11;q++) {
		vierer_temp.clear();
		for (v_ull_it_vierer1=bm_vec[i].begin();
		     v_ull_it_vierer1!=last2_int[i];
		     v_ull_it_vierer1++) {
			for (v_ull_it_vierer2=bm_vec[j].begin();
			     v_ull_it_vierer2!=last2_int[j];
			     v_ull_it_vierer2++)
			if (!(*v_ull_it_vierer1 & *v_ull_it_vierer2)) {
				ui_temp=*v_ull_it_vierer1 | *v_ull_it_vierer2;
				for (v_ull_it_vierer3=bm_vec[k].begin();
				     v_ull_it_vierer3!=last2_int[k];
				     v_ull_it_vierer3++)
				if (!(ui_temp & *v_ull_it_vierer3)) {
	ui_temp2 = ui_temp | *v_ull_it_vierer3;
	for (v_ull_it_vierer4=bm_vec[q].begin();
	     v_ull_it_vierer4!=last2_int[q];
	     v_ull_it_vierer4++)
	if (!(ui_temp2 & *v_ull_it_vierer4)) {
	
					vierer_temp.push_back(ui_temp2 | *v_ull_it_vierer4);
	}
				}
			}
		}
		sort(vierer_temp.begin(),vierer_temp.end());
		vierer_temp.push_back(0xCFFFFFFF);
		p=0;
		pos=0;
		vierer_ptr[i][j][k][q][0]=new unsigned int[vierer_temp.size()+1];
		vierer_num[i][j][k][q][0]=new unsigned int[vierer_temp.size()+1];
		last=0xFFFFFFFF;
		for (v_ui_it=vierer_temp.begin(); v_ui_it!=vierer_temp.end();
		     v_ui_it++) {
			if (*v_ui_it == last) vierer_num[i][j][k][q][0][pos-1]++;
			else {
				vierer_ptr[i][j][k][q][0][pos]=*v_ui_it;
				vierer_num[i][j][k][q][0][pos]=1;
				while (((*v_ui_it & 0x3FF00000)>>20) > p) {
					p++;
					vierer_ptr[i][j][k][q][p]=vierer_ptr[i][j][k][q][0]+pos;
					vierer_num[i][j][k][q][p]=vierer_num[i][j][k][q][0]+pos;
				}
				pos++;
				last=*v_ui_it;
			}
		}
		for (p=0;p<=1024;p++) {
			perm4[0]=i;
			perm4[1]=j;
			perm4[2]=k;
			perm4[3]=q;
			do {
				vierer_ptr[perm4[0]][perm4[1]][perm4[2]][perm4[3]][p]=vierer_ptr[i][j][k][q][p];
				vierer_num[perm4[0]][perm4[1]][perm4[2]][perm4[3]][p]=vierer_num[i][j][k][q][p];
			} while (next_permutation(perm4.begin(),perm4.end()));
		}
	}

#endif

// Mit der Funktion first_pieces() werden alle Mglichen Kombinationen
// fr das erste Teil berechnet, und in start_bitmaps gespeichert.

#ifdef DEBUG
	set <pair <pair <unsigned long long, int>, string> > start_bitmaps;
	vector <int> sym(NUM_SYM,1);
	start_bitmaps=first_pieces(0ULL,bm,11,sym);
	set <pair <pair <unsigned long long, int>, string> >::iterator s_p_ull_i_it;
#else
	set <pair <unsigned long long, int> > start_bitmaps;
	vector <int> sym(NUM_SYM,1);
	start_bitmaps=first_pieces(0ULL,bm,11,sym);
	set <pair <unsigned long long, int> >::iterator s_p_ull_i_it;
#endif
	solutions=0;
	unsigned long long filled;

// Und nun wird fr jedes der Elemente von start_bitmaps die Funktion
// solve<N>() aufgerufen, die den Rest erledigt.

#ifdef THREADS
	done1=1; done2=1;
	pthread_t thread1, thread2;
	threadargs args1, args2;
	solution1=0;
	solution2=0;
#endif
	for (s_p_ull_i_it=start_bitmaps.begin();
	     s_p_ull_i_it!=start_bitmaps.end();
	     s_p_ull_i_it++) {
#ifdef THREADS
#ifndef DEBUG
		while (!done1 && !done2) usleep(1000);
		filled=0xF000000000000000ULL | (*s_p_ull_i_it).first;
		if (done1) {
			solutions+=solution1;
			solution1=0;
			done1=0;
			for (i=0;i<12;i++) permutation1[i]=i;
			args1.filled=filled;
			args1.n=(*s_p_ull_i_it).second;
			pthread_create(&thread1, NULL, solve1a, (void *)(&args1));
		} else if (done2) {
			solutions+=solution2;
			solution2=0;
			done2=0;
			for (i=0;i<12;i++) permutation2[i]=i;
			args2.filled=filled;
			args2.n=(*s_p_ull_i_it).second;
			pthread_create(&thread2, NULL, solve2a, (void *)(&args2));
		}
#endif
#else

#ifdef DEBUG
		filled=0xF000000000000000ULL | ((*s_p_ull_i_it).first).first;
		for (i=0;i<12;i++) permutation[i]=i;
		solution=(*s_p_ull_i_it).second;
		switch (((*s_p_ull_i_it).first).second)
#else
		filled=0xF000000000000000ULL | (*s_p_ull_i_it).first;
		for (i=0;i<12;i++) permutation[i]=i;
		switch ((*s_p_ull_i_it).second)
#endif
{
			case 11: solve<11>(filled); break;
			case 10: solve<10>(filled); break;
			case 9: solve<9>(filled); break;
			case 8: solve<8>(filled); break;
			case 7: solve<7>(filled); break;
			case 6: solve<6>(filled); break;
			case 5: solve<5>(filled); break;
			case 4: solve<4>(filled); break;
			case 3: solve<3>(filled); break;
			case 2: solve<2>(filled); break;
			case 1: solve<1>(filled); break;
		}
#endif
	}

#ifdef THREADS
	while (!done1 || !done2) usleep(1000);
	solutions+=solution1+solution2;
#endif

	cout << solutions << endl;
	return 0;
}

