/*
	Programm zur vollstndigen Suche nach allen Mglichkeiten
	der Anordnung von 12 Puzzleteilen aus 4-6 Einheitswrfeln
	(9 Teile entsprechen dem klassischen Pentomino)
	zu einem Quader mit den Seitenlngen 3 x 4 x 5
	(c) 4/2003 Heinz Repp

	Vorberlegungen:
	Der Quader besteht aus 60 Einheitswrfeln, fr jeden steht
	das x+3y+12z-ste Bit eines long long int, wobei x die 3er-
	Seite, y die 4er-Seite und z die 5er-Seite bezeichnet.
	Jedes Teil in jeder mglichen Position lt sich als 60-Bit-
	Zahl darstellen. Damit ist ein Hinzufgen zum teilweise gefllten
	Quader eine OR-, der Test, ob das mglich ist, eine AND-Operation.
	Leider sind atomare 64-Bit-Befehle dafr beim Pentium nur als
	unhandliche MMX-Anweisungen vorhanden.

	Um die Anzahl der zu prfenden Mglichkeiten einzuschrnken, werden
	folgende Anstze verwendet:

	1.	Der Quader wird Wrfel fr Wrfel gefllt, nicht Teil fr Teil.
		Aus praktischen Grnden wird beim hintersten obersten rechten
		Wrfel, also Bit 59, begonnen. Die Fllreihenfolge ist also von
		rechts nach links, von hinten nach vorne und von oben nach unten.

	2.	Damit sollte immer der Wrfel als nchstes gefllt werden, der
		die wenigsten Anlagemglichkeiten bietet. Ein Kriterium dafr ist
		die Begrenztheit des umgebenden Raumes: am wenigsten Platz ist
		an den Ecken der 3-er-Kante der 3x4-Grenzflche; das legt nahe,
		zuerst in Richtung der 3-er-Seite, dann erst in Richtung der
		4-er-Seite und erst zum Schlu entlang der 5-er-Seite zu gehen.
		Das legt auch die o.g. Bitcodierung nahe.

	3.	Um symmetrische Lsungen zu eliminieren, wird von einem Teil statt
		weniger Orientierungen, die sich auf jeder Stufe gleichmig aus-
		wirken wrden, nur Plazierungen in einem Teil des Quaders genommen.
		Voraussetzung dafr ist, da alle Lagen des Teils in eine von vier
		wendesymmetrischen Bereichen des Quaders eingeordnet werden knnen
		und jede Lage durch Wendung des Quaders in einen anderen Bereich
		fllt. Das geht hier nur mit den 2x2x2-Teilen. Wird von diesem Teil
		der Bereich gewhlt, der beim Fllen zuletzt erreicht wird, sind
		bei der ersten Iterationen berhaupt keine Lagen dieses Teils zu
		bercksichtigen. Es lohnt sich also, das Teil mit den meisten
		Lagen, glcklicherweise auch im 2x2x2-Format, dafr zu verwenden
		(hier als 'Q' bezeichnet)

	4.	Frbt man alle Wrfel des Quaders so mit einer von zwei Farben,
		etwa schwarz und wei, ein, da benachbarten Wrfel immer verschie-
		denfarbig sind, entsteht ein Schachbrettmuster. Jedes Teil belegt
		davon in jeder Lage immer eine bestimmte Zahl weier und schwarzer
		Teilwrfel, wobei je nach Lage die ein oder andere Farbe berwiegt,
		die Differenz jedoch immer gleich bleibt und ein Charakteristikum
		dieses Teils darstellt. Hier fllt auf:
		- es gibt nur ein Teil mit der Differenz 0, das 4-wrflige, hier 'v'
		- die meisten Teile haben die Differenz 1
		- es gibt nur ein Teil mit der Differenz 2, das 6-wrflige, hier 'F'
		- nur ein Teil hat die Differenz 3: das '+'-frmige, hier 'x'.
		Da jede 3x4-Ebene gleichviel schwarze und weie Wrfel enthlt, mu
		das durch ein 'x' eingebrachte Ungleichgewicht bereits im Nahbereich
		wieder ausgeglichen werden - was etwa bedeuten knnte, das fr 3
		folgende Teile mit 1-er Differenz nur die Hlfte der an dieser Stelle
		mglichen Lagen in Frage kommt: das wre eine Reduzierung um 7/8!

	5.	Der Effekt aus Punkt 4 wirkt sich erwartungsgem umso strker aus,
		je frher das 'x'-Element gelegt wird. Allerdings ist es mglich, fr
		die ungnstigen unteren Lagen von unten, etwa von hinten links, mit dem
		Fllen zu beginnen. Dadurch sind aber die 'Q'-Lagen relativ frh zu
		bercksichtigen. Aufgrund der Unabhngigkeit der oben/unten-Symmetrie
		und der Symmetrie in x/y-Richtung kann zwar auch hier der Teil der
		'Q'-Lagen der unteren zwei Ebenen verwendet werden, der am weitesten
		vorne rechts liegt - das 'Q'-Teil ist damit erst ab dem 5. Wrfel zu
		bercksichtigen, bei den ersten vielleicht zwei bis vier Iterationen
		noch nicht - der Effekt drfte aber erheblich geringer sein als bei der
		Lage am Schlu. Daher soll diese umgekehrte Fllreihenfolge nur bei den
		'x'-Lagen angewendet werden, bei denen der Mittelpunkt des Teils auf
		einer der beiden untersten Ebenen liegt. In der Praxis wird diese
		Fllreihenfolge von hinten unten links dadurch erreicht, da der ganze
		Quader um die y-Achse gewendet wird, hinten unten links wird so wieder
		zu hinten oben rechts, die 'Q'-Lagen vorne unten rechts werden zu
		vorne oben links und die 'x'-Lagen mit Mittelpunkten in den untersten
		beiden Ebenen werden zu Lagen mit Mittelpunkten in den obersten beiden
		Ebenen. Das 'x'-Teil hat aufgrund seiner mehrfachen Symmetrie auch die
		wenigsten Lagen insgesamt: um den Punkt-4-Effekt auszuntzen bietet es
		sich an, als erstes dieses Teil zu legen und den Quader dann von oben
		mit den restlichen Teilen zu fllen.

	6.	Da die Bits von oben her gefllt werden, kommen fr das nchste freie
		Bit nur die Lagen in Betracht, fr die dieses Bit das hchstwertige
		1-er Bit ist. Tabellen mit fr jedes Bit mglichen Lagen lassen sich
		bereits am Anfang aufstellen.

	7.	Jede Legereihenfolge der Teile stellt eine bestimmte Permutation der
		Teile dar: wenn nur Permutionen getestet werden, werden nur Teile
		betrachtet, die noch nicht verwendet wurden, damit entfllt eine
		entsprechende Prfung. Dagegen steht der Aufwand, die Permutation immer
		konsistent zu halten. Da der Aufwand proportional zur Anzahl der noch
		nicht gelegten Teile ist, die alternative Prfung, ob ein Teil bereits
		verwendet wurde, immer fr alle Teile notwendig ist, ist ein Gewinn erst
		in den tieferen Rekursionsebenen zu erwarten.
*/

#include <stdio.h>
#include <stdlib.h>
#ifdef DEBUG
	#include <assert.h>
#endif

#define QUADWORD __int64
#define QUADER_X 3
#define QUADER_Y 4
#define QUADER_Z 5
#define TEILEZAHL 12

/*
	die Bitoperationen fr 90- und 180-Grad-Drehungen um jede Achse
	werden als Makros implementiert
	zwei TURNs reichen, da TURN_X = TURN_Y + TURN_Z
*/

#define TURN_Y(q) \
	((01111i64 & q) << 50 | \
	 (02222i64 & q) << 48 | \
	 (04444i64 & q) << 46 | \
	 (011110000i64 & q) << 26 | \
	 (022220000i64 & q) << 24 | \
	 (044440000i64 & q) << 22 | \
	 (0111100000000i64 & q) << 2 | \
	 (0222200000000i64 & q)      | \
	 (0444400000000i64 & q) >> 2 | \
	 (01111000000000000i64 & q) >> 22 | \
	 (02222000000000000i64 & q) >> 24 | \
	 (04444000000000000i64 & q) >> 26 | \
	 (011110000000000000000i64 & q) >> 46 | \
	 (022220000000000000000i64 & q) >> 48 | \
	 (044440000000000000000i64 & q) >> 50)

#define TURN_Z(q) \
	((010001000100010001i64 & q) << 11 | \
	 (020002000200020002i64 & q) << 9 | \
	 (040004000400040004i64 & q) << 7 | \
	 (0100010001000100010i64 & q) << 5 | \
	 (0200020002000200020i64 & q) << 3 | \
	 (0400040004000400040i64 & q) << 1 | \
	 (01000100010001000100i64 & q) >> 1 | \
	 (02000200020002000200i64 & q) >> 3 | \
	 (04000400040004000400i64 & q) >> 5 | \
	 (010001000100010001000i64 & q) >> 7 | \
	 (020002000200020002000i64 & q) >> 9 | \
	 (040004000400040004000i64 & q) >> 11)

#define ROTATE_X(q) \
	((07i64 & q) << 9 | \
	 (070i64 & q) << 18 | \
	 (0700i64 & q) << 27 | \
	 (07000i64 & q) << 36 | \
	 (070000i64 & q) >> 6 | \
	 (0700000i64 & q) << 3 | \
	 (07000000i64 & q) << 12 | \
	 (070000000i64 & q) << 21 | \
	 (0700000000i64 & q) >> 21 | \
	 (07000000000i64 & q) >> 12 | \
	 (070000000000i64 & q) >> 3 | \
	 (0700000000000i64 & q) << 6 | \
	 (07000000000000i64 & q) >> 36 | \
	 (070000000000000i64 & q) >> 27 | \
	 (0700000000000000i64 & q) >> 18 | \
	 (07000000000000000i64 & q) >> 9)

#define ROTATE_Y(q) \
	((01111i64 & q) << 24 | \
	 (02222i64 & q) << 11 | \
	 (04444i64 & q) >>  2 | \
	 (011110000i64 & q) << 13 | \
	 (022220000i64 & q)       | \
	 (044440000i64 & q) >> 13 | \
	 (0111100000000i64 & q) << 2 | \
	 (0222200000000i64 & q) >> 11 | \
	 (0444400000000i64 & q) >> 24)

#define ROTATE_Z(q) \
	((010001000100010001i64 & q) << 2 | \
	 (020002000200020002i64 & q) << 4 | \
	 (040004000400040004i64 & q) << 6 | \
	 (0100010001000100010i64 & q) >> 2 | \
	 (0200020002000200020i64 & q)      | \
	 (0400040004000400040i64 & q) << 2 | \
	 (01000100010001000100i64 & q) >> 6 | \
	 (02000200020002000200i64 & q) >> 4 | \
	 (04000400040004000400i64 & q) >> 2)

#define LEFTADJUST(x) while (!(x & 011111111111111111111i64)) x >>= 1
#define FRONTADJUST(x) while (!(x & 070007000700070007i64)) x >>= 3
#define BOTTOMADJUST(x) while (!(x & 07777i64)) x >>= 12

#define XCENTER(x) ((x << 1 & x >> 1 | x << 3 & x >> 3) & x)

/* Globals */

int Loesungen = 0, Permutation[11] = {0,1,2,3,4,5,6,7,8,9,10}, TeilLoesungen;
struct _Teil {
	int Anzahl;
	QUADWORD *Lage;
	struct _Verteilung {
		QUADWORD *Erste;
		int Anzahl;
	} Verteilung[63];
} Teil[TEILEZAHL];
#ifdef DEBUG
	int gelegt;
	QUADWORD Plazierungen = 0;
#endif


QUADWORD *SchiebeTeile (QUADWORD *src, QUADWORD *base, int blk, int x, int y, int z)
{
	QUADWORD *dst;
	int c;

	x = QUADER_X + 1 - x;
	y = QUADER_Y + 1 - y;
	z = QUADER_Z + 1 - z;

	/* Ecklagen in Teil-Lagen-Array kopieren */
	dst = base;
	c = blk;
	do
		*dst++ = *src++;
	while (--c);

	/* auf alle x-Positionen */
	c = (x-1)*blk;
	src = base;
	while (c--)
		*dst++ = *src++ << 1;

	/* auf alle y-Positionen */
	c = (y-1)*x*blk;
	src = base;
	while (c--)
		*dst++ = *src++ << 3;

	/* auf alle z-Positionen */
	c = (z-1)*y*x*blk;
	src = base;
	while (c--)
		*dst++ = *src++ << 12;

	return dst;
}


/*
	Generische Teile-Initialisierung
	setzt Ausgangslage nicht-Null und ursprungsbndig voraus
*/

void AlleLagen (struct _Teil *dieses, QUADWORD Ausgangslage)
{
	int i, c, blk = 1, t_x, t_y, t_z;
	QUADWORD Ecklage[6][4], Ebene, temp, *next;

	/* bestimme Teil-Dimensionen */
	t_x = 1;
	Ebene = 022222222222222222222i64;
	while (Ausgangslage & Ebene && ++t_x < QUADER_X)
		Ebene <<= 1;

	t_y = 1;
	Ebene = 0700070007000700070i64;
	while (Ausgangslage & Ebene && ++t_y < QUADER_Y)
		Ebene <<= 3;

	t_z = 1;
	Ebene = 077770000i64;
	while (Ausgangslage & Ebene && ++t_z < QUADER_Z)
		Ebene <<= 12;

	/* Ausgangslagen mit allen Wendungen */

	/* Ecklage[0] = nicht rotiert (xyz) */
	for (c = 1; c < 6; c++)
		Ecklage[c][0] = 0i64;
	temp = Ecklage[0][0] = Ausgangslage;
	temp = TURN_Z (temp);
	LEFTADJUST (temp);
	FRONTADJUST(temp);
	if (temp != Ecklage[0][0])	// achsensymmetrisch zu z?
	{
		Ecklage[0][blk++] = temp;
		temp = TURN_Y (temp);
		LEFTADJUST(temp);
		BOTTOMADJUST (temp);
		if (temp != Ecklage[0][0] &&	// achsensymmetrisch zu x?
			temp != Ecklage[0][1])	// achsensymmetrisch zu y?
		{
			Ecklage[0][blk++] = temp;
			temp = TURN_Z (temp);
			LEFTADJUST (temp);
			FRONTADJUST(temp);
			Ecklage[0][blk++] = temp;
		}
	}
	else
	{
		temp = TURN_Y (temp);
		BOTTOMADJUST (temp);
		LEFTADJUST(temp);
		if (temp != Ecklage[0][0])	// achsensymmetrisch zu y?
			Ecklage[0][blk++] = temp;
	}
	dieses->Anzahl = blk *
		(QUADER_X+1 - t_x) * (QUADER_Y+1 - t_y) * (QUADER_Z+1 - t_z);

	/* alle mglichen und nicht symmetrischen Rotationen */

	/* x-Rotation mglich? */
	if (t_y <= QUADER_Z && t_z <= QUADER_Y)
	{
		temp = Ecklage[0][0];
		temp = ROTATE_X (temp);
		FRONTADJUST(temp);
		c = 0;
		while (temp != Ecklage[0][c] && ++c < blk);
		/* nicht rotationssymmetrisch um x? */
		if (c == blk)
		{
			/* Ecklage[1] = Rotation um x-Achse (xzy) */
			Ecklage[1][0] = temp;
			for (c = 1; c < blk; c++)
			{
				temp = Ecklage[0][c];
				temp = ROTATE_X (temp);
				FRONTADJUST(temp);
				Ecklage[1][c] = temp;
			}
			dieses->Anzahl += blk *
				(QUADER_X+1 - t_x) * (QUADER_Y+1 - t_z) * (QUADER_Z+1 - t_y);

			/* y-Rotation mglich? */
			if (t_x <= QUADER_Z && t_z <= QUADER_X)
			{
				temp = Ecklage[0][0];
				temp = ROTATE_Y (temp);
				BOTTOMADJUST (temp);
				c = 0;
				while (temp != Ecklage[0][c] && ++c < blk);
				/* nicht rotationssymmetrisch um y? */
				if (c == blk)
				{
					c = 0;
					while (temp != Ecklage[1][c] && ++c < blk);
					/* nicht rotationssymmetrisch um Raumdiagonale? */
					if (c == blk)
					{
						/* Ecklage[3] = Rotation um y-Achse (zyx) */
						Ecklage[3][0] = temp;
						for (c = 1; c < blk; c++)
						{
							temp = Ecklage[0][c];
							temp = ROTATE_Y (temp);
							BOTTOMADJUST (temp);
							Ecklage[3][c] = temp;
						}
						dieses->Anzahl += blk *
							(QUADER_X+1 - t_z) * (QUADER_Y+1 - t_y) * (QUADER_Z+1 - t_x);

						/* z-Rotation mglich? */
						if (t_x <= QUADER_Y && t_y <= QUADER_X)
						{
							temp = Ecklage[0][0];
							temp = ROTATE_Z (temp);
							LEFTADJUST(temp);
							c = 0;
							while (temp != Ecklage[0][c] && ++c < blk);
							/* nicht rotationssymmetrisch um z? */
							if (c == blk)
							{
								/* Ecklage[5] = Rotation um z-Achse (yxz) */
								Ecklage[5][0] = temp;
								for (c = 1; c < blk; c++)
								{
									temp = Ecklage[0][c];
									temp = ROTATE_Z (temp);
									LEFTADJUST(temp);
									Ecklage[5][c] = temp;
								}
								dieses->Anzahl += blk *
									(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_z);

								/* Ecklage[2] = Rotation um z-Achse nach x-Rotation (zxy) */
								for (c = 0; c < blk; c++)
								{
									temp = Ecklage[1][c];
									temp = ROTATE_Z (temp);
									LEFTADJUST(temp);
									Ecklage[2][c] = temp;
								}
								dieses->Anzahl += blk *
									(QUADER_X+1 - t_z) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_y);

								/* Ecklage[4] = Rotation um z-Achse nach y-Rotation (yzx) */
								for (c = 0; c < blk; c++)
								{
									temp = Ecklage[3][c];
									temp = ROTATE_Z (temp);
									LEFTADJUST(temp);
									Ecklage[4][c] = temp;
								}
								dieses->Anzahl += blk *
									(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_z) * (QUADER_Z+1 - t_x);
							}
						}
						/* keine z-Rotation */
						else if (t_x <= QUADER_Y)
						{
							/* Ecklage[2] = Rotation um z-Achse nach x-Rotation (zxy) */
							for (c = 0; c < blk; c++)
							{
								temp = Ecklage[1][c];
								temp = ROTATE_Z (temp);
								LEFTADJUST(temp);
								Ecklage[2][c] = temp;
							}
							dieses->Anzahl += blk *
								(QUADER_X+1 - t_z) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_y);
						}
						else
						{
							/* Ecklage[4] = Rotation um z-Achse nach y-Rotation (yzx) */
							for (c = 0; c < blk; c++)
							{
								temp = Ecklage[3][c];
								temp = ROTATE_Z (temp);
								LEFTADJUST(temp);
								Ecklage[4][c] = temp;
							}
							dieses->Anzahl += blk *
								(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_z) * (QUADER_Z+1 - t_x);
						}
					}
				}
				/* rotationssymmetrisch um y; z-Rotation mglich? */
				else if (t_y <= QUADER_X)	// t_x <= QUADER_Y, da t_x == t_z
				{
					/* Ecklage[5] = Rotation um z-Achse (yxz) */
					for (c = 0; c < blk; c++)
					{
						temp = Ecklage[0][c];
						temp = ROTATE_Z (temp);
						LEFTADJUST(temp);
						Ecklage[5][c] = temp;
					}
					dieses->Anzahl += blk *
						(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_z);
				}
			}
			/* keine y-Rotation; z-Rotation mglich? */
			else if (t_x <= QUADER_Y && t_y <= QUADER_X)
			{
				temp = Ecklage[0][0];
				temp = ROTATE_Z (temp);
				LEFTADJUST(temp);
				c = 0;
				while (temp != Ecklage[0][c] && ++c < blk);
				/* nicht rotationssymmetrisch um z? */
				if (c == blk)
				{
					/* Ecklage[5] = Rotation um z-Achse (yxz) */
					Ecklage[5][0] = temp;
					for (c = 1; c < blk; c++)
					{
						temp = Ecklage[0][c];
						temp = ROTATE_Z (temp);
						LEFTADJUST(temp);
						Ecklage[5][c] = temp;
					}
					dieses->Anzahl += blk *
						(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_z);

					if (t_z <= QUADER_X)
					{
						/* Ecklage[2] = Rotation um z-Achse nach x-Rotation (zxy) */
						for (c = 0; c < blk; c++)
						{
							temp = Ecklage[1][c];
							temp = ROTATE_Z (temp);
							LEFTADJUST(temp);
							Ecklage[2][c] = temp;
						}
						dieses->Anzahl += blk *
							(QUADER_X+1 - t_z) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_y);
					}
					else
					{
						/* Ecklage[4] = Rotation um x-Achse nach z-Rotation (yzx) */
						for (c = 0; c < blk; c++)
						{
							temp = Ecklage[5][c];
							temp = ROTATE_X (temp);
							FRONTADJUST(temp);
							Ecklage[4][c] = temp;
						}
						dieses->Anzahl += blk *
							(QUADER_X+1 - t_z) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_y);
					}
				}
			}
		}
		/* rotationssymmetrisch um x; y-Rotation mglich? */
		else if (t_x <= QUADER_Z && t_z <= QUADER_X)
		{
			temp = Ecklage[0][0];
			temp = ROTATE_Y (temp);
			BOTTOMADJUST (temp);
			c = 0;
			while (temp != Ecklage[0][c] && ++c < blk);
			/* nicht rotationssymmetrisch um y? */
			if (c == blk)
			{
				/* Ecklage[3] = Rotation um y-Achse (zyx) */
				Ecklage[3][0] = temp;
				for (c = 1; c < blk; c++)
				{
					temp = Ecklage[0][c];
					temp = ROTATE_Y (temp);
					BOTTOMADJUST (temp);
					Ecklage[3][c] = temp;
				}
				dieses->Anzahl += blk *
					(QUADER_X+1 - t_z) * (QUADER_Y+1 - t_y) * (QUADER_Z+1 - t_x);

				/* z-Rotation mglich? */
				if (t_x <= QUADER_Y)	// t_y <= QUADER_X, da t_y == t_z
				{
					/* Ecklage[5] = Rotation um z-Achse (yxz) */
					for (c = 0; c < blk; c++)
					{
						temp = Ecklage[0][c];
						temp = ROTATE_Z (temp);
						LEFTADJUST(temp);
						Ecklage[5][c] = temp;
					}
					dieses->Anzahl += blk *
						(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_z);
				}
			}
		}
		/* rotationssymmetrisch um x, keine y-Rotation; z-Rotation mglich? */
		else if (t_x <= QUADER_Y && t_y <= QUADER_X)
		{
			/* Ecklage[5] = Rotation um z-Achse (yxz) */
			for (c = 0; c < blk; c++)
			{
				temp = Ecklage[0][c];
				temp = ROTATE_Z (temp);
				LEFTADJUST(temp);
				Ecklage[5][c] = temp;
			}
			dieses->Anzahl += blk *
				(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_z);
		}
	}
	/* keine x-Rotation; y-Rotation mglich? */
	else if (t_x <= QUADER_Z && t_z <= QUADER_X)
	{
		temp = Ecklage[0][0];
		temp = ROTATE_Y (temp);
		BOTTOMADJUST (temp);
		c = 0;
		while (temp != Ecklage[0][c] && ++c < blk);
		/* nicht rotationssymmetrisch um y? */
		if (c == blk)
		{
			/* Ecklage[3] = Rotation um y-Achse (zyx) */
			Ecklage[3][0] = temp;
			for (c = 1; c < blk; c++)
			{
				temp = Ecklage[0][c];
				temp = ROTATE_Y (temp);
				BOTTOMADJUST (temp);
				Ecklage[3][c] = temp;
			}
			dieses->Anzahl += blk *
				(QUADER_X+1 - t_z) * (QUADER_Y+1 - t_y) * (QUADER_Z+1 - t_x);

			/* z-Rotation mglich? */
			if (t_x <= QUADER_Y && t_y <= QUADER_X)
			{
				temp = Ecklage[0][0];
				temp = ROTATE_Z (temp);
				LEFTADJUST(temp);
				c = 0;
				while (temp != Ecklage[0][c] && ++c < blk);
				/* nicht rotationssymmetrisch um z? */
				if (c == blk)
				{
					/* Ecklage[5] = Rotation um z-Achse (yxz) */
					Ecklage[5][0] = temp;
					for (c = 1; c < blk; c++)
					{
						temp = Ecklage[0][c];
						temp = ROTATE_Z (temp);
						LEFTADJUST(temp);
						Ecklage[5][c] = temp;
					}
					dieses->Anzahl += blk *
						(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_z);

					if (t_y <= QUADER_Z)
					{
						/* Ecklage[2] = Rotation um x-Achse nach y-Rotation (zxy) */
						for (c = 0; c < blk; c++)
						{
							temp = Ecklage[3][c];
							temp = ROTATE_X (temp);
							FRONTADJUST(temp);
							Ecklage[2][c] = temp;
						}
						dieses->Anzahl += blk *
							(QUADER_X+1 - t_z) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_y);
					}
					else
					{
						/* Ecklage[4] = Rotation um x-Achse nach z-Rotation (yzx) */
						for (c = 0; c < blk; c++)
						{
							temp = Ecklage[5][c];
							temp = ROTATE_X (temp);
							FRONTADJUST(temp);
							Ecklage[4][c] = temp;
						}
						dieses->Anzahl += blk *
							(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_z) * (QUADER_Z+1 - t_x);
					}
				}
			}
		}
		/* rotationssymmetrisch um y; z-Rotation mglich? */
		else if (t_x <= QUADER_Y && t_y <= QUADER_X)
		{
			/* Ecklage[5] = Rotation um z-Achse (yxz) */
			for (c = 0; c < blk; c++)
			{
				temp = Ecklage[0][c];
				temp = ROTATE_Z (temp);
				LEFTADJUST(temp);
				Ecklage[5][c] = temp;
			}
			dieses->Anzahl += blk *
				(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_z);
		}
	}
	/* weder x- oder y-Rotation; z-Rotation mglich? */
	else if (t_x <= QUADER_Y && t_y <= QUADER_X)
	{
		temp = Ecklage[0][0];
		temp = ROTATE_Z (temp);
		LEFTADJUST(temp);
		c = 0;
		while (temp != Ecklage[0][c] && ++c < blk);
		/* nicht rotationssymmetrisch um z? */
		if (c == blk)
		{
			/* Ecklage[5] = Rotation um z-Achse (yxz) */
			Ecklage[5][0] = temp;
			for (c = 1; c < blk; c++)
			{
				temp = Ecklage[0][c];
				temp = ROTATE_Z (temp);
				LEFTADJUST(temp);
				Ecklage[5][c] = temp;
			}
			dieses->Anzahl += blk *
				(QUADER_X+1 - t_y) * (QUADER_Y+1 - t_x) * (QUADER_Z+1 - t_z);
		}
	}

	next = dieses->Lage = calloc (sizeof (QUADWORD), dieses->Anzahl);

	/* alle xyz */
	next = SchiebeTeile (Ecklage[0], next, blk, t_x, t_y, t_z);

	/* alle xzy */
	if (Ecklage[1][0])
		next = SchiebeTeile (Ecklage[1], next, blk, t_x, t_z, t_y);

	/* alle zxy */
	if (Ecklage[2][0])
		next = SchiebeTeile (Ecklage[2], next, blk, t_z, t_x, t_y);

	/* alle zyx */
	if (Ecklage[3][0])
		next = SchiebeTeile (Ecklage[3], next, blk, t_z, t_y, t_x);

	/* alle yzx */
	if (Ecklage[4][0])
		next = SchiebeTeile (Ecklage[4], next, blk, t_y, t_z, t_x);

	/* alle yxz */
	if (Ecklage[5][0])
		SchiebeTeile (Ecklage[5], next, blk, t_y, t_x, t_z);
}


int CompareQuads (QUADWORD *qw1, QUADWORD *qw2)
{
	return *qw1 > *qw2 ? 1 : -1;
}


int CompareXCenters (QUADWORD *qw1, QUADWORD *qw2)
{
	return XCENTER (*qw1) < XCENTER (*qw2) ? 1 : -1;
}


void Initialisiere (void)
{
	int i, j, k, l;
	QUADWORD Schwelle;

	AlleLagen (&Teil[ 0], 010033);	// Q 001 000 000 011 011
	AlleLagen (&Teil[ 1],    073);	// p 111 011
	AlleLagen (&Teil[ 2],   0172);	// 4 001 111 010
	AlleLagen (&Teil[ 3], 010023);	// v 001 000 000 010 011 symm.
	AlleLagen (&Teil[ 4],    057);	// c 101 111 y-achsensymm.
	AlleLagen (&Teil[ 5],  01313);	// F 001 011 001 011
	AlleLagen (&Teil[ 6],  01131);	// y 001 001 011 001
	AlleLagen (&Teil[ 7],  01113);	// L 001 001 001 011
	AlleLagen (&Teil[ 8],   0174);	// s 001 111 100 z-achsensymm.
	AlleLagen (&Teil[ 9],   0171);	// T 001 111 001 x-achsensymm.
	AlleLagen (&Teil[10],   0136);	// w 001 011 110 xy-diagonalsymm.
	AlleLagen (&Teil[11],   0272);	// x 010 111 010 achsen- und rotationssymm.

	/* Von Teil Q nur die Lagen vorne unten links (fr Rckwrtssuche)
	   und vorne oben links (fr Vorwrtssuche) verwenden */
	for (i = j = 0; i < Teil[0].Anzahl; i++)
		if ((Teil[0].Lage[i] & 000000000011701170117i64) &&
			!(Teil[0].Lage[i] & 077777777740074007400i64) ||
			 (Teil[0].Lage[i] & 001170117011700000000i64) &&
			!(Teil[0].Lage[i] & 074007400740077777777i64))
			Teil[0].Lage[j++] = Teil[0].Lage[i];
	Teil[0].Anzahl = j;

	for (i = 0; i < 11; i++)
	{
		/* Teilepositionen nach hchstwertigem Bit sortieren */
		qsort (Teil[i].Lage, Teil[i].Anzahl, sizeof (QUADWORD),
			(int (*)(const void *,const void *)) CompareQuads);

		/* Tabelle der mit gleichem Bit beginnenden Lageserien aufstellen */
		Teil[i].Verteilung[0].Anzahl = 0;
		j = 0;
		Schwelle = 2;
		while (Schwelle <= Teil[i].Lage[0])
		{
			Teil[i].Verteilung[j].Erste = NULL;
			Teil[i].Verteilung[j].Anzahl = 0;
			Schwelle <<= 1;
			j++;
		}
		Teil[i].Verteilung[j].Erste = Teil[i].Lage;
		l = 0;
		k = 1;
		for (;;)
		{
			if (Teil[i].Lage[k] >= Schwelle)
			{
				Teil[i].Verteilung[j].Anzahl = k - l;
				Teil[i].Verteilung[++j].Erste = Teil[i].Lage + k;
				l = k;
				Schwelle <<= 1;
				if (j > 58) break;
			}
			else
				if (++k >= Teil[i].Anzahl) break;
		}
		Teil[i].Verteilung[j].Anzahl = Teil[i].Anzahl - l;
		while (++j < 60)
		{
			Teil[i].Verteilung[j].Erste = NULL;
			Teil[i].Verteilung[j].Anzahl = 0;
		}
	}

	/* Sonderbehandlung Teil 11 ('x') */
	/* absteigend nach Mittelpunkt-z sortieren */
	qsort (Teil[11].Lage, Teil[11].Anzahl, sizeof (QUADWORD),
		(int (*)(const void *,const void *)) CompareXCenters);
	/* Tabelle der Lagen mit in der gleichen z-Ebene liegenden
		Mittelpunkten aufstellen; nur z-Ebenen 4 bis 2 bentigt,
		da 1 und 0 gewendet 3 und 4 entsprechen; Ebenen 3 und 4
		kommen in Verteilung[0], Ebene 2 in Verteilung [1] */
	Schwelle = 01000000000000i64;	// unterstes Bit Ebene 3
	Teil[11].Verteilung[0].Erste = Teil[11].Lage;
	for (j = 0, k = 1, l = 0; j < 2; k++)
		if (XCENTER(Teil[11].Lage[k]) < Schwelle)
		{
			Teil[11].Verteilung[j].Anzahl = k - l;
			Teil[11].Verteilung[++j].Erste = Teil[11].Lage + k;
			l = k;
			Schwelle >>= 12;
		}
}


void LegeTeileMitQAbBit (int Bit, QUADWORD Quader, int Teile)
#ifdef HIGHSPEED
	;
#else
{
	register QUADWORD *AktLage, QuaderNeu;
	register int j, dran, i, BitNeu;

	i = --Teile;
	do
	{
		dran = Permutation[i];
		Permutation[i] = Permutation[Teile];

		AktLage = (QUADWORD *) &Teil[dran].Verteilung[Bit];
		if (j = ((struct _Verteilung *) AktLage)->Anzahl)
		{
			AktLage = ((struct _Verteilung *) AktLage)->Erste;
			do
			{
				if (!(*AktLage & Quader))
				{
#ifdef DEBUG
					gelegt++;
#endif
					if (Teile)
					{
						QuaderNeu = Quader | *AktLage;
						BitNeu = Bit;
						while (QuaderNeu & (1i64 << --BitNeu));

						LegeTeileMitQAbBit (BitNeu, QuaderNeu, Teile);
					}
					else
					{
						TeilLoesungen++;
#ifdef DEBUG
						assert ((*AktLage | Quader) == 0xFFFFFFFFFFFFFFFFi64);
						printf("\r%8d", TeilLoesungen);
#endif
						break;
					}
				}
				AktLage++;
			} while (--j);
		}
		Permutation[i] = dran;
	} while (i--);
	/* Schleife mit erstem Permutationselement! */
}
#endif


void LegeTeileAbBit (int Bit, QUADWORD Quader, int Teile)
#ifdef HIGHSPEED
	;
#else
{
	register QUADWORD *AktLage, QuaderNeu;
	register int j, dran, i, BitNeu;

	/* Schleife ohne erstes Permutationselement! */
	i = Teile--;
	while (--i)
	{
		dran = Permutation[i];
		Permutation[i] = Permutation[Teile];

		AktLage = (QUADWORD *) &Teil[dran].Verteilung[Bit];
		if (j = ((struct _Verteilung *) AktLage)->Anzahl)
		{
			AktLage = ((struct _Verteilung *) AktLage)->Erste;
			do
			{
				if (!(*AktLage & Quader))
				{
#ifdef DEBUG
					gelegt++;
#endif
					QuaderNeu = Quader | *AktLage;
					BitNeu = Bit;
					while (QuaderNeu & (1i64 << --BitNeu));

					if (BitNeu > 31)
						LegeTeileAbBit (BitNeu, QuaderNeu, Teile);
					else
						LegeTeileMitQAbBit (BitNeu, QuaderNeu, Teile);
				}
				AktLage++;
			} while (--j);
		}
		Permutation[i] = dran;
	}
}
#endif


void LegeTeileGewendetAbBit (int Bit, QUADWORD Quader, int Teile)
#ifdef HIGHSPEED
	;
#else
{
	register QUADWORD *AktLage, QuaderNeu;
	register int j, dran, i, BitNeu;

	i = --Teile;
	do
	{
		dran = Permutation[i];
		Permutation[i] = Permutation[Teile];

		AktLage = (QUADWORD *) &Teil[dran].Verteilung[Bit];
		if (j = ((struct _Verteilung *) AktLage)->Anzahl)
		{
			AktLage = ((struct _Verteilung *) AktLage)->Erste;
			do
			{
				if (!(*AktLage & Quader))
				{
#ifdef DEBUG
					gelegt++;
#endif
					QuaderNeu = Quader | *AktLage;
					BitNeu = Bit;
					while (QuaderNeu & (1i64 << --BitNeu));

					if (dran == 0)
						LegeTeileMitQAbBit (BitNeu, QuaderNeu, Teile);
					else if (BitNeu >= 36)
						LegeTeileGewendetAbBit (BitNeu, QuaderNeu, Teile);
				}
				AktLage++;
			} while (--j);
		}
		Permutation[i] = dran;
	} while (i--);
	/* Schleife mit erstem Permutationselement! */
}
#endif


int main(void)
{
	int i;

	Initialisiere ();

#ifndef DEBUG
	printf ("... los geht's, suche ...");
#endif

	for (i = 0; i < 2; i++)
	{
		do
		{
			/* 'x'-Mittelpunkt-Ebenen z = 4, 3, 2 */
			TeilLoesungen = 0;
#ifdef DEBUG
			gelegt = 0;
#endif

			LegeTeileAbBit (59,
				Teil[11].Verteilung[i].Erste[--Teil[11].Verteilung[i].Anzahl] |
				0xF000000000000000i64, 11);
			Loesungen += TeilLoesungen;

#ifdef DEBUG
			Plazierungen += gelegt;
			printf (", %d\n", gelegt);
#else
			printf ("\r... schon %d Lsungen ...", Loesungen);
#endif

			/* 'x'-Mittelpunkt-Ebenen z = 0, 1 */
			if (i == 0)
			{
				TeilLoesungen = 0;
#ifdef DEBUG
				gelegt = 0;
#endif

				LegeTeileGewendetAbBit (59,
				Teil[11].Verteilung[i].Erste[Teil[11].Verteilung[i].Anzahl] |
					0xF000000000000000i64, 11);
				Loesungen += TeilLoesungen;

#ifdef DEBUG
				Plazierungen += gelegt;
				printf (", %d\n", gelegt);
#else
				printf ("\r... schon %d Lsungen ...", Loesungen);
#endif
			}
		} while (Teil[11].Verteilung[i].Anzahl);
	}

	printf ("\rEs gibt %d echt unterschiedliche Lsungen"
#ifdef DEBUG
		" (%.0f Plazierungen)"
#endif
		"!\n", Loesungen
#ifdef DEBUG
		, (double) Plazierungen
#endif
		);
	return 0;
}
