/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
/* This Header is built automatically. Do not change.               */
/* $Id: main.c,v 1.28 2003/04/30 20:32:50 raap Exp $ */
/*<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/

/* This program is (c) 2003 Horst Raap and may be freely distributed under the GNU GPL */

/********************************************************************/
/* 
Dieses Programm lst das c't Puzzle.
Folgende Ideen zur Zeitoptimierung wurden zustzlich zu den im Artikel beschriebenen eingebaut:

- Es wird nicht immer versucht, das nchste Loch im Klotz aufzufllen, sondern je nach Gegebenheit wird
	entweder das Loch mit den wenigsten Fll-Mglichkeiten durchprobiert oder aber es wird versucht,
	das Teil mit den wenigsten Mglichkeiten an irgendeine Stelle des Klotzes zu packen.
	Quasi nebenbei werden so sehr schnell unfllbare Lcher und/oder nicht mehr platzierbare Kltze entdeckt.

- Es wird eine relativ komplizierte Datenstruktur "FillList" aufgebaut, die zu jedem Loch im Klotz angibt, welche Teile in welchen
	Lagen noch dort hineinpassen. Bis zu einer gewissen Tiefe wird diese Struktur stndig aktualisiert, damit man zgig die o. g.
	Informationen ermitteln kann. 
	
-  Auch bei herausgerechneten Symmetrien ergeben sich bei obigem Vorgehen Situationen, die bereits durchprobiert wurden,
	d. h. dieselbe Menge von Teilen belegt die gleichen Teilwrfel. Fr solche Situationen, die keine Lsungen liefern 
	(das sind die meisten), wird eine Hash-Tabelle angelegt, so dass bei einer Wiederholung nicht mehr erneut gesucht werden muss.

- Einige Situationen knnen recht frhzeitig als unlsbar erkannt werden, wenn man ein Paritts-Argument benutzt:
	Die Teilwrfel des Klotzes werden dabei abwechselnd schwarz und wei gefrbt. Wenn der Klotz ganz gefllt ist, sind genauso
	viele schwarze wie weie Teilwrfel belegt, d. h. die Paritt muss bei jeder Lsung 0 sein.
	Jetzt stelle man sich vor, es sind noch drei weie und drei schwarze Teilwrfel frei und das einzufgende Teil ist
	das "T". Das "T" hat aber in jeder Lage eine Paritt von 2 oder -2, so dass es keine Lsung geben kann.
	Dieser Algorithmus ist hier etwas verallgemeinert implementiert.
*/
/********************************************************************/

//#define DEBUG
//#define EARLYABORT

/*=================================================================*/
/*              Include files needed are following here !          */
/*=================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#define _GNU_SOURCE
#include <getopt.h>


/*=================================================================*/
/*              End of includes !                                  */
/*=================================================================*/


/* /////////////////////////////////////////////////////////////// */
/*              External Variables and Functions                   */
/* /////////////////////////////////////////////////////////////// */

int getopt(int argc, char * const argv[],
                  const char *optstring);

extern char *optarg;
extern int optind, opterr, optopt;

      
/* \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\ */
/*              Global Variables                                   */
/* \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\ */



/* Verbosity */
int Verb = 0;

/* Is set to 1 iff we must print solutions */
int PrtSol = 0;

#ifdef EARLYABORT
	int mindepth = 0;
#endif

#define HashDepth 7
#define UPDATEFILLDEPTH 5

/* This part will be treated differently due to symmetry */
#define SYMMPART 2

/* Is 1 iff the part currently is in use */
int Used[12];
/* Same as above but with bits */
unsigned int UseMap;

/* Used to find the lowest 0-Bit in an integer fast */
unsigned char LeastZero[(1<<16)];


int solutions;

/* Preference ordering of the parts */
int TryList[12]={11,10,7,5,3,2,1,0,9,8,6,4}; 

#ifdef DEBUG
int Nodes[12];
int MinHoleTries[12];
int SpecialHoleTries[12];
int PartTries[12];
int ParityFails[12];
int Repetitions[12];
int ZeroHoles[12];
#endif

/*\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
/*              End of Global Variables                            */
/*\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

/*------------------------ Local Variables ------------------------*/

/* Allow automatic identification by RCS */
static char RcsId[]="$Id: main.c,v 1.28 2003/04/30 20:32:50 raap Exp $";


/*------------------ Forward Function definitions -----------------*/

void CleanExit();    /* terminate program gracefully */
static void Usage();        /* give a usage message */

/*-----------------------------------------------------------------*/

/* Just for convenience */
#define bool int


/* A 60 bit BitMap represents the cubes of the block
	or a part at a spcific placement within the block
*/  
typedef struct {
	unsigned long w1;
	unsigned long w2;
} BitMap; 

/* -------------------------- General functions ------------------------------------- */
void NewLine() {
	printf("\n");
};

static inline
int abs(int a){
	if (a < 0) return -a;
	else return a;
};


/* Mapping from coordinates to an integer 0..59 */
int Coord2Index(int x, int y, int z){
 return(z + y * 3 + x * 12); 
};


/* --------------------------- Bitmap handling --------------------------------------- */

/* For each integer below 2**16 store the index of the lowest 0-bit 
	 Stores 16 if there are no 0-bits 
*/
void InitLeastZero(){
	int i,j;
	
	for (i=0; i<(1<<16); i++){
		for (j=0; j<16; j++)
			if (0 == (i & (1<<j)))
				break;
		LeastZero[i]=j;
	};
};

/* For each unsigned integer below 2**32 returns the index of the lowest 0-bit */
static inline
int LeastZero32(unsigned long w){
	if ((w & 0xffffL) != 0xffffL)
		return LeastZero[w & 0xffffL];
	else
		return LeastZero[w >> 16] + 16;
};


/* Returns index of the first empty space in given bitmap */
static inline
int FindFirstHole(BitMap *m){
	if (m->w1 != 0xffffffff)
		return (LeastZero32(m->w1));
	else
		return (LeastZero32((m->w2)) + 32);
};


/* Returns 0 iff the given bitmaps are disjoint (have no common bits set) */
static inline 
bool disjoint (BitMap *m1, BitMap *m2){
	return ( ((m1->w1 & m2->w1) == 0)  &&
			 ((m1->w2 & m2->w2) == 0) 
	);
};


/* Anding two bitmaps together storing result in second bitmap */
static inline 
void SetAnd (BitMap *m1, BitMap *m2){
	m2->w1 &= m1->w1;
	m2->w2 &= m1->w2;
};


/* Returns <> 0 iff bit i is set in the given bitmap */
bool GetBit(int i, BitMap *m) {
	if (i<32) 
		return (m->w1 & (1<<i));
	else
		return (m->w2 & (1<<(i-32)));
};

void PrtBitMap(BitMap *m) {
	int x, y, z;
	NewLine();
	
	for (z=2; z>=0; z--){
		for (y=3; y>=0; y--){
			for (x=0; x<5; x++)
				printf("%2d ", GetBit(Coord2Index(x,y,z),m)?1:0);
			NewLine();
		};
		NewLine();
	}
};


/* Sets a bitmap to just the given bit */
void IndexToBit (int idx, BitMap *m){
	if (idx < 32) {
		m->w1 = 1 << idx;
		m->w2 = 0;
	} else {
		m->w1 = 0;
		m->w2 = 1 << (idx-32);
	};
};


/* Returns number of bits set in the given bitmap */
int CntBits (BitMap *m){
	int i, cnt=0;
	for (i=0; i<60; i++)
		if (GetBit(i,m)) cnt++;
	return cnt;
};


/* Returns TRUE iff the given bitmap is 1 at index i */
static inline 
bool IsSet (BitMap *m, int i){
	if (i<32) return ( m->w1 & (1<<i)) ;
	else return ( m->w2 & (1 << (i-32))); 
};


/* ------------------------------------------- Hash table handling ---------------------------------- */

/* The information to be stored in a hash entry */
typedef struct {
	BitMap	m;
	unsigned short use;
	short depth;
} State;
 

State *HashTable = NULL; 
/* Number of entries in hash table */
int	HashTableLim;
unsigned long HashMask;
/* some random bits used for hashing */
long int Rand[1<<16];

void InitHash(){
	int i;
	srandom(1);
	for (i=0; i<(1<<16); i++)
		Rand[i] = random();
	/* We dont know how much memory we can get for the hash table, try some powers of two */
	for (i=24; i>0; i--){
		HashTableLim = 1<<i;
		HashTable = (State *) calloc(HashTableLim,sizeof(State));
		if (HashTable != 0) break;
	};
	if (HashTable == 0)
		CleanExit("Not enough memory for hash table\n",20);
	HashMask = HashTableLim - 1;

	for (i=0; i<HashTableLim; i++){
		(HashTable+i)->depth = 99;
		(HashTable+i)->use = 0xffff;
	};		
};


/* Check if this situation is already in HashTable 
	and return TRUE
	else return 0
*/
bool Recall (BitMap *bm){
	unsigned int i;
	State *s;
	unsigned long hv, hv1;
	
	/* Compute HashValue */
	hv1 = bm->w1 ^ bm->w2;
	hv = Rand[hv1 & 0xffff] ^ Rand[hv1 >> 16] ^ Rand[UseMap];

	i = hv & HashMask;
	s = HashTable+i;
	if ((s->use == UseMap) && (s->m.w1 == bm->w1) && (s->m.w2 == bm->w2) )
		return 1;

	i = (hv + 1) & HashMask;
	s = HashTable+i;
	return ((s->use == UseMap) && (s->m.w1 == bm->w1) && (s->m.w2 == bm->w2) );
};


/* Store this situation in HashTable 
	but do not override situations at lower depth
*/
bool StoreHash(BitMap *bm, int depth){
	unsigned int i;
	State *s;
	unsigned long hv, hv1;
	
	/* Compute HashValue */
	hv1 = bm->w1 ^ bm->w2;
	hv = Rand[hv1 & 0xffff] ^ Rand[hv1 >> 16] ^ Rand[UseMap];	

	i = hv & HashMask;
	s = HashTable+i;		
	if (s->depth >= depth) {
		s->use = UseMap;
		s->m = *bm;
		s->depth = depth;
		return 0;
	};	
	i = (hv + 1) & HashMask;
	s = HashTable+i;		
	if (s->depth >= depth) {
		s->use = UseMap;
		s->m = *bm;
		s->depth = depth;
		return 0;
	};
return 0;
};


/* -------------------------- Cube handling ----------------------------------------- */

/* A cube is defined by its 3D position */
typedef struct {
	int x;
	int y;
	int z;
	BitMap m;
} Cube;

/* Returns an index from 0 to 59 for each cube */
int CubeIndex (Cube *c){
	return ( Coord2Index(c->x, c->y, c->z)); 
};

/* Sets a bit in the bitmap according to given cube */
void SetBit(Cube *c, BitMap *m) {
	int i;
	i = CubeIndex(c);
	if (i<32) 
		m->w1 |= 1<<i;
	else
		m->w2 |= 1<<(i-32);
};

void PrtCube (Cube *c) {
	printf("(%d,%d,%d)", c->x, c->y, c->z);
};


/* Shift Cube c by a specified amount */
void ShiftCube(Cube *c, int dx, int dy, int dz){
	c->x += dx;
	c->y += dy;
	c->z += dz;
};

/* Turn a cube 90 degrees around z-axis */
void TurnCubeZ(Cube *c){
	int h;
	h = c->y;
	c->y = c->x;
	c->x = -h;
};

/* Turn a cube 90 degrees around y-axis */
void TurnCubeY(Cube *c){
	int h;
	h = c->z;
	c->z = c->x;
	c->x = -h;
};

/* Turn a cube 90 degrees around x-axis */
void TurnCubeX(Cube *c){
	int h;
	h = c->z;
	c->z = c->y;
	c->y = -h;
};

/* Rotate the block around the given axis
	and set Cube to new position
*/
void RotateCube(Cube *c, int rx, int ry, int rz){
	if (rx) {
		c->z = 2 - c->z;
		c->y = 3 - c->y;
	};
	if (ry) {
		c->x = 4 - c->x;
		c->z = 2 - c->z;
	};
	if (rz) {
		c->x = 4 - c->x;
		c->y = 3 - c->y;
	};
};


/* --------------------------------------- Parity Handling ---------------------------------- */
/*
	We assign a color (black = 0, white = 1) to every cube in the block
	The colors are assigned alternately black and white along each axis of the block.
	starting with black at (0,0,0).
	Whenever we place a part in the block, it will cover a number of white and a number of black cubes.
	We call the parity of a part the difference between these two numbers.
	The absolute parity is independent of the placement of the part within the block.
	If we have found a solution, the total parity of the parts must equal the parity of the filled block,
	which is 0.
	By checking the parity of the unused parts we can sometimes detect unsolvable situations early on.
*/

/* Maximum possible parity which can be achieved by the unused parts */
int MaxParity;

/* For each part its absolute parity */
int AbsParity[12];

/* Will be filled with a pattern so that two neighbours in any direction are always 0 and 1*/
BitMap ParityMap;

/* Sets the global ParityMap */
void InitParity(){
	Cube c;
	
	for (c.x=0; c.x<5; c.x++)
		for (c.y=0; c.y<4; c.y++)
			for (c.z=0; c.z<3; c.z++){
				if ((c.x+c.y+c.z) & 1) SetBit(&c, &ParityMap);	
	};
};


/* Returns parity of the given bitmap m1 (i.e. the number of white cubes set - number of black cubes set) */
int Parity (BitMap *m1){
	BitMap sum;
	BitMap *pm = &ParityMap;
	int cnt;
	
	sum.w1 = (m1->w1 & pm->w1); 
	sum.w2 = (m1->w2 & pm->w2);
	
	cnt = CntBits(&sum);
	 
	sum.w1 = (m1->w1 & ~(pm->w1)); 
	sum.w2 = (m1->w2 & ~(pm->w2));
	cnt -= CntBits(&sum);
	return cnt;
};

/* ---------------------------------------- Part handling ------------------------------------- */

/* A part consists of at most 6 cubes, a bitmap description and its parity at this point */
typedef struct {
	BitMap m;
	int Parity; 
	int n; /* number of cubes */
	Cube c[6];
	int PartNo;  /* Original part number */
} Part;
  

/* The original parts */
Part Teil[12];

/* Initialize the global array of basic parts */
void InitTeile( Part *Teil){
	Teil[0].n = 5;
	Teil[0].c[0] = (Cube) {0,0,0};
	Teil[0].c[1] = (Cube) {1,0,0};
	Teil[0].c[2] = (Cube) {0,1,0};
	Teil[0].c[3] = (Cube) {0,2,0};
	Teil[0].c[4] = (Cube) {1,2,0};
	Teil[0].PartNo = 0;
	
	Teil[1].n = 5;
	Teil[1].c[0] = (Cube) {0,0,0};
	Teil[1].c[1] = (Cube) {1,0,0};
	Teil[1].c[2] = (Cube) {1,1,0};
	Teil[1].c[3] = (Cube) {1,2,0};
	Teil[1].c[4] = (Cube) {2,2,0};
	Teil[1].PartNo = 1;
		
	Teil[2].n = 5;
	Teil[2].c[0] = (Cube) {0,0,0};
	Teil[2].c[1] = (Cube) {1,0,0};
	Teil[2].c[2] = (Cube) {1,1,0};
	Teil[2].c[3] = (Cube) {1,2,0};
	Teil[2].c[4] = (Cube) {1,3,0};
	Teil[2].PartNo = 2;
	Teil[3].n = 5;
	Teil[3].c[0] = (Cube) {0,0,0};
	Teil[3].c[1] = (Cube) {0,1,0};
	Teil[3].c[2] = (Cube) {0,2,0};
	Teil[3].c[3] = (Cube) {0,3,0};
	Teil[3].c[4] = (Cube) {1,2,0};
	Teil[3].PartNo = 3;
	
	Teil[4].n = 5;
	Teil[4].c[0] = (Cube) {0,0,0};
	Teil[4].c[1] = (Cube) {0,1,0};
	Teil[4].c[2] = (Cube) {1,0,0};
	Teil[4].c[3] = (Cube) {1,1,0};
	Teil[4].c[4] = (Cube) {1,1,1};
	Teil[4].PartNo = 4;
	
	Teil[5].n = 5;
	Teil[5].c[0] = (Cube) {1,0,0};
	Teil[5].c[1] = (Cube) {1,1,0};
	Teil[5].c[2] = (Cube) {1,2,0};
	Teil[5].c[3] = (Cube) {0,2,0};
	Teil[5].c[4] = (Cube) {2,2,0};
	Teil[5].PartNo = 5;
		
	Teil[6].n = 5;
	Teil[6].c[0] = (Cube) {0,0,0};
	Teil[6].c[1] = (Cube) {0,1,0};
	Teil[6].c[2] = (Cube) {1,0,0};
	Teil[6].c[3] = (Cube) {1,1,0};
	Teil[6].c[4] = (Cube) {1,2,0};
	Teil[6].PartNo = 6;
	
	Teil[7].n = 5;
	Teil[7].c[0] = (Cube) {0,0,0};
	Teil[7].c[1] = (Cube) {1,0,0};
	Teil[7].c[2] = (Cube) {1,1,0};
	Teil[7].c[3] = (Cube) {2,1,0};
	Teil[7].c[4] = (Cube) {2,2,0};
	Teil[7].PartNo = 7;
	
	Teil[8].n = 4;
	Teil[8].c[0] = (Cube) {0,0,0};
	Teil[8].c[1] = (Cube) {1,0,0};
	Teil[8].c[2] = (Cube) {1,1,0};
	Teil[8].c[3] = (Cube) {0,0,1};
	Teil[8].PartNo = 8;
	
	Teil[9].n = 5;
	Teil[9].c[0] = (Cube) {0,1,0};
	Teil[9].c[1] = (Cube) {0,2,0};
	Teil[9].c[2] = (Cube) {1,0,0};
	Teil[9].c[3] = (Cube) {1,1,0};
	Teil[9].c[4] = (Cube) {2,1,0};
	Teil[9].PartNo = 9;
	
	Teil[10].n = 5;
	Teil[10].c[0] = (Cube) {0,1,0};
	Teil[10].c[1] = (Cube) {1,0,0};
	Teil[10].c[2] = (Cube) {1,1,0};
	Teil[10].c[3] = (Cube) {1,2,0};
	Teil[10].c[4] = (Cube) {2,1,0};
	Teil[10].PartNo = 10;
	
	Teil[11].n = 6;
	Teil[11].c[0] = (Cube) {0,0,0};
	Teil[11].c[1] = (Cube) {1,0,0};
	Teil[11].c[2] = (Cube) {0,1,0};
	Teil[11].c[3] = (Cube) {0,2,0};
	Teil[11].c[4] = (Cube) {1,2,0};
	Teil[11].c[5] = (Cube) {0,3,0};
	Teil[11].PartNo = 11;
};


/* Returns TRUE iff both parts are composed of identical cubes */
bool EqPart (Part *p1, Part *p2) {
	
	return ((p1->m.w1 == p2->m.w1) && (p1->m.w2 == p2->m.w2));
};


/* Print representation of part */
void PrtPart (Part *p) {
	int	i;
	printf("[");
	for (i=0; i<p->n; i++){
		PrtCube(&(p->c[i]));
		if (i != p->n - 1) printf(",");
	};
	printf("]");
};

/* smallest x-coordinate of all cubes in p */
int MinX (Part *p){
	int i,m=9999;
	for (i=0; i<p->n; i++){
		if (p->c[i].x < m) m=p->c[i].x;
	};
	return m;
};


/* smallest y-coordinate of all cubes in p */
int MinY (Part *p) {
	int i,m=9999;
	for (i=0; i<p->n; i++){
		if (p->c[i].y < m) m=p->c[i].y;
	};
	return m;
};


/* smallest z-coordinate of all cubes in p */
int MinZ (Part *p){
	int i,m=9999;
	for (i=0; i<p->n; i++){
		if (p->c[i].z < m) m=p->c[i].z;
	};
	return m;
};


/* Shift complete part by the given amount */
void ShiftPart(Part *p, int dx, int dy, int dz){
	int i;
	for (i=0; i<p->n; i++){
		ShiftCube(&(p->c[i]), dx, dy, dz);
	};
};

/* Set the bitmap and parity of the given part to the correct values */
void SetPartMap(Part *p) {
	int i;
	p->m.w1 = 0;
	p->m.w2 = 0;
	for (i=0; i<p->n; i++)
		SetBit( &(p->c[i]), &(p->m));
	p->Parity = Parity(&(p->m));
};

/* Shift Part, so that the lowest coordinates are 0 */
void NormPart (Part *p){
	int dx, dy, dz;
	
	dx = -MinX(p);
	dy = -MinY(p);
	dz = -MinZ(p);
	ShiftPart(p, dx, dy, dz);
};



/* Turn complete part by given amounts (90 degree turns) in each axis */
void TurnPart (Part *p, int tx, int ty, int tz){
	int i, t;
	for (t=0; t<tx; t++)
		for (i=0; i<p->n; i++) 
			TurnCubeX(&(p->c[i]));

	for (t=0; t<ty; t++)
		for (i=0; i<p->n; i++) 
			TurnCubeY(&(p->c[i]));;

	for (t=0; t<tz; t++)
		for (i=0; i<p->n; i++) 
			TurnCubeZ(&(p->c[i]));
			
};

/* Rotate complete part (180 degrees) around given axis of block */
void RotatePart (Part *p, int rx, int ry, int rz){
	int i;
	
	for (i=0; i<p->n; i++) 
		RotateCube(&(p->c[i]), rx, ry, rz);
	SetPartMap(p);
}


/* Copy Part p1 to p2 */
void CopyPart (Part *p1, Part *p2){
	int i;
	
	p2->n = p1->n;
	p2->m = p1->m;
	p2->Parity = p1->Parity;
	p2->PartNo = p1->PartNo;
	for (i=0; i<p1->n; i++){
		p2->c[i] = p1->c[i];
	}
};



/* Returns 1 iff the given part fits completely inside puzzle area */ 
bool Fits(Part *p){
	int i;
	for (i=0; i<p->n; i++){
		if (p->c[i].x > 4) return 0;
		if (p->c[i].y > 3) return 0;
		if (p->c[i].z > 2) return 0;
	};
	return 1;
};



/* ---------------------------------------------- Handling of part lists ------------------------------------- */

/* A PartList can hold at most MAXPARTLIST parts */
#define MAXPARTLIST 1023
typedef struct {
	int n;
	Part p[MAXPARTLIST];
} PartList;

/* A pplist can hold at most MAXPPLIST pointers to parts */
#define MAXPPLIST 121
typedef struct {
	int n;
	Part *pp[MAXPPLIST];
} ppList;

/* A ppList for each space in the block is called a FillHoleList */
typedef ppList FillHoleList[64];

/* A FillHoleList for each part number is called a FillList */
typedef FillHoleList FillList[12];


 /* We define an array which holds for each part and for each cube index a list with pointers to those part placements,
    which can fill the block at this hole.
*/

/* This array holds a FillList for each depth of Try() */
FillList FillsArray[12];

/* Sets contents of given FillList to correct values */
void InitFills (PartList *Lagen, FillList *fills){
	int i, j, k, count;
	Part *curPart;
	BitMap m;
	
	for (i=0; i<12; i++) {
		for (j=0; j<60; j++){
			count = 0;
			for (k=0; k<Lagen[i].n; k++){
				curPart = &(Lagen[i].p[k]);
				IndexToBit(j, &m);
				if (! disjoint(&(curPart->m), &m)) {
					(*fills)[i][j].pp[count] = curPart;
					count++;
				};
			};
			(*fills)[i][j].n = count;
		};
	};
};

/* We have a current FillList *oldf
	All placements which are still possible within current block
	will be copied to the new FillList *newf

	Ignore unused parts and already filled hole
*/
void UpdateFill (BitMap *m, FillList *oldf, FillList *newf){
	int i,j,k,count, pplim;
	Part *curPart;
	ppList *curppList;
	
	for (i=0; i<12; i++) {
		if (Used[i]) continue;
		for (j=0; j<60; j++){
			if (IsSet(m,j)) continue;
			count = 0;
			curppList = &( (*oldf)[i][j] );
			pplim = curppList->n;
			for (k=0; k < pplim; k++){
				curPart = curppList->pp[k];
				if ( disjoint(&(curPart->m), m)) {
					(*newf)[i][j].pp[count++] = curPart;
				};
			};
			(*newf)[i][j].n = count;
		};
	};
}

/* Returns 1 if the given part is already a member of the partlist */
bool Member (Part *p, PartList *pl){
	int i;
	for (i=0; i<pl->n; i++)
		if (EqPart(p, &(pl->p[i])))
			return 1;
	return 0;
};


/* Returns TRUE, iff any reflection of the given part along the blocks axis
	results in a part which is already in the partlist
*/
bool CheckRotations (Part *p, PartList *pl){
	int rx,ry,rz;
	Part pnew;
	
	for (rx=0; rx<2; rx++)
		for (ry=0; ry<2; ry++)
			for (rz=0; rz<2; rz++){
				CopyPart(p,&pnew);
				RotatePart(&pnew,rx,ry,rz);
				if (Member(&pnew,pl)) 
					return 1;
			};
	return 0;
}

void InsertPart(Part *pnew, PartList *pl){
	if (pl->n >= MAXPARTLIST) 
		CleanExit("Too many parts in partlist",20);
	CopyPart(pnew,&(pl->p[pl->n]));
	(pl->n)++;
};

/* Generate all turns of a given part and insert them normalized into the given PartList if not already there */
void GenTurns(int pn, Part *p, PartList *pl){
	Part pnew;
	int i, j, k;
	
	for (i=0; i<4; i++)
		for (j=0; j<4; j++)
			for (k=0; k<4; k++) {
				CopyPart(p, &pnew);
				TurnPart(&pnew, i, j, k);
				NormPart(&pnew);
				if (!Fits(&pnew))
					continue;
				SetPartMap(&pnew);
				if (pn == SYMMPART)
					if (CheckRotations(&pnew, pl)) 
						continue;
				if ( ! Member(&pnew, pl) )
					InsertPart(&pnew,pl);
			};
};


/* Print representation of a PartList */
void PrtPartList(PartList *pl){
	int i;
	for (i=0; i<pl->n; i++) {
		PrtPart(&(pl->p[i]));
		NewLine();
	};
	NewLine();
};


/* Generate all shifts of the Parts in the PartList 
   We assume they are all normalized 
  */
void GenShifts (int pn, PartList *pl) {
	Part pnew;
	int i, np, dx, dy, dz;
	
	np = pl->n;
	
	for (i=0; i<np; i++) 	
		for (dx=0; dx < 5; dx++)
			for (dy=0; dy < 4; dy++)
				for (dz=0; dz < 3; dz++) {
					CopyPart(&(pl->p[i]), &pnew);
					ShiftPart(&pnew, dx, dy, dz);
					if (!Fits(&pnew))
						continue;
					SetPartMap(&pnew);
					if (pn == SYMMPART)
						if (CheckRotations(&pnew, pl)) 
							continue;
					if ( ! Member(&pnew, pl) ) 
						InsertPart(&pnew,pl); 
				};
};
  

 PartList Lagen[12];
   
/* Fill the array with all possible placements of all 12 basic parts */
void InitLagen (PartList *Lagen, Part *Teil){

	int i;
	for (i=0; i<12; i++) {
		Lagen[i].n = 1;
		Lagen[i].p[0] = Teil[TryList[i]];
		SetPartMap( &(Lagen[i].p[0]) );

		GenTurns(i, &(Lagen[i].p[0]), &(Lagen[i]));
		GenShifts(i, &(Lagen[i]));
	};
};


/* Initialize the absolute parity for each part number */
void InitAbsParity() {
	int pn;
	
	for (pn=0; pn<12; pn++)
		AbsParity[pn] = abs( Lagen[pn].p[0].Parity );
};



/* -------------------------------------------- block handling functions */

/* Bitmap and list of parts which fill this block */
typedef struct {
	/* Every hole already filled has its bit set */
	BitMap m;
	/* The current parity */
	int Parity;
	/* The placements of the parts */
	Part *p[12];
} Block;

Block b;

/* The block is initially empty */
void InitBlock (Block *b){
	int i;

	b->m.w1 = 0;
	b->m.w2 = 0;
	b->Parity = 0;
	for (i=0; i<12; i++)
		b->p[i] = 0;
};


/* Print the block in human readable form */
void PrtBlock(Block *b){
	int i, j, x, y, z;
	int PartNums[5][4][3];
	Part *p;
	
	/* Clear the PartNums array */ 
	for (z=2; z>=0; z--)
		for (y=3; y>=0; y--)
			for (x=0; x<5; x++)
				PartNums[x][y][z] = 0;

	/* Which Parts are where ? */
	for (i=0; i<12; i++) {
		if (b->p[i]) {
			p = b->p[i];
			/* Mark each cube of this part in array */
			for (j=0; j < p->n; j++) 
				PartNums[p->c[j].x][p->c[j].y][p->c[j].z] = p->PartNo + 1;
		};
	};
	NewLine();
	for (z=2; z>=0; z--){
		for (y=3; y>=0; y--){
			for (x=0; x<5; x++)
				printf("%2d ", PartNums[x][y][z]);
			NewLine();
		};
		NewLine();
	}
};


/* Mark this part as used */
static inline 
void Use(int pn){
	Used[pn] = 1;
	UseMap ^= 1<<pn;
	MaxParity -= AbsParity[pn];
};

/* Mark this part as not used anymore */
static inline
void UnUse(int pn){
	Used[pn] = 0;
	UseMap ^= 1<<pn;
	MaxParity += AbsParity[pn];
};


/* Store this part in the block */
static inline
void Store (Part *p, Block *b){
	b->m.w1 ^= p->m.w1;
	b->m.w2 ^= p->m.w2;
	b->Parity += p->Parity;
	b->p[p->PartNo] = p;
};

/* Remove part from block */
static inline
void Remove (Part *p, Block *b){
	b->Parity -= p->Parity;
	b->m.w1 ^= p->m.w1;
	b->m.w2 ^= p->m.w2;
	b->p[p->PartNo] = 0;
};


/* Returns 1 iff the given block is empty at index i */
static inline 
bool empty (Block *b, int i){
	if (i<32) return (( b->m.w1 & (1<<i)) == 0);
	else return (( b->m.w2 & (1 << (i-32))) == 0); 
};


/* Returns TRUE iff the given part fits in empty space of block */
static inline 
bool FitInBlock (Part *p, Block *b){
	return ( disjoint(&(b->m), &(p->m)) ); 
};	



/* We have found a solution, 
	count it and maybe print it
*/
void FoundSolution (Block *b) {
	solutions++;
	if (PrtSol){
		PrtBlock(b);
	};
};


/*  Stores part placement in *ResPart 
	iff there is a hole which can be filled only one way 
	Does detect holes which can never be filled
	and sets *h==-1 in this case
	Sets *partno to the one part filling in the special hole only if there is one
	Returns 1 iff a special hole has been found, else 0
	!!! Currently does not prefer unfillable holes !!!
*/
bool SpecialHole (Block *b, FillList *fl, int *partno, Part **ResPart){
	int pn, ln, ci, pnum, lnlim;
	int numfills;
	ppList *ppl;
	Part *resp;	
	
	/* For every empty cube */
	for (ci=0;ci<60; ci++) {
		if (!empty(b, ci)) continue;
		
		/* Can this hole be filled two or more ways? */
		numfills=0;	
		for (pn=0; pn < 12; pn++) {
			if (Used[pn]) continue;
			ppl = &((*fl)[pn][ci]);
			lnlim = ppl->n;
			for (ln=0; ln < lnlim; ln++) {
				if (FitInBlock(ppl->pp[ln],b)){
					if ((numfills++) > 0) 
						goto toomuch;
					else {
						pnum = pn;
						resp = ppl->pp[ln];
					};
				};
			};
		};
		/* Coming here there were 0 or 1 possible placements for this hole */
		if (numfills){
			*partno = pnum;
			*ResPart = resp;
		} else {
			*partno = -1;
		};
		return 1;
				
	/* Coming here we have found more than 1 possible placement for this hole
		Try the next hole
	 */
	toomuch:
	};
	return 0;
};


/*  Look for an unfillable hole
	Returns TRUE iff one is found
*/
bool ZeroHole (Block *b, FillList *fl){
	int pn, ln, ci, pnum, lnlim;
	int numfills;
	ppList *ppl;
	
	/* For every empty cube */
	for (ci=0;ci<60; ci++) {
		if (!empty(b, ci)) continue;
		
		/* Can this hole be filled? */
		for (pn=0; pn < 12; pn++) {
			if (Used[pn]) continue;
			ppl = &((*fl)[pn][ci]);
			lnlim = ppl->n;
			for (ln=0; ln < lnlim; ln++) {
				if (FitInBlock(ppl->pp[ln],b))
					goto toomuch;
			};
		};
		/* Coming here there was no possible placement for this hole */
		return 1;
				
	/* Coming here we have found more than 1 possible placement for this hole
		Try the next hole
	 */
	toomuch:
		continue;
	};
	return 0;
};


/* Returns hole index with minimal number of ways to fill it 
	2 holes with equal counts are sorted according to the number of parts
	Does not try to find holes with more than CurMin possibilities
*/
int MinHole (Block *b, FillList *fl, int *HoleIndex, int CurMin){
	Part *p;
	int pn, ln, ci, pinc, pcnt;
	int min = CurMin, pmax=0, numfills;
	ppList *ppl;
		
	/* For every empty cube */
	for (ci=0;ci<60; ci++) {
		if (!empty(b, ci)) continue;
	
		numfills=0;
		pcnt=0;
		
		/* Find all parts which fill this hole */	
		/* and are not already used */
		for (pn=0; pn <12; pn++) {
			if (Used[pn]) continue;

			ppl = &((*fl)[pn][ci]);
			pinc = 0;
			for (ln=0; ln < ppl->n; ln++) {
				p = ppl->pp[ln];
				if (FitInBlock(p,b)){
					pinc = 1; 
					if (++numfills > min) break; /* No need to search further at this hole */
				}; 
			};
			pcnt += pinc;
			if (numfills > min) break;
		};
		if (numfills < min) {
			min = numfills;
			*HoleIndex = ci;
			pmax = pcnt;
			if (min == 0) return min;
		} else {
			if ((numfills == min) && (pcnt > pmax)) {
				min = numfills;
				*HoleIndex = ci;
				pmax = pcnt;
			};
		};
	};
	return min;
};



/* Returns part placement iff part fits into given hole 
	Tries possible placements for this hole
	Returns 0 if no fit
*/
Part *FitsHole(Block *b, int pn, int hole, FillList *fl) 
{
	ppList *ppl = &((*fl)[pn][hole]);
	int ln;
	Part *p;
	
	for (ln=0; ln < ppl->n; ln++) {
		p = ppl->pp[ln];
		if (FitInBlock(p,b)) 
			return p;
	};
	return (Part *) 0;
};



void PrtDepth(int depth){
	int i;
	printf("%02d:",depth);
	for (i=0; i<depth; i++)
		printf("  ");
};


/* Returns the first unused part */
static inline 
int FindUnusedPart(){
	return LeastZero[UseMap];
};



/* Returns number of placements so that part fits into block 
	Does not check more than max placements per part
*/
int NumFits(Block *b, int pn, int max){
	PartList *pl = &(Lagen[pn]);
	int ln, cnt=0;
	Part *p;	
	
	for (ln=0; ln < pl->n; ln++) {
		p = &(pl->p[ln]);
		if (FitInBlock(p,b)){
			if (++cnt >= max) break;
		};
	};
	return cnt;
};


/* Examine all unused parts
   Returns part index which should be tried next 
   Sets min to the number of possible placements this part has
   min is set to 0 iff no part can be placed
  */
int ExamineParts(Block *b, int *min){
	int i,pn, h;
	
	*min = 99999;
	
	/* For every unused part */
	for (i=0; i<12;i++){
		if (Used[i]) continue;
		h = NumFits(b,i, *min);
		if (h < *min) {
			*min=h;
			pn=i;
		};
	}
	return pn;
};

/* Returns 0 iff parity is impossible to reach */
int CheckParity(Block *b, int depth){
	int pn, p1, p2;
	
	if  (abs(b->Parity) > MaxParity) {
#ifdef DEBUG			
			ParityFails[depth]++;
#endif	
		return 0;
	};
	
	if ( (depth == 10) ){
	/* Find the first part which is still unused */
		pn = FindUnusedPart();
		p1 = abs(Lagen[pn].p[0].Parity);
		
	/* Find the second part which is still unused */
		for (pn++; pn <12; pn++) 
			if (!Used[pn]) break;		
		p2 = abs(Lagen[pn].p[0].Parity);
	
		if ( (p1+p2 != abs(b->Parity)) && ( abs(p1 - p2) != abs(b->Parity)) ){
#ifdef DEBUG			
			ParityFails[depth]++;
#endif			
			return 0;
		};
	};
	return 1;
};


int TryLastPart(Block *b, FillList *fl){
	int i, pn;
	Part *p;
	
	if (abs(b->Parity) != MaxParity) {
#ifdef DEBUG			
			ParityFails[11]++;
#endif			
		return 0;
	};
	/* Find first empty cube */
	i = FindFirstHole(&(b->m));
	
	/* Find the part which is still unused */
	pn = FindUnusedPart(); 

	if ( (p = FitsHole(b, pn, i, fl)) ) {
		Store (p, b);
		FoundSolution(b);
		Remove (p,b);
		return 1;
	};
	return 0;
};

#define hashstore {if ( (depth<=HashDepth) && (sol == 0)) StoreHash(&(b->m), depth);}

/* This array tells Try() which FillList it shall use at which depth */
FillList  *TryFills[12];

/* Forward declaration */
int Try (Block *b, int depth);


/* Try all placements of the given part 
	and return number of solutions
*/
int TryPart(Block *b, int pn, int depth){
	int ln,sol=0;
	PartList *pl;
	Part *p;
	
#ifdef DEBUG
	PartTries[depth]++;
#endif
	
	Use(pn);
	pl = &(Lagen[pn]);
	for (ln=0; ln < pl->n; ln++) {
		p = &(pl->p[ln]);

		if (!FitInBlock(p,b)) continue;
			
		/* Store Part in block */

		Store (p, b);

		/* Recursively try the rest of the block */
		sol += Try(b, depth+1);

#ifdef EARLYABORT	
		if (depth == mindepth) CleanExit("",0);
#endif

		/* Remove part from block */
		Remove(p,b);
	}	

	UnUse(pn);
	hashstore;
	return sol;
};

/* This routine handles the last stages of the search
	It tries to find a solution very fast by always filling the first empty hole
*/
int TryFirstHole (Block *b, FillList *fl, int depth){
	int sol=0, pn, ln, h;
	Part *p;
	ppList *ppl;

#ifdef DEBUG
	Nodes[depth]++;
#endif
	if (depth == 11) 
		return TryLastPart(b, fl);

	/* Check if the parity is still valid */
	if (CheckParity(b, depth) == 0) 
		return 0;
	
	h = FindFirstHole(&(b->m));
	
		for (pn=0; pn <12; pn++) {
			if (Used[pn]) continue;
		
			Use(pn);
			ppl = &( (*fl)[pn][h]);
			for (ln=0; ln < ppl->n; ln++) {
				p = ppl->pp[ln];
				if (FitInBlock(p,b)) { 
					/* Store Part in block */
					Store (p, b);
					
					sol += TryFirstHole(b, fl, depth+1);

					/* Remove part from block */
					Remove(p,b);

				}
			};
			UnUse(pn);
		};
	return sol;
};


/* Given a hole h, tries all possibilities to fill it 
	Returns number of solutions
*/

int TryHole(Block *b, int h, int depth){
	int sol=0, pn, ln;
	Part *p;
	ppList *ppl;
		
#ifdef DEBUG
		MinHoleTries[depth]++;
#endif

	for (pn=0; pn <12; pn++) {
		if (Used[pn]) continue;
	
		Use(pn);
		ppl = &( (*(TryFills[depth]))[pn][h]);

		for (ln=0; ln < ppl->n; ln++) {
			p = ppl->pp[ln];
			if (FitInBlock(p,b)) { 
				/* Store Part in block */
				Store (p, b);
				
				sol += Try(b, depth+1);

				/* Remove part from block */
				Remove(p,b);
			}
		};
		UnUse(pn);
	};
	hashstore;
	return sol;
}


/* Recursively try to solve the puzzle
	Returns number of solutions found 
*/
int Try (Block *b, int depth){
	int pn, ln;
	int  min, h,  minh, sol;

	Part *p;
	ppList *ppl;

#ifdef DEBUG
	Nodes[depth]++;
#endif

	/* Check if we have already searched this situation and found no solution */
	if ( (depth <= HashDepth) && Recall(&(b->m)) )  {
#ifdef DEBUG
		Repetitions[depth]++;
#endif		
		return 0;
	};

	sol = 0;
	
	/* Is this the last part to fit ? */
	if (depth == 11) 
		return TryLastPart(b, TryFills[10]);

	/* Check if the parity is still valid */
	if (CheckParity(b, depth) == 0) 
		return 0;

	/* Update the list of possible moves per hole at shallow depths */
	if (depth <= UPDATEFILLDEPTH) {
		UpdateFill(&(b->m), &(FillsArray[depth]), &(FillsArray[depth+1]) );
		TryFills[depth] = &FillsArray[depth+1];
	} else {
		TryFills[depth] = TryFills[depth-1];
	};

	if ((depth > 9) && ZeroHole(b,TryFills[depth])){
#ifdef DEBUG
		ZeroHoles[depth]++;
#endif
		return 0;
	};
	
	/* Look for a hole which is unfillable or can only be filled one way */
	if (  SpecialHole(b, TryFills[depth], &pn, &p) ) {
#ifdef DEBUG
		SpecialHoleTries[depth]++;
#endif
		/* Is there an unfillable hole ? */
		if (pn == -1) return 0; 
		/* Now we have a hole which can be filled only one way */
		/* Store Part in block */
		Use(pn);
		Store (p, b);

		sol = Try(b, depth+1);
	
		/* Remove part from block */
		Remove(p,b);
		UnUse(pn);
		hashstore;
		return sol;
	};	
	
	/* Select a part which will possibly be tried */
	min = 99999;
	/* At shallow depths find the part with minimum possible placements */
	if (depth < 6) {
		pn=ExamineParts(b, &min);
		/* Do we have a part which can not be placed anywhere ? */
		if (min == 0) 
			return 0;
	} else {
		/* At higher depths simply take the first unused part */
		pn=FindUnusedPart();
	};
	
	/* Find the hole with the minumum number of placements possible
		Make sure we are below the minimum number possible for a part
	*/
	if ( (depth < 7) && (min > 4)  && ((minh=MinHole(b, TryFills[depth], &h, min-3)) < (min - 3) ) ) {
		return TryHole(b, h, depth);

	};
	if (depth > 6) return TryFirstHole(b, TryFills[depth], depth);
	
	return TryPart(b,pn,depth);
};


#ifdef DEBUG
void InitStatistics() {
	int i;
	for (i=0; i<12; i++){
		Nodes[i] = 0;
		MinHoleTries[i] = 0;
		SpecialHoleTries[i] = 0;
		PartTries[i] = 0;
		ParityFails[i] = 0;
		Repetitions[i] = 0;	
		ZeroHoles[i] = 0;
	};	
};

/*<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/
void Statistics(){
	int i, sum=0;	
	printf("Solutions: %d\n", solutions);
	printf("\nDepth\t\tNodes\t\tSpecial\t\tMinHole\t\tPart\t\tRepetitions\tParity\tZeroHole");
	for (i=0; i<12; i++){
		printf("\n%3d: \t",i);
		printf("%10d\t", Nodes[i]);
		printf("%10d\t", SpecialHoleTries[i]);
		printf("%10d\t", MinHoleTries[i]);
		printf("%10d\t", PartTries[i]);
		printf("%10d\t",Repetitions[i]);	
		printf("%10d\t",ParityFails[i]);
		printf("%10d\n",ZeroHoles[i]);
		sum += Nodes[i];
	};
	printf("\nTotal Nodes: %d\n",sum);
};
#endif

/* -------------------------------------------------- Main Program --------------------------------------- */
	
int main(int argc, char *argv[])
{
  /* Variable needed for option processing */
  int option;

  int i;
  
    
/*+++++++++++++++++++ Process Options ++++++++++++++++++++++++++++++++++*/


  while ( (option = getopt(argc,argv,"v:s")) != EOF)
  {
    switch (option)
    {
      
		case 'v':
        	Verb = atoi(optarg);
        	break;

		case 's':
			PrtSol = 1;
			break;
				
     case '?':
        Usage(argv[0]);
        CleanExit("", 5);
    }
  }



/*====================================================================*/
/*              User Initialization Routines go here!                 */
/*====================================================================*/
	InitLeastZero();
	InitHash();
	InitParity();
	MaxParity = 14;  /* Hand computed */

	InitTeile(Teil);

	InitLagen(Lagen, Teil);

	InitAbsParity();
	 
#ifdef DEBUG 
	for (i=0; i<12; i++){
		printf("%d, ",Lagen[i].n);
		printf("%d, ",AbsParity[i]);
	};
	NewLine();
#endif

	InitFills (Lagen, &(FillsArray[0]));

	InitBlock (&b);

	for (i=0; i<12; i++)
		Used[i]=0;
	UseMap = 0;

	solutions = 0;

#ifdef DEBUG
	InitStatistics();
#endif

/*====================================================================*/
/*              User Initialization is done!                          */
/*====================================================================*/


/*||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||*/
/*              M A I N    L O O P                                    */
/*||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||*/

	Try(&b,0);
	printf("\nSolutions: %d\n", solutions);


/*====================================================================*/
/*              User CleanUp Procedures are Here !                    */
/*====================================================================*/
  
 	CleanExit("",0);
 	return(0);  /* Never executed, just to make compiler happy */
}


/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
/* Deallocate all used resources                                     */
/* Terminate the program returning RetCode                           */
/*<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/

void 
CleanExit(char *pCh, int RetCode)
{
#ifdef DEBUG
	Statistics();
#endif
	if (HashTable) free(HashTable);
	fprintf(stderr, "%s\n", pCh);
	exit(RetCode);
}


static void 
Usage(char *ProgName)
{
  fprintf(stderr,"Usage: %s\n",ProgName);
  fprintf(stderr,"Options: -v<i> :Set verbosity to <i>\n");
  fprintf(stderr,"         -s    :Show solutions\n"); 
}


  /* THE END */


