// Copyright (C) 2003 by Michael Pichler.
// Partially based on code Copyright (C) 2003 by Harald Bgeholz.
// See copympi.txt for further information.
//
// created: mpichler, 20030411
// changed: mpichler, 20030414


#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>

#include "point.h"
#include "part.h"
#ifndef LOGO
#  include "tuple.h"
#endif

/*
 * get current time as timestamp in milliseconds (32 bit precision)
 */
static unsigned int gettimemillis ()
{
  static timeval tstamp;  // tv_sec, tv_usec (micro seconds)
  gettimeofday (&tstamp, NULL);
  //#fprintf (stderr, "tstamp: %ld.%06ld\n", tstamp.tv_sec, tstamp.tv_usec);
  // NOTE: value overflow ignored, just used for timing and difference is OK
  return ((unsigned int) tstamp.tv_sec) * 1000 +
    (unsigned int) (tstamp.tv_usec / 1000);
}


/*
 * parts data (REMIND: should be encapsulated by a class)
 */
static const int numparts = 12;
static Part* part = NULL;  // part array
static const Part* permut[numparts];  // permutation array
#ifdef LOGO
static int numsolutions = 0;
#endif
static unsigned int tstart = 0;  // timing helper
#ifdef LOGO
static int partpos[numparts];  // position of part i
#endif


/*
 * recursive solve method.
 * build permutation (into permut[]) starting at index level;
 * try each of the remaining parts at index level, in each position
 * that fills the next unset bit in bvfield but no other of its bits.
 * Param nextbit tells where to start search for smallest unset bit
 * (each level sets at least the next unset bit) in bvfield.
 * Param nbval is always identical to (BIT_ONE << nextbit).
 */
int Part::solve (Bitvector bvfield, int level, int nextbit, BIT_TYPE nbval)
{
#ifndef LOGO
  int numsolutions = 0;
#endif

  // smallest unset bit in bvfield
  while (bvfield.testValue (nbval))
  {
    ++nextbit;
    nbval <<= 1;
  }

  // try each part of permut[level] to permut[numparts-1]
  const Part* permlevel = permut[level];
  for (int j = level;  j < numparts;  ++j)
  {
    // use part p = permut[j] at this level, swap with permut[level]
    const Part* p = permut[level] = permut[j];
    permut[j] = permlevel;

    // try all suitable (i. e. smallest bit nextbit set) part positions
    // w/o preprocessing:  for (int pos = 0;  pos < pp.numpos_;  ++pos)
    // and pp.bvpos_[pos].test (nextbit) had to be checked inside the loop
    int lastpos = p->start_[nextbit+1];
    for (int pos = p->start_[nextbit];  pos < lastpos;  ++pos)
    {
      if (bvfield.testN (p->bvpos_[pos]))
      {
#ifdef LOGO
        partpos[p-part] = pos;
        if (level < numparts-1)
          solve (Bitvector (bvfield, p->bvpos_[pos]), level+1, nextbit+1, nbval<<1);
        else  // found solution
          printSolution (++numsolutions);
#else
        if (level < numparts-1)
          numsolutions +=
          solve (Bitvector (bvfield, p->bvpos_[pos]), level+1, nextbit+1, nbval<<1);
        else  // found solution
          ++numsolutions;
#endif
      }
    }
    permut[j] = p;  // restore array
    // permut[level] = permlevel;  // overwritten inside loop, restored at end
  }
  permut[level] = permlevel;
  return numsolutions;
}


#ifdef LOGO
/*
 * print a solution.
 */
void Part::printSolution (int solnum)
{
  // no optimizations done herein.
  printf ("solution %d:\n", solnum);
  for (int z = ZLEN-1;  z >= 0;  --z)  // top to bottom z layer
  {
    if (ZLEN > 1)
      printf ("  z==%d\n", z);
    for (int y = YLEN-1;  y >=0;  --y)  // top to bottom y row
    {
      printf ("    ");
      for (int x = 0;  x < XLEN;  ++x)  // left to right
      {
        int bit = pos2bit (x, y, z);  // assert: not shuffled
        char partname = '?';  // check which part has set this bit
        for (int i = 0;  i < numparts;  ++i)
        {
          if (part[i].bvpos_[partpos[i]].test (bit))
          {
            partname = part[i].getName ();
            break;
          }
        }
        putchar (partname);
      }
      printf ("\n");
    }
  }
}
#endif


/*
 * main program.
 */
int main (int argc, char** argv)
{
  printf ("ctbox (C) 2003 by Michael Pichler, partially (C) 2003 Harald Bgeholz\n");

  // init
  unsigned int t1 = gettimemillis ();
  // check compiler settings for Bitvector data
  if (NUM_BITS != (8 * sizeof (BIT_TYPE)) || NUM_BITS < (XLEN * YLEN * ZLEN))
  {
    fprintf (stderr, "error: Bitvector length (8*%d) wrong (%d) or too small for this puzzle.\n",
      sizeof (BIT_TYPE), NUM_BITS);
    return 1;
  }
  // points not needed after Part creation
  // coordinate system: plane (x right, y back), z up; as shown in location of pXY vars
  // z == 0
#ifdef PENTOMINO
  Point3i p03 (0,3,0);  Point3i p04 (0,4,0);
#else
  Point3i p03 (0,3,0);  Point3i p13 (1,3,0);
#endif
  Point3i p02 (0,2,0);  Point3i p12 (1,2,0);  Point3i p22 (2,2,0);
  Point3i p01 (0,1,0);  Point3i p11 (1,1,0);  Point3i p21 (2,1,0);
  Point3i p00 (0,0,0);  Point3i p10 (1,0,0);  Point3i p20 (2,0,0);
#ifndef PENTOMINO
  // z == 1
  Point3i p001(0,0,1);  Point3i p101(1,0,1);  Point3i p011(0,1,1);
#endif

#ifdef PENTOMINO
  // "Pentominoes" puzzle.
  // See "Algorithms from P to NP", vol 1, p. 34;
  // by B.M. Moret et Henry D. Shapiro.
  Point3i points0[] = { p00, p01, p02, p03, p04 };
  Point3i points1[] = { p00, p10, p01, p02, p03 };
  Point3i points2[] = { p00, p01, p11, p02, p03 };
  Point3i points3[] = { p10, p11, p01, p02, p03 };
  Point3i points4[] = { p10, p20, p11, p12, p02 };
  Point3i points5[] = { p02, p12, p11, p21, p20 };
  Point3i points6[] = { p00, p01, p11, p02, p12 };
  Point3i points7[] = { p10, p01, p11, p12, p22 };
  Point3i points8[] = { p10, p11, p02, p12, p22 };
  Point3i points9[] = { p00, p10, p11, p12, p02 };
  Point3i points10[]= { p00, p10, p20, p01, p02 };
  Point3i points11[]= { p10, p01, p11, p21, p12 };
#else
  // the 12 parts (ordered as shown on the image)
  Point3i points0[] = { p00, p10, p01, p02, p12 };  // "c"
  Point3i points1[] = { p00, p10, p11, p12, p22 };
  Point3i points2[] = { p00, p10, p11, p12, p13 };
  Point3i points3[] = { p00, p01, p02, p03, p12 };
  // Point3i points4[] = { p00, p10, p01, p11, p001 };
  Point3i points4[] = { p00, p10, p01, p001, p011 };  // "'"
  Point3i points5[] = { p10, p11, p02, p12, p22 };
  Point3i points6[] = { p00, p10, p01, p11, p12 };
  Point3i points7[] = { p10, p20, p01, p11, p02 };
  Point3i points8[] = { p00, p01, p001, p101 };
  Point3i points9[] = { p10, p01, p11, p21, p02 };
  Point3i points10[]= { p10, p01, p11, p21, p12 };
  Point3i points11[]= { p00, p10, p01, p02, p12, p03 };  // "t"
#endif
  // use one part as symmetry "anchor"
  Part::Orientations or[numparts] =
  {
    Part::all, Part::all, Part::all, Part::all, Part::all, Part::all,
    Part::all, Part::all, Part::all, Part::all, Part::all, Part::all
  };

  // optional: choose one for partial solutions (faster testing)
  bool stop = false;

#ifdef LOGO
# ifdef PENTOMINO
  or[1] = Part::z2;  // on (asymmetric) part 1 orientations xz and xy only
# else
  or[0] = or[4] = or[11] = Part::none;  // manual placement of "c't" logo parts
# endif
#else
  if (argc == 2)  // commandline option (ignored for logo puzzle)
  {
    char* opt = argv[1];
         if (!strcmp (opt, "-stop"))  stop = true;
    else if (!strcmp (opt, "-all"))   or[11] = Part::all;  // 224
    else if (!strcmp (opt, "-zx"))  or[11] = Part::zx;  // 12
    else if (!strcmp (opt, "-zy"))  or[11] = Part::zy;  // 18
    else if (!strcmp (opt, "-x"))   or[11] = Part::xz;  // 10
    else if (!strcmp (opt, "-y"))   or[11] = Part::yz;  // 16
    // symmetry lock on another part
    // parts 0, 1, 5, 7, 8, 10 are right out because of their symmetry
    else if (!strcmp (opt, "-2"))   or[2] = Part::sym;
    else if (!strcmp (opt, "-3"))   or[3] = Part::sym;
    else if (!strcmp (opt, "-4"))   or[4] = Part::sym;
    else if (!strcmp (opt, "-6"))   or[6] = Part::sym;  // default
    else if (!strcmp (opt, "-9"))   or[9] = Part::sym;
    else if (!strcmp (opt, "-11"))  or[11]= Part::sym;
    else
    {
      fprintf (stderr,
        "usage: ctbox [Option]\n"
        "Possible values for Option:\n"
        "  -stop ... stop after preprocessing\n"
        "  -all ... disregard symmetries\n"
        "  -zx, -zy, -x, -y ... partial solution (test only)\n"
        "  -2, -3, -4, -6, -9, -11 ... symmetry anchor part (def.: 11)\n"
      );
      return 0;
    }
  }
  else  // default: symmetry lock on part 6
  {
    or[6] = Part::sym;
  }
#endif

#define PARTPOINTS(i)  i, points##i, (sizeof(points##i)/sizeof(Point3i)), or[i]
  Part partarray[] = {
    Part (PARTPOINTS (0)),
    Part (PARTPOINTS (1)),
    Part (PARTPOINTS (2)),
    Part (PARTPOINTS (3)),
    Part (PARTPOINTS (4)),
    Part (PARTPOINTS (5)),
    Part (PARTPOINTS (6)),
    Part (PARTPOINTS (7)),
    Part (PARTPOINTS (8)),
    Part (PARTPOINTS (9)),
    Part (PARTPOINTS(10)),
    Part (PARTPOINTS(11))
  };
  part = partarray;
#undef PARTPOINTS

  int totalpos = 0;
  for (int i = 0;  i < numparts;  i++)
    totalpos += part[i].getNumberOfPositions ();
  printf ("positioning of %d parts completed: %d positions.\n", numparts, totalpos);

#ifdef LOGO
# ifdef CTLOGO
  // place the logo parts where they belong, apostrophe on part 4 (cheat ;-)
  part[0].setName ('c');
  const Bitvector* place0 = part[0].place (points0, 0, 0, 2);
  part[4].setName ('\'');
  const Bitvector* place4 = part[4].place (points4, 2, 2, 1);
  part[11].setName ('t');
  const Bitvector* place11 = part[11].place (points11, 3, 0, 2);
# endif
#endif

//#  Part::shuffleBits (part, numparts, stop);

  unsigned int t2 = tstart = gettimemillis ();
  if (stop)
  {
    printf ("Prepocessing time: %.3f s. Stop.\n", (t2-t1)*.001);
    return 0;
  }

  // first permutation
  for (int i = 0;  i < numparts;  ++i)
    permut[i] = part+i;

#ifdef LOGO
  Bitvector bvbox;
# ifdef CTLOGO
  // place those 3 parts in permut
  permut[0] = part+0;
  permut[1] = part+4;  permut[4] = part+1;
  permut[2] = part+11;  permut[11] = part+2;
  // use their first (and only) position
  partpos[0] = partpos[4] = partpos[11] = 0;
  // set the placement bits in bvbox
  if (!place0 || !place4 || !place11)
  {
    fprintf (stderr, "internal error: could not place logo parts.\n");
    return 1;
  }
  bvbox.set (*place0);
  bvbox.set (*place4);
  bvbox.set (*place11);
  // and start recursion on level 3 (global var numsolutions)
  Part::solve (bvbox, 3, 0, BIT_ONE);
# else
  Part::solve (bvbox, 0, 0, BIT_ONE);
# endif
#else
  // solve, starting with tuples of 4 elements
  int numsolutions = 0;
  const Part* temp;
//    int countdown = numparts * (numparts-1) / 2;
//    fprintf (stderr, "solution countdown: %d", countdown);
  for (int i = 0;  i < numparts;  ++i)
  {
    permut[0] = part+i;
    permut[i] = part+0;
    for (int j = i+1;  j < numparts;  ++j)
    {
      temp = permut[1];
      permut[1] = part+j;
      permut[j] = temp;
      for (int k = j+1;  k < numparts;  ++k)
      {
        temp = permut[2];
        permut[2] = part+k;
        permut[k] = temp;
        for (int l = k+1;  l < numparts;  ++l)
        {
          temp = permut[3];
          permut[3] = part+l;
          permut[l] = temp;
          // 0 <= i < j < k < l < numparts
          // considers all permutations of (i, j, k, l)
          // calls Part::solve starting at level 4
          numsolutions += Tuple::solveTuple (permut[0], permut[1], permut[2], permut[3]);
          // progress ticker
//            fprintf (stderr, "#[%d,%d,%d,%d](%d/%d)%c", i, j, k, l,
//              numsolutions, (gettimemillis () - tstart),
//              /*(j==numparts-1 && i<8) ? '\n' :*/ ' ');
          temp = permut[3];  // restore permut
          permut[3] = permut[l];
          permut[l] = temp;
        }
        temp = permut[2];  // restore permut
        permut[2] = permut[k];
        permut[k] = temp;
      }
//        fprintf (stderr, " %d", --countdown);
      temp = permut[1];  // restore permut
      permut[1] = permut[j];
      permut[j] = temp;
    }
    permut[0] = 0;  // restore permut
    permut[i] = part+i;
  }
//    fprintf (stderr, " finished!\n");
#endif

  // done
  unsigned int t3 = gettimemillis ();
  printf ("Time init: %.3f, solve: %.3f, total: %.3f s.\n",
    (t2-t1)*.001, (t3-t2)*.001, (t3-t1)*.001);

  printf ("number of solutions: %d\n", numsolutions);

  return 0;
}
