// Copyright (C) 2003 by Michael Pichler.
// Partially based on code Copyright (C) 2003 by Harald Bgeholz.
// See copympi.txt for further information.
//
// created: mpichler, 20030412
// changed: mpichler, 20030425


#include "point.h"
#include "part.h"
//#include "bits.h"

#include <stdio.h>

/*
 * mapping of a position (3D point inside puzzle box) to a bit
 */
#ifdef POS2BIT
// if you find a better bit ordering, you may code it manually here
int Part::pos2bit_[XLEN][YLEN][ZLEN] =
{
  // 12x + 3y + z
  { {  0,  1,  2 }, {  3,  4,  5 }, {  6,  7,  8 }, {  9, 10, 11 } }, // x = 0, y = 0..3
  { { 12, 13, 14 }, { 15, 16, 17 }, { 18, 19, 20 }, { 21, 22, 23 } }, // x = 1, y = 0..3
  { { 24, 25, 26 }, { 27, 28, 29 }, { 30, 31, 32 }, { 33, 34, 35 } }, // x = 2, y = 0..3
  { { 36, 37, 38 }, { 39, 40, 41 }, { 42, 43, 44 }, { 45, 46, 47 } }, // x = 3, y = 0..3
  { { 48, 49, 50 }, { 51, 52, 53 }, { 54, 55, 56 }, { 57, 58, 59 } }  // x = 4, y = 0..3
};
#endif

/*
 * Constructor. Turn part into all given orientations, move and place it.
 */
Part::Part (int partnum, Point3i* point, int numpoints, Orientations orientations)
: point_ (point), numpoints_ (numpoints), numpos_ (0)
{
  name_ = (partnum < 10) ? ('0' + partnum) : ('A' + (partnum-10));
  for (int i = 0;  i < NUM_BITS;  ++i)
    start_[i] = 0;
  if (orientations != none)
    twist (orientations);
  fprintf (stderr, "%c: %3d pos.%c", name_, numpos_,
    ((partnum%6)==5) ? '\n' : ' ');
}


/*
 * Turn ("twist") part around into each "up" direction
 * Parameter dirs: orientations to be generated.
 */
void Part::twist (Orientations dirs)
{
  // unit vectors, pos. axis
  const Point3i ex (1, 0, 0);
  const Point3i ey (0, 1, 0);
  const Point3i ez (0, 0, 1);
  // unit vectors, neg. axis
  const Point3i e_x (-1,  0,  0);
  const Point3i e_y ( 0, -1,  0);
  const Point3i e_z ( 0,  0, -1);
  // temporary target array for transformed points
  Point3i* xp = new Point3i[numpoints_];

  // part facing upside in +/-z direction
  if (dirs & zx)  turn (xp, ez, ex);
  if (dirs & zy)  turn (xp, ez, ey);
  if (dirs & zz)
  {
    turn (xp, ez, e_x);  turn (xp, ez, e_y);
    turn (xp, e_z, ex);  turn (xp, e_z, ey);  turn (xp, e_z, e_x);  turn (xp, e_z, e_y);
  }

  // part facing upside in +/-x direction
  if (dirs & xy)  turn (xp, ex, ey);
  if (dirs & xz)  turn (xp, ex, ez);
  if (dirs & xx)
  {
    turn (xp, ex, e_y);  turn (xp, ex, e_z);
    turn (xp, e_x, ey);  turn (xp, e_x, ez);  turn (xp, e_x, e_y);  turn (xp, e_x, e_z);
  }

  // part facing upside in +/-y direction
  if (dirs & yx)  turn (xp, ey, ex);
  if (dirs & yz)  turn (xp, ey, ez);
  if (dirs & yy)
  {
    turn (xp, ey, e_x);  turn (xp, ey, e_z);
    turn (xp, e_y, ex);  turn (xp, e_y, ez);  turn (xp, e_y, e_x);  turn (xp, e_y, e_z);
  }

  delete xp;  // free temp array
}


/*
 * Turn part to the specified "up" and "right" vectors
 * i. e. up is the image of ez and right the image of ex
 * Parameter xfpt is (re)used as temporary array for transformed points
 * (overwritten by this method)
 */
void Part::turn (Point3i* xfpt, const Point3i& up, const Point3i& right)
{
  const Point3i& nx = right;
  const Point3i& nz = up;
  Point3i ny = Point3i::cross (up, right);
  // now (nx, ny, nz) form a righthanded system (no part mirroring)
  // transform all points into the target orientation, compute bbox
  int n = numpoints_;
  const Point3i* s = point_;
  Point3i* d = xfpt;
  Point3i min (100, 100, 100);
  Point3i max (-100, -100, -100);
  for (int i = 0;  i < n;  ++i)
  {
    s->xform (nx, ny, nz, d);
    d->updateBoundings (min, max);
    s++;
    d++;
  }
  // move transformed part around ...
  move (xfpt, min, max);
}


/*
 * Move transformed (rotated) part to all positions that lie inside the 
 * (specific) target puzzle, build bitvector, and record new positions
 */
void Part::move (const Point3i* xfpt, const Point3i& min, const Point3i& max)
{
  // move along x (for dx) such that 0 <= min.x+dx <= x+dx <= max.x+dx < XLEN
  // thus: -min.x <= dx < (XLEN-max.x); the same way move along y and z;
  // possible placements for this part orientation exist only for min_x < max_x etc.

  int min_x = - min.x ();  int max_x = XLEN - max.x ();
  int min_y = - min.y ();  int max_y = YLEN - max.y ();
  int min_z = - min.z ();  int max_z = ZLEN - max.z ();

  for (int dx = min_x;  dx < max_x;  ++dx)
    for (int dy = min_y;  dy < max_y;  ++dy)
      for (int dz = min_z;  dz < max_z;  ++dz)
      {
        place (xfpt, dx, dy, dz);
      }
}


/*
 * Place a part: build bitvector and record the new position.
 * called for all or selected part orientations and translations.
 * Returns the Bitvector* of the moved position if it was a new one, 0 otherwise.
 */
const Bitvector* Part::place (const Point3i* xfpt, int dx, int dy, int dz)
{
  // build the position bitvector of all translated points
  // assert: fit into XLEN, YLEN, ZLEN box

  int n = numpoints_;
  Point3i tpt;  // translated point
  Bitvector newpos;  // bits set by translated part
  for (int i = 0;  i < n;  ++i)
  {
    xfpt[i].translate (dx, dy, dz, tpt);
    const int x = tpt.x (), y = tpt.y (), z = tpt.z ();
    // verify assertion: tpt must fit into puzzle box
    if ( x < 0 || x >= XLEN
      || y < 0 || y >= YLEN
      || z < 0 || z >= ZLEN)
    {
      fprintf (stderr, "Error: attempt to place part outside box\n");
      return 0;
    }
    // newpos.set (tpt.toBit ());
    newpos.set (pos2bit (x, y, z));
  }

  // NOTE: start_[i] also helps faster finding duplicates, but is otherwise
  // unused (and has to be rebuilt) when shuffleBits is called later on.

  // INVARIANT: keep pos_ elements sorted by smallest bit set and let start_[i]
  // point to first index in pos_ where bit i is the smallest bit set.
  unsigned int sb = 0;  // smallest bit in newpos
  while (!newpos.test (sb))
    ++sb;
  if (sb >= NUM_BITS-1)  // assert: sb < NUM_BITS-1
  {
    fprintf (stderr, "Error on searching smallest bit set: %d\n", sb);
    return 0;
  }

  int t = start_[sb+1];  // insert newpos at index t (if it is new)

  // check whether newpos is really new
  for (int i = start_[sb];  i < t;  ++i)
  {
    if (newpos == bvpos_[i])  // seen before
      return 0;
  }

  if (numpos_ >= MAX_PARTPOS)
  {
    fprintf (stderr, "internal error: exceeded MAX_PARTPOS limit %d\n", MAX_PARTPOS);
    exit (1);
  }

  // move elements t to numpos_-1 one behind
  for (int i = numpos_-1;  i >= t;  --i)
    bvpos_[i+1] = bvpos_[i];

  // store newpos at index t
  bvpos_[t] = newpos;

  while (++sb < NUM_BITS)  // update all start indices > sb
    ++(start_[sb]);

  ++numpos_;
  return bvpos_+t;
}
