/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	PuzzleClasses.cpp
*/
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"

#include "PuzzleClasses.h"

#include <algorithm>

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

// namespace to wrap puzzle-related classes
namespace Puzzle{

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	Block: implementation
*/
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	constructors
*/
Block::Block()
{
	// nothing, data is left uninitialized
}

Block::Block(int x,int y,int z)
{
	m_Data[0] = x; m_Data[1] = y; m_Data[2] = z;
}

Block::Block(const Block& B)
{
	// same as assignment
	operator=(B);
}

/////////////////////////////////////////////////////////////////////////////
/*
	assignment
*/
Block& Block::operator=(const Block& B)
{
	for(int i=0;i<DIMENSIONS;++i)
	{
		m_Data[i] = B.m_Data[i];
	}
	return *this;
}

/////////////////////////////////////////////////////////////////////////////
/*
	comparison
*/
bool Block::operator==(const Block& B)const
{
	for(int i=0;i<DIMENSIONS;++i)
	{
		if(m_Data[i]!= B.m_Data[i])return false;
	}
	return true;
}

/////////////////////////////////////////////////////////////////////////////
/*
	access to data
	non-const and const version
*/
int& Block::operator[](int i)
{
	ASSERT(i>=0 && i<DIMENSIONS);
	return m_Data[i];
}

const int& Block::operator[](int i) const
{
	ASSERT(i>=0 && i<DIMENSIONS);
	return m_Data[i];
}

/////////////////////////////////////////////////////////////////////////////
/*
	Rotate
	function to rotate block

	up to 24 different directions in 3-dimensional space possible
*/
Block Rotate(const Block& B,int Rot)
{
	/////////////////////////////////////////////////////////////////////////////

	ASSERT(Rot>=0 && Rot<24);

	/////////////////////////////////////////////////////////////////////////////

	// any rotation in space can be described as a sequence of swapping + sign change
	// the scheme used here minimizes code
	int SwapPattern = (Rot/2)%3;		// what swap pattern to use
	int NegatePattern = (Rot/6)%4;		// negate pattern to use

	bool Revert = ((Rot%2)!=0);			// wether to revert complete pattern or not
	
	/////////////////////////////////////////////////////////////////////////////

	Block R;	// return value

	switch(SwapPattern)
	{
	case 0:		// 012 => 012
		{
			R[0] = B[0]; 
			R[1] = B[1]; 
			R[2] = B[2];
		}
		break;

	case 1:		// 012 => 201
		{
			R[0] = B[2];
			R[1] = B[0]; 
			R[2] = B[1]; 
		}
		break;

	case 2:		// 012 => 120
		{
			R[0] = B[1]; 
			R[1] = B[2]; 
			R[2] = B[0];
		}
		break;
	}

	/////////////////////////////////////////////////////////////////////////////

	switch(NegatePattern)
	{
//	case 0:		// +++ => +++, nothing to do

	case 1:		// +++ => +--
		{
			R[1] = -R[1]; 
			R[2] = -R[2];
		}
		break;

	case 2:		// +++ => -+-
		{
			R[0] = -R[0]; 
			R[2] = -R[2];
		}
		break;

	case 3:		// +++ => --+
		{
			R[0] = -R[0]; 
			R[1] = -R[1];
		}
		break;
	}

	/////////////////////////////////////////////////////////////////////////////

	if(Revert == true)
	{
		// Revert: 012 => 210, +++ => ---
		std::swap(R[0],R[2]);

		R[0] = -R[0]; 
		R[1] = -R[1]; 
		R[2] = -R[2]; 
	}

	/////////////////////////////////////////////////////////////////////////////

	return R;

	/////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	DrawSide
	primitive drawing routine for one single cube side
*/
void DrawSide(CDC* pDC,int XOffset,int YOffset,int Width,int Side,COLORREF C)
{
	/////////////////////////////////////////////////////////////////////////////

	ASSERT(Side>=0 && Side < 3);

	/////////////////////////////////////////////////////////////////////////////

	// init points to draw polygon
	POINT P[4];

	P[0].x = XOffset;	// anchor, always the same
	P[0].y = YOffset;

	switch(Side)
	{
	case 0:		// front (back)
		{
			P[1].x = P[0].x + Width;
			P[1].y = P[0].y;

			P[2].x = P[1].x;
			P[2].y = P[1].y + Width;

			P[3].x = P[2].x - Width;
			P[3].y = P[2].y;
		}
		break;

	case 1:		// top (bottom)
		{
			P[1].x = P[0].x + Width;
			P[1].y = P[0].y;

			P[2].x = P[1].x + Width/2;
			P[2].y = P[1].y - Width/2;

			P[3].x = P[2].x - Width;
			P[3].y = P[2].y;
		}
		break;

	case 2:		// left (right)
		{
			P[1].x = P[0].x + Width/2;
			P[1].y = P[0].y - Width/2;

			P[2].x = P[1].x;
			P[2].y = P[1].y + Width;

			P[3].x = P[2].x - Width/2;
			P[3].y = P[2].y + Width/2;
		}
		break;
	}

	/////////////////////////////////////////////////////////////////////////////

	CPen Pen(PS_SOLID,0,RGB(64,64,64));
	CBrush Brush(C);

	CPen* OldPen     = pDC->SelectObject(&Pen);
	CBrush* OldBrush = pDC->SelectObject(&Brush);

	pDC->Polygon(P,4);

	pDC->SelectObject(OldBrush);
	pDC->SelectObject(OldPen);

	/////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	Stone, implementation
*/
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	constructors
*/
Stone::Stone()
{
	// empty
}

Stone::Stone(const Stone& B)
	:BaseClass(B)
{
	;
}

Stone::Stone(int (*Init)[3])
{
	// init from field

	// count field
	for(int i=0;/*empty*/;++i)
	{
		if(Init[i][0] <0) break;		// termination marker: negative number
	}

	resize(i);
	for(int j=0;j<i;++j)
	{
		operator[](j) = Block(Init[j][0],Init[j][1],Init[j][2]);
	}
}

/////////////////////////////////////////////////////////////////////////////
/*
	Rotate
*/
void Stone::Rotate(int Rot)
{
	for(iterator It=begin(); It!=end(); ++It)
	{
		*It = Puzzle::Rotate(*It,Rot);
	}
}

/////////////////////////////////////////////////////////////////////////////
/*
	Align
*/
void Stone::Align()
{
	Block B = GetMin();
	for(iterator It=begin(); It!=end(); ++It)
	{
		for(int i=0;i<B.DIMENSIONS;++i)
		{
			(*It)[i] -= B[i];
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
/*
	Shift
*/
void Stone::Shift(Block B)
{
	for(iterator It=begin(); It!=end(); ++It)
	{
		for(int i=0;i<B.DIMENSIONS;++i)
		{
			(*It)[i] += B[i];
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
/*
	GetMin / GetMax
*/
Block Stone::GetMin()const
{
	Block R(INT_MAX,INT_MAX,INT_MAX);
	for(const_iterator It=begin(); It!=end(); ++It)
	{
		for(int i=0;i<R.DIMENSIONS;++i)
		{
			R[i] = min(R[i],(*It)[i]);
		}
	}
	return R;
}

Block Stone::GetMax()const
{
	Block R(INT_MIN,INT_MIN,INT_MIN);
	for(const_iterator It=begin(); It!=end(); ++It)
	{
		for(int i=0;i<R.DIMENSIONS;++i)
		{
			R[i] = max(R[i],(*It)[i]);
		}
	}
	return R;
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

// basic shapes used for puzzle
int BasicShape[][7][3] = 
{
	{ 
		{0,0,0},	{0,1,0},	{1,0,0},	{2,0,0},	{2,1,0},				{-1,-1,-1},
	},
	{ 
		{0,0,0},	{0,1,0},	{1,1,0},	{2,1,0},	{2,2,0},				{-1,-1,-1},
	},																			
	{																			
		{0,0,0},	{0,1,0},	{0,2,0},	{0,3,0},	{1,3,0},				{-1,-1,-1},
	},																			
	{																			
		{0,0,0},	{0,1,0},	{0,2,0},	{0,3,0},	{1,1,0},				{-1,-1,-1},
	},
	{ 
		{0,0,0},	{0,1,0},	{1,0,0},	{1,1,0},	{1,1,1},				{-1,-1,-1},
	},
	{ 
		{0,0,0},	{1,0,0},	{2,0,0},	{1,1,0},	{1,2,0},				{-1,-1,-1},
	},
	{ 
		{0,0,0},	{1,0,0},	{2,0,0},	{1,1,0},	{2,1,0},				{-1,-1,-1},
	},
	{ 
		{0,0,0},	{1,0,0},	{1,1,0},	{2,1,0},	{2,2,0},				{-1,-1,-1},
	},
	{ 
		{0,0,0},	{0,1,0},	{1,1,0},	{1,1,1},							{-1,-1,-1},
	},
	{ 
		{0,0,0},	{0,1,0},	{1,1,0},	{2,1,0},	{1,2,0},				{-1,-1,-1},
	},
	{ 
		{0,1,0},	{2,1,0},	{1,0,0},	{1,1,0},	{1,2,0},				{-1,-1,-1},
	},
	{ 
		{0,0,0},	{0,1,0},	{0,2,0},	{0,3,0},	{1,1,0},	{1,3,0},	{-1,-1,-1},
	},
/*
	{ 
		{0,0,0},	{1,0,0},	{2,0,0},	
		{0,1,0},	{1,1,0},	{2,1,0},	
		
		
		{-1,-1,-1},
	},
*/
};

const int NumberOfBasicShapes = sizeof(BasicShape)/sizeof(BasicShape[0]);

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	BlockToBitpos/BitposToBlock
	version 1: simple
*/
int BlockToBitpos(const Block& B)
{
	if(B[0]<0 || B[0]>=BitpatternExtX || B[1]<0 || B[1]>=BitpatternExtY || B[2]<0 || B[2]>=BitpatternExtZ)
	{
		return -1;
	}

	return B[0] + BitpatternExtX*(B[1] + BitpatternExtY*B[2]);
}

Block BitposToBlock(int Bitpos)
{
	Block B(	Bitpos%BitpatternExtX,
				(Bitpos/BitpatternExtX)%BitpatternExtY,
				Bitpos/(BitpatternExtX*BitpatternExtY)
			);

	return B;
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	StoneToBitpattern
*/
tBitPattern StoneToBitpattern(const Stone& S)
{
	/////////////////////////////////////////////////////////////////////////////
	
	tBitPattern R;

	for(int i=0;i<S.size();++i)
	{
		// check: are all blocks in range?
		int Bitpos = BlockToBitpos(S[i]);
		if(Bitpos<0)
		{
			return tBitPattern();	// empty pattern
		}

		R.set(Bitpos);
	}
	return R;

	/////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	...Color
	color codings for drawing
*/
COLORREF LeftRightColor(COLORREF C)
{
	int Red = GetRValue(C);
	int Green = GetGValue(C);
	int Blue = GetBValue(C);

	return RGB(Red*3/4,Green*3/4,Blue*3/4);
}

COLORREF TopBottomColor(COLORREF C)
{
	int Red = GetRValue(C);
	int Green = GetGValue(C);
	int Blue = GetBValue(C);

	return RGB(Red/2,Green/2,Blue/2);
}

COLORREF FrontBackColor(COLORREF C)
{
	return C;
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	DrawBitpattern
*/
void DrawBitpattern(CDC* pDC,int XOffset,int YOffset,int Width,tBitPattern Pattern,COLORREF C)
{
	/////////////////////////////////////////////////////////////////////////////

	// intermediate fields
	int Cube[BitpatternExtX][BitpatternExtY][BitpatternExtZ];
	memset(Cube,0,sizeof(Cube));

	int Back[BitpatternExtX][BitpatternExtY];
	int Bottom[BitpatternExtX][BitpatternExtZ];
	int Left[BitpatternExtY][BitpatternExtZ];

	memset(Back,0,sizeof(Back));
	memset(Bottom,0,sizeof(Bottom));
	memset(Left,0,sizeof(Left));

	/////////////////////////////////////////////////////////////////////////////

	int x,y,z;

	/////////////////////////////////////////////////////////////////////////////

	// transfer bit positions to interm. fields
	for(x=0;x<BitpatternExtX;++x)
	for(y=0;y<BitpatternExtY;++y)
	for(z=0;z<BitpatternExtZ;++z)
	{
		int BitPos = BlockToBitpos(Block(x,y,z));

		int Value = ( Pattern.test(BitPos) ? 1 : 0);

		Cube[x][y][z] = Value;
		Back[x][y]    |= Value;
		Bottom[x][z]  |= Value;
		Left[y][z]    |= Value;
	}

	/////////////////////////////////////////////////////////////////////////////

	// colors used for background
	COLORREF Free(RGB(192,192,64));
	COLORREF Occupied(RGB(128,128,64));

	/////////////////////////////////////////////////////////////////////////////

	// draw background and cubes
	for(z=BitpatternExtZ-1;z>=0;--z)
	for(y=BitpatternExtY-1;y>=0;--y)
	for(x=0;x<BitpatternExtX;++x)
	{
		/////////////////////////////////////////////////////////////////////////////

		int X1 = XOffset + x*Width + z*Width/2;
		int Y1 = YOffset + y*Width - z*Width/2;

		/////////////////////////////////////////////////////////////////////////////

		// background
		if(x==0)
		{
			DrawSide(pDC,X1,Y1,Width,2,LeftRightColor(Left[y][z]?Occupied:Free));
		}

		if(y==BitpatternExtY-1)
		{
			DrawSide(pDC,X1,Y1+Width,Width,1,TopBottomColor(Bottom[x][z]?Occupied:Free));
		}

		if(z==BitpatternExtZ-1)
		{
			DrawSide(pDC,X1+Width/2,Y1-Width/2,Width,0,FrontBackColor(Back[x][y]?Occupied:Free));
		}

		/////////////////////////////////////////////////////////////////////////////

		// cube
		if(Cube[x][y][z])
		{
			DrawSide(pDC,X1,Y1,Width,1,TopBottomColor(C));
			DrawSide(pDC,X1,Y1,Width,0,FrontBackColor(C));
			DrawSide(pDC,X1+Width,Y1,Width,2,LeftRightColor(C));
		}

		/////////////////////////////////////////////////////////////////////////////
	}

	/////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	GetFirstBitFrom
*/
int GetFirstBitFrom(tBitPattern Pattern,int StartBit,bool Condition)
{
	for(int i=StartBit;i<Pattern.size();++i)
	{
		if(Pattern.test(i) == Condition)return i;
	}
	return -1;		// no set bit found
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	PatternHeap, implementation
*/
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	construction
*/
PatternHeap::PatternHeap()
{
	/////////////////////////////////////////////////////////////////////////////

	DWORD Start = GetTickCount();

	/////////////////////////////////////////////////////////////////////////////

	m_Pattern.resize(NumberOfBasicShapes);

	/////////////////////////////////////////////////////////////////////////////

	// intermediate vector keeping all rotations per stone
	std::vector<tBitPattern> Rotations;

	/////////////////////////////////////////////////////////////////////////////

	for(int StoneNr=0; StoneNr<NumberOfBasicShapes; ++StoneNr)
	{
		/////////////////////////////////////////////////////////////////////////////

		m_Pattern[StoneNr].resize(BitsTotal);	

		/////////////////////////////////////////////////////////////////////////////

		// get basic shape nr n
		Stone S(BasicShape[StoneNr]);

		/////////////////////////////////////////////////////////////////////////////

		// prepare intermed. array
		Rotations.reserve(24);
		Rotations.resize(0);

		/////////////////////////////////////////////////////////////////////////////

		int MaxRot = 24;

		// basic shapes 4/6/9 have 24 different rotations
		// filter out rotations 6..23 for one of them to remove symetric solutions 
		if(StoneNr==4)MaxRot=6;
//		if(StoneNr==6)MaxRot=6;
//		if(StoneNr==9)MaxRot=6;

		for(int Rot=0; Rot<MaxRot; ++Rot)
		{
			/////////////////////////////////////////////////////////////////////////////

			// calculate all rotations
			Stone S1 = S;

			S1.Rotate(Rot);	S1.Align();
			tBitPattern P1 = StoneToBitpattern(S1);

			if(P1.none() == true)continue;	// failed, continue with next rotation
			
			/////////////////////////////////////////////////////////////////////////////

			// check: is this rotation already in vector (if yes, skip it)
			if(std::find(Rotations.begin(),Rotations.end(),P1) != Rotations.end())continue;

			/////////////////////////////////////////////////////////////////////////////

			Rotations.push_back(P1);

			/////////////////////////////////////////////////////////////////////////////

			// x/y/z min of S1 are 0 (due to S1.Align)
			Block B = S1.GetMax();

			// create all possible shifts of pattern
			for(int x=0;x<BitpatternExtX-B[0];++x)
			for(int y=0;y<BitpatternExtY-B[1];++y)
			for(int z=0;z<BitpatternExtZ-B[2];++z)
			{
				Stone S2 = S1;
				S2.Shift(Block(x,y,z));

				tBitPattern P2 = StoneToBitpattern(S2);

				if(P2.none() == true)
				{
					ASSERT(FALSE);	// must not happen, bounds in loop should guarantee existence of pattern
					continue;
				}

				int Bit = GetFirstBitFrom(P2,0,true);
				if(Bit<0 || Bit>=m_Pattern[StoneNr].size())
				{
					ASSERT(FALSE);	// OOPS, something rotten in the state of Denmark
					continue;
				}

				// use upper bits for hash key
				m_Pattern[StoneNr][Bit].push_back(P2);
			}

			/////////////////////////////////////////////////////////////////////////////
		}

		/////////////////////////////////////////////////////////////////////////////

		TRACE(_T("Stone%d has %d different rotations\n"),StoneNr,Rotations.size());

		/////////////////////////////////////////////////////////////////////////////
	}

	/////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	Solver, implementation
*/
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	construction, destruction
*/
Solver::Solver()
	:m_ThreadRunning(FALSE,TRUE),m_TerminateThread(FALSE,TRUE)
{
	m_ThreadState = IDLE;
	m_ThreadHandle = INVALID_HANDLE_VALUE;
	m_Solutions = 0;
	m_NumberOfTries = 0;

	m_SolveStart = 0;
	m_SolveEnd = m_SolveStart - 1;
}

Solver::~Solver()
{
	HANDLE H = m_ThreadHandle;
	
	if(H != INVALID_HANDLE_VALUE)
	{
		Terminate();	
		WaitForSingleObject(H,INFINITE);
	}
}

/////////////////////////////////////////////////////////////////////////////
/*
	Start
*/
void Solver::Start()
{
	/////////////////////////////////////////////////////////////////////////////

	LockPublicData();

	if(m_ThreadState == RUNNING)	// already running
	{
		UnlockPublicData();
		return;
	}

	/////////////////////////////////////////////////////////////////////////////
	
	// prepare statistics
	
	m_Solutions = 0;
	m_NumberOfTries = 0;
	m_SolveStart = GetTickCount();
	m_SolveEnd = m_SolveStart - 1;

	/////////////////////////////////////////////////////////////////////////////

	// prepare free stones

	m_FreeStones.reserve(m_PatternHeap.P().size());
	m_FreeStones.resize(0);
	for(int i=0;i<m_PatternHeap.P().size();++i)
	{
		m_FreeStones.push_back(i);
	}

	/////////////////////////////////////////////////////////////////////////////

	// prepare used stones

	m_UsedStones.reserve(m_PatternHeap.P().size());
	m_UsedStones.resize(0);

	/////////////////////////////////////////////////////////////////////////////

	// prepare fill state

	m_FillState = tBitPattern();
	m_NextFreeBitInFillState = 0;

	/////////////////////////////////////////////////////////////////////////////

	UnlockPublicData();

	// start worker thread
	m_ThreadState = RUNNING;
	m_ThreadRunning.SetEvent();
	m_TerminateThread.ResetEvent();

	m_ThreadHandle = AfxBeginThread(&Run,this)->m_hThread;

	/////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////
/*
	Run
*/
UINT Solver::Run(LPVOID PointerToThis)
{
	Solver* This = reinterpret_cast<Solver*>(PointerToThis);
	This->Solve();

	This->m_ThreadState = IDLE;
	This->m_ThreadHandle = INVALID_HANDLE_VALUE;
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
/*
	Terminate
*/
void Solver::Terminate()
{
	Continue();
	HANDLE H = m_ThreadHandle;
	if(H != INVALID_HANDLE_VALUE)
	{
		m_TerminateThread.SetEvent();
		WaitForSingleObject(H,INFINITE);
	}
}

/////////////////////////////////////////////////////////////////////////////
/*
	Solve
*/
void Solver::Solve()
{
	/////////////////////////////////////////////////////////////////////////////

	// wait, if run flag is down
	CSingleLock L(&m_ThreadRunning,TRUE);
	L.Unlock();

	/////////////////////////////////////////////////////////////////////////////

	// check, wether termination flag is up
	CSingleLock L1(&m_TerminateThread,FALSE);
	if(L1.Lock(0) == TRUE)return;
	L1.Unlock();

	/////////////////////////////////////////////////////////////////////////////

	int FreeStones = m_FreeStones.size();

	if(FreeStones == 0)
	{
		/////////////////////////////////////////////////////////////////////////////

		// update stats + solution
		LockPublicData();

		m_Solutions += 1;

		m_LastSolution.reserve(m_UsedStones.size());
		m_LastSolution.resize(0);

		for(int i=0;i<m_UsedStones.size();++i)
		{
			tSolutionEntry Sol;
			Sol.StoneNr = m_UsedStones[i].StoneNr;
			Sol.Pattern = m_PatternHeap.P()[m_UsedStones[i].StoneNr][m_UsedStones[i].BitNr][m_UsedStones[i].PatternIndex];

			m_LastSolution.push_back(Sol);
		}

		UnlockPublicData();

		/////////////////////////////////////////////////////////////////////////////

		return;		// no need to go on, no more free stones

		/////////////////////////////////////////////////////////////////////////////
	}

	/////////////////////////////////////////////////////////////////////////////

	for(int StoneIndex=0; StoneIndex<FreeStones;++StoneIndex)
	{
		/////////////////////////////////////////////////////////////////////////////

		int StoneNr = m_FreeStones[StoneIndex];

		std::vector<tBitPattern>& P = m_PatternHeap.P()[StoneNr][m_NextFreeBitInFillState];

		/////////////////////////////////////////////////////////////////////////////

		int ExistingPatterns = P.size();

		for(int PatternIndex=0; PatternIndex<ExistingPatterns;++PatternIndex)
		{
			/////////////////////////////////////////////////////////////////////////////

			LockPublicData();
			m_NumberOfTries += 1;
			UnlockPublicData();

			/////////////////////////////////////////////////////////////////////////////

			tBitPattern Current = m_FillState;

			tBitPattern New = P[PatternIndex];
			Current &= New;

			if(Current.none()==true)
			{
				/////////////////////////////////////////////////////////////////////////////

				// intersection empty => pattern fits: insert pattern

				/////////////////////////////////////////////////////////////////////////////

				int NextFreeBitInFillStateBeforeInsertion = m_NextFreeBitInFillState;
				tBitPattern FillStateBeforeInsertion = m_FillState;

				tEntry E;
				E.StoneIndex = StoneIndex;
				E.StoneNr = StoneNr;
				E.BitNr = m_NextFreeBitInFillState;
				E.PatternIndex = PatternIndex;

				m_UsedStones.push_back(E);

				/////////////////////////////////////////////////////////////////////////////

				m_FillState |= New;
				m_NextFreeBitInFillState = GetFirstBitFrom(m_FillState,m_NextFreeBitInFillState+1,false);

				/////////////////////////////////////////////////////////////////////////////

				std::copy(m_FreeStones.begin()+StoneIndex+1,m_FreeStones.end(),m_FreeStones.begin()+StoneIndex);
				m_FreeStones.pop_back();

				/////////////////////////////////////////////////////////////////////////////

				Solve();	// descend

				/////////////////////////////////////////////////////////////////////////////

				// remove pattern, try next one

				/////////////////////////////////////////////////////////////////////////////

				m_FreeStones.push_back(0);
				std::copy_backward(m_FreeStones.begin()+StoneIndex,m_FreeStones.end()-1,m_FreeStones.end());
				m_FreeStones[StoneIndex] = StoneNr;

				m_FillState = FillStateBeforeInsertion;
				m_NextFreeBitInFillState = NextFreeBitInFillStateBeforeInsertion;

				m_UsedStones.pop_back();

				/////////////////////////////////////////////////////////////////////////////
			}
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	AsmSolver, implementation
*/
namespace AsmSolver
{

/////////////////////////////////////////////////////////////////////////////

tPatternInfoAsm* PatternHeap = NULL;
int PatternHeapStones;
int PatternHeapBitIndizes;

/////////////////////////////////////////////////////////////////////////////

void CreatePatternHeap()
{
	Puzzle::PatternHeap H;

	PatternHeapStones = NumberOfBasicShapes;
	PatternHeapBitIndizes = BitsTotal;

	PatternHeap = new tPatternInfoAsm[PatternHeapStones*PatternHeapBitIndizes];

	for(int i=0;i<PatternHeapStones*PatternHeapBitIndizes;++i)
	{
		// memory order: PatternHeap[Stone][Bit]
		int Stone = i%PatternHeapStones;
		int Bit   = i/PatternHeapStones;

		ASSERT(Stone>=0 && Stone<H.P().size());
		ASSERT(Bit>=0 && Bit<H.P()[Stone].size());

		int Patterns = H.P()[Stone][Bit].size();

		PatternHeap[i].Count = Patterns;
		PatternHeap[i].Pattern = new __int64[Patterns];

		for(int j=0;j<Patterns;++j)
		{
			PatternHeap[i].Pattern[j] = 0;

			tBitPattern P = H.P()[Stone][Bit][j];

			for(int k=0;k<P.size();++k)
			{
				if(P.test(k))
				{
					PatternHeap[i].Pattern[j] |= (1I64<<k);
				}
			}
		}
	}
}

void DeletePatternHeap()
{
	for(int i=0;i<PatternHeapStones*PatternHeapBitIndizes;++i)
	{
		delete[] PatternHeap[i].Pattern;
	}
	delete[] PatternHeap;
	PatternHeap = NULL;
}

/////////////////////////////////////////////////////////////////////////////

__int64 FillState = 0;
int NextFreeBitInFillState = 0;
	
int FreeStones[128];
int FreeStonesCount = 0;

int Solutions = 0;
__int64 NumberOfTries = 0;

DWORD SolveStart = 0;
HWND OutputWindow[3] = {NULL,NULL,NULL};

/////////////////////////////////////////////////////////////////////////////
/*
	Start:
	prepare variables
*/
void Start(HWND W0,HWND W1, HWND W2)
{
	SolveStart = GetTickCount();

	FillState = 0;
	NextFreeBitInFillState = 0;

	FreeStonesCount = NumberOfBasicShapes;
	ASSERT(FreeStonesCount<=sizeof(FreeStones)/sizeof(FreeStones[0]));
	for(int i=0;i<NumberOfBasicShapes;++i)
	{
		FreeStones[i] = i;
	}

	Solutions = 0;
	NumberOfTries = 0;

	OutputWindow[0] = W0;
	OutputWindow[1] = W1;
	OutputWindow[2] = W2;
}

/////////////////////////////////////////////////////////////////////////////
/*
	Solve
	do a recursive descent to find solutions
*/
void Solve(int Recurse)
{
	/////////////////////////////////////////////////////////////////////////////

	if(FreeStonesCount == 0)
	{
		Solutions += 1;
		return;
	}

	/////////////////////////////////////////////////////////////////////////////

	for(int StoneIndex=0; StoneIndex<FreeStonesCount;++StoneIndex)
	{
		/////////////////////////////////////////////////////////////////////////////

		int StoneNr = FreeStones[StoneIndex];
		tPatternInfoAsm* P = PatternHeap + StoneNr + NextFreeBitInFillState*PatternHeapStones;

		/////////////////////////////////////////////////////////////////////////////

		bool CurrentStoneIndexRemoved = false;

		/////////////////////////////////////////////////////////////////////////////

		int ExistingPatterns = P->Count;

		for(int PatternIndex=0; PatternIndex<ExistingPatterns;++PatternIndex)
		{
			/////////////////////////////////////////////////////////////////////////////

			NumberOfTries += 1;

			/////////////////////////////////////////////////////////////////////////////

			__int64 New = P->Pattern[PatternIndex];

			if((FillState & New) == 0)
			{
				/////////////////////////////////////////////////////////////////////////////

				// intersection empty => pattern fits: insert pattern

				/////////////////////////////////////////////////////////////////////////////

				// remove stone, if not already done
				if(CurrentStoneIndexRemoved == false)
				{
					/////////////////////////////////////////////////////////////////////////////

					for(int i=StoneIndex+1;i<FreeStonesCount;++i)
					{
						FreeStones[i-1] = FreeStones[i];
					}
					FreeStonesCount -= 1;

					CurrentStoneIndexRemoved = true;

					/////////////////////////////////////////////////////////////////////////////
				}

				/////////////////////////////////////////////////////////////////////////////

				// store current state of pattern
				int NextFreeBitInFillStateBeforeInsertion = NextFreeBitInFillState;
				__int64 FillStateBeforeInsertion = FillState;

				/////////////////////////////////////////////////////////////////////////////

				// fill in pattern, determine next free bit
				_asm
				{
					/////////////////////////////////////////////////////////////////////////////

					// FillState |= New
					MOV EAX, DWORD PTR FillState
					MOV EBX, DWORD PTR FillState + 4

					OR  EAX, DWORD PTR New
					OR  EBX, DWORD PTR New+4

					MOV DWORD PTR FillState, EAX
					MOV DWORD PTR FillState + 4, EBX

					/////////////////////////////////////////////////////////////////////////////

					// scan for least significant bit cleared
					NOT EAX
					BSF ECX, EAX

					JZ TestHigh
					MOV NextFreeBitInFillState, ECX
					JMP EndOfBittest
				
				TestHigh:
					NOT EBX
					BSF ECX, EBX
					JZ NoBitFound

					ADD ECX, 32
					MOV NextFreeBitInFillState, ECX
					JMP EndOfBittest

				NoBitFound:
					MOV NextFreeBitInFillState, -1

				EndOfBittest:
					/////////////////////////////////////////////////////////////////////////////
				}

				/////////////////////////////////////////////////////////////////////////////

				// output

				if(Recurse<3)
				{
					CString S;

					if(OutputWindow[0] != NULL)
					{
						DWORD Now = GetTickCount();
						DWORD Delay = Now - SolveStart;

						S.Format(_T("running for %d s, solutions so far: %d (%d/s)"),Delay/1000,Solutions,Solutions*1000/max(Delay,1));

						::SetWindowText(OutputWindow[0],S);
					}

					if(Recurse == 1 && OutputWindow[1] != NULL)
					{
						S.Format(_T("push (%2d): %2d,(%2d of %2d)"),Recurse,StoneNr,PatternIndex,ExistingPatterns);

						::SetWindowText(OutputWindow[1],S);
						if(OutputWindow[2] != NULL)
						{
							::SetWindowText(OutputWindow[2],_T(""));
						}
					}
					if(Recurse == 2 && OutputWindow[2] != NULL)
					{
						S.Format(_T("push (%2d): %2d,(%2d of %2d)"),Recurse,StoneNr,PatternIndex,ExistingPatterns);

						::SetWindowText(OutputWindow[2],S);
					}
				}

				/////////////////////////////////////////////////////////////////////////////

				Solve(Recurse+1);	// descend

				/////////////////////////////////////////////////////////////////////////////

				// remove pattern, try next one

				/////////////////////////////////////////////////////////////////////////////

				FillState = FillStateBeforeInsertion;
				NextFreeBitInFillState = NextFreeBitInFillStateBeforeInsertion;

				/////////////////////////////////////////////////////////////////////////////
			}

		}

		/////////////////////////////////////////////////////////////////////////////

		// put stone back, if it was removed
		if(CurrentStoneIndexRemoved == true)
		{
			/////////////////////////////////////////////////////////////////////////////

			for(int i=FreeStonesCount-1;i>=StoneIndex;--i)
			{
				FreeStones[i+1] = FreeStones[i];
			}
			FreeStonesCount += 1;
			FreeStones[StoneIndex] = StoneNr;

			/////////////////////////////////////////////////////////////////////////////
		}

		/////////////////////////////////////////////////////////////////////////////
	}

	/////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////

} // end of namespace AsmSolver

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

}	// end of namespace Puzzle

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

