/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	PuzzleClasses.h
*/
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

#pragma once

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

// includes required for header
#include <vector>
#include <bitset>

#include <afxmt.h>

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

// namespace to wrap puzzle-related classes
namespace Puzzle{

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	Block: one single block in a stone
	basically just three coordinates, wrapped with class to allow a "copy semantic" on
	blocks (not possible with simple arrays under C/C++)
*/
class Block
{
public:
	enum{DIMENSIONS = 3};	// keep number of dimensions as enum

	// construction, copy
	Block();
	Block(int,int,int);
	Block(const Block&);
	Block& operator=(const Block&);

	// comparison
	bool operator==(const Block&)const;

	// access to data
	int& operator[](int);	
	const int& operator[](int)const;	

private:
	int m_Data[DIMENSIONS];
};

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	Rotate:
	rotation function for a single block. Rot = [0..24[
*/
Block Rotate(const Block& B,int Rot);

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	DrawSide
	primitive drawing routine for one single cube side
*/
void DrawSide(CDC* pDC,int XOffset,int YOffset,int Width,int Side,COLORREF C);

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	Stone: array of Blocks
	helper class to deal with operations on stones, that can be coded more natural
	by using arrays
*/
class Stone : private std::vector<Block>
{
	typedef std::vector<Block> BaseClass;
public:
	//construction
	Stone();
	Stone(const Stone&);
	Stone(int (*Init)[3]);

	// rotation
	void Rotate(int Rot);

	// alignment (min x/y/z coordinates = 0)
	void Align();

	// shift 
	void Shift(Block B);

	// determine size
	Block GetMin()const;
	Block GetMax()const;

	// access to single blocks (just forwards of vector)
	int size()const {return BaseClass::size();}
	Block& operator[](int n){ return BaseClass::operator[](n);}
	const Block& operator[](int n) const{ return BaseClass::operator[](n);}
};

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

// basic shapes used for puzzle
extern int BasicShape[][7][3];
extern const int NumberOfBasicShapes;

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	dimensions of one single bitpattern 
*/
enum
{
//	BitpatternExtX = 5,
//	BitpatternExtY = 3,
//	BitpatternExtZ = 4,

	BitpatternExtX = 3,
	BitpatternExtY = 4,
	BitpatternExtZ = 5,

	BitsTotal = BitpatternExtX*BitpatternExtY*BitpatternExtZ,
};

typedef std::bitset<BitsTotal> tBitPattern;

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	BlockToBitpos:

	mapping between block position and bit position 
*/
int BlockToBitpos(const Block& B);
Block BitposToBlock(int Bitpos);

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	StoneToBitpattern:
	conversion from Stone into bitpattern
	returns empty pattern, if conversion fails (if one block exceeds space reserved for bits)
*/
tBitPattern StoneToBitpattern(const Stone& S);

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	...Color:
	color schemes used for drawing: sides top/bottom have shadings relative to front 
*/
COLORREF LeftRightColor(COLORREF C);
COLORREF TopBottomColor(COLORREF C);
COLORREF FrontBackColor(COLORREF C);

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	DrawBitpattern:
	draw pattern at position in given color
*/
void DrawBitpattern(CDC* pDC,int XOffset,int YOffset,int Width,tBitPattern Pattern,COLORREF C);

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	GetFirstBitFrom:
	scans for first bit in pattern matching condition (set or cleared), beginning search from StartBit
*/
int GetFirstBitFrom(tBitPattern Pattern,int StartBit,bool Condition);

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	PatternHeap:

	create required patterns to solve the puzzle
*/
class PatternHeap 
{
public:
	PatternHeap();

	// keep all possible patterns, indexed by stone number, and bit position of first bit occupied by pattern
	std::vector<std::vector<std::vector<tBitPattern> > > m_Pattern;

	// shortcut
	std::vector<std::vector<std::vector<tBitPattern> > >& P(){return m_Pattern;}
};

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	Solver:

	realizes algorithm to solve the puzzle
*/
class Solver
{
public:
	/////////////////////////////////////////////////////////////////////////////

	Solver();
	~Solver();
	void Start();

	void Suspend(){m_ThreadRunning.ResetEvent();}
	void Continue(){m_ThreadRunning.SetEvent();}

	void Terminate();

	bool IsSuspended()
	{
		CSingleLock L(&m_ThreadRunning,FALSE);
		return L.Lock(0) == FALSE;
	}

	/////////////////////////////////////////////////////////////////////////////

	// public data showing state of solver, read only from outside!
	// must be locked, when read
	void LockPublicData(){m_AccessToPublicData.Lock();}
	void UnlockPublicData(){m_AccessToPublicData.Unlock();}

	enum ThreadState
	{
		IDLE, RUNNING
	};
	ThreadState m_ThreadState;

	struct tSolutionEntry
	{
		int StoneNr;		
		tBitPattern Pattern;
	};

	std::vector<tSolutionEntry> m_LastSolution;		// last solution found

	/////////////////////////////////////////////////////////////////////////////

	// statistics
	int m_Solutions;						// number of solutions found
	__int64 m_NumberOfTries;				// total number of patterns tried
	DWORD m_SolveStart;						// start time
	DWORD m_SolveEnd;						// end time

	/////////////////////////////////////////////////////////////////////////////
private:
	/////////////////////////////////////////////////////////////////////////////

	static UINT Run(LPVOID PointerToThis);
	void Solve();		// recursive solve function

	/////////////////////////////////////////////////////////////////////////////

	// sync objects
	CCriticalSection m_AccessToPublicData;		// critical section to control access to public data

	CEvent m_ThreadRunning;		// if thread is active: if set => thread runs, if down => thread suspended
	CEvent m_TerminateThread;	// set from main thread, if we want to kill worker thread

	HANDLE m_ThreadHandle;

	/////////////////////////////////////////////////////////////////////////////

	PatternHeap m_PatternHeap;						// existing patterns

	/////////////////////////////////////////////////////////////////////////////

	struct tEntry
	{
		int StoneIndex;		// index in stack of free stones 

		// 'address' in m_PatternHeap
		int StoneNr;		
		int BitNr;
		int PatternIndex;
	};
	std::vector<tEntry> m_UsedStones;		// used stones in an stack-like used vector

	/////////////////////////////////////////////////////////////////////////////

	tBitPattern m_FillState;				// current fill state of solution
	int m_NextFreeBitInFillState;			// next free bit to be found in fill state

	/////////////////////////////////////////////////////////////////////////////

	std::vector<int> m_FreeStones;			// keep free stones in an array

	/////////////////////////////////////////////////////////////////////////////
};

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/*
	AsmSolver:

	basically same as solver, just optimized by using assembler code in some parts and without support
	for multithreading
*/
namespace AsmSolver
{
	/////////////////////////////////////////////////////////////////////////////

	void CreatePatternHeap();
	void DeletePatternHeap();

	struct tPatternInfoAsm
	{
		int Count;
		__int64* Pattern;
	};
	extern tPatternInfoAsm* PatternHeap;

	extern int PatternHeapStones;
	extern int PatternHeapBitIndizes;

	/////////////////////////////////////////////////////////////////////////////

	void Start(HWND W0,HWND W1, HWND W2);
	void Solve(int Recurse);

	extern __int64 FillState;
	extern int NextFreeBitInFillState;
	
	extern int FreeStones[128];
	extern int FreeStonesCount;

	/////////////////////////////////////////////////////////////////////////////

	extern int Solutions;
	extern __int64 NumberOfTries;

	extern DWORD SolveStart;
	extern HWND OutputWindow[3];

	/////////////////////////////////////////////////////////////////////////////

} // end of namespace AsmSolver

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

}	// end of namespace Puzzle

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
