#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>

/*
 * Bit 0: stdout on/off
 * Bit 1: string ouput on/off
 * Bit 2: file on/off
 */
#define MODE 0				/* disable output */

#define MODE_STDOUT 0x1		/* print on stdout */
#define MODE_STR 0x2		/* 0: cube, 1: flat string */
#define MODE_FILE 0x4		/* 1: string to file */

#define INIT_BIT_CHAR '.'

#define MAX_X 3
#define MAX_Y 4
#define MAX_Z 5
#define MAX_CT 4

#define N_BITS (MAX_X * MAX_Y * MAX_Z)		/* bits in vector */

#define MAX_POS (N_BITS * 24)				/* maximal positions in cube */

#define C_BITS 0							/* precompared bits (0 disables feature) */
#define C_BITS_COUNT (1 << C_BITS)

#define N_PARTS 12

typedef unsigned long long int FIELD;
typedef union {
	FIELD v;
	long int vt[2];
} UFIELD;

/* a PART consist of a name and a bit vector */
typedef struct {
	char name;
	FIELD v;
} PART;

/* the solution */
typedef PART SOLUTION[N_PARTS];

/* permutated part */
typedef struct {
	char name;
	int used;
	int perm;
	int rot;
	int min_start[N_BITS];
	int min_end[N_BITS];
	FIELD v[MAX_POS];
	#if (C_BITS > 0)
	int cmin_start[C_BITS_COUNT][N_BITS];
	int cmin_end[C_BITS_COUNT][N_BITS];
	FIELD cv[C_BITS_COUNT][MAX_POS];
	#endif
	int count[MAX_POS];
} PART_PERM;

typedef PART_PERM PARTS_PERM[N_PARTS];

typedef struct {
	int min_bit;
	FIELD v;
} PART_SORT;

typedef PART_SORT PARTS_SORT[MAX_POS];

/* count the solutions */
unsigned long long int solution_count = 0ll;

unsigned long long int test_count = 0ll;
unsigned long long int test_count1 = 0ll;

unsigned long long int next_bit_count = 0ll;
unsigned long long int next_bit_count16 = 0ll;

/* define the parts of the puzzle */
const SOLUTION init_parts = {
	{'A', (FIELD)0x2F},		/* c */
	{'B', (FIELD)0x139},		/* first ' */
	{'C', (FIELD)0x24B},
	{'D', (FIELD)0x259},
	{'E', (FIELD)0x101B},		/* second ' */
	{'F', (FIELD)0x97},
	{'G', (FIELD)0x1F},
	{'H', (FIELD)0x5E},
	{'I', (FIELD)0x10013},		/* third ' */
	{'J', (FIELD)0xB3},
	{'K', (FIELD)0xBA},
	{'L', (FIELD)0x2CB}		/* t */
};


char min_0_array[65536];

PARTS_PERM parts_perm;

FIELD mask_array[N_BITS];

FIELD cmask_array[N_BITS];

char *prg;
struct tm *s_time_tm;

time_t s_time;

#define TIME_STR 512
char s_time_str[TIME_STR];

const char ext[] = "out";

char fname[FILENAME_MAX];

FILE *out = NULL;
FILE *in = NULL;

char reduce_name = 'J';

clock_t start_clock;

SOLUTION global_solution;

void error (char *function_name, char *msg) {
	printf("ERROR: %s (function %s)\n", msg, function_name);
	exit(1);
};

/* print bit representation of cube */
void field_print (FIELD f) {
	char a [MAX_X][MAX_Y][MAX_Z];
	int i, j, k;
	int h;

	/* init the output vector */
	for (i = 0; i < MAX_X; i++) {
		for (j = 0; j < MAX_Y; j++) {
			for (k = 0; k < MAX_Z; k++) {
				a[i][j][k] = INIT_BIT_CHAR;
			};
		};
	};
	/* fill parts into output vector */
	for (i = 0; i < N_BITS; i++) {
		if (f & mask_array[i]) {
			a[i%MAX_X][(i/MAX_X)%MAX_Y][i/(MAX_X*MAX_Y)] = '1';
		};
	};

	/* show output vector */
	for (i = 0; i < MAX_X; i++) {
		for (j = 0; j < MAX_Y; j++) {
			for (h = 0; h < j; h++) {
				printf(" "); /* */
			};
			for (k = 0; k < MAX_Z; k++) {
				printf("%c ", a[i][j][k]); /* */
			};
			printf("\n"); /* */
		};
	};
};

void solution_init (SOLUTION s) {
	int j;

	for (j = 0; j < N_PARTS; j++) {
		s[j].name = ' ';
		s[j].v = (FIELD)0;
	};
};

int solution_read (SOLUTION s) {
	char c;
	int i;
	int result;
	char res_str[N_BITS+1];

	solution_init(s);
	if ((result = fscanf(in, "%60s", res_str)) > 0) {
		/* printf("XXX: %s\n", res_str); /* */
	}
	else {
		return(0);
	};
	solution_init(s);
	for (i = 0; i < N_PARTS; i++) {
		s[i].name = 'A' + i;
	};
	for (i = 0; i < N_BITS; i++) {
		c = res_str[i];
		s[c - 'A'].v |= mask_array[i];
	};
	return(result);
};

void solution_print (SOLUTION s, int mode) {
	char a [MAX_X][MAX_Y][MAX_Z];
	int i, j, k;
	int h, r;
	char res_str[N_BITS+1];

	/* mode:
	 *	bit 0: = 1 stdout on, = 0 stdout off
	 *  bit 1: = 1 file on, = 0 file off
	 */
	/* init the output vector */
	for (i = 0; i < MAX_X; i++) {
		for (j = 0; j < MAX_Y; j++) {
			for (k = 0; k < MAX_Z; k++) {
				a[i][j][k] = INIT_BIT_CHAR;
			};
		};
	};
	/* fill parts into output vector */
	for (j = 0; j < N_PARTS; j++) {
		if (s[j].name != ' ') {		/* skip empty parts */
			for (i = 0; i < N_BITS; i++) {
				if (s[j].v & mask_array[i]) {
					a[i%MAX_X][(i/MAX_X)%MAX_Y][i/(MAX_X*MAX_Y)] = s[j].name;
				};
			};
		};
	};
	if (solution_count != 0ll) {
		/* if ((mode & MODE_STDOUT) && !(mode & MODE_STR)) printf("\nsolution_count = %llu\n", solution_count); /* */
	};
	/* show output vector */
	r = 0;
	for (i = 0; i < MAX_X; i++) {
		for (j = 0; j < MAX_Y; j++) {
			for (h = 0; h < j; h++) {
				if ((mode & MODE_STDOUT) && !(mode & MODE_STR)) printf(" "); /* */
			};
			for (k = 0; k < MAX_Z; k++) {
				if ((mode & MODE_STDOUT) && !(mode & MODE_STR)) printf("%c ", a[i][j][k]); /* */
			};
			if ((mode & MODE_STDOUT) && !(mode & MODE_STR)) printf("\n"); /* */
		};
	};
	r = 0;
	for (k = 0; k < MAX_Z; k++) {
		for (j = 0; j < MAX_Y; j++) {
			for (i = 0; i < MAX_X; i++) {
				res_str[r++] = a[i][j][k];
			};
		};
	};
	res_str[r] = '\0';
	#if ((MODE & MODE_FILE) != 0)
	if (mode & MODE_FILE) fprintf(out, "%s\n", res_str);
	#endif
	if ((mode & MODE_STDOUT) && !(mode & MODE_STR)) printf("\n"); /* */
	if ((mode & MODE_STDOUT) && (mode & MODE_STR)) printf("%s\n", res_str);
};

void test_FIELD (void) {
	FIELD f;
	int j;
	
	/* TEST: ensure that FIELD defintion contain at least 60 bits */
	j = 0;
	for (f = (FIELD)1; f != 0; f <<= 1) {
		j++;
	};
	if (j <  N_BITS) {
		error("test_FIELD", "not enough bits in FIELD");
	};
};

void box_get_max (FIELD v, int *max_x, int *max_y, int *max_z) {
	int x, y, z;
	int i;

	*max_x = *max_y = *max_z = 0;
	for (i = 0; i < N_BITS; i++) {
		if (v & mask_array[i]) {
			x = i % MAX_X;
			y = (i / MAX_X) % MAX_Y;
			z = i / (MAX_X * MAX_Y);
			if (x > *max_x) {
				*max_x = x;
			};
			if (y > *max_y) {
				*max_y = y;
			};
			if (z > *max_z) {
				*max_z = z;
			};
		};
	};
};

void box_get_min (FIELD v, int *min_x, int *min_y, int *min_z) {
	int x, y, z;
	int i;

	*min_x = MAX_X - 1;
	*min_y = MAX_Y - 1;
	*min_z = MAX_Z - 1;
	for (i = 0; i < N_BITS; i++) {
		if (v & mask_array[i]) {
			x = i % MAX_X;
			y = (i / MAX_X) % MAX_Y;
			z = i / (MAX_X * MAX_Y);
			if (x < *min_x) {
				*min_x = x;
			};
			if (y < *min_y) {
				*min_y = y;
			};
			if (z < *min_z) {
				*min_z = z;
			};
		};
	};
};

int min_bit_calc (FIELD v) {
	int i;
	FIELD mask;
	
	mask = (FIELD)1;
	for (i = 0; i < N_BITS; i++) {
		if (v & mask) {
			return(i);
		};
		mask <<= 1;
	};
	return(-1);
};

void part_print (PART_PERM *p_perm) {
	SOLUTION s;
	int max_x, max_y, max_z;
	int j;
	int rot;
	int i, k;
	int bits;
	FIELD val, mask;
	int c[32];
	int max_min_bit;

	printf("\n\n\nPart: %c\n", p_perm->name);
	rot = p_perm->rot;
	max_min_bit = 0;
	for (j = 0; j < p_perm->perm; j++) { /* */
		k = min_bit_calc(p_perm->v[j]);
		if (k > max_min_bit) {
			max_min_bit = k;
		};
		if (j >= rot) continue;
		/* printf("index = %3d, ", j);
		box_get_max(p_perm->v[j], &max_x, &max_y, &max_z);
		printf("max_x=%d, max_y=%d, max_z=%d\n", max_x, max_y, max_z);
		solution_init(s);
		s[0].name = p_perm->name;
		s[0].v = p_perm->v[j];
		solution_print(s,(MODE | MODE_STDOUT) & ~MODE_FILE); /* */
	};
	printf("%d permutations, %d rotations, %d moves, max_min_bit = %d\n",j,rot,j-rot,max_min_bit);
	if ((max_min_bit + C_BITS) >= N_BITS) {
		printf("WARNING: C_BITS too large\n");
	};

	return;

	/* TEST: print array of bit indexes */
	bits = 4;
	for (j = 0; j < N_BITS; j++) {
		if (p_perm->min_start[j] == -1) {
			continue;
		};
		printf("%c: %2d", p_perm->name, j);
		/* printf(", start=%3d, end=%3d",
			p_perm->min_start[j], p_perm->min_end[j]); /* */
		printf(", anz=%2d", p_perm->min_end[j] - p_perm->min_start[j]);
		mask = (FIELD)0;
		for (i = 0; i < bits; i++) {
			mask = (mask << 1) + 1;
		};
		mask <<= (j + 1);
		for (i = 0; i < (1 << bits); i++) {
			c[i] = 0;
		};
		for (i = 0; i < (1 << bits); i++) {
			val = (FIELD)i << (j + 1);
			for (k = p_perm->min_start[j]; k < p_perm->min_end[j]; k++) {
				if ((p_perm->v[k] & mask) == val) {
					c[i]++;
				};
			};
		};
		for (i = 0; i < (1 << bits); i++) {
			printf(", %2d",c[i]);
		};
		printf("\n");
	};
};

void bit_set (FIELD *v,
	int x, int y, int z,
	int max_x, int max_y, int max_z,
	char axis, int angle) {

	int new_x, new_y, new_z;
	int i;

	new_x = new_y = new_z = 0;
	switch (axis) {
		case 'x':
			switch (angle) {
				case 1:
					new_y = max_z - z;
					new_z = y;
					break;
				case 2:
					new_y = max_y - y;
					new_z = max_z - z;
					break;
				case 3:
					new_y = z;
					new_z = max_y - y;
					break;
			};
			new_x = x;
			break;
		case 'y':
			switch (angle) {
				case 1:
					new_x = max_z - z;
					new_z = x;
					break;
				case 2:
					new_x = max_x - x;
					new_z = max_z - z;
					break;
				case 3:
					new_x = z;
					new_z = max_x - x;
					break;
			};
			new_y = y;
			break;
		case 'z':
			switch (angle) {
				case 1:
					new_x = max_y - y;
					new_y = x;
					break;
				case 2:
					new_x = max_x - x;
					new_y = max_y - y;
					break;
				case 3:
					new_x = y;
					new_y = max_x - x;
					break;
			};
			new_z = z;
			break;
		default:
			error("bit_set", "unsupported axis");
			break;
	};
	i = new_z * (MAX_X * MAX_Y) + new_y * MAX_X + new_x;
	*v |= mask_array[i];
};

FIELD rotate (FIELD v, char axis, int angle, int max_set) {

/* angle: 1=90, 2=180, 3=270 degree */
	FIELD result;
	int max_x, max_y, max_z;
	int x, y, z;
	int i;

	result = (FIELD)0;
	if (v == (FIELD)0) {
		return(result);
	};
	if (max_set != 0) {
		box_get_max(v, &max_x, &max_y, &max_z);
	}
	else {
		max_x = MAX_X - 1;
		max_y = MAX_Y - 1;
		max_z = MAX_Z - 1;
	};
	switch (axis) {
		case 'x':
			if (((max_y >= MAX_Z) || (max_z >= MAX_Y)) && (angle & 1)) {
				return(result);
			};
			break;
		case 'y':
			if (((max_z >= MAX_X) || (max_x >= MAX_Z)) && (angle & 1)) {
				return(result);
			};
			break;
		case 'z':
			if (((max_y >= MAX_X) || (max_x >= MAX_Y)) && (angle & 1)) {
				return(result);
			};
			break;
	};
	switch (angle) {
		case 0: return(v);
		case 1:
		case 2:
		case 3:
			for (x = 0; x <= max_x; x++) {
				for (y = 0; y <= max_y; y++) {
					for (z = 0; z <= max_z; z++) {
						i = z * MAX_X * MAX_Y + y * MAX_X + x;
						if (v & mask_array[i]) {
							bit_set(&result,x,y,z,max_x,max_y,max_z,axis,angle);
						};
					};
				};
			};
			break;
		default:
			error("rotate_z", "unsupported angle");
			break;
	};
	return(result);
};

void part_permutate (PART p, PART_PERM *p_perm) {
	PART_PERM tmp_perm;
	int i, j, k, max_j, min_j;
	int x, y, z;
	int x_off, y_off, z_off;
	int max_x, max_y, max_z;
	FIELD min;
	FIELD rot[3];
	char msg[512];

	p_perm->name = p.name;
	p_perm->used = 0;

	/* create all rotations */
	tmp_perm.perm = 1;
	tmp_perm.v[0] = p.v;
	for (j = 1; j <= 3; j++) {
		tmp_perm.v[j] = rotate(tmp_perm.v[0],'z',j,1);
		tmp_perm.perm++;
	};
	tmp_perm.v[tmp_perm.perm++] = rotate(tmp_perm.v[0],'x',1,1);
	for (j = 1; j <= 3; j++) {
		tmp_perm.v[j+4] = rotate(tmp_perm.v[4],'z',j,1);
		tmp_perm.perm++;
	};
	tmp_perm.v[tmp_perm.perm++] = rotate(tmp_perm.v[0],'x',2,1);
	for (j = 1; j <= 3; j++) {
		tmp_perm.v[j+8] = rotate(tmp_perm.v[8],'z',j,1);
		tmp_perm.perm++;
	};
	tmp_perm.v[tmp_perm.perm++] = rotate(tmp_perm.v[0],'x',3,1);
	for (j = 1; j <= 3; j++) {
		tmp_perm.v[j+12] = rotate(tmp_perm.v[12],'z',j,1);
		tmp_perm.perm++;
	};
	tmp_perm.v[tmp_perm.perm++] = rotate(tmp_perm.v[0],'y',1,1);
	for (j = 1; j <= 3; j++) {
		tmp_perm.v[j+16] = rotate(tmp_perm.v[16],'z',j,1);
		tmp_perm.perm++;
	};
	tmp_perm.v[tmp_perm.perm++] = rotate(tmp_perm.v[0],'y',3,1);
	for (j = 1; j <= 3; j++) {
		tmp_perm.v[j+20] = rotate(tmp_perm.v[20],'z',j,1);
		tmp_perm.perm++;
	};

	/* copy only disjunct parts */
	for (i = 0; i < tmp_perm.perm; i++) {
		if (tmp_perm.v[i] != (FIELD)0) {
			for (j = 0; j < p_perm->perm; j++) {
				if (tmp_perm.v[i] == p_perm->v[j]) {
					break;
				};
			};
			if (j == p_perm->perm) {
				p_perm->v[p_perm->perm++] = tmp_perm.v[i];
			};
		};
	};

	/* eliminate unnecessary rotations */
	if (p.name == reduce_name) {
		if ((p_perm->perm / 4) < 4) {
			sprintf(msg,"can not reduce part %c (only %d rotations)", p.name, p_perm->perm);
			error("part_permutate", msg);
		};
		tmp_perm.perm = 0;
		for (i = 0; i < p_perm->perm; j++) {
			min_j = -1;
			min = (FIELD)0xFFFFFFFFFFFFFFFF;
			for (j = 0; j < p_perm->perm; j++) {
				if (p_perm->v[j] && (p_perm->v[j] < min)) {
					min = p_perm->v[j];
					min_j = j;
				};
			};
			if (min_j == -1) {
				break;
			};
			tmp_perm.v[tmp_perm.perm++] = min;
			for (k = 0; k < 3; k++) {
				rot[k] = rotate(min,'x' + k,2,1);
			};
			p_perm->v[min_j] = (FIELD)0;
			for (j = 0; j < p_perm->perm; j++) {
				for (k = 0; k < 3; k++) {
					if (p_perm->v[j] == rot[k]) {
						p_perm->v[j] = (FIELD)0;
					};
				};
			};
		};
		for (i = 0; i < tmp_perm.perm; i++) {
			p_perm->v[i] = tmp_perm.v[i];
		};
		p_perm->perm = tmp_perm.perm;
	};

	p_perm->rot = p_perm->perm;

	/* move the part within the cube */
	max_j = p_perm->perm;
	for (j = 0; j < max_j; j++) {
		box_get_max(p_perm->v[j], &max_x, &max_y, &max_z);
		for (z_off = 0; z_off < (MAX_Z - max_z); z_off++) {
			for (y_off = 0; y_off < (MAX_Y - max_y); y_off++) {
				for (x_off = 0; x_off < (MAX_X - max_x); x_off++) {
					if ((x_off == 0) && (y_off == 0) && (z_off == 0)) {
						continue;
					};
					for (z = 0; z <= max_z; z++) {
						for (y = 0; y <= max_y; y++) {
							for (x = 0; x <= max_x; x++) {
								i = z * (MAX_X * MAX_Y) + y * MAX_X + x;
								if (p_perm->v[j] & mask_array[i]) {
									i = (z+z_off) * (MAX_X * MAX_Y) + (y+y_off) * MAX_X + (x+x_off);
									p_perm->v[p_perm->perm] |= mask_array[i];
								};
							};
						};
					};
					p_perm->perm++;
				};
			};
		};
	};

	/* reset counter */
	max_j = p_perm->perm;
	for (j = 0; j < max_j; j++) {
		p_perm->count[j] = 0;
	};
};

void solution_rotate (SOLUTION s) {
	int i, j;
	SOLUTION tmp;

	solution_print(s,MODE);
	for (i = 0; i < 3; i++) {
		memcpy(&tmp,s,sizeof(tmp));
		for (j = 0; j < N_PARTS; j++) {
			tmp[j].v = rotate(tmp[j].v,'x' + i,2,0);
		};
		solution_print(tmp,MODE); /* */
	};
};

char min_0_calc_TEST01 (FIELD v) {
	int i;
	FIELD mask;
	
	mask = (FIELD)1;
	for (i = 0; i < N_BITS; i++) {
		if ((v & mask) == (FIELD)0) {
			return(i);
		};
		mask <<= 1;
	};
	return(-1);
};

void cmask_array_init (void) {
	int i;
	FIELD mask;
	
	mask = (FIELD)0;
	for (i = 0; i < C_BITS; i++) {
		mask = (mask << 1) + 1;
	};
	for (i = 0; i < N_BITS; i++) {
		mask <<= 1;
		cmask_array[i] = mask;
	};
};

void mask_array_init (void) {
	int i;
	FIELD mask;
	
	mask = (FIELD)1;
	for (i = 0; i < N_BITS; i++) {
		mask_array[i] = mask;
		mask <<= 1;
	};
};

int min_0_calc_TEST02 (FIELD v, char last_next_bit) {
	int i;
	
	for (i = 0; i < N_BITS; i++) {
		if ((v & mask_array[i]) == (FIELD)0) {
			return(i);
		};
	};
	return(-1);
};

int min_0_calc_TEST03 (FIELD v) {
	int i,j;
	FIELD m = 0xFFFF;
	
	for (j = 0; j < 4; j++) {
		m <<= 16;
		if ((v & m) == m) {
			continue;
		};
		for (i = (j << 4); i < N_BITS; i++) {
			if ((v & mask_array[i]) == (FIELD)0) {
				return(i);
			};
		};
	};
	return(-1);
};

void min_0_array_init (void) {
	FIELD i;

	for (i = 0; i < 65536; i++) {
		min_0_array[i] = min_0_calc_TEST01(i);
	};
};

char min_0_calc_TEST04 (FIELD v, char last_next_bit) {
	FIELD i;

	i = (v & (FIELD)0x00000000FFFFFFFF);
	if ((v & (FIELD)0x00000000FFFFFFFF) != (FIELD)0x00000000FFFFFFFF) {
		if ((i & (FIELD)0x000000000000FFFF) != (FIELD)0x000000000000FFFF) {
			return(min_0_array[i & 0xFFFF]);
		}
		else {
			return(min_0_array[(i >> 16) & 0xFFFF]+16);
		};
	}
	else {
		i = (v & (FIELD)0xFFFFFFFF00000000);
		if ((i & (FIELD)0x0000FFFF00000000) != (FIELD)0x0000FFFF00000000) {
			return(min_0_array[(i >> 32) & 0xFFFF]+32);
		}
		else {
			return(min_0_array[(i >> 48) & 0xFFFF]+48);
		};
	};
};

char min_0_calc (FIELD v, int last_next_bit) {
	unsigned int j;
	char result;

	if ((j = (v & 0xFFFFFFFF)) != 0xFFFFFFFF) {
		if ((j & 0x0000FFFF) != 0x0000FFFF) {
			result = (min_0_array[j & 0xFFFF]);
		}
		else {
			result = (min_0_array[(j >> 16) & 0xFFFF]+16);
		};
	}
	else {
		if (((j = ((v >> 32) & 0xFFFFFFFF)) & 0x0000FFFF) != 0x0000FFFF) {
			result = (min_0_array[j & 0xFFFF]+32);
		}
		else {
			result = (min_0_array[(j >> 16) & 0xFFFF]+48);
		};
	};
	next_bit_count++;
	if (next_bit_count == 0ll) {
		error("min_0_calc","next_bit_count overflow");
	};
	if ((result - last_next_bit) > 17) {
		next_bit_count16++;
		if (next_bit_count16 == 0ll) {
			error("min_0_calc","next_bit_count16 overflow");
		};
	};
	return(result);
};

char min_0_calc_TEST06 (FIELD v, char last_next_bit) {
	int i, j;
	char result;

	if (last_next_bit < 0) {
		result = (min_0_calc_TEST02(v,last_next_bit));
		return(result);
	};
	result = 0;
	last_next_bit++;
	v >>= last_next_bit;
	if ((i = (v & 0xFFFF)) != 0xFFFF) {
		result = (min_0_array[i] + last_next_bit);
	}
	else {
		if ((i = (v & 0xFFFF0000)) != 0xFFFF0000) {
			result = (min_0_array[(i>>16) &  0xFFFF] + 16 + last_next_bit);
		}
		else {
			j = ((v >> 32) & 0xFFFFFFFF);
			if ((i = (j & 0xFFFF)) != 0xFFFF) {
					result = (min_0_array[i] + 32 + last_next_bit);
			}
			else {
				if ((i = (j & 0xFFFF0000)) != 0xFFFF0000) {
					result = (min_0_array[(i>>16) &  0xFFFF] + 48 + last_next_bit);
				};
			};
		};
	};
	next_bit_count++;
	if (next_bit_count == 0ll) {
		error("min_0_calc","next_bit_count overflow");
	};
	if ((result - last_next_bit) > 17) {
		next_bit_count16++;
		if (next_bit_count16 == 0ll) {
			error("min_0_calc","next_bit_count16 overflow");
		};
	};
	return(result);
};


char min_0_calc_TEST07 (FIELD v, char last_next_bit) {
	int j;
	char result;

	if (last_next_bit < 32) {
		j = (v & 0xFFFFFFFF);
		if ((j & 0x0000FFFF) != 0x0000FFFF) {
			result = (min_0_array[j & 0xFFFF]);
		}
		else {
			result = (min_0_array[(j >> 16) & 0xFFFF]+16);
		};
	}
	else {
		j = ((v >> 32) & 0xFFFFFFFF);
		if ((j & 0x0000FFFF) != 0x0000FFFF) {
			result = (min_0_array[j & 0xFFFF]+32);
		}
		else {
			result = (min_0_array[(j >> 16) & 0xFFFF]+48);
		};
	};
	next_bit_count++;
	if (next_bit_count == 0ll) {
		error("min_0_calc","next_bit_count overflow");
	};
	if ((result - last_next_bit) > 17) {
		next_bit_count16++;
		if (next_bit_count16 == 0ll) {
			error("min_0_calc","next_bit_count16 overflow");
		};
	};
	return(result);
};

char min_0_calc_TEST08 (UFIELD uf, char last_next_bit) {
	union {
		unsigned int j;
		unsigned short int jt[2];
	} uj;
	char result;

	if ((uj.j = uf.vt[0]) != 0xFFFFFFFF) {
		if ((uj.jt[0] & 0x0000FFFF) != 0x0000FFFF) {
			result = (min_0_array[uj.jt[0]]);
		}
		else {
			result = (min_0_array[uj.jt[1]]+16);
		};
	}
	else {
		if (((uj.j = uf.vt[1]) & 0x0000FFFF) != 0x0000FFFF) {
			result = (min_0_array[uj.jt[0]]+32);
		}
		else {
			result = (min_0_array[uj.jt[1]]+48);
		};
	};
	next_bit_count++;
	if (next_bit_count == 0ll) {
		error("min_0_calc","next_bit_count overflow");
	};
	if ((result - last_next_bit) > 17) {
		next_bit_count16++;
		if (next_bit_count16 == 0ll) {
			error("min_0_calc","next_bit_count16 overflow");
		};
	};
	return(result);
};

char min_0_calc_TEST09 (FIELD v, int last_next_bit) {
	unsigned int i, j;
	unsigned int a, b, c, d;
	char result;

	j = (v & 0xFFFFFFFF);
	a = (min_0_array[j & 0xFFFF]);
	if (a != 16) {
		result = a;
	}
	else {
		b = (min_0_array[(j >> 16) & 0xFFFF]);
		if (b != 16) {
			result = b+16;
		}
		else {
			/* i = ((v >> 32) & 0xFFFFFFFF); /* */
			i = ((v >> 32)); /* */
			c = (min_0_array[i & 0xFFFF]);
			if (c != 16) {
				result = c+32;
			}
			else {
				d = (min_0_array[(i >> 16) & 0xFFFF]);
				if (d != 16)
					result = d+48;
				else
					result = 64;
			};
		};
	};
	next_bit_count++;
	if (next_bit_count == 0ll) {
		error("min_0_calc","next_bit_count overflow");
	};
	if ((result - last_next_bit) > 17) {
		next_bit_count16++;
		if (next_bit_count16 == 0ll) {
			error("min_0_calc","next_bit_count16 overflow");
		};
	};
	return(result);
};

char min_0_calc_TEST10 (FIELD v, int last_next_bit) {
    union {
        float fnum;
        unsigned int inum;
    } f;
    int r;

    if (v) {
        f.fnum = (float)v;
        r = ((f.inum >> 23) & 0xFF) - 127;
        return(r < 24? r + 1: r);
    }
    else {
        return(0);
    };
};

int max_bit_calc (FIELD v) {
	int i;
	FIELD mask;
	
	mask = (FIELD)1;
	for (i = 1; i < N_BITS; i++) {
		mask <<= 1;
	};
	for (i = N_BITS - 1; i > -1; i--) {
		if ((v & mask) != (FIELD)0) {
			return(i);
		};
		mask >>= 1;
	};
	return(-1);
};

int max_0_calc (FIELD v) {
	int i;
	FIELD mask;
	
	mask = (FIELD)1;
	for (i = 1; i < N_BITS; i++) {
		mask <<= 1;
	};
	for (i = N_BITS - 1; i > -1; i--) {
		if ((v & mask) == (FIELD)0) {
			return(i);
		};
		mask >>= 1;
	};
	return(-1);
};

int comp_func_min (const void  *e1, const void  *e2) {
	int result;

	result = (((PART_SORT *)e1)->min_bit) - (((PART_SORT *)e2)->min_bit);
	if (result == 0) {
		if ((((PART_SORT *)e1)->v) > (((PART_SORT *)e2)->v)) {
			result = 1;
		}
		else {
			result = -1;
		};
	};
	return(result);
};

void perm_sort (PART_PERM *p_perm) {
	PARTS_SORT p_sort;
	int j, max_j;
	int last_min_bit;

	/* init indexes */
	for (j = 0; j < N_BITS; j++) {
		p_perm->min_start[j] = p_perm->min_end[j] = -1;
	};
	/* sort min bits */
	max_j = p_perm->perm;
	for (j = 0; j < max_j; j++) {
		p_sort[j].v = p_perm->v[j];
		p_sort[j].min_bit = min_bit_calc(p_sort[j].v);
	};
	qsort(&p_sort[0], max_j, sizeof(p_sort[0]),comp_func_min);
	last_min_bit = 9999;
	for (j = 0; j < max_j; j++) {
		p_perm->v[j] = p_sort[j].v;
		if (last_min_bit != p_sort[j].min_bit) {
			if (j > 0) {
				p_perm->min_end[last_min_bit] = j;
			};
			last_min_bit = p_sort[j].min_bit;
			p_perm->min_start[last_min_bit] = j;
		};
	};
	p_perm->min_end[last_min_bit] = j;
};

void perm_precmp (PART_PERM *p_perm) {		/* pre compare parts */
	int i, j;
	int end;
	int next_bit;
	int next_index;
	FIELD mask;

	#if (C_BITS > 0)
	for (i = 0; i < C_BITS_COUNT; i++) {
		next_index = 0;
		for (next_bit = 0; next_bit < N_BITS; next_bit++) {
			mask = ((FIELD)i << (next_bit + 1));
			p_perm->cmin_start[i][next_bit] = -1;
			p_perm->cmin_end[i][next_bit] = -1;
			if ((end = p_perm->min_end[next_bit]) < 0) {
				continue;
			};
			for (j = p_perm->min_start[next_bit]; j < end; j++) {
				if (((p_perm->v[j] & cmask_array[next_bit]) & mask) == (FIELD)0) {
					if (p_perm->cmin_start[i][next_bit] < 0) {
						p_perm->cmin_start[i][next_bit] = next_index;
					};
					if (p_perm->cmin_end[i][next_bit] < 0) {
						p_perm->cmin_end[i][next_bit] = next_index + 1;
					}
					else {
						p_perm->cmin_end[i][next_bit]++;
					};
					p_perm->cv[i][next_index] = p_perm->v[j];
					next_index++;
				};
			};
		};
	};
	#endif


	#if (0 == 1)
	/* test output */
	printf("Part: %c\n", p_perm->name);
	for (next_bit = 0; next_bit < N_BITS; next_bit++) {
		printf("x %02d %3d %3d ", next_bit, p_perm->min_start[next_bit], p_perm->min_end[next_bit]);
		printf("\n");
	};
	for (i = 0; i < C_BITS_COUNT; i++) {
		for (next_bit = 0; next_bit < N_BITS; next_bit++) {
			mask = ((FIELD)i << (next_bit + 1));
			printf("%01X ", i);
			printf("%02d %3d %3d ", next_bit, p_perm->cmin_start[i][next_bit], p_perm->cmin_end[i][next_bit]);
			printf("%016llX ", mask);
			printf("\n");
			if ((end = p_perm->cmin_end[i][next_bit]) < 0) {
				continue;
			};
			for (j = p_perm->cmin_start[i][next_bit]; j < end; j++) {
				if ((p_perm->cv[i][j] & mask) != (FIELD)0) {
					printf("Z");
				};
			};
			printf("\n");
		};
	};
	printf("\n\n");
	#endif
};

FIELD part_normalize (FIELD v) {
	FIELD result;
	int i;
	int x, y, z;
	int max_x, max_y, max_z;
	int min_x, min_y, min_z;

	result = (FIELD)0;

	box_get_min(v, &min_x, &min_y, &min_z);
	box_get_max(v, &max_x, &max_y, &max_z);

	for (x = min_x; x <= max_x; x++) {
		for (y = min_y; y <= max_y; y++) {
			for (z = min_z; z <= max_z; z++) {
				i = z * MAX_X * MAX_Y + y * MAX_X + x;
				if (v & mask_array[i]) {
					i = (z-min_z) * MAX_X * MAX_Y + (y-min_y) * MAX_X + (x-min_x);
					result |= mask_array[i];
				};
			};
		};
	};

	return(result);
};

void parts_get (SOLUTION s) {
	FIELD f;
	int i, j, k;
	int max_k;

	for (i = 0; i < N_PARTS; i++) {
		j = s[i].name - 'A';
		f = part_normalize(s[i].v);
		max_k = parts_perm[j].perm;
		for (k = 0; k < max_k; k++) {
			if (parts_perm[j].v[k] == f) {
				(parts_perm[j].count[k])++;
				break;
			};
		};
	};
};

void parts_count_print (void) {
	SOLUTION s;
	int i, j;
	int max_j;

	for (i = 0; i < N_PARTS; i++) {
		max_j = parts_perm[i].perm;
		for (j = 0; j < max_j; j++) {
			if (parts_perm[i].count[j] > 0) {
				solution_init(s);
				s[0].name = parts_perm[i].name;
				s[0].v = parts_perm[i].v[j];
				printf("part count = %d\n", parts_perm[i].count[j]);
				solution_print(s, MODE);
			};
		};
	};
};

void solve (FIELD f, SOLUTION s, int level, int last_next_bit);

#if (C_BITS > 0)
void solve (FIELD f, SOLUTION s, int level, int last_next_bit) {
	int i, j;
	int end;
	int next_bit;
	int bits;
	FIELD *p;

	/* field_print(f); /* */
	next_bit = min_0_calc(f,last_next_bit);
	bits = (f & cmask_array[next_bit]) >> (next_bit + 1);
	for (j = 0; j < N_PARTS; j++) {
		if (parts_perm[j].used != 0) {
			continue;
		};
		if ((end = parts_perm[j].cmin_end[bits][next_bit]) < 0) {
			continue;
		};
		for (i = parts_perm[j].cmin_start[bits][next_bit], p = &parts_perm[j].cv[bits][i]; i < end; i++, p++) {
			test_count++;
			if (f & *p) {
				continue;
			};
			test_count1++;
			parts_perm[j].used = 1;				/* mark part as used */
			#if (MODE != 0)
			s[level].v = *p;
			s[level].name = parts_perm[j].name;	/* update solution */
			#endif
			/* printf("%2d  ", level); /* */
			/* printf("level = %2d, next_bit = %2d\n", level, next_bit); /* */
			/* solution_print(&s[0],MODE); /* */
			/* field_print(f1); /* */
			if (level == (N_PARTS - 1)) {
				solution_count++;
				/* if ((solution_count % 100) == 0) {
					printf("%llu\n", solution_count);
				};
				if (solution_count > 1000ll) {
					exit(99);
				}; /* */
				#if (MODE != 0)
	   			solution_print(s,MODE); /* */
				#endif
			}
			else {
				solve(f | *p, s, level + 1,next_bit);
			};
			parts_perm[j].used = 0;				/* mark part as unsed */
		};
	};
};

#else

void solve (FIELD f, SOLUTION s, int level, int last_next_bit) {
	int i, j;
	int end;
	int next_bit;
	FIELD *p;

	/* field_print(f); /* */
	next_bit = min_0_calc(f,last_next_bit);
	for (j = 0; j < N_PARTS; j++) {
		if (parts_perm[j].used != 0) {
			continue;
		};
		if ((end = parts_perm[j].min_end[next_bit]) < 0) {
			continue;
		};
		for (i = parts_perm[j].min_start[next_bit], p = &parts_perm[j].v[i]; i < end; i++, p++) {
			test_count++;
			if (f & *p) {
				continue;
			};
			test_count1++;
			parts_perm[j].used = 1;				/* mark part as used */
			#if (MODE != 0)
			s[level].name = parts_perm[j].name;	/* update solution */
			s[level].v = *p;
			#endif
			/* printf("%2d  ", level); /* */
			/* printf("level = %2d, next_bit = %2d\n", level, next_bit); /* */
			/* solution_print(&s[0],MODE); /* */
			/* field_print(f1); /* */
			if (level == (N_PARTS - 1)) {
				solution_count++;
				/* if ((solution_count % 100) == 0) {
					printf("%llu\n", solution_count);
				};
				if (solution_count > 1000ll) {
					exit(99);
				}; /* */
				#if (MODE != 0)
	   			solution_print(s,MODE); /* */
				#endif
			}
			else {
				solve(f | *p, s, level + 1,next_bit); /* */
			};
			parts_perm[j].used = 0;				/* mark part as unsed */
		};
	};
};

#endif


void out_close(void) {
	if (out != (FILE *)NULL) {
		fclose(out);
	};
};

void in_close(void) {
	if (in != (FILE *)NULL) {
		fclose(in);
	};
};

void next_bit_count_print (void) {
	printf("next_bit_count = %llu\n", next_bit_count);
	printf("next_bit_count16 = %llu\n", next_bit_count16);
};

void stats_print (void) {
	printf("\nLaufzeit: %.2fs \n",(double)(clock()-start_clock)/CLOCKS_PER_SEC);
	printf("\nTests: %llu \n",test_count);
	printf("\nTest1: %llu \n",test_count1);
};

void puzzle (void) {
	SOLUTION s;
	int i, j;
	FIELD f;

	atexit(next_bit_count_print);

	/* f = (FIELD)0;
	j = -1;
	for (i = 0; i < 66; i++) {
		printf("%2d: %016llX, %2d\n", i, f, min_0_calc((FIELD)f,i-1));
		j = 0;
		f = (f << 1) + 1;
	};
	/* return; /* check min_0_calc */


	/* sort permutated parts by used min bit */
	for (j = 0; j < N_PARTS; j++) {
		perm_sort(&parts_perm[j]);
		perm_precmp(&parts_perm[j]);
	};

/* return; /* */

	/* solve the puzzle */
	solution_init(s);
	f = (FIELD)0;
	solution_count = 0;
	for (j = 0; j < N_PARTS; j++) {
		parts_perm[j].used = 0;
	};
	solve(f,s,0,-1);
	printf("\nSolutions: %llu\n", solution_count);
};

void sol_print (void) {
	SOLUTION s;

	while (solution_read(s) > 0) {
		solution_print(s,MODE); /* */
		/* solution_rotate(s); /* rotate solutions */
		/* parts_get(s); /*  count normalized parts */
	};
	parts_count_print();
};

int main(int argc, char *argv[]) {
	int j;

	start_clock = clock();
	test_FIELD();			/* ensure that FIELD has enough bits */

	mask_array_init();
	cmask_array_init();
	min_0_array_init();

	if ((prg = strrchr(argv[0],'/')) != NULL) {
		prg++;			/* remove pathname from 1. argument (prg name) */
	}
	else {
		prg = argv[0];
	};

	#if ((MODE & MODE_FILE) != 0)
	s_time = time((time_t) NULL);
	if (strftime(&s_time_str[0], TIME_STR, "%Y%m%d_%H%M%S", localtime(&s_time))) {
		/* printf("%s\n", s_time_str); /* */
	};
	if ((strlen(prg)+strlen(s_time_str)+strlen(ext)+2+1) < FILENAME_MAX) {
		strcpy(fname,prg);
		strcat(fname,".");
		strcat(fname,s_time_str);
		strcat(fname,".");
		strcat(fname,ext);
		printf("fname = %s\n", fname);
	};
	
	if ((out = fopen(fname, "w")) != (FILE *)NULL) {
		atexit(out_close);
	}
	else {
		error("main","can not open file");
	};
	#endif

	if (strcmp(prg,"sol_print") == 0) {
		for (j = 0; j < N_PARTS; j++) { /* */
		/* for (j = 0; j < 1; j++) { /* */
		/* for (j = 7; j < 8; j++) { /* */
			part_permutate(init_parts[j], &parts_perm[j]);
			part_print(&parts_perm[j]); /* */
		};

/* return(99); /* */

		if (argc < 2) {
			error("main", "parameter missing");
		};
		if ((in = fopen(argv[1], "r")) != 0) {
			atexit(in_close);
		}
		else {
			error("main", "can not open file");
		};
		sol_print();
		return(0);
	};

	if (argc >= 2) {
		reduce_name = *argv[1];
	};
	printf("reducing part: %c\n", reduce_name);
/* return(99); /* */
	for (j = 0; j < N_PARTS; j++) { /* */
	/* for (j = 0; j < 1; j++) { /* */
	/* for (j = 7; j < 8; j++) { /* */
		part_permutate(init_parts[j], &parts_perm[j]);
		part_print(&parts_perm[j]); /* */
	};

/* return(99); /* */

	atexit(stats_print);

	puzzle();
	return(0);
};
