        page    132,60
;****************************************************************************
;       60HZDEMO.ASM
;       Example program to demonstrate the difference between 24-bit data
;       capabilities and 16-bit data capabilities
;
;       Copywrite (c) MOTOROLA 1994
;                     Semiconductor Products Sector 
;                     Digital Signal Processing Division
;
;       ver. 1.1 3/16/95 Placed coefficients on modulo boundaries.  Moved the 
;                        "state" variable locations. Set sine amplitude to 0.3.
;
;******************************************************************************
; This code demonstrates the benefits of a 24-bit architecture over that of
; a 16-bit architecture.  The demonstration runs on the DSP56002EVM evaluation
; module.  An external audio signal is input through the microphone connector
; on the EVM.  A 60 Hz tone is generated by the DSP (via a digital oscillator)
; and then added to the digitized audio signal.  The resulting data is then
; sent through a filter with one of three sets of coefficients.  The first
; set of coefficients is located at location y:no_filter and performs no
; filtering at all, simply allowing the corrupted signal to pass.  This first
; set of coefficients are used by the filter when the demo first begins or
; following the non-maskable interrupt (NMI).  The second set of coefficients
; (located at y:coef_24) are 24-bit coefficients that make up a 60 Hz notch
; filter which removes the 60 Hz portion of the corrupted signal.  This second
; set of coefficients are used following the external interrupt A (IRQA).  The
; final set of coefficients (located at y:coeff_16) are the same coefficients
; as those for the 24-bit filter, only rounded to 16 bits.  These 16-bit
; coefficients are used by the filter following the external interrupt B 
; (IRQB).
;******************************************************************************

Fs      set     48000.0                         ;Specify sampling frequency.
PI      set     2.0*@asn(1.0)                   ;Compute PI as 2.0*arcsin(1.0)
factor  set     PI/180.0                        ;Multiplier for degrees to radians
eighteen set    18
hex_twenty  set   $20

;******************************************************************************
; Specification for tone a.
freq_a  set     60.0                        ;Specify frequency in Hertz.
phi_a   set     360.0*(freq_a/Fs)               ;Compute phi
phase_a set     0.0                             ;Specify the phase angle in 
                                                ; degrees (-180 -- +180).
amp_a   set     0.3                             ;Specify amplitude (0-1).
theta2_a        set     (phase_a-(2.0*phi_a))   ;Compute theta2
theta1_a        set     (phase_a-phi_a)         ;Compute theta1
s2_a    set     amp_a*@sin(factor*theta2_a)     ;Compute s2
s1_a    set     amp_a*@sin(factor*theta1_a)     ;Compute s1
coeff_a set     @cos(factor*phi_a)              ;Compute rcoef in 2:14 format


;******************************************************************************
; These three interrupts (IRQA, IRQB, and NMI) load the program memory location
; of the appropriate filter routine (24-bit, 16-bit, or NMI).
; 
        org     p:$8                    ; IRQA--Filter in 24 bit mode.
        move    #coef_24+1,r4           ; Load code pointer.
        org     p:$a                    ; IRQB--Filter in 16 bit mode.
        move    #coef_16+1,r4           ; Load code pointer.
        dup  eighteen
        nop
        ENDM
        org     p:$1e                   ; NMI--no filtering
        move    #no_filter+1,r4         ; Load code pointer.
        dup  hex_twenty
        nop
        ENDM

;******************************************************************************

        include 'codec.asm'

;******************************************************************************
TONE_OUTPUT     EQU     HEADPHONE_EN+LINEOUT_EN+(4*LEFT_ATTN)+(4*RIGHT_ATTN)   
TONE_INPUT      EQU     MIC_IN_SELECT+(15*MONITOR_ATTN)
nsec   equ     3               ;number of second order sections
scount equ     1               ;final shift count

;******************************************************************************
       org     x:      
DOSC_BUFF_BASE  EQU     *
coeff           ds      1       ; data location for osc. a's coeff.
s1              ds      1       ; data location for osc. a's sr1.
s2              ds      1       ; data location for osc. a's sr2.
LEFT_HUM        ds      1       ;storage for Left Signal + Hum
RIGHT_HUM       ds      1       ;storage for Left Signal + Hum
        org     x:$10
state1          dsm     nsec
        org     x:$1c
state1_r        dsm     nsec
        org     x:$20
state2          dsm     nsec
        org     x:$2c
state2_r        dsm     nsec

;******************************************************************************
       org     y:
coef_24            
       dc    $3FFD61        ;b(*,0)/2    =0.49992001      section number  1
       dc    $800641        ;b(*,1)/2    =-.99980915      section number  1
       dc    $7FF9C0        ;a(*,1)/2    =0.99980927      section number  1
       dc    $3FFD61        ;b(*,2)/2    =0.49992001      section number  1
       dc    $C0053E        ;a(*,2)/2    =-.49984002      section number  1
       dc    $3E3C48        ;b(*,0)/2    =0.48621464      section number  2
       dc    $83886C        ;b(*,1)/2    =-.97239923      section number  2
       dc    $7FFDB8        ;a(*,1)/2    =0.99993038      section number  2
       dc    $3E3C48        ;b(*,2)/2    =0.48621464      section number  2
       dc    $C0013F        ;a(*,2)/2    =-.49996197      section number  2
       dc    $20E7A1        ;b(*,0)/ 4   =0.25706875      section number  3
       dc    $BE3142        ;b(*,1)/ 4   =-.51412177      section number  3
       dc    $7FFDCE        ;a(*,1)/ 2   =0.99993300      section number  3
       dc    $20E7A1        ;b(*,2)/ 4   =0.25706875      section number  3
       dc    $C00136        ;a(*,2)/ 2   =-.49996305      section number  3
       dc    0

coef_16
       dc    $3FFD00        ;b(*,0)/2    =0.49991         section number  1
       dc    $800600        ;b(*,1)/2    =-.99982         section number  1
       dc    $7FFA00        ;a(*,1)/2    =0.99982         section number  1
       dc    $3FFD00        ;b(*,2)/2    =0.49991         section number  1
       dc    $C00500        ;a(*,2)/2    =-.49985         section number  1
       dc    $3E3C00        ;b(*,0)/2    =0.48621         section number  2
       dc    $838800        ;b(*,1)/2    =-.97241         section number  2
       dc    $7FFE00        ;a(*,1)/2    =0.99994         section number  2
       dc    $3E3C00        ;b(*,2)/2    =0.48621         section number  2
       dc    $C00100        ;a(*,2)/2    =-.49997         section number  2
       dc    $20E800        ;b(*,0)/ 4   =0.25708         section number  3
       dc    $BE3100        ;b(*,1)/ 4   =-.51413         section number  3
       dc    $7FFE00        ;a(*,1)/ 2   =0.99994         section number  3
       dc    $20E800        ;b(*,2)/ 4   =0.25708         section number  3
       dc    $C00100        ;a(*,2)/ 2   =-.49997         section number  3
       dc    0

no_filter
       dc    $400000        ;b(*,0)/2    =0.50000000      section number  1
       dc    $000000        ;b(*,1)/2    =0.00000000      section number  1
       dc    $000000        ;a(*,1)/2    =0.00000000      section number  1
       dc    $000000        ;b(*,2)/2    =0.00000000      section number  1
       dc    $000000        ;a(*,2)/2    =0.00000000      section number  1
       dc    $400000        ;b(*,0)/2    =0.50000000      section number  2
       dc    $000000        ;b(*,1)/2    =0.00000000      section number  2
       dc    $000000        ;a(*,1)/2    =0.00000000      section number  2
       dc    $000000        ;b(*,2)/2    =0.00000000      section number  2
       dc    $000000        ;a(*,2)/2    =0.00000000      section number  2
       dc    $200000        ;b(*,0)/ 4   =0.25000000      section number  3
       dc    $000000        ;b(*,1)/ 4   =0.00000000      section number  3
       dc    $000000        ;a(*,1)/ 2   =0.00000000      section number  3
       dc    $000000        ;b(*,2)/ 4   =0.00000000      section number  3
       dc    $000000        ;a(*,2)/ 2   =0.00000000      section number  3

left_a          ds     1
left_y0         ds     1
right_a         ds     1
right_y0        ds     1



;******************************************************************************
        org     p:
main
; Initialize SC0, SC1 as GPIO inputs    
    bclr    #0,x:PCC                 ; SC0 - GPIO (PC0)
    bclr    #1,x:PCC                 ; SC1 - GPIO (PC1)
    bclr    #0,x:PCDDR               ; SC0 (PC0) - input
    bclr    #1,x:PCDDR               ; SC1 (PC1) - input
    

; Initialize Filter Parameters    
    move    #state1,r3                ;point to filter state1
    move    #state2,r1                ;point to filter state2
    move    #no_filter,r4             ;Initialize for unity filter.
    move    #5*nsec-1,m4              ;addressing modulo 5*nsec
    clr     a                         ;initialize internal state storage
    move    a,y:left_a
    move    a,y:right_a
    rep     #4                   ;*    zero state1        (BAK)
    move    a,x:(r3)+                 ;*
    rep     #4                   ;*    zero state2        (BAK)
    move    a,x:(r1)+                 ;*
    move    #state1_r,r3                ;point to filter state1  (BAK)
    move    #state2_r,r1                ;point to filter state2  (BAK)
    rep     #4                   ;*    zero state1_r            (BAK)
    move    a,x:(r3)+                 ;*                          (BAK)
    rep     #4                   ;*    zero state2_r            (BAK)
    move    a,x:(r1)+                 ;*                          (BAK)
    move                  y:(r4)+,y0  ;a must be initially zero ,y0=b10/2
    move    y0,y:left_y0
    move    y0,y:right_y0
        move    #filter,r7              ; Load filter location.
        
; Initialize Digital Oscillator Parameters        
        movep   #$fb00,x:CRB            ; Enable the SSI interrupts.
        move    #coeff_a,x0
        move    x0,x:DOSC_BUFF_BASE        ; Load coeff. for osc. a.
        move    #s1_a,x0
        move    x0,x:DOSC_BUFF_BASE+1      ; Load s1 for osc. a.
        move    #s2_a,x0
        move    x0,x:DOSC_BUFF_BASE+2      ; Load s2 for osc. a.

; Main Loop
loop_1 
        movep   #$303f,x:IPR            ; IRQA/IRQB/SSI are level 3 interrupts. edge sensitive
        jset    #2,x:SSISR,*            ; Wait for frame sync to pass.
        jclr    #2,x:SSISR,*            ; Wait for frame sync.
        
        move    #TONE_OUTPUT,y0         ; headphones, line out, mute spkr, no attn.
        move            y0,x:TX_BUFF_BASE+2
        move    #TONE_INPUT,y0          ; no input gain, monitor mute
        move            y0,x:TX_BUFF_BASE+3
        move    #DOSC_BUFF_BASE,r5      ; Load pointer to osc's coeff, sr1 and sr2
        jsr     dosc_sin                ; Call oscillator routine.
        move    a,x0                    ; Copy new tone into x0.
        move    x:RX_BUFF_BASE,a        ; Load the left channel input.
        add     x0,a                    ; Add the tone.
        move    a,x:LEFT_HUM
        move    x:RX_BUFF_BASE+1,b      ; Load the right channel input.
        add     x0,b                    ; Add the tone.
        move    b,x:RIGHT_HUM
        
        move    #state1,r3
        move    #state2,r1
        jsr     bst60ev6_l              ;bandstop filter at 60 Hz (Elliptical)
        move    a,x:TX_BUFF_BASE        ; Put value in left channel tx.
        move    #state1_r,r3
        move    #state2_r,r1
        jsr     bst60ev6_r              ;bandstop filter at 60 Hz (Elliptical)
        move    b,x:TX_BUFF_BASE+1      ; Put value in right channel tx.  
        
        jmp     loop_1                  ; Loop back.


;*************************************************************************************

;---------------------------------------------------------------------------


;---------------------------------------------------------------------------
; The following subroutine calculates the next sinusoidal output value as a 
; function by the digital oscillator given that r5 points to the memory 
; location that contains the "coeff" value followed by the memory location 
; that contains the "s1" value, followed by the memory location that contains
; the "s2" value.  The formula and block diagram of the oscillator are:
;
;               s1[n] = coeff*s1[n-1] - s2[n-1] = coeff*s1[n-1] - s1[n-2]
;
;                 _______         _______
;                |       |  s1   |       |  s2
;           +--->|  z^-1 |--+--->|  z^-1 |----+
;           |    |_______|  |    |_______|    |
;           |               |                 |
;           |            ___V___           ___V___
;           |           |       |         |       |
;           |           | coeff |         |  -1   |  
;           |           |_______|         |_______|
;           |               |                 |
;           |               |                 |
;           |               |                 |
;           |               +----->( + )<-----+
;           |                        |
;           |                        |
;           +------------------------+---------> sine output
dosc_sin

        move    x:(r5)+,x0              ; Load coeff
        move    x:(r5)+,y0              ; Load s1 into a.
        move    x:(r5)-,b               ; Load s2 into b, r5 points to s1.
        mpy     x0,y0,a                 ;Get coef*s1 in a in 2:14 format
        subl    b,a          ;y0,b      ;Get (coef*s1 -s2)=sin_val in a
                                        ;  in fractional format
        move    a,x:(r5)+               ; Save new s1.
        move    y0,x:(r5)               ; Save new s2.
        rts
        
;BST60EL6

;
; This code segment implements cascaded biquad sections in transpose form
; The "bit_24" section of the code implements the filter with 24 bit data 
; and 24 bit coefficients.  The "bit_16" section of the code implements the 
; filter with coefficients that are truncated to 16 bits.  The "bit_16" code
; also truncates the data to 16 bits before writing any accumulator value
; to memory.  This "bit_16" code is to simulate the performance of a 16 bit 
; processor implementing the same filter.  In actuality, the filtering 
; performed by "bit_16" will yield a better performance than that which
; a 16 bit processor would yield.
; 
;

;
;  multiple shift left macro
;
mshl       macro    scount,acc
           if       scount
           rep      #scount
           asl      acc                
           endif
           endm



bst60ev6_l                              ;Bandstop filter at 60Hz (Elliptical
        ori     #$08,mr                 ;set scaling mode
        move    x:LEFT_HUM,y1           ;load left signal + Hum
    
        move    y:left_a,a
        move    y:left_y0,y0
        jsr     (r7)                    ; Call cascade biquad routine.
        move    y0,y:left_y0
        move    a,y:left_a
        move    y1,a
        mshl    scount,a                   ;bring gain back to 0 dB
        andi    #$f7,mr                   ;disable scaling mode?
        rts

bst60ev6_r                            ;Bandstop filter at 60Hz (Elliptical        
        ori     #$08,mr                   ;set scaling mode
        move    x:RIGHT_HUM,y1            ;load right signal + Hum
    
        move   y:right_a,a
        move   y:right_y0,y0
        jsr    (r7)                    ; Call cascade biquad routine.
        move   y0,y:right_y0
        move   a,y:right_a
        move   y1,b
        mshl   scount,b                   ;bring gain back to 0 dB
        andi    #$f7,mr                   ;disable scaling mode?
        rts
        

;     assumes each section's coefficients are divided by 2
;
filter  EQU     *
        do     #nsec,_end_filter                        ;do each section
        macr   y0,y1,a   x:(r1),b   y:(r4)+,y0          ;a=x(n)*bi0/2+wi1/2,b=wi2,y0=bi1/2
        asr    b         a,x0                           ;b=wi2/2,x0=y(n)
        mac    y0,y1,b              y:(r4)+,y0          ;b=x(n)*bi1/2+wi2/2,y0=ai1/2
        macr   x0,y0,b              y:(r4)+,y0          ;b=b+y(n)*ai1/2,y0=bi2/2
        mpy    y0,y1,b   b,x:(r3)+  y:(r4)+,y0          ;b=x(n)*bi2/2,save wi1,y0=ai2
        macr   x0,y0,b   x:(r3),a   a,y1                ;b=b+y(n)*ai2/2,a=next iter wi1,
                                        ;y1=output of section i
        asr    a         b,x:(r1)+  y:(r4)+,y0  ;a=next iter wi1/2,save wi2,
                                                ;y0=next iter bi0
_end_filter
        rts                                     ; Return from filter routine.
    end
