/* Display Routines for the CT-Player-Project,version 1.0.

	Copyright (C) 1999 Peter Santo, Marcus Schwatke, Andreas Kemper

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2, or (at your option)
	any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <conio.h>
#include <dos.h>
#include <stdio.h>

#include "parms.h"
#include "masisr.h"
#include "display.h"
#include "regio.h"

/* Element of the LCD-Queue */
typedef struct tag_lcdQueue_el
{
	char data;	/* Databyte to be sent to LCD */
	char reg;	/* Corresponding register */
} lcdQueue_el;

lcdQueue_el lcdQueue[LCDQUEUE_SIZE];	/* LCD-Queue */
unsigned int lcdQ_start, lcdQ_end;		/* Buffer position vars */
unsigned int lcdQ_empty, lcdQ_full;		/* Buffer status vars */

extern MASLPTPARAMS lptP;
extern lcdemu;

/*
Function     : Initializes LCD queue, clears CRT if necessary
Arguments    : none
Return Values: none
*/
void Displ_Init()
{
	lcdQ_start = 0;			/* Queue start position */
	lcdQ_end = 0;				/* Queue end position */
	lcdQ_empty = TRUE;		/* Queue is empty after startup */
	lcdQ_full = FALSE;		/* Queue isn't full at the beginning */

	/* Clear LCD by transmitting special sequence after short pause */
	delay(20);

	Reg_Write(DISPLAY_CMD, 0x30);
	Reg_Write(DISPLAY_CMD, 0x30);
	Reg_Write(DISPLAY_CMD, 0x30);
	Reg_Write(DISPLAY_CMD, 0x38);
	Reg_Write(DISPLAY_CMD, 0x08);
	Reg_Write(DISPLAY_CMD, 0x01);
	Reg_Write(DISPLAY_CMD, 0x06);
	Reg_Write(DISPLAY_CMD, 0x0d);

	/* Clear CRT */
	clrscr();

	/* Clear screen output if emulation mode is active */
	if(lcdemu == TRUE)
	{
		gotoxy(1, 1);
		printf("                ");
		gotoxy(1, 2);
		printf("                ");
		gotoxy(1, 3);
		printf("                ");
		gotoxy(1, 4);
		printf("                ");
	}
}

/*
Function    : Appends a character/register-pair to the the queue (if possible)
				  and sets global variables lcdQ_empty and lcdQ_full accordingly.
Arguments   : data: the byte to be enqueued.
				  reg : the register of the byte to be sent to the LCD
Return Value: SUCCESS on success, FAIL on failure (queue full)
Bugs fixed  : - Return value did not indicate success or failure correctly.
				  - Modulo operation on negative values may not be well defined.
				  - Function did not set lcdQ_empty=FALSE on enqueue.
*/
int Displ_Enqueue(char data, char reg)
{

	if(lcdQ_full==TRUE)
		return(FAIL);
	lcdQueue[lcdQ_end].data = Convert_Special_Char(data);
	lcdQueue[lcdQ_end].reg  = reg;
	lcdQ_end = ++lcdQ_end % LCDQUEUE_SIZE;
	lcdQ_empty = FALSE;

	/* Check if is buffer full */
	if(((lcdQ_end+LCDQUEUE_SIZE-lcdQ_start) % LCDQUEUE_SIZE) == (LCDQUEUE_SIZE-1))
		lcdQ_full = TRUE;

	return(SUCCESS);
}

/*
Function    : Calls WriteToLCD for queue entry (if possible) and sets
				  global variables lcdQ_empty and lcdQ_full accordingly.
Arguments   : none
Return Value: SUCCESS on success, FAIL on failure (queue empty)
Bugs fixed  : - Moved lcdQ_empty-check. The bug dequeued old entries!
				  - Return value did not indicate success or failure correctly.
*/
int Displ_Dequeue()
{
	if(lcdQ_empty==FALSE)
	{
		/* Disable INTs during output to be not interrupted by MP3-data */
		asm cli;

		/* Send byte to LCD */
		Reg_Write(lcdQueue[lcdQ_start].reg, lcdQueue[lcdQ_start].data);

		/* Enable INTs again */
		asm sti;

		lcdQ_start = ++lcdQ_start % LCDQUEUE_SIZE;
		if (lcdQ_start == lcdQ_end)
			lcdQ_empty = TRUE;
		lcdQ_full = FALSE;
		return(SUCCESS);
	}
	return(FAIL);
}

/* Write string to display queue and CRT if option is set */
int Display_String(int line, char *str)
{
	Set_Cursor(line, 0);
	if(lcdemu == TRUE)
	{
		gotoxy(1, line + 1);
		printf("%s", str);
	}
	while(*str)
		Displ_Enqueue(*str++, DISPLAY_DATA);
	return(SUCCESS);
}

/*
Function    : Sets the cursor to the given position.
Arguments   : line  : the line (0 - ?).
				  column: the column (0 - ?).
Return Value: ERR_DSPLRANGE or return value of Displ_Enqueue().
Comments    : This function will only work on a Hitachi 44780 compatible
				  display. While other functions may work on other controllers
				  as well, this one is unlikely to do so.
*/
void Set_Cursor(int line, int column)
{
	int l;

	switch(line) {
	case 0:
		l = 0;
		break;
	case 1:
		l = 64;
		break;
	case 2:
		l = 16;
		break;
	case 3:
		l = 80;
		break;
	default:
		l = 0;
		break;
	}
	Displ_Enqueue(128 + l + column, DISPLAY_CMD);
}

/*
Function    : Converts special characters like German Umlaute () into
				  the corresponding characters defined in the Hitachi charset.
Arguments   : A character to convert.
Return Value: The (possibly converted) character.
Notes       : Please report if you find characters that are missing or do
				  not get converted correctly.
*/
char Convert_Special_Char(char c){
		switch(c)
		{
			case 0xe4:		/* '' */
				return 0xe1;
			case 0xf6:		/* '' */
				return 0xef;
			case 0xfc:		/* '' */
				return 0xf5;
			case 0xc4:		/* '' */
				return 0xe1;
			case 0xd6:		/* '' */
				return 0xef;
			case 0xdc:		/* '' */
				return 0xf5;
			case 0xdf:		/* '' */
				return 0xe2;
			case '~':      /* Tilde->Infinity */
				return 0xf3;
			case '\\':		/* Backslash->Slash */
				return 0x2f;
		}
		return c;
}