/*
This source code is copyrighted Jan. 1999 by Peter Santo.
Please read the license (GPL) before using it, because it describes the
terms and conditions to use this source code.

In dieser Datei sind einige Funktionen zusammengefasst, die in der
einfachen Version des Players keine Verwendung finden, die jedoch dazu
dienen koennen, mehr aus dem Player (insbesondere aus dem Display)
herauszuholen.
Die Funktionen werden einfach mitgeliefert, ohne den Anspruch oder gar
das Versprechen, dass sie mit dem Rest auf Anhieb oder problemlos laufen.
Wer sich eintschliesst, sie zu verwenden, sollte zumindest wissen, wo
zu suchen ist, wenn der Compiler eine "unknown Symbol"-Fehlermeldung
ausgibt - weiterfuehrende Programmierkenntnisse sind von Vorteil.

Viel Spass damit.

Peter Santo
<peter.santo@post.rwth-aachen.de>

FUNCTIONS IN THIS FILE
======================

Clear_Display()                  Clears the display.

Cursor_Mode(blink)					Sets display of Cursor (CUR_OFF, CUR_ON
											or CUR_BLINK)

Display_nString(string, number)  Pushes NUMBER characters of STRING into
											queue, not stopping at a nullbyte.

Display_Line(line, string)       Queues STRING to display in line LINE and
											schedules scrolling if necessary.

Display_DoScroll()	            Displays scroll buffer shifted by 1.

Define_Char(nr, string)    Defines userdef character NR (0-7) using
									STRING (8 bytes).

MP3_Logo()                 Defines a "MP3"-Logo and queues it.

Init_Bignum()              Initializes the userdefined character set
									for Display_BigTime()

Display_BigTime(min, sec)  Displays a big time display in the form
									min:sec.

Display_Hand(value)        Shows one hand of a analog clock in the
									direction VALUE (0 = 12 o'clock, 0.25 =
									3 o'clock) using userdefined characters.

Display_ProgInd(line, col, size, alt, value)
									Displays a progress indicator in line
									LINE and column COL of the size SIZE.
									The fill level is determined by VALUE
									with 0 = empty, 1 = full.
									Normally, ALT is set to 0, but if you
									need to display a second progress
									indicator on the same display, set to 1.

Display_StdProgInd(line, value)
									Displays a progress indicator in line
									LINE with additional numeric display.
									The fill level is determined by VALUE
									with 0 = empty, 1 = full.

round(value)               Rounds the float VALUE to the nearest
									integer.

DB_Parse(field)            Parses a song database file and fills FIELD.
*/

#include <stdio.h>
#include <dos.h>
#include <string.h>
#include <time.h>
#include "lcdbasic.h"
#include "lpt_defs.h"
#include "displ_l.h"
#include "error.h"
#include <stdlib.h>
#include <conio.h>
#include "lcdmisc.h"
#include <math.h>




struct tagLCD_buf	LCD_buf;	// DEFINITION !

const	int fl[SONGDB_NUM_FIELDS] = {128, 64, 64, 64, 4, 64, 20, 3};	// field sizes

BYTE emu_gx, emu_gy;

clock_t scroll_timer;


/*
Function    : Clears the display.
Arguments   : -
Return Value: The return value of Displ_Enqueue().
*/
int Clear_Display(void) {
	return(Displ_Enqueue(1, DISPLAY_CONTROL));
}




/*
Function    : Sets cursor mode.
Arguments   : CUR_OFF, CUR_ON (a line) or CUR_BLINK (blinking block)
Return Value: The return value of Displ_Enqueue().
*/
int Cursor_Mode(BYTE blink)
{
	return(Displ_Enqueue(blink, DISPLAY_CONTROL));
}




/*
Function    : Defines the character nr in the set of userdefined characters.
				  These can be used by just displaying the character with the
				  ASCII value nr.
Arguments   : nr : The number of the character to define.
				  def: 8 bytes of data. First byte: top line, MSBit: right pixel.
Return Value: SUCCESS, ERR_DEFCHAR (wrong NR), ERR_DEFVAL (invalid data)
				  or the error code of Displ_Enqueue().
Todo        : This function assumes a character height of 8 pixels.
Comments    : This function changes the current cursor position.
*/
int Define_Char(BYTE nr, BYTE *def) {
  int i, ret;

	if(nr > 7)
		return(ERR_DEFCHAR);

	// Set CG RAM Address.
	if((ret = Displ_Enqueue(64 + 8 * nr, DISPLAY_CONTROL)) != SUCCESS)
		return(ret);

	// Enqueue data.
	for(i = 0; i < 8; i++)
	{
		if(def[i] > 31)
			return(ERR_DEFVAL);
		if((ret = Displ_Enqueue(def[i], DISPLAY_DATA)) != SUCCESS)
			return(ret);
	}
	return(SUCCESS);
}



/*
Function    : Displays the given string at the current cursor position. You
				  have to ensure, the string fits on the display, no check or
				  scrolling is done.
Arguments   : str: Pointer to string to display.
				  i  : Length of string (like strlen(str)).
Return Value: SUCCESS or the return value of Displ_Enqueue().
Notes       : For text to be scrolled use the function Display_Line().
				  If you provide a length bigger than the space allocated for
				  the string, the function will crash. Sorry about that, but
				  there is no way to detected this error.
Comments    : This function is provided to enable display of strings
				  containing a '\0' character (userdef character).
*/
int Display_nString(BYTE *string, BYTE j)
{
	int ret;
	while(j--)
		if((ret = Displ_Enqueue(*string++, DISPLAY_DATA)) != SUCCESS)
			return(ret);
	return(SUCCESS);
}



/*
Function    : Displays a big "MP3"-Logo on a 4-line display.
Arguments   : -
Return Value: -
Comments    : Note that all usedef characters are overwritten.
Todo        : Use return value.
*/
int MP3_Logo(){

//	if(DISPL_LINES < 4) return(ERR_DISPLRANGE);
	Clear_Display();
	Define_Char(0, "\037\037\037\017\007\007\003\001");
	Define_Char(1, "\000\000\000\020\030\034\034\036");
	Define_Char(2, "\000\000\000\001\003\007\007\017");
	Define_Char(3, "\037\037\037\036\034\034\030\020");
	Define_Char(4, "\037\037\037\037\037\000\000\000");
	Define_Char(5, "\030\034\036\036\037\037\037\037");
	Define_Char(6, "\037\037\037\037\036\036\034\030");
	Define_Char(7, "\000\000\000\037\037\037\037\037");
	Set_Cursor(0, 0);
	Display_nString("\377\000\001\002\003\377\024\377\004\004\005\024\004\004\004\005", 16);
	Set_Cursor(1, 0);
	Display_nString("\377\024\000\003\024\377\024\377\007\007\006\024\024\007\007\377", 16);
	Set_Cursor(2, 0);
	Display_nString("\377\024\024\024\024\377\024\377\024\024\034\024\024\024\024\377", 16);
	Set_Cursor(3, 0);
	Display_nString("\377\024\024\024\024\377\024\377\024\024\024\024\007\007\007\006", 16);
	return(SUCCESS);
}




/*
Function    : Initializes the userdef charset for display of big (3 lines)
				  numbers.
Arguments   : -
Return Value: Success or the return value of Define_Char().
Comments    : Previous userdef characters are overwritten.
*/
int Init_Bignum() {
	int ret;

	if((ret = Define_Char(0, "\016\016\016\016\016\016\016\016")) != SUCCESS)
		return(ret);
	if((ret = Define_Char(1, "\000\000\037\037\037\000\000\000")) != SUCCESS)
		return(ret);
	if((ret = Define_Char(2, "\000\000\034\036\036\016\016\016")) != SUCCESS)
		return(ret);
	if((ret = Define_Char(3, "\016\016\036\036\034\000\000\000")) != SUCCESS)
		return(ret);
	if((ret = Define_Char(4, "\000\000\007\017\017\016\016\016")) != SUCCESS)
		return(ret);
	if((ret = Define_Char(5, "\016\016\017\017\007\000\000\000")) != SUCCESS)
		return(ret);
	if((ret = Define_Char(6, "\016\016\036\036\036\016\016\016")) != SUCCESS)
		return(ret);
	if((ret = Define_Char(7, "\000\000\007\007\007\000\000\000")) != SUCCESS)
		return(ret);
	return(SUCCESS);
}



/*
Function    : Displays a big (3 line) time display in the form mm:ss.
Arguments   : min: Minutes
				  sec: Seconds
Return Value:
Comments    : You must call Init_Bignum() before calling Display_BigTime().
Todo        : Make code smaller, implement return values.
*/
int Display_BigTime(BYTE min, BYTE sec)
{
	BYTE min1, min2, sec1, sec2, lin;
BYTE big_num[10][3][3] = {
{{'\004', '\001', '\002'}, {'\000', '\024', '\000'}, {'\005', '\001', '\003'}},	// 0
{{'\024', '\002', '\024'}, {'\024', '\000', '\024'}, {'\024', '\005', '\024'}},	// 1
{{'\007', '\001', '\002'}, {'\004', '\001', '\003'}, {'\005', '\001', '\001'}},	// 2
{{'\007', '\001', '\002'}, {'\024', '\001', '\006'}, {'\007', '\001', '\003'}},	// 3
{{'\004', '\024', '\004'}, {'\005', '\001', '\006'}, {'\024', '\024', '\003'}},	// 4
{{'\004', '\001', '\024'}, {'\005', '\001', '\002'}, {'\007', '\001', '\003'}},	// 5
{{'\004', '\001', '\024'}, {'\000', '\001', '\002'}, {'\005', '\001', '\003'}},	// 6
{{'\007', '\001', '\002'}, {'\024', '\004', '\003'}, {'\024', '\003', '\024'}},	// 7
{{'\004', '\001', '\002'}, {'\000', '\001', '\006'}, {'\005', '\001', '\003'}},	// 8
{{'\004', '\001', '\002'}, {'\005', '\001', '\006'}, {'\007', '\001', '\003'}},	// 9
};


//	if(DISPLAY_LINES < 3) return(ERR_DISPLRANGE);

	min1 = min/10;
	min2 = min%10;
	sec1 = sec/10;
	sec2 = sec%10;

	Set_Cursor(0, 1);
	Displ_Enqueue(big_num[min1][0][0], 1);
	Displ_Enqueue(big_num[min1][0][1], 1);
	Displ_Enqueue(big_num[min1][0][2], 1);
	Displ_Enqueue(big_num[min2][0][0], 1);
	Displ_Enqueue(big_num[min2][0][1], 1);
	Displ_Enqueue(big_num[min2][0][2], 1);
	Displ_Enqueue('\240', 1);
	Displ_Enqueue('\241', 1);
	Displ_Enqueue(big_num[sec1][0][0], 1);
	Displ_Enqueue(big_num[sec1][0][1], 1);
	Displ_Enqueue(big_num[sec1][0][2], 1);
	Displ_Enqueue(big_num[sec2][0][0], 1);
	Displ_Enqueue(big_num[sec2][0][1], 1);
	Displ_Enqueue(big_num[sec2][0][2], 1);

	Set_Cursor(1, 1);
	Displ_Enqueue(big_num[min1][1][0], 1);
	Displ_Enqueue(big_num[min1][1][1], 1);
	Displ_Enqueue(big_num[min1][1][2], 1);
	Displ_Enqueue(big_num[min2][1][0], 1);
	Displ_Enqueue(big_num[min2][1][1], 1);
	Displ_Enqueue(big_num[min2][1][2], 1);
	Displ_Enqueue('\240', 1);
	Displ_Enqueue('\240', 1);
	Displ_Enqueue(big_num[sec1][1][0], 1);
	Displ_Enqueue(big_num[sec1][1][1], 1);
	Displ_Enqueue(big_num[sec1][1][2], 1);
	Displ_Enqueue(big_num[sec2][1][0], 1);
	Displ_Enqueue(big_num[sec2][1][1], 1);
	Displ_Enqueue(big_num[sec2][1][2], 1);

	Set_Cursor(2, 1);
	Displ_Enqueue(big_num[min1][2][0], 1);
	Displ_Enqueue(big_num[min1][2][1], 1);
	Displ_Enqueue(big_num[min1][2][2], 1);
	Displ_Enqueue(big_num[min2][2][0], 1);
	Displ_Enqueue(big_num[min2][2][1], 1);
	Displ_Enqueue(big_num[min2][2][2], 1);
	Displ_Enqueue('\240', 1);
	Displ_Enqueue('\337', 1);
	Displ_Enqueue(big_num[sec1][2][0], 1);
	Displ_Enqueue(big_num[sec1][2][1], 1);
	Displ_Enqueue(big_num[sec1][2][2], 1);
	Displ_Enqueue(big_num[sec2][2][0], 1);
	Displ_Enqueue(big_num[sec2][2][1], 1);
	Displ_Enqueue(big_num[sec2][2][2], 1);
	return(SUCCESS);
}




/*
Function    : Displays the given string in the given LCD line. If the string
				  does not fit into the line, the string is copied into the scroll
				  buffer and the scroll routine is scheduled for updates.
Arguments   : line  : The line to display the string in.
				  string: The string to be displayed.
Return Value: Not yet implemented.
Notes       : The functions assumes that the time is periodically checked and
				  Do_Scroll() is called when it is time to scroll the line.
Todo        : If the line has to be scrolled, there is no need to queue
				  anything here. Better just copy it to the scroll buffer and
				  schedule an immediate update. (Maybe Problem with END_DELAY).
*/
int Display_Line(BYTE line, BYTE *str)
{
	extern struct tagLCD_buf LCD_buf;
	extern clock_t scroll_timer;
	int sl, wrap_separator_length, col;

	Convert_Special(str);
	if((sl = strlen(str)) <= LCD_COLS) {	// line fits in LCD line.
		Set_Cursor(line, 0);
		Display_String(str);
	}

	else {                              // line must be scrolled.
		wrap_separator_length = strlen(WRAP_SEPARATOR);
		LCD_buf.pos[line]	=	0;
		#if(SCROLL_MODE == SCROLL_WRAP)
		sl +=	wrap_separator_length;
		#endif
		if(sl	> LCD_BUF_COLS) {
			return(ERR_2BIG4LINEBUF);
		}		// if very long line
		#if(SCROLL_MODE == SCROLL_WRAP)
		strcpy(LCD_buf.ch[line], str); // Put line in buffer
		strcat(LCD_buf.ch[line], WRAP_SEPARATOR); // Append separator
		#else
		strcpy(LCD_buf.ch[line], str);					// Put line in buffer
		#endif

		str[LCD_COLS] = '\0';	// Truncate the string.
		Set_Cursor(line, 0);
		Display_String(str);

		scroll_timer = clock() * 1000 / CLK_TCK +	SCROLL_DELAY;
		LCD_buf.delay[line]	=	SCROLL_DELAY_END_FACTOR;
														// Set timer for scroll updates
//		if(++LCD_buf.pos[line] > sl) { 		// Set next position to scroll from
			LCD_buf.pos[line]	%= sl;
//		}
	}	// Line must be scrolled
	return(SUCCESS);
}



/*
Function    : Does the scrolling using the scroll buffer
Arguments   : None
Return Value: None
Comment     : This function has to be called periodically in the main
				  loop of your player routine to do the updating (scrolling).
Todo        : SCROLL_MODE should be adjustable within the program.
*/
void Display_DoScroll(void)
{
	int line, col, sl;
	extern struct tagLCD_buf LCD_buf;
	#ifdef USE_EMU
	extern BYTE emu_gx, emu_gy;
	#endif

	scroll_timer += SCROLL_DELAY;
	for(line = 0;	line < LCD_LINES;	line++)
	{
		if((sl = strlen(LCD_buf.ch[line])) !=	0) 		// Line to scroll
		{
			if(LCD_buf.delay[line] > 0)
				(LCD_buf.delay[line])--;
			else
			{
				#if(SCROLL_MODE == SCROLL_WRAP)
				Set_Cursor(line, 0);
				#ifdef USE_LCD
				for(col = 0; col < LCD_COLS; col++)
					Displ_Enqueue(LCD_buf.ch[line][(LCD_buf.pos[line]+col) % sl], DISPLAY_DATA);
				#endif	// USE_LCD
				#ifdef USE_EMU
				for(col = 0; col < LCD_COLS; col++)
					EMU_PrintAt(emu_gy, emu_gx++, LCD_buf.ch[line][(LCD_buf.pos[line]+col) % sl]);
				#endif	// USE_EMU
				LCD_buf.pos[line]	+= LCD_buf.step[line];
				#endif	// SCROLL_WRAP
				#if(SCROLL_MODE == SCROLL_PING_PONG)
				Set_Cursor(line, 0);
				for(col	=	0; col < LCD_COLS; col++)
					Displ_Enqueue(LCD_buf.ch[line][LCD_buf.pos[line]+col], DISPLAY_DATA);
				LCD_buf.pos[line]	+= LCD_buf.step[line];
				if(LCD_buf.pos[line] < 0)
				{
					LCD_buf.step[line] = abs(LCD_buf.step[line]);
					LCD_buf.pos[line]	=	abs(LCD_buf.step[line]);
					LCD_buf.delay[line]	=	SCROLL_DELAY_END_FACTOR;
				}
				else if(LCD_buf.pos[line]	>	(sl	-	LCD_COLS))
				{
					LCD_buf.step[line] = (-1)	*	abs(LCD_buf.step[line]);
					LCD_buf.pos[line]	=	2	*	sl - LCD_buf.pos[line] - 2 * LCD_COLS;
					LCD_buf.delay[line]	=	SCROLL_DELAY_END_FACTOR;
				}
				#endif	// SCROLL_PING_PONG
			}
		}	// if(line to scroll)
	}	// for(line)
}




/*
Function    : This is a "fun" function not intended for serious use.
				  It shows one hand of a analog clock using userdefined
				  characters. Nice demonstration though.
Arguments   : phi: direction of arm, 0 = 12 o'clock, 0.25 = 3 o'clock, ..
Return Value: The respective return values of the called functions
				  Define_Char(), Set_Cursor() and Display_nString().
Comments    : The hand is displayed in 4 lines in the first 6 columns.
Bugs        : - Sometimes, the line has steps.
				  - No checks are performed whatsoever, so be careful.
*/
int Display_Hand(float phi)
{
	float r;
	int ret;
	BYTE sx, sy, x, y, z;
	BYTE pixel[35][35];
	BYTE segment[6][4][8];	// cols, lines, pixel lines
	BYTE used = 0, pins;
	BYTE line[6];

//	if(DISPLAY_LINES < 4) return(ERR_DISPLRANGE);

	for(x = 0; x < 35; x++)
		for(y = 0; y < 35; y++)
			pixel[x][y] = 0;
	for(x = 0; x < 6; x++)
		for(y = 0; y < 4; y++)
			for(z = 0; z < 8; z++)
				segment[x][y][z] = 0;
	line[6] = '\0';

	phi = phi * 2 * M_PI;	// M_PI = 3.141592...

	for(r = 0; r <= 16; r += H_RESOLUTION)
	{
		x = round(sin(phi) * r + H_CENTER_X);
		y = round(-cos(phi) * r + H_CENTER_Y);
		// use for 1-Pixel line
		pixel[x][y]++;
		// use for 2-Pixel line
		pixel[x][y+1]++;
		pixel[x+1][y]++;
		pixel[x+1][y+1]++;
/*		// use for 3-Pixel line
		pixel[x-1][y-1]++;
		pixel[x-1][y]++;
		pixel[x-1][y+1]++;
		pixel[x][y-1]++;
		pixel[x+1][y-1]++;
*/
	}

	for(sy = 0; sy <= 3; sy++)
	{
		for(sx = 0; sx <= 5; sx++)
		{
			pins = 0;
			for(y = 0; y < 8; y++)
			{
				for(x = 0; x < 5; x++)
				{
					if(pixel[x+6*sx][y+9*sy] > 2)	// Threshold
					{
						segment[sx][sy][y] += (1 << (4 - x));
						pins = 1;
					}
				}
			}

			if((pins == 1) && (used < 8))
			{
				if((ret = Define_Char(used, segment[sx][sy])) != SUCCESS)
					return(ret);
				line[sx] = used++;
			}
			else
				line[sx] = '\040';
		} // for sx
		if((ret = Set_Cursor(sy, 0)) != SUCCESS)
			return(ret);
		if((ret = Display_nString(line, 6)) != SUCCESS)
			return(ret);
	}	// for sy
	return(SUCCESS);
}



/*
Description : Displays a progress indicator of given position and size.
Arguments   : line : Line to draw progress indicator in
				  col  : The first column to use for progress indicator
				  size : Size of the progress indicator in fields
				  alt  : Set to 0 or 1, so two progress indicators on one
							display do not use the same userdef characters.
				  value: 0 = 0% (empty), 1 = 100% (full)
Return Value: Define_Char(), Displ_Enqueue().
Comments    : If you do repeated additions in a function calling this
				  function, assume EPSILON as the biggest possible error
				  in each addition.
Bugs        : The progress indicator does not maintain state, so
				  everything has to be drawn on every call, even if only
				  some pixels have changed (usually the case with progress
				  indicators). This uses some bandwidth on the display
				  interface (The Display_StdProgInd() call transmits 62
				  bytes to the LCD).
				  Note that in general, if you need to save bandwidth, it
				  is best to maintain the state of the display in your
				  software and send only necessary data.
*/
int Display_ProgInd(BYTE line, BYTE col, BYTE size, BYTE alt, float value)
{
	int i, j, pos = 0;
	float frac;
	BYTE def[8];
	BYTE ind[16];
	BYTE tmp;
	int px;

	if(
		(line > 3) || (col > 15) ||
		(size < 2) || (size + col > 16) ||
		(value < 0)|| (value > 1.0 + EPSILON * 100.0) || (alt > 1)
	  )
		return(ERR_PROGIND);

	px = size * 6 - 5;
	frac = round(value * (float)px);

	Define_Char(0, "\037\000\000\000\000\000\000\037");
	Define_Char(1, "\037\000\037\037\037\037\000\037");

	def[0] = def[7] = 31;
	def[1] = def[6] = 16;
	tmp = 16;
	for(i = 0; (i < 3) && (i < frac); i++)
		tmp += (1 << (2 - i));
	for(i = 2; i <= 5; i++)
		def[i] = tmp;
	frac -= 4;
	Define_Char(2 + alt, def);
	ind[pos++] = 2 + alt;

	for(i = 2; i < size; i++)
	{
		if(frac <= 0)
			ind[pos++] = 0;
		else if(frac > 5)
			ind[pos++] = 1;
		else
		{
			def[0] = def[7] = 31;
			def[1] = def[6] = 0;
			tmp = 0;
			for(j = 0; (j < 5) && (j < frac); j++)
				tmp += (1 << (4 - j));
			for(j = 2; j <= 5; j++)
				def[j] = tmp;
			Define_Char(4 + alt, def);
			ind[pos++] = 4 + alt;
		}
		frac -= 6;
	}

	def[0] = def[7] = 31;
	def[1] = def[6] = 1;
	tmp = 1;
	for(i = 0; (i < 3) && (i < frac); i++)
		tmp += (1 << (4 - i));
	for(i = 2; i <= 5; i++)
		def[i] = tmp;
	Define_Char(6 + alt, def);
	ind[pos] = 6 + alt;

	Set_Cursor(line, col);
	Display_nString(ind, size);

	return(SUCCESS);
}



/*
Description : A wrapper for Display_ProgInd() to create a standard
				  progress indicator plus numeric percentage display.
Arguments   : line : Line to draw progress indicator in
				  value: 0 = 0% (empty), 1 = 100% (full)
Return Value: (not done)
*/
int Display_StdProgInd(BYTE line, float value)
{
	char cent[5];

	if(Display_ProgInd(line, 0, 12, 0, value) == ERR_PROGIND)
		Set_Cursor(line, 12);
	sprintf(cent, "%3d%%", round(value * 100.0));
	return(Display_String(cent));
}



/*
Function    : Rounds the float A to the nearest integer.
Arguments   : A: a floating point number.
Return Value: The integer nearest to A
Comments    : Possibly, a function round() exists in some programming
				  environments, so the linker complains about a redefinition.
				  roud() is only needed by the progress indicator and
				  Display_Hand() functions.
*/
int round(float a)
{
	int b = floor(a);
	return(a - b < 0.5 ? b : b + 1);
}



/*
  DB_Parse parses a track database (described in the documentation) and
  returns the number of entries found. The pointer given as an argument
  points to an array of pointers to lines, which in turn point to pointers to
  the fields:

  field_ptr -> line1 -> field1
								field2
								field3
								.
								.

					line2 -> field1
								field2
								.
								.
					.
					.
This function is probably still quite buggy - use with care!
*/
int DB_Parse(char ****field_ptr)	// 4 asterisks... scary ;)
{
	FILE *dbfile;
	char *linebuf;
	char *tmp_field[SONGDB_NUM_FIELDS];
	int	i, ScanRet, sum = 0,	lencnt = 0, sl;
	int line, col;
	int ecol;
	char errorflag = 0;
	char *tmp_ptr;

// Calculate maximal line length.
	for(i = 0; i < SONGDB_NUM_FIELDS; i++)
	{
		sum += fl[i] + 1;
// Allocate space where the parsed fields can be stored temporarily.
		if((tmp_field[i] = (char*) malloc((fl[i] + 1) * sizeof(char))) == NULL)
			return(ERR_NOMEM);
		*tmp_field[i] = '\0';
	} // for

// Allocate a line buffer.
	linebuf = (char *) malloc(sizeof(char) * sum);

	if((dbfile = fopen(DATABASE_FILE, "r")) == NULL)
		return(ERR_OPENDB);
	line = 0;
	*field_ptr = (char***)malloc(sizeof(char**));

// Uh-oh! the "300" ist not a good idea, but I hadn't a better one. (todo).
	while(((ScanRet	=	fscanf(dbfile, "%300[^\n]\r\n",	linebuf)) !=	EOF)
		&&	(ScanRet !=	0))
	{
		col = 0;
		lencnt = 0;

// Allocate space for another line pointer.
		*field_ptr = (char***)realloc(*field_ptr, sizeof(char**) * (line+1));
// Alocate space for new column pointers.
		if((*(*field_ptr + line) = (char **)malloc(sizeof(char*) * SONGDB_NUM_FIELDS)) == NULL)
			return(ERR_NOMEM);

		tmp_ptr = tmp_field[0];
		while(*linebuf !=	'\0')
		{
			if(*linebuf	== '\t') 	// Next field.
			{
				if(errorflag) 				// Error occurred in previous field.
				{
					errorflag = 0;
				}	// if(errorflag)
				*tmp_ptr = '\0';
				col++;
				tmp_ptr = tmp_field[col];
				lencnt = 0;
			}	// if(*linebuf == '\t')
			else
			{
				if(lencnt++ < fl[col])
				{
					*tmp_ptr = *linebuf;
					tmp_ptr++;
				}
				else
				{
//					ecol = col;
					errorflag = 1;
				}
			}
			linebuf++;
		}	// while chars in line
		*tmp_ptr = '\0';

		if(errorflag)
		{
//			printf("Field #%d in line %d too long, truncated.\n", ecol, line);
			errorflag = 0;
		}


/* Ok, the fields of one line are read into tmp_field[].
	Now, let's allocate the minimum amount of memory for them and move
	them there.
*/

		for(i	=	0; i < SONGDB_NUM_FIELDS;	i++)
		{
			sl = strlen(tmp_field[i]);

// Alocate space for new fields.
			if((*(*((*field_ptr) + line) + i) =
				(char *)malloc(sizeof(char) * (sl + 1))) == NULL)
				return(ERR_NOMEM);

// Copy the field to the allocated area.
			strcpy(*(*(*field_ptr + line) + i), tmp_field[i]);
// Make sure strings are 0-terminated.
			*(*(*(*field_ptr + line) + i) + sl) = '\0';
		}	// for columns
		line++;
	} // while scanf

// Release memory chunks not needed any more.
	free(linebuf);
	for(i = 0; i < SONGDB_NUM_FIELDS; i++)
		free(tmp_field[i]);
	return(line);
}	// function

