package ioio.examples.hardware.hacks;

import ioio.examples.hello.R;
import ioio.lib.api.DigitalOutput;
import ioio.lib.api.PwmOutput;
import ioio.lib.api.exception.ConnectionLostException;
import ioio.lib.util.AbstractIOIOActivity;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.ServerSocket;
import java.net.Socket;

import org.json.JSONException;
import org.json.JSONObject;

import android.app.Service;
import android.os.Bundle;
import android.widget.EditText;
import android.widget.SeekBar;
import android.widget.TextView;
import android.widget.ToggleButton;

/**
 * This is the main activity of the Hardware Hacks example application.
 */
public class MainActivity extends AbstractIOIOActivity {
    private ToggleButton button_;
    private EditText editText_;
    private TextView textView_;
    private ToggleButton ledOnOffButton_;
    private ToggleButton motorOnOffButton_;

    private final int PAN_PIN = 6;
    private final int PWM_FREQ = 100;

    private SeekBar mPanSeekBar;
    private static boolean buildFailed = false;

    /**
     * Called when the activity is first created. Here we initialize our GUI.
     */
    @Override
    public void onCreate(final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);
        button_ = (ToggleButton) findViewById(R.id.button);
        editText_ = (EditText) findViewById(R.id.message);

        ledOnOffButton_ = (ToggleButton) findViewById(R.id.ledOnOffButton);
        motorOnOffButton_ = (ToggleButton) findViewById(R.id.motorOnOffButton);

        mPanSeekBar = (SeekBar) findViewById(R.id.panSeekBar);
        enableUi(false);

        // Here we start our Socket Server
        this.startSocketServer();
    }

    /**
     * This is the thread on which all the IOIO activity happens. It will be run
     * every time the application is resumed and aborted when it is paused. The
     * method setup() will be called right after a connection with the IOIO has
     * been established (which might happen several times!). Then, loop() will
     * be called repetitively until the IOIO gets disconnected.
     */
    class MyIOIOThread extends AbstractIOIOActivity.IOIOThread {
        private static final int LED_PIN = 3;
        private static final int LED_PIN2 = 11;

        private DigitalOutput led_;
        private DigitalOutput ledDiode_;
        private DigitalOutput ledDiode2_;

        private PwmOutput panPwmOutput;

        /**
         * Called every time a connection with IOIO has been established.
         * Typically used to open pins.
         * 
         * @throws ConnectionLostException
         *             When IOIO connection is lost.
         * 
         * @see ioio.lib.util.AbstractIOIOActivity.IOIOThread#setup()
         */
        @Override
        protected void setup() throws ConnectionLostException {
            led_ = ioio_.openDigitalOutput(0, true);
            ledDiode_ = ioio_.openDigitalOutput(LED_PIN, false);
            ledDiode2_ = ioio_.openDigitalOutput(LED_PIN2, false);

            try {
                panPwmOutput = ioio_.openPwmOutput(PAN_PIN, PWM_FREQ);
                enableUi(true);
            } catch (final ConnectionLostException e) {
                enableUi(false);
                throw e;
            }
        }

        /**
         * Called repetitively while the IOIO is connected.
         * 
         * @throws ConnectionLostException
         *             When IOIO connection is lost.
         * 
         * @see ioio.lib.util.AbstractIOIOActivity.IOIOThread#loop()
         */
        @Override
        protected void loop() throws ConnectionLostException {
            handleAttachedLED();
            handleServo();
        }

        private void handleServo() throws ConnectionLostException {
            if (motorOnOffButton_.isChecked() || buildFailed) {
                try {
                    panPwmOutput.setPulseWidth(500 + mPanSeekBar.getProgress() * 2);
                    sleep(10);
                } catch (InterruptedException e) {
                    ioio_.disconnect();
                } catch (ConnectionLostException e) {
                    enableUi(false);
                    throw e;
                }
            } else {
                panPwmOutput.setPulseWidth(0);
            }
        }

        private void handleAttachedLED() throws ConnectionLostException {
            if (ledOnOffButton_.isChecked() && buildFailed) {
                ledDiode_.write(true);
                ledDiode2_.write(true);
            } else {
                ledDiode_.write(false);
                ledDiode2_.write(false);
                buildFailed = false;
            }
        }
    }

    /**
     * A method to create our IOIO thread.
     * @see ioio.lib.util.AbstractIOIOActivity#createIOIOThread()
     */
    @Override
    protected AbstractIOIOActivity.IOIOThread createIOIOThread() {
        return new MyIOIOThread();
    }

    private void enableUi(final boolean enable) {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                mPanSeekBar.setEnabled(enable);
            }
        });
    }

    /**
     * Start a thread for the SocketServer to listen on port 30033.
     * */
    private void startSocketServer() {
        new Thread(new Runnable() {
            public void run() {
                try {
                    final ServerSocket socket = new ServerSocket(30033);
                    boolean listening = true;
                    while (listening) {
                        final Socket s = socket.accept();
                        final BufferedReader reader = new BufferedReader(new InputStreamReader(s.getInputStream()));
                        final String jenkinsMessage = reader.readLine();
                        if (null != jenkinsMessage) {
                            final JSONObject jsonMessage = new JSONObject(jenkinsMessage);
                            final JSONObject buildJson = jsonMessage.getJSONObject("build");
                            final String phase = buildJson.getString("phase");

                            String status = null;
                            if (!buildJson.isNull("status")) {
                                status = buildJson.getString("status");
                            }

                            if ("FAILURE".equals(status) && "FINISHED".equals(phase)) {
                                buildFailed = true;
                                System.out.println(jenkinsMessage);
                            }
                        } else {
                            System.out.println("Message must not be null.");
                        }
                    }
                } catch (final IOException e) {
                    e.printStackTrace();
                } catch (final JSONException e) {
                    e.printStackTrace();
                }
            }
        }).start();
    }
}