/*
 * Sehr einfaches Snake-Spiel
 * Florian SchÃ¤ffer, Redaktion Make: http://www.make-magazin.de/
 * VII/2016
 * 
 * Per Nunchuck kann die Schlange ueber das Display gesteuert werden.
 * Aepfel werden zufaellig gesetzt. Essendieser bringt extra Punkte.
 * Die Funktion, um den Displayinhalt zwischen zu speichern, um zu erkennen, 
 * wo die Schlange schon war, ist etwas trickreich.
 */


const uint8_t MIDDLE = 180;
const uint8_t XMAX = 140;
const uint8_t YMAX = 24;
const uint8_t LEVELSTART = 200;   // Pause zwischen zwei Bewegungsschritten zu Beginn in ms

uint8_t xpos;
uint8_t ypos;
int8_t xdir;
int8_t ydir;
uint8_t playzone[140][3];
uint8_t appleX;
uint8_t appleY;
uint8_t level;
uint16_t score = 0;

int8_t snakeMove(void);
void playzoneClear(void);
int8_t playzoneCheck (uint8_t, uint8_t);
int8_t playzoneSet (uint8_t, uint8_t);
void setApple (void);
void outZahl (uint16_t);
void initsnake (void);

/*
 * @brief   Bewegt die Snake, prueft ob legale Bewegung
 * @param   none
 * @return  Ergebnis. 0=OK, -1=nicht erlaubt
 */
int8_t snakeMove(void)
{
  score++;
  nunchuck_get_data();

  // Mittelpos Nunchuck Joystick etwa bei 180
  if (nunchuck_joyx() < (70))         // links
  {
    xdir = -1;
    ydir=0;
  }
  else if (nunchuck_joyx() > (200))   // rechts
  {
    xdir = +1;
    ydir=0;
  }
  else if (nunchuck_joyy() < (70))    // kleinere Werte => aufwaerts
  {
    xdir=0;
    ydir = +1;
  }
  else if (nunchuck_joyy() > (200))   // down
  {
    xdir=0;
    ydir = -1;
  }

/*
  Serial.print(nunchuck_joyx(),DEC);
  Serial.print(",");
  Serial.print(nunchuck_joyy(),DEC);
  Serial.print(",");
  Serial.print(xdir,DEC);
  Serial.print(",");
  Serial.println(ydir,DEC);
*/  
  
  xpos += xdir;   // in die gewuenschte Richtung bewegen
  ypos += ydir;  

  if ((xpos < 0) || (xpos > XMAX) || (ypos < 0) || (ypos > YMAX))     // Spielfeld verlassen?
    return -1;    // => raus
  
  flipdot_pixel (xpos, ypos, 1);      // Pixel an die neue Stelle setzen

  if ((xpos == appleX) && (ypos == appleY))     // Apfel gefressen?
  {
    if (level > 20)   // wenn noch moeglich:
      level -= 20;    // level = Pause zwischen zwei Bewegungsschritten verringern
    score *= 2;       // Spielstand
    setApple();   
  }
  return playzoneSet (xpos, ypos);    // Neue Position in FlipDot-Abbild setzen und pruefen, wenn nicht erlaubt => Ende (-1)
}

/*
 * @brief   FlipDot-Abbild leeren
 * @param   none
 * @return  none
 */
void playzoneClear(void)
{
  uint8_t x, y;

  for (y = 0; y < 3; y++)
  {
    for (x = 0; x < 140; x++)
      playzone[x][y] = 0;
  }
}

/*
 * @brief   Pruefen, ob Position im FlipDot-Abbild schon gesetzt ist
 * @param   Position x, y
 * @return  Ergebnis. 0=Nein, -1=schon belegt
 */
int8_t playzoneCheck (uint8_t x, uint8_t y)
{
  /*
   * Das FlipDot-Abbild ist ein Array mit 140 Spalten zu je 3 Zeilen-Bytes (=3 x 8 Byte = 24 Bit = 24 Zeilen)
   * Das erste Byte einer Zeile ist oben, das dritte ist unten. 
   * HSB im ersten Byte = 1. Zeile
   * LSB im dritten Byte = 24. Zeile
   */

  // setze eine 32-Bit Variable aus den drei 8-Bit Zeilen-Bytes zusammen. 
  // Die oberen 8 Bit beleiben in der Variable ungenutzt.
  // So enthaelt die Variable die Infos fuer die ganze Spalte.
  // LSB in dieser Variablen = 24. Zeile
  uint32_t alt = ((uint32_t)playzone[x][0] << 16) | ((uint32_t)playzone[x][1] << 8) | (playzone[x][2]);   // bisheriges FlipDot-Abbild fuer die Zeile
  uint32_t neu = alt | ((uint32_t)1 << (23-y));   // setze an der zu pruefenden Stelle ein Bit. So soll das Display aussehen

  if (alt == neu)   // kein Aenderung, beide Variablen sind gleich => Pixel war schon mal gesetzt => Game over
    return -1;
  else
    return 0;
}

/*
 * @brief   Pruefen, ob Position im FlipDot-Abbild schon gesetzt ist und wenn nicht setze Punkt
 * @param   Position x, y
 * @return  Ergebnis. 0=Nein, -1=schon belegt
 */
int8_t playzoneSet (uint8_t x, uint8_t y)
{
  if (playzoneCheck (x, y) == 0)
  {
    uint32_t alt = ((uint32_t)playzone[x][0] << 16) | ((uint32_t)playzone[x][1] << 8) | (playzone[x][2]);
    uint32_t neu = alt | ((uint32_t)1 << (23-y));

    // Zerlege die 32-Bit Variable wieder in drei Bytes. Es werden nur die unteren 24 Bit benutzt
    playzone[x][0] = (uint8_t)(neu >> 16);    // casten, damit 8-Bit
    playzone[x][1] = (uint8_t)(neu >> 8);
    playzone[x][2] = (uint8_t)(neu);
    return 0;
  }
  return -1;
}

/*
 * @brief   Erzeugt einen Apfel
 * @param   none
 * @return  none
 */
void setApple (void)
{
  do  // erzeuge solange eine zufaellige Apfelposition, bis der Apfel nicht irgendwo liegt, wo schon ein Pixel im  FlipDot-Abbild belegt ist
  {
    appleX = random(XMAX);
    appleY = random(YMAX);
  } while (playzoneCheck (appleX, appleY) == -1);

  // lasse den Apfel aufblinken, jaja, Schleife...
  flipdot_pixel (appleX, appleY, 1);
  delay(300);
  flipdot_pixel (appleX, appleY, 0);
  delay(300);
  flipdot_pixel (appleX, appleY, 1);
  delay(300);
  flipdot_pixel (appleX, appleY, 0);
  delay(300);
  flipdot_pixel (appleX, appleY, 1);
}

/*
 * @brief   Schreibt eine Zahl (max. 99999) auf das Display
 * @param   Zahl
 * @return  none
 */
void outZahl (uint16_t zahl)
{
  writechar (zahl/10000+48, 40, 15);  // +48 fuer ASCII-Tabelle = Position von "0"
  zahl = zahl % 10000;
  writechar (zahl/1000+48, 48, 15);
  zahl = zahl % 1000;
  writechar (zahl/100+48, 56, 15);
  zahl = zahl % 100;
  writechar (zahl/10+48, 64, 15);
  zahl = zahl % 10;
  writechar (zahl+48, 72, 15);
  
}

/*
 * @brief   Snake Spiel endlos
 * @param   none
 * @return  none
 */
void initsnake (void)
{
  unsigned long zeit;
  int8_t alive;
  
  DISPLAYDELAY = 400;   // FlipDot langsamer ansteuern fuer bessere Darstellung
  nunchuck_init();      // send the initilization handshake fuer Nunchuck
  randomSeed (analogRead(0));   // ZZ-Genarator init
  delay(200);             // Nunchuck ist zickig
  nunchuck_get_data();    // verwerfen, da erste Ergebnis immer Muell

  while (1)
  {
    xpos = -1;
    ypos = 12;
    xdir = 1;
    ydir = 0;
    score = 0;
    level = LEVELSTART;
    zeit = millis();
    
    playzoneClear();
    flipdot_cls(0);
  
    writechar ('S', 40, 10);
    writechar ('n', 48, 8);
    writechar ('a', 56, 7);
    writechar ('k', 64, 10);
    writechar ('e', 72, 8);
  
    writechar ('3', 90, 8);
    delay(1000);
    writechar ('2', 90, 8);
    delay(1000);
    writechar ('1', 90, 8);
    delay(1000);
    writechar ('0', 90, 8);
  
    writechar (' ', 40, 10);
    writechar (' ', 48, 8);
    writechar (' ', 56, 7);
    writechar (' ', 64, 10);
    writechar (' ', 72, 8);
  
    writechar (' ', 90, 8);

    setApple();
    
    do    // Endlos bis Spielende, koennte auch eleganter mit Timer sein
    {
      if ((millis() - zeit) >= level)
      {
        zeit = millis();
        alive = snakeMove();
      }
    } while (alive == 0);
  
    writechar ('G', 40, 1);
    writechar ('a', 48, 1);
    writechar ('m', 56, 1);
    writechar ('e', 64, 1);
    writechar (' ', 72, 1);
    writechar ('o', 80, 1);
    writechar ('v', 88, 1);
    writechar ('e', 96, 1);
    writechar ('r', 104, 1);

    outZahl(score);
    while (!nunchuck_zbutton())    // warte auf Z-Button pressed
      nunchuck_get_data();
  }  
}


