/*
 * Make: 4/2018
 * Minimalistisches Beispiel zur Ansteuerung eines TM1638 mit 8-Digit 7-Segment-LED per Software-SPI, 8 LEDs und 8 Buttons
 * Idee: https://github.com/MartyMacGyver/TM1638-demos-and-examples
 */

#include <SPI.h>

// Wir nutzen Software-SPI. Pins frei wählbar:
const uint8_t TM1638_CS=A0;
const uint8_t TM1638_CLK=A1;
const uint8_t TM1638_DATA=A2;

/*
 * @brief   Sendet Wert per SPI, togglet CLK
 * @param   Wert
 * @return  none
 */
void sendCommand(uint8_t value)
{
  digitalWrite(TM1638_CS, LOW);    // Modul CS
  shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, value);
  digitalWrite(TM1638_CS, HIGH);
}

/*
 * @brief   Initialisiert Display
 * @param   none
 * @return  none
 */
void sendReset()
{
  // Flowchart for program design in the modes of auto address increment by 1 and fixed address:
  sendCommand(0x40);  // Set command for writing data into display memory, in the mode of auto address increment by 1
  
  digitalWrite(TM1638_CS, LOW);     // Besonderheit: kein CS Toggle waehrend der Uebertragung
  shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, 0xC0);  // set starting address (0x0C0)
  
  for(uint8_t i = 0; i < 16; i++)
    shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, 0x00);    // 16x 0 senden

  digitalWrite(TM1638_CS, HIGH);
}

/*
 * @brief   Gibt die Zahlen 0..9 nacheinander gleichzeitg auf allen Stellen aus
 *          Definiert die anzuzeigenden Segmente fuer jede Zahl
 * @param   none
 * @return  none
 */
void counting()
{
                       //0// //1// //2// //3// //4// //5// //6// //7// //8// //9//
  uint8_t digits[] = { 0x3f, 0x06, 0x5b, 0x4f, 0x66, 0x6d, 0x7d, 0x07, 0x7f, 0x6f };
  uint8_t value;

  for (uint8_t value=0; value <= 9; value++)
  {
    sendCommand(0x40);    //Set command for writing data into display memory, in the mode of auto address increment by 1
    
    digitalWrite(TM1638_CS, LOW);   // Besonderheit: kein CS Toggle waehrend der Uebertragung
    shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, 0xC0);  // set starting address (0x0C0)
    
    for(uint8_t i = 0; i < 8; i++)
    {
      shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, digits[value]);  // Wert
      shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, 0x00);  
    }
    digitalWrite(TM1638_CS, HIGH);
    delay (500);
  }
}

/*
 * @brief   Gibt Text aus. Funktioniert im Prinzip wie Zahlenausgabe, nur mit eigenen Zeichen
 *          Definiert die anzuzeigenden Segmente fuer jede Zahl
 * @param   none
 * @return  none
 */
void text()
{
  uint8_t scrollText[] =
  {
    //H// //E// //L// //L// //O// //.// //.// //.//
    0x76, 0x79, 0x38, 0x38, 0x3f, 0x80, 0x80, 0x80
  };

  sendCommand(0x40);
  
  digitalWrite(TM1638_CS, LOW);
  shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, 0xc0);

  for(int i = 0; i < 8; i++)
  {
    shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, scrollText[i]);  // Wert
    shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, 0x00);  
  }
  digitalWrite(TM1638_CS, HIGH);
  delay (3000);
}

uint8_t readButtons(void)
{
  uint8_t buttons = 0;
  
  digitalWrite(TM1638_CS, LOW);
  shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, 0x42);

  // Ab jetzt Daten empfangen:
  pinMode(TM1638_DATA, INPUT);

  for (uint8_t i = 0; i < 4; i++)
  {
    uint8_t v = shiftIn (TM1638_DATA, TM1638_CLK, LSBFIRST) << i;    // Daten einlesen, kein MISO sondern gleicher Pin wie MOSI
    buttons |= v;
  }

  pinMode(TM1638_DATA, OUTPUT);     // wieder als Eingang
  digitalWrite(TM1638_CS, HIGH);
  return buttons;
}


/*
 * @brief   Schaltet eine LED an (1) oder aus (0)
 * @param   Zustand
 *          LED (0=links)
 * @return  none
 */
void setLed(uint8_t value, uint8_t position)
{
  sendCommand(0x44);
  digitalWrite(TM1638_CS, LOW);
  shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, 0xC1 + (position << 1));
  shiftOut (TM1638_DATA, TM1638_CLK, LSBFIRST, value);
  digitalWrite(TM1638_CS, HIGH);
}

/*
 * @brief   Laesst die LEDs nacheinader leuchten
 * @param   none
 * @return  none
 */
void lauflicht ()
{
  for (uint8_t i=0; i<=7; i++)
  {
    setLed (1, i);
    delay (200);
    setLed (0, i);
  }
}

/*
 * @brief   Fragt die Taster ab und laesst die passende LED leuchten
 * @param   none
 * @return  none
 */
void taster ()
{
  uint8_t buttons = readButtons();
  uint8_t position, mask;

  for(position = 0; position < 8; position++)
  {
    mask = 0x01 << position;
    setLed(buttons & mask ? 1 : 0, position);
  }
}
  
void setup()
{
  digitalWrite(TM1638_CS, HIGH);
  digitalWrite(TM1638_DATA, LOW);
  digitalWrite(TM1638_CLK, LOW);
  pinMode(TM1638_CS, OUTPUT);
  pinMode(TM1638_DATA, OUTPUT);
  pinMode(TM1638_CLK, OUTPUT);

  sendCommand(0x8F);  // Display ON, max. Helligkeit
  sendReset();
}

void loop()
{
  counting();
  text();
  lauflicht();
  sendCommand(0x88);    // Display ON, min. Helligkeit (Datenblatt Kap. 5.3)
  while (1)
    taster();
}



