/*
 * Make 6/2018 https://www.heise.de/make/
 * 
 * Gutscheinbuch
 * 
 * QMC5883 Magnetfeldsensor
 * GPS-Modul
 * ACHTUNG: Daten-Leitung GPS -> Arduino beim Programmiervorgang trennen
 * 
 * Florian Schäffer
 * 
 * V 1.0
 *
 */
 
#include "MechaQMC5883.h"       // https://github.com/mechasolution/Mecha_QMC5883L
#include <math.h>
#include <Wire.h>               // I2C Arduino Library
#include <Servo.h>

//#define SERVOTEST       	// wenn definiert, werden nur die Servopositionen endlos angefahren
#define DEBUG             	// wenn definiert, werden die GPS-Daten an die serielle Schnittstelle gesendet
// #define SIMULATE        	// wenn definiert, wird ein GPS-Empfang mit unten angegebenen Koordinaten simuliert

const double ZielLatitude =   52.38896;         // Breitengrad φ
const double ZielLongitude =  9.811608;         // Laengengrad λ
const uint8_t AbweichungZielEntfernung = 10;     // Wie nah muss der Sucher am Ziel sein. In gleicher EInheit wie rEarth

const uint8_t ServoDistanzPin = 9;       	// Pin fuer Servo zur Entfernungsanzeige    
const uint8_t ServoDistanz10m = 145;    	// = "10m"
const uint8_t ServoDistanz100m = 113;   	// = "100m"
const uint8_t ServoDistanz1km = 80;    		// = "1km"
const uint8_t ServoDistanz10km = 50;     	// = "10 km"
const uint8_t ServoDistanzError = 20;     	// = "???"

const uint8_t ServoKursPin = 10;       		// Pin fuer Servo zur Kursanzeige    
const uint16_t ServoKurs270 = 180;    		// Drehscheibe zeigt nach Westen
const uint16_t ServoKurs90 = 0;    		// Westen

const uint8_t ServoAuswurfPin = 11;       	// Pin fuer Servo zum Auswurf der Gutscheinkarte
const uint16_t ServoAuswurfIn = 180;    	// Servoposition in der die Karte eingezogen ist
const uint16_t ServoAuswurfOut = 0;    		// Gutschein ausgeschoben

const double PIval = M_PI;                      // math.h: #define M_PI 3.14159265358979323846
const double deg2rad = PIval / 180.0;           // Degree to Radiant: Einheit Winkelmass aendern
const double rad2deg = 180/PIval;           	// Radiant to Degree: Einheit Winkelmass aendern
const double rEarth = 6378137.0;                // Aequatorradius in m. Kann auch in km oder mil angegeben werden. https://de.wikipedia.org/wiki/Erdradius

const uint8_t MAXLINELENGTH = 120;   		// max NMEA-Message

Servo myServoDistanz;       	// create servo object to control a servo
Servo myServoKurs;        	// create servo object to control a servo
Servo myServoAuswurf;       	// create servo object to control a servo
MechaQMC5883 qmc;         	// Objekt erzeugen

void setup() 
{
  Serial.begin (9600);   	// Debug und GPS
  Wire.begin ();         	// I²C fuer Kompass
  qmc.init ();			// Kompass Init
  myServoDistanz.attach (ServoDistanzPin);	// Servos anbinden
  myServoKurs.attach (ServoKursPin);
  myServoAuswurf.attach (ServoAuswurfPin);

  myServoDistanz.write (ServoDistanzError);     // Ausgangspositionen einnehmen
  myServoKurs.write (0);
  myServoAuswurf.write (ServoAuswurfIn);
}

/**
  @brief  Such im Quellstring nach dem n-ten vorkommen des Zeichen ',' und kopiert den String ab 
          dem darauffolgenden Zeichen bis zum nächsten Zeichen ',' (ohne dieses).
          Liefert leeren String, wenn zwischen beiden ',' kein Zeichen steht.
  @param  Zielstring
          Quellstring
          ab dem wievielten ',' kopieren?
  @return 1=gefunden, 0=leer
*/
uint8_t getTeilstring (char *dest, char *from, uint8_t pos)
{
  uint8_t cnt=0, detect=0;

  while (*from != '\0')     // Zeichenkette durchlaufen bis Ende
  {
    if (*from == ',')       // wenn gesuchtes Zeichen gefunden wurde
      cnt++;                // das wievielte mal wurde das gesuchte Zeichen gefunden?
    else                    // erst beim naechsten Zeichen kopieren, damit "," am Anfang nicht dabei ist
    {
      if (cnt == pos)       // wenn das gesuchte Zeichen zum n-ten mal gefunden wurde: kopiere String bis zum nächsten auftreten
      {
        detect = 1;
        *dest = *from;    // Zeichen kopieren
        dest++;           // nächstes zeichen im Zielstring wählen
      }
    }
    from++;                 // nächstes Zeichen im Quellstring wählen
  }
  *dest = '\0';             // Zielstring mit Endmarkierung versehen
  return detect;
}

/**
  @brief  Zerlegt einen NMEA-String in Teile und liefert long, lat. Prueft vorab, ob passender String und ob Daten valide
  @param  NMEA
          Laengengrad in Dezimalgrad
          Breitengrad in Dezimalgrad
  @return 1=OK, 0=keine Daten
*/
uint8_t GPS_parse (char* nmea, double* longitude, double* latitude) 
{
  char temp[15];    // max. Zeichen pro Datenblock
  char temp2[4];
  double dezimalgrad;
  uint8_t grad;

/*
 * 
 *  $RMC Recommended minimum specific GPS/Transit data or Glonass/Galileo/Baidou  http://www.catb.org/gpsd/NMEA.html#_talker_ids
 *  http://catb.org/gpsd/NMEA.html#_rmc_recommended_minimum_navigation_information
 *  $GPRMC,hhmmss.ss,A,llll.ll,a,yyyyy.yy,a,x.x,x.x,ddmmyy,x.x,a*hh\r\n
 *  1    = UTC of position fix
 *  2    = Data status (V=navigation receiver warning/invalid; A=OK)
 *  3    = Latitude of fix
 *  4    = N or S
 *  5    = Longitude of fix
 *  6    = E or W
 *  7    = Speed over ground in knots
 *  8    = Track made good in degrees/True course
 *  9    = UT date
 *  10   = Magnetic variation degrees (Easterly var. subtracts from true course)
 *  11   = NMEA Positioning System Mode Indicator (NMEA 2.3 and later): A = Autonomous mode, S = Simulated Mode, N = Data Not Valid
 *  12   = Checksum
 *  13   = String Termination \r\n
 *  
 */

  // first look if we even have GPS data
  if ((nmea[strlen(nmea)-3] != '*') || (strncmp(nmea, "$GNRMC", 6) != 0))     // 4 Zeichen von rechts muss ein "*" sein und "$GPRMC" oder GNRMC (Glonass/Galileo) muss zu finden sein
      return 0;		// kein NMEA

  if (getTeilstring (temp, nmea, 2))      // data status vorhanden?
  {
    if (temp[0] == 'A')    // status = A => valid
    {
      getTeilstring (temp, nmea, 3);    // Long als Dezimalminuten, bspw.: 5223.1363 
                                        // Konvertierung in Decimalgrad: xx.yyyyy°
      temp2[0] = temp[0];               // Grad-Teil: ersten 2 Zeichen - ist einfacher und verstaendlicher als strncpy und dann noch ein \0 ranhaengen
      temp2[1] = temp[1];
      temp2[2] = '\0';
      grad = atoi (temp2);              
      dezimalgrad =  atof (temp) - (grad * 100.0);      // Minuten-Teil = Dezimalminuten - Grad * 100 = 5223.1363 - 5200 = 23.1363
      *latitude = grad + (dezimalgrad / 60.0);         	// 52.3856°
      getTeilstring (temp, nmea, 4);                   	// N/S
      if (temp[0] == 'S')                           	// Suedhalbkugel = negative Koordinaten
        *latitude *= -1.0;
        
      getTeilstring (temp, nmea, 5);    // Lat, bspw.: 00948.5821
      temp2[0] = temp[0];               // Grad-Teil: ersten 3 Zeichen - ist einfacher und verstaendlicher als strncpy und dann noch ein \0 ranhaengen
      temp2[1] = temp[1];
      temp2[2] = temp[2];
      temp2[3] = '\0';
      grad = atoi (temp2);              
      dezimalgrad =  atof (temp) - (grad * 100.0);    // Minuten-Teil = Dezimalminuten - Grad * 100 = 00948.5821 - 900 = 48.5821
      *longitude = grad + (dezimalgrad / 60.0);       // 9,8097°
      getTeilstring (temp, nmea, 6);                  // W/E
      if (temp[0] == 'W')                             // westlich Nullmeriadian = negative Koordinaten
        *longitude *= -1.0;

      return 1;
    }
  }
  return 0;
}

/*
 * @brief   Liest GPS-Daten von serieller Schnittstelle. wartet ewig
 * @param   Char-Array
 * @return  none
 */
void GPSread(char* nmea) 
{
  char c;
  uint8_t idx;
  
  idx = 0; // start at beginning
  while (1) 
  {
    c = Serial.read();
    if (c != -1)
    {
      if (c != '\n')
      {
        if ((idx == MAXLINELENGTH-1) || (c == '\r'))    // Datensatz komplett
        {
          nmea[idx++] = '\0';
          return ;
        }
        nmea[idx++] = c;
      }
    }
  }
}

/**
  @brief  Berechnet die Entfernung zweier Punkte auf einer Kugel. Reihenfolge der Punkte egal
  @param  Punkt 1 Laengengrad, Breitengrad in Dezimalgrad
          Punkt 2 Laengengrad, Breitengrad in Dezimalgrad
  @return Distanz. Einheit ist abhaengig von oben angegebener Konstante fuer Erdumfang
*/
double calcDistance (double long1, double lat1, double long2, double lat2) 
{
  // Alles in Radiant (Bogenmaß) umrechnen
  long1 *= deg2rad;
  lat1 *= deg2rad;
  long2 *= deg2rad;
  lat2 *= deg2rad;

  // https://en.wikipedia.org/wiki/Haversine_formula
  return 2.0 * rEarth * asin (sqrt (sq ((sin ((lat2 - lat1) / 2.0) ) ) + (cos(lat1) * cos(lat2) * sq ((sin ((long2 - long1) / 2.0))))));
}

/**
  @brief  Berechnet die Kursrichtung in Grad vom Startpunkt aus gesehen zum Zielpunkt entlang der Orthodrome
  @param  Punkt 1 Laengengrad, Breitengrad in Dezimalgrad
          Punkt 2 Laengengrad, Breitengrad in Dezimalgrad
  @return Distanz in Grad 0..359
*/
uint16_t calcBearing (double long1, double lat1, double long2, double lat2) 
{
  // Alles in Radiant (Bogenmaß) umrechnen
  long1 *= deg2rad;
  lat1 *= deg2rad;
  long2 *= deg2rad;
  lat2 *= deg2rad;

  // https://de.wikipedia.org/wiki/Positionswinkel , https://www.movable-type.co.uk/scripts/latlong.html
  float brng = atan2(sin(long2-long1) * cos(lat2),cos(lat1)*sin(lat2) - sin(lat1)*cos(lat2)*cos(long2-long1));
  brng = rad2deg * brng;         		// radians to degrees
  brng = ((int16_t)brng + 360) % 360;           // atan2 = 0..180 im Uhrzeigersinn und dann -180..-0 weiter , jetzt => 0..360

  return brng;  
}

void loop() 
{
  uint16_t x,y,z;
  char nmea[MAXLINELENGTH];
  double latitude = 52.38557;
  double longitude = 9.810242;
  double abstand;
  int16_t bearing;
  int16_t v;
  int16_t AzimuthIst;
  int16_t AzimuthSoll;
  int16_t AzimuthIst0;
  int16_t AzimuthSoll0;

// Testet die Servopositionen
#ifdef SERVOTEST
  while (1)
  {
    myServoAuswurf.write (ServoAuswurfIn);  
    myServoDistanz.write (ServoDistanz10m);
    myServoKurs.write (ServoKurs270);
    delay (3000);
    myServoDistanz.write (ServoDistanz100m);
    myServoKurs.write (ServoKurs90);
    myServoAuswurf.write (ServoAuswurfOut);  
    delay (3000);
    myServoDistanz.write (ServoDistanz1km);
    delay (3000);
    myServoDistanz.write (ServoDistanz10km);
    delay (3000);
  }
#endif

#ifndef SIMULATE
  GPSread(nmea);
  if (GPS_parse (nmea, &longitude, &latitude))
#endif
  {
    qmc.read(&x, &y, &z, &AzimuthIst);       	// Auslesen des Magnetsensor. uns interessiert nur die Richtung 0..359
    abstand = calcDistance (longitude, latitude, ZielLongitude, ZielLatitude);
    AzimuthSoll = calcBearing (longitude, latitude, ZielLongitude, ZielLatitude);

    if (AzimuthIst > 180)
      AzimuthIst0 = (360 - AzimuthIst) * -1;
    else 
      AzimuthIst0 = AzimuthIst;
      
    if (AzimuthSoll > 180)
      AzimuthSoll0 = (360 - AzimuthSoll) * -1;
    else
      AzimuthSoll0 = AzimuthSoll;

    if (AzimuthSoll0 > AzimuthIst0)
      bearing = AzimuthSoll0 - AzimuthIst0;
    else
      bearing = (AzimuthIst0 - AzimuthSoll0) * -1;
    
#ifdef DEBUG
    Serial.print ("Ort: ");
    Serial.print (longitude,4);
    Serial.print (" | ");
    Serial.print (latitude,4);
    
    Serial.print (" *** Distanz: ");
    Serial.print (abstand);
    
    Serial.print (" *** Kurs Soll: ");
    Serial.print (AzimuthSoll);

    Serial.print (" *** Kurs Ist: ");
    Serial.print (AzimuthIst);

    Serial.print (" *** Kurs Abweichung: ");
    Serial.println (bearing);
#endif

    // Distanz zum Ziel
    if (abstand < 50)
      v = ServoDistanz10m;
    else if (abstand < 300)
      v = ServoDistanz100m;
    else if (abstand < 3000)
      v = ServoDistanz1km;
    else if (abstand >= 3000)
      v = ServoDistanz10km;

    // Abstand zum Ziel
    myServoDistanz.write (v);   
    if (abstand <= AbweichungZielEntfernung)      // Ziel erreicht
      myServoAuswurf.write (ServoAuswurfOut);     // Karte auswerfen
 
    // Richtung zum Ziel
    v = 90 + bearing;     		// 90 = Mitte => negative Werte = Zeiger nach rechts
    
    if ((v < 10) || (v > 170))          // zu grosse Abweichung um anzuzeigen
      v = 0;                     	// => links unterhalb Sichtbereich fahren

    v = map (v, 0, 180, ServoKurs270, ServoKurs90);
    myServoKurs.write (v);
    delay (500);
  }
}
