/*
   Liest einen Binärdump von der Logamatic-Diagnosebuchse Pin 1 und filtert nach verschiedenen
   Sequenzen. Wenn eine gesuchte Sequenz gefunden ist, wird der entsprechende
   Wert ausgegeben.
   Mögliche Sequenzen:

   af 82 9e 06 xx .. .. aa .. ..	xx ist Speichertemperatur unten in °C, aa könnte signed byte sein, vlt. Regeldifferenz Heizkreis?
   af 82 80 00 .. .. aa xx .. ..	aa ist Heizkreis 1 soll, xx ist Heizkreisvorlauf 1 in °C
   af 82 84 00 .. .. .. xx .. ..	xx ist Speichertemperatur oben in °C
   af,82,85,00 aa xx .. .. .. ..	xx ist Zulauftemp Mischer HK1 in °C, aa könnte Mindesttemperatur soll sein
   af 82 89 00 xx .. .. .. .. ..	xx ist die Außentemperatur
   af 82 92 00 .. xx .. .. .. .. 	xx ist die relative Brennerleistung	

   af 82 89 00 02 03 09 00 00 00 09	Bit 0 von Byte 0 könnte Status Umschaltventil Speicher / Therme sein 

   Um das Logfile zu kürzen, reagiert das Programm auf SIGHUP, indem
   es
   	1) die Datei datalog.csv schließt, 
   	2) sie umbenennt in datalog.tmp.csv
   	3) die letzten 48h in eine neue datalog.csv kopiert
   	4) und das logging fortführt
   	5) die nicht in die neue Datei kopierten Zeilen an die Logdatei datalog_bak.csv anhängt
*/
 
// Change this to match the machine's serial port. /dev/ttyS0 for pi zero w, /dev/ttyAMA0 for pi 2
#define SERDEV "/dev/ttyS0"
//Maximale Zeilenlänge des Logfiles. Sollte Luft nach oben lassen. Newline und 0 am Ende nicht vergessen!
#define MAXLOGLINELEN 80

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <termios.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <time.h>
#include <signal.h>

char dologtrim=0;		//set to !=0 if log should be trimmed. Happens by receiving SIGHUP

void initComPort(int* sfd, char* device)
{
  //initialisiere die serielle Schnittstelle
  struct termios options;

  *sfd = open(device, O_RDWR | O_NOCTTY | O_NDELAY);
  if (*sfd == -1)
  {
    fprintf(stderr, "unable to open %s\n",device);
    exit(1);
  }
  else
  {
    fcntl(*sfd, F_SETFL, FNDELAY);
  }

  tcgetattr(*sfd, &options);

  cfsetispeed(&options, B1200);
  cfsetospeed(&options, B1200);

  cfmakeraw(&options);

  options.c_cflag &= ~(PARENB | CSTOPB | CSIZE | CRTSCTS);
  options.c_cflag |= (CLOCAL | CREAD | CS8);

  options.c_oflag &= ~OPOST;
  options.c_cc[VMIN] = 1;
  options.c_cc[VTIME] = 0;

  tcsetattr(*sfd, TCSANOW, &options);
}

void trimlog(int sig) {
	printf("SIGHUP empfangen\n");
	dologtrim=1;	//Setze globale Variable dologtrim, um das trimming bei passender Gelegenheit durchzuführen
}

int main(int argc, char* argv[]) {

	int sfd;
	int startLogging=63;    //die 6 LSB sind gesetzt. Jedes Bit steht für eine
	                        //Botschaft. Bei Empfang einer Botschaft wird das
                                //zugehörige Bit genullt. Es müssen alle Botschaften
	                        //empfangen worden sein, bevor das Loggen beginnt.
	int32_t n;
	u_int32_t bytes;
	u_int8_t buff[10000];
	char linebuf[MAXLOGLINELEN];	//Zum Logfiletrimmen

	FILE *out,*outlong,*outnew;
	int c,i,j;

	time_t now,loglinedate,trimtotime;
	
	typedef struct {
		const unsigned char byte[8];
		int offset;
		unsigned char value[6];
	}  sequenz;
	
	const int botschaftszahl=6;	//Anzahl der interessanten Botschaften
	sequenz datenblock[]={
		{ {0xaf,0x82,0x84,0x00}, 0, {0,0,0,0,0,0} },	//Speichertemperatur oben Botschaft 0x84, Adresse 0-5, Byte 3
		{ {0xaf,0x82,0x9e,0x06}, 0, {0,0,0,0,0,0} },	//Speichertemperatur unten Botschaft 0x9e, Adresse 6-11, Byte 0
		{ {0xaf,0x82,0x80,0x00}, 0, {0,0,0,0,0,0} },	//Heizkreisvorlauf, Botschaft 0x80, Adresse 0-5, Byte 2 Ist, Byte 3 Soll
		{ {0xaf,0x82,0x85,0x00}, 0, {0,0,0,0,0,0} },	//Brennervorlauf, Botschaft 0x85, Adresse 0-5, Byte 1
		{ {0xaf,0x82,0x92,0x00}, 0, {0,0,0,0,0,0} },	//rel. Brennerleistung, Botschaft 0x92, Adresse 0-5, Byte 1
		{ {0xaf,0x82,0x89,0x00}, 0, {0,0,0,0,0,0} }	//Außentemperatur Botschaft 0x89, Adresse 0-5, Byte 0
	};

	int spo,spu,hkvs,hkvi,brv,brl,auss;
	int spo_old,spu_old,hkvs_old,hkvi_old,brv_old,brl_old,auss_old;
	
	// Initialize the serial port
	initComPort(&sfd, SERDEV);

	//register SIGHUP handling for logfile trimming
	if(signal(SIGHUP,trimlog)==SIG_ERR) {
		fprintf(stderr,"Kann SIGHUP nicht registrieren!\n");
		return(-1);
	}
	
	out=fopen("datalog.csv","a");
	if (out==NULL) {
		fprintf(stderr,"Error opening log file datalog.csv\n");
		return(-1);
	}

	
	while (1) {
		// Check if there's any data available to read
		ioctl(sfd, FIONREAD, &bytes);
		if (bytes > 0) {
			// Read what we can
			n = read(sfd, buff, 10000);
			if (n < 0) {
				fprintf(stderr, "read failed\n");
			}
			if (n > 0) {
				//n bytes ab buff
				for(i=0;i<n;i++) {
					c=buff[i];
					//Für alle interessanten Botschaften prüfen, ob sie gerade empfangen wird:
					for(j=0;j<botschaftszahl;j++) {
						if (datenblock[j].offset<4) {
							//Die ersten 4 Bytes müssen stimmen, sonst rücksetzen
							if (c==datenblock[j].byte[datenblock[j].offset]) {
								datenblock[j].offset++;
							} else {
								datenblock[j].offset=0;
							}
						} else {
							//Die ersten 4 Bytes haben gestimmt, ab hier 6 Datenbytes
							if ( (datenblock[j].offset>=4) && (datenblock[j].offset<=9) ) {
								datenblock[j].value[datenblock[j].offset-4]=c;
								if (datenblock[j].offset<9) {
									datenblock[j].offset++;
								} else {
									//Alle Datenbytes angekommen. Rücksetzen
									datenblock[j].offset=0;
									startLogging=(startLogging&(~(1<<j)));	//Bit der entspr. Botschaft löschen
								}
							} else {
								datenblock[j].offset=0;
							}
						}
					}
					spo=datenblock[0].value[3];
					//printf("Speicher oben: %i\n",spo);
					spu=datenblock[1].value[0];
					//printf("Speicher unten: %i\n",spu);
					hkvs=datenblock[2].value[2];
					//printf("Heizkreisvorlauf Soll: %i\n",hkvs);
					hkvi=datenblock[2].value[3];
					//printf("Heizkreisvorlauf Ist: %i\n",hkvi);
					brv=datenblock[3].value[1];
					//printf("Brennervorlauf: %i\n",brv);
					brl=datenblock[4].value[1];
					//printf("Brennerleistung: %i\n",brl);
					auss=datenblock[5].value[0];
					if (auss>127) {
						auss=auss-256;	//vermutlich sind negative Außentemperaturen vorgesehen, deshalb ist der Wert eigentlich ein signed byte
					}
					//printf("Außentemperatur: %i\n",auss);
				}
			}
			//Schreibe erst dann in die Logdatei, wenn alle relevanten Botschaften mindestens einmal aufgetaucht sind
			if (startLogging==0) {
				//Schreibe aktuelle Zeit und alle Werte, sobald sich mindestens ein Wert geändert hat
				if ( (spo!=spo_old) || (spu!=spu_old) || (brv!=brv_old) || (hkvs!=hkvs_old) || (hkvi!=hkvi_old) || (brl!=brl_old) || (auss!=auss_old) ) {
					time(&now);
					fprintf(out,"%ld\t",now);
					fprintf(out,"%i\t",spo);
					fprintf(out,"%i\t",spu);
					fprintf(out,"%i\t",brv);
					fprintf(out,"%i\t",hkvs);
					fprintf(out,"%i\t",hkvi);
					fprintf(out,"%i\t",brl);
					fprintf(out,"%i\n",auss);
					spo_old=spo;
					spu_old=spu;
					brv_old=brv;
					hkvs_old=hkvs;
					hkvi_old=hkvi;
					brl_old=brl;
					auss_old=auss;
					fflush(out);
				}
			}
		}
		if(dologtrim==1) {
			dologtrim=0;
			/* Logfile trimmen. An dieser Stelle 
				*rewind machen, zeilen lesen und in dauerlog anhängen bis zu Zeit-48h
				*dann alle kommenden zeilen in neue logdatei schreiben
				*dann alte datei schliessen, löschen
				*dann neue logdatei in alte umbenennen und mit append öffnen.
			*/
			fclose(out);
			out=fopen("datalog.csv","r");
			if(out==NULL) {
				fprintf(stderr,"Error opening logfile datalog.csv as read for trimming\n");
				return(-1);
			}
			
			outlong=fopen("datalog_all.csv","a");	//gets older lines appended
			if(out==NULL) {
				fprintf(stderr,"Error opening logfile datalog_all.csv for trimming\n");
				return(-1);
			}
			outnew=fopen("datalog_new.csv","w");	//will contain trimmed logfile
			if(out==NULL) {
				fprintf(stderr,"Error opening logfile datalog_new.csv for trimming\n");
				return(-1);
			}

			//FIXME: Wenn now noch nicht gesetzt ist, da noch keine Zeile geschrieben wurde, wird alles ins neueübernommen. Kein Trimmung!
			trimtotime=now-48*3600-1800;	//Unix-Timestamp, bis zu dem die Logdatei getrimmt werden soll: letztes Datum-48,5 Stunden

			while( (fgets(linebuf,MAXLOGLINELEN,out))!=0 ) {
				sscanf(linebuf,"%ld\t",&loglinedate);
				//for safety null the last character of linebuf...
				//..because fputs will not stop until 0 is encountered
				linebuf[MAXLOGLINELEN-1]=0;
				if(loglinedate<trimtotime) {
					//Append line older than trimtotime to long logfile.
					fputs(linebuf,outlong);
				} else {
					//Write line newer or sametime than trimtotime to new logfile.
					fputs(linebuf,outnew);
				}
			}
			fclose(outlong);
			fclose(outnew);
			fclose(out);
			if (rename("datalog.csv","datalog_bak.csv")!=0) {
				fprintf(stderr,"Error renaming datalog.csv to datalog_bak.csv\n");
				return(-1);
			}
			if (rename("datalog_new.csv","datalog.csv")!=0) {
				fprintf(stderr,"Error renaming datalog_new.csv to datalog.csv\n");
				return(-1);
			}
			out=fopen("datalog.csv","a");
			if(out==NULL) {
				fprintf(stderr,"Error reopening trimmed logfile datalog.csv\n");
				return(-1);
			}
		}
		usleep(1000);
	}
	
	return(0);
}
