/*
 * Chainst - Konfigurationsprogramm fuer ChaiNT.
 * Copyright (c) 1997 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * Dieses Programm ist Freie Software im Sinne der GNU General Public
 * License (GPL), Version 2 (siehe Datei COPYING).
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#ifndef DEFAULT_INPUT_FILE
#define DEFAULT_INPUT_FILE	"chaint"
#endif

#define OFFSET_DRIVE	4
#define OFFSET_PART	5
#define OFFSET_TYPE	6
#define OFFSET_ACTIVE	7
#define OFFSET_STRING	8
#define OFFSET_SIG1	510
#define OFFSET_SIG2	511

void
process(const char *ostr, const char *istr, int part, int type, int active) {
    unsigned char buffer[512];
    FILE *fp;

    /*
     * Eingabedatei lesen
     */
    if (strcmp(istr, "-") == 0) {
	fp = stdin;
	istr = "<stdin>";
    }
    else if (!(fp = fopen(istr, "rb"))) {
	fprintf(stderr, "Kann Eingabedatei `%s' nicht oeffnen.", istr);
	exit(1);
    }
    if (fread(buffer, 1, 512, fp) != 512) {
	fprintf(stderr, "Fehler beim Lesen von %s.\n", istr);
	exit(1);
    }
    if (fp != stdin) {
	fclose(fp);
    }
    /*
     * Datei ueberpruefen
     */
    if (buffer[OFFSET_SIG1] != 0x55
     || buffer[OFFSET_SIG2] != 0xaa
     || strncmp(&buffer[OFFSET_STRING], "ChaiNT", 6) != 0) {
	fprintf(stderr, "Fehlerhafte Eingabedatei.\n");
	exit(1);
    }
    /*
     * Parameter setzen
     */
    buffer[OFFSET_PART] = part;
    buffer[OFFSET_TYPE] = type;
    buffer[OFFSET_ACTIVE] = active;
    /*
     * Ausgabedatei schreiben
     */
    if (strcmp(ostr, "-") == 0) {
	fp = stdout;
	ostr = "<stdout>";
    }
    else if (!(fp = fopen(ostr, "wb"))) {
	fprintf(stderr, "Kann Ausgabedatei `%s' nicht oeffnen.", ostr);
	exit(1);
    }
    if (fwrite(buffer, 1, 512, fp) != 512) {
	fprintf(stderr, "Fehler beim Schreiben von %s.\n", ostr);
	exit(1);
    }
    if (fp != stdout) {
	fclose(fp);
    }
}

void
usage(int ret) {
    fprintf(stderr, "Benutzung: \
chainst [ -a ] [ -p <part> | -t <type> ] <outfile> [ <infile> ]\n");
    exit(ret);
}

int
set_argument(char **arg, char name, int argc, char **argv, int i) {
    if (argv[i][2]) {
	*arg = &argv[i][2];
    }
    else if (++i < argc) {
	*arg = argv[i];
    }
    else {
	fprintf(stderr, "Option `-%c' benoetigt ein Argument.\n", name);
	usage(1);
    }
    return i;
}

int
main(int argc, char **argv) {
    char *pstr = NULL;
    char *tstr = NULL;
    char *istr = DEFAULT_INPUT_FILE;
    char *ostr = NULL;
    unsigned part = 0;
    unsigned type = 0;
    unsigned active = 0;
    char *tmp;
    int i;

    for (i = 1; i < argc && argv[i][0] == '-' && argv[i][1]; i++) {
	if (argv[i][1] == '-') {
	    /* end of options */
	    i++;
	    break;
	}
	switch (argv[i][1]) {
	    case 'p':
		i = set_argument(&pstr, 'p', argc, argv, i);
		break;
	    case 't':
		i = set_argument(&tstr, 't', argc, argv, i);
		break;
	    case 'a':
		active = 1;
		break;
	    case 'h':
	    case '?':
		usage(0);
		break;
	    default:
		fprintf(stderr, "Ungueltiger Parameter `%s'.\n", argv[i]);
		usage(1);
		break;
	}
    }
    if (tstr) {
	type = strtol(tstr, &tmp, 0);
	if (tmp == tstr || (tmp && *tmp) || type < 1 || type > 255) {
	    fprintf(stderr, "Ungueltiges Argument: `%s'\n", tstr);
	    usage(1);
	}
    }
    if (pstr) {
	part = strtol(pstr, &tmp, 0);
	if (tmp == pstr || (tmp && *tmp) || part < 0 || part > 3) {
	    fprintf(stderr, "Ungueltiges Argument: `%s'\n", pstr);
	    usage(1);
	}
    }
    if (tstr && pstr) {
	fprintf(stderr, "Warnung: `-t' hat Vorrang vor `-p'.\n");
    }
    if (i >= argc) {
	fprintf(stderr, "Name der Ausgabedatei fehlt.\n");
	usage(1);
    }
    ostr = argv[i++];
    if (i < argc) {
	istr = argv[i++];
    }
    if (i < argc) {
	fprintf(stderr, "Zu viele Argumente.\n");
	usage(1);
    }
    process(ostr, istr, part, type, active);
    fprintf(stderr, "Fertig.\n");
    exit(0);
}
