#include <stdio.h>
#ifdef WIN32
#include <windows.h>
#include <wincrypt.h>
#include <io.h>
#define MY_ENCODING_TYPE	(PKCS_7_ASN_ENCODING | X509_ASN_ENCODING)
#else
#include <crypt.h>
#endif
#include <stdlib.h>
#include <errno.h>
#include <zlib.h>
#include <sys/timeb.h>

#define	true	1
#define	false	0

#define MD5LEN  16

#define	DEFAULT_BLOCKSIZE_KB 8192
#define	DEFINE_RAND_SEED	0x1709
#define	MIN_BLOCKSIZE_KB 4
#define	MAX_BLOCKSIZE_KB 65536
#define	MAX_THREADS	8

typedef unsigned char bool;
typedef unsigned char uchar;
typedef unsigned long ulong;

typedef struct {
	int		blockNo;
	uchar	*inputBlock;
	int		inputSize;
	uchar	*outputBlock;
	int		outputSize;
	FILE	*ofp;
} BP_S, *PBP_S;

static char *progName="FileCompress", *outFileNamePattern="outFile__%012x.fcc";
static unsigned char magicHdr[] = { 0xfd, 0xe8, 0xdf, 0x80 };

int		allocateThread();
int		flushBlockToFile(PBP_S);
int		processBlock(PBP_S);
int		processDecompression(char *);
int		processFile(char *);
int		processFileList(char **, int);
int		str2BlockSizeKB(char *);
uchar	*getMD5Hash(uchar *, int);
void	usage();
void	waitForBlockNoWritten(int);

bool	isStdIn=false, isDebug=false, isDecompress=false, isForce=false, isVerbose=false, isTest=false, isCrypt=false, isRemoveSource=false;
char	*passwd=NULL;
int		threadActiveCount=0, blocksWrittenCount=0, blockSizeKB=DEFAULT_BLOCKSIZE_KB, *blocksWrittenList=NULL;
#ifdef WIN32
HANDLE	blocksWrittenMutex=NULL;
#endif

int main(int argc, char *argv[]) {
	int		argCnt=1, lastOpt=0, fileNameListCount=0;
	char	*args, **fileNameList=NULL;
	bool	argFollows=false;

	if (argc > 1) {
		while (--argc) {
			args=argv[argCnt++];
			if (*args=='-' && strlen(args)>1) {
				int idx, length=strlen(args);
				bool isBreak=false;

				for (idx=1;idx<length;idx++) {
					switch (lastOpt=args[idx]) {
					case 'b': // blocksizeKB follows
						if ((idx+1) < length) {
							if ((blockSizeKB=str2BlockSizeKB(&args[idx+1]))<0) usage();
							isBreak=true;
						} else {
							argFollows=true;
						}
						break;
					case 'd': // decompress
						isDecompress=true; break;
					case 'D': // debug
						isDebug=true; break;
					case 'f': // force overwriting
						isForce=true; break;
					case 'p': // password follows - implicitly sets the crypt flag
						isCrypt=true;
						if ((idx+1) < length) {
							passwd=&args[idx+1]; isBreak=true;
						} else argFollows=true;
						break;
					case 'r': // remove source file
						isRemoveSource=true; break;
					case 't': // test file
						isTest=true; break;
					case 'v': // sets the verbose flag
						isVerbose=true;	break;
					default: usage();	
					}
					if (argFollows || isBreak) break;
				}
				continue;
			}
			if (argFollows) {
				switch (lastOpt) {
				case 'b': if ((blockSizeKB=str2BlockSizeKB(args))<0) usage();
					break;
				case 'p': passwd=args; break;
				}
				argFollows=false; continue;
			}
			if (access(args, 0)==-1) {
				fprintf(stderr,"unable to access \"%s\"\n", args);
				exit(2);
			}
			fileNameList=realloc(fileNameList,(1+fileNameListCount)*sizeof(char *));
			fileNameList[fileNameListCount++]=args;
		}
	} else isStdIn=true;
#ifdef WIN32
	blocksWrittenMutex=CreateMutex(NULL, false, NULL);
#endif
	if (fileNameListCount==0) {
		isStdIn=true;
	} else processFileList(fileNameList, fileNameListCount);
}

int processFileList(char **fileList, int fileListCount) {
	int i;

	for (i=0;i<fileListCount;i++) {
		processFile(fileList[i]);
	}
	return fileListCount;
}

int processFile(char *fileName) {
	FILE *ifp, *ofp;
	char	outFileName[300];
	struct	timeb tp;
	int		i, l, ch, blockNo, lastBlockNo;
	long tics;

	srand(DEFINE_RAND_SEED);
	if (isDecompress)
		return processDecompression(fileName);
	ftime(&tp);
	tics=1000*(long)tp.time+tp.millitm;
	if ((ifp=fopen(fileName,"rb"))==NULL) {
		fprintf(stderr,"unable to open file \"%s\"\nfopen(): %s\n", fileName, strerror(errno));
		return -1;
	}
	sprintf(outFileName,outFileNamePattern,tics);
	if ((ofp=fopen(outFileName,"wb"))==NULL) {
		fprintf(stderr,"unable to open file \"%s\"\nfopen(): %s\n", outFileName, strerror(errno));
		return -1;
	}
	/**
	 * write the file header to the output file right here
	 */
	for (i=0;i<4;i++) {
		ch=magicHdr[i];
		if (i==3 && isCrypt) ch++;
		fputc(ch, ofp);
	}
	if (isCrypt) {
		/**
		 * store 16 byte MD5 sum of password here ...
		 */
		uchar *md5Hash=getMD5Hash(passwd, strlen(passwd));
		fwrite(md5Hash, 1, MD5LEN, ofp);
	}
	/**
	 * store length of filename
	 * and the filename simply scrambled by some random numbers ...
	 */
	fputc(l=strlen(fileName), ofp);
	for (i=0;i<l;i++) {
		fputc(fileName[i] ^ ~rand(), ofp);
	}
	/**
	 * then store the block size in KB to give the decompressor a hint of
	 * how many bytes should be allocated for the destination buffer
	 */
	fputc(blockSizeKB & 0xff, ofp);
	fputc((blockSizeKB >> 8) & 0xff, ofp);
	if (isVerbose) fprintf(stderr,"processing %s ...\n", fileName);
	for (blockNo=0;;blockNo++) {
		DWORD dwTid;
		PBP_S pbp_s=calloc(1,sizeof(BP_S));

		uchar	*inputBlock=malloc(1024*blockSizeKB);
		int nbRead=fread(inputBlock, 1, 1024*blockSizeKB, ifp);
		fprintf(stderr,"Read %2d. block with %7d bytes length\n", blockNo+1, nbRead);
		if (nbRead<=0) break;
		pbp_s->blockNo=lastBlockNo=blockNo;
		pbp_s->inputBlock=inputBlock;
		pbp_s->inputSize=nbRead;
		pbp_s->ofp=ofp;
		allocateThread();
		fprintf(stderr,"Creating processBlock_Thread for block #%d and %d input bytes ...\n", blockNo, nbRead);
		if (CreateThread((LPSECURITY_ATTRIBUTES)NULL,0, (LPTHREAD_START_ROUTINE)processBlock,(LPVOID)pbp_s,0, &dwTid)==NULL) {
			fprintf(stderr,"CreateThread(processBlock()) failed!\n");
			fclose(ofp); unlink(outFileName);
			return -1;
		}
	}
	/**
	 * waiting for the threads to terminate ...
	 */
	fclose(ifp);
	waitForBlockNoWritten(lastBlockNo);
	fclose(ofp);
	if (isRemoveSource) unlink(fileName);
	return 0;
}

int processDecompression(char *fileName) {
	FILE	*ifp, *ofp;
	int		i, l, ch;
	DWORD	dwDestLen;
	uchar	hdr[MD5LEN], originalFileName[300], *outputBuffer;

	if ((ifp=fopen(fileName,"rb"))==NULL) {
		fprintf(stderr,"unable to open file \"%s\"\nfopen(): %s\n", fileName, strerror(errno));
		return -1;
	}
	fread(hdr, 1, 3, ifp);
	if (memcmp(hdr, magicHdr, 3) || ((ch=fgetc(ifp))!=0x80 && ch!=0x81)) {
		fprintf(stderr,"invalid magic header - probably not compressed with this program!\n");
		fclose(ifp);
		return -1;
	}
	isCrypt=(ch == 0x81);
	if (isCrypt) {
		if (!passwd) {
			fprintf(stderr,"File %s is encrypted but no password has been given in the command line\n",
				fileName);
			fclose(ifp);
			return -1;
		}
		fread(hdr, 1, MD5LEN, ifp);
		if (memcmp(hdr, getMD5Hash(passwd, strlen(passwd)), MD5LEN)) {
			fprintf(stderr,"wrong password!\n");
			fclose(ifp);
			return -1;
		}
	}
	/**
	 * now create the output file
	 * extract the filename for the original file ...
	 */
	l=fgetc(ifp);
	for (i=0;i<l;i++) {
		originalFileName[i] = fgetc(ifp) ^ ~rand();
	}
	originalFileName[i]=0;
	if (access(originalFileName, 0)>=0 && !isForce) {
		fprintf(stderr,"File \"%s\" exists!\nUse -f option to force overwrite!\n", originalFileName);
		fclose (ifp);
		return -1;
	}
	if ((ofp=fopen(originalFileName, "wb"))==NULL) {
		fprintf(stderr,"unable to open file \"%s\"\nfopen(): %s\n", originalFileName, strerror(errno));
		return -1;
	}
	/**
	 * now read the input BlockSize
	 */
	blockSizeKB=(fgetc(ifp)|(fgetc(ifp))<<8);
	outputBuffer=malloc(blockSizeKB*1024);
	if (isDebug) fprintf(stderr,"Input Block Length = %d KB\n", blockSizeKB);
	for (;;) {
		uchar *inputBuffer;
		int blockLength=0;

		if (fread(hdr, 1, 4, ifp)<4) break;	// EOF
		if (isDebug)
			fprintf(stderr,"block length bytes: %02x %02x %02x %02x\n", hdr[0], hdr[1], hdr[2], hdr[3]);
		for (i=3;i>=0;i--) {
			blockLength <<= 8;
			blockLength |= hdr[i];
		}
		if ((inputBuffer=malloc(blockLength))==NULL) {
			fprintf(stderr,"unable to allocate %d bytes for the input buffer!\nmalloc() returned %d: %s\n",
				blockLength, errno, strerror(errno));
			fclose(ifp);
			return -1;
		}
		if (fread(inputBuffer, 1, blockLength, ifp)<blockLength) { // unexp. EOF
			fprintf(stderr,"unexpected EOF reading from input file!\n");
			fclose(ifp);
			return -1;
		}
		dwDestLen=blockSizeKB*1024;
		if (isCrypt) {
			/**
			 * block has to be de-crypted here ...
			 */
#ifdef WIN32
			bool	bResult;
			DWORD	dwSize;
			HCRYPTPROV	hProv;
			HCRYPTHASH	hHash;
			HCRYPTKEY	hKey;

			if (!CryptAcquireContext(&hProv, NULL, MS_DEF_PROV, PROV_RSA_FULL, 0)) {
				fprintf(stderr,"CryptAcquireContext() FAILED!\n");
				return -1;
			}
			if (!CryptCreateHash(hProv, CALG_MD5, 0, 0, &hHash)) {
				fprintf(stderr,"CryptCreateHash() FAILED!\n");
				return -1;
			}
			if (!CryptHashData(hHash, passwd, strlen(passwd), 0)) {
				fprintf(stderr,"CryptHashData() FAILED!\n");
				return -1;
			}
			if (!CryptDeriveKey(hProv, CALG_RC4, hHash, CRYPT_EXPORTABLE, &hKey)) {
				fprintf(stderr,"CryptDeriveKey() FAILED!\n");
				return -1;
			}
			dwSize=blockLength;
			if (!CryptDecrypt(hKey, 0, TRUE, 0, inputBuffer, &dwSize)) {
				fprintf(stderr,"CryptDecrypt(2) FAILED!\n");
				return -1;
			}
			blockLength=dwSize;
			CryptDestroyKey(hKey);
			CryptDestroyHash(hHash);
			CryptReleaseContext(hProv, 0);
#endif
		}
		if ((l=uncompress(outputBuffer, &dwDestLen, inputBuffer, blockLength))!=Z_OK) {
			fprintf(stderr,"uncompress() failed with %d - dwDestLen=%d!\n", l, dwDestLen);
			fclose(ifp);
			return -1;
		}
		fwrite(outputBuffer, 1, dwDestLen, ofp);
		free(inputBuffer);
	}
	fclose(ifp);
	fclose(ofp);
	return 1;
}

int allocateThread() {
	int iCount;

	for (;;) {
#ifdef WIN32
		WaitForSingleObject(blocksWrittenMutex, INFINITE);
#endif
		iCount=threadActiveCount;
#ifdef WIN32
		ReleaseMutex(blocksWrittenMutex);
#endif
		if (iCount < MAX_THREADS) break;
		Sleep(50);
	}
#ifdef WIN32
		WaitForSingleObject(blocksWrittenMutex, INFINITE);
#endif
		threadActiveCount++;
#ifdef WIN32
		ReleaseMutex(blocksWrittenMutex);
#endif
	return 1;
}

int	processBlock(PBP_S pbp_s) {
	DWORD dwTid;

	/**
	 * given is the inputBlock with inputSize bytes length
	 * and an outputBlock which is at least allocated with 2 times of inputSize length
	 * the processed outputBlock length will be returned in *outputSize
	 */
	int ret, outputSize=2*pbp_s->inputSize;
	uchar *outputBlock=malloc(outputSize);

	if ((ret=compress2(outputBlock, (ulong *)&outputSize, pbp_s->inputBlock, pbp_s->inputSize, Z_BEST_COMPRESSION))!=Z_OK) {
		fprintf(stderr,"compress2() returned %d and failed\n",ret);
		return -1;
	}
	if (isCrypt) {
#ifdef WIN32
		/**
		 * to be thread - safe it is necessary to initialize an own context for each thread
		 */
		bool	bResult;
		DWORD	dwSize;
		HCRYPTPROV	hProv;
		HCRYPTHASH	hHash;
		HCRYPTKEY	hKey;

		if (!CryptAcquireContext(&hProv, NULL, MS_DEF_PROV, PROV_RSA_FULL, 0)) {
			fprintf(stderr,"CryptAcquireContext() FAILED!\n");
			return -1;
		}
		if (!CryptCreateHash(hProv, CALG_MD5, 0, 0, &hHash)) {
			fprintf(stderr,"CryptCreateHash() FAILED!\n");
			return -1;
		}
		if (!CryptHashData(hHash, passwd, strlen(passwd), 0)) {
			fprintf(stderr,"CryptHashData() FAILED!\n");
			return -1;
		}
		if (!CryptDeriveKey(hProv, CALG_RC4, hHash, CRYPT_EXPORTABLE, &hKey)) {
			fprintf(stderr,"CryptDeriveKey() FAILED!\n");
			return -1;
		}
		dwSize=outputSize;
		if (!CryptEncrypt(hKey, 0, TRUE, 0, NULL, &dwSize, outputSize)) {
			fprintf(stderr,"CryptEncrypt(1) FAILED!\n");
			return -1;
		}
		if (!CryptEncrypt(hKey, 0, TRUE, 0, outputBlock, &dwSize, dwSize)) {
			fprintf(stderr,"CryptEncrypt(2) FAILED!\n");
			return -1;
		}
		outputSize=dwSize;
		CryptDestroyKey(hKey);
		CryptDestroyHash(hHash);
		CryptReleaseContext(hProv, 0);
#endif
	}
	pbp_s->outputBlock=outputBlock;
	pbp_s->outputSize=outputSize;
	if (CreateThread((LPSECURITY_ATTRIBUTES)NULL,0, (LPTHREAD_START_ROUTINE)flushBlockToFile,(LPVOID)pbp_s,0, &dwTid)==NULL) {
		fprintf(stderr,"CreateThread(flushBlockToFile()) failed!\n");
		fclose(pbp_s->ofp);
		return -1;
	}
	free(pbp_s->inputBlock);
	WaitForSingleObject(blocksWrittenMutex, INFINITE);
	threadActiveCount--;
	ReleaseMutex(blocksWrittenMutex);
	return 0;
}

int flushBlockToFile(PBP_S pbp_s) {
	int i, os=pbp_s->outputSize;

	if (pbp_s->blockNo > 0) {
		waitForBlockNoWritten(pbp_s->blockNo-1);
	}
	if (pbp_s->outputSize <= 0) return -1;
	for (i=0;i<4;i++) {
		fputc(os & 0xff, pbp_s->ofp); os >>= 8;
	}
	if (fwrite(pbp_s->outputBlock, 1, pbp_s->outputSize, pbp_s->ofp)<pbp_s->outputSize) {
		fprintf(stderr,"Could not write buffer - filesystem full?\nprocessFile() failed!\n");
		fclose(pbp_s->ofp);
		return -1;
	}
#ifdef WIN32
	WaitForSingleObject(blocksWrittenMutex, INFINITE);
	blocksWrittenList=realloc(blocksWrittenList, (1+blocksWrittenCount)*sizeof(int));
	blocksWrittenList[blocksWrittenCount]=pbp_s->blockNo;
	blocksWrittenCount++;
	ReleaseMutex(blocksWrittenMutex);
	free(pbp_s->outputBlock);
#endif
	return 1;
}

void waitForBlockNoWritten(int blockNo) {
#ifdef WIN32
	int i;

	for (;;) {
		WaitForSingleObject(blocksWrittenMutex, INFINITE);
		for (i=0;i<blocksWrittenCount;i++) {
			if (blocksWrittenList[i]==blockNo) break;
		}
		ReleaseMutex(blocksWrittenMutex);
		if (i < blocksWrittenCount) break;
		Sleep(50);
	}
#endif
}

int str2BlockSizeKB(char *s) {
	int bs=atoi(s);
	if (bs < MIN_BLOCKSIZE_KB || bs > MAX_BLOCKSIZE_KB) {
		fprintf(stderr,"Illegal blocksize [%d]\n", bs); return -1;
	}
	return bs;
}

uchar *getMD5Hash(uchar *inputBuffer, int inputLength) {
#ifdef WIN32
	DWORD dwStatus = 0;
	bool bResult = FALSE;
	HCRYPTPROV hProv = 0;
	HCRYPTHASH hHash = 0;
	HANDLE hFile = NULL;
	BYTE *rgbHash=malloc(MD5LEN);
	DWORD cbHash = MD5LEN;
	CHAR rgbDigits[] = "0123456789abcdef";

	if (!CryptAcquireContext(&hProv, NULL, NULL, PROV_RSA_FULL, CRYPT_VERIFYCONTEXT)) {
		dwStatus = GetLastError();
		fprintf(stderr,"CryptAcquireContext failed: %d\n", dwStatus); 
        return NULL;
    }

	if (!CryptCreateHash(hProv, CALG_MD5, 0, 0, &hHash)) {
		dwStatus = GetLastError();
		fprintf(stderr,"CryptAcquireContext failed: %d\n", dwStatus); 
		CryptReleaseContext(hProv, 0);
		return NULL;
	}
	if (!CryptHashData(hHash, inputBuffer, inputLength, 0)) {
		dwStatus = GetLastError();
		fprintf(stderr,"CryptHashData failed: %d\n", dwStatus); 
		CryptReleaseContext(hProv, 0);
		CryptDestroyHash(hHash);
		CloseHandle(hFile);
		return NULL;
	}
    if (!CryptGetHashParam(hHash, HP_HASHVAL, rgbHash, &cbHash, 0)) {
    	return NULL;
    }
    return rgbHash;
#endif
}

void usage() {
	fprintf(stderr,"usage: %s [-options] file ...\n", progName);
	fprintf(stderr,"valid options are:\n");
	fprintf(stderr,"\t-b blockSize in KB (%d-%d) DEFAULT=%d\n",MIN_BLOCKSIZE_KB,MAX_BLOCKSIZE_KB,DEFAULT_BLOCKSIZE_KB);
	fprintf(stderr,"\t-d = decompress\n");
	fprintf(stderr,"\t-D = debug level on!\n");
	fprintf(stderr,"\t-f = force overwriting output file\n");
	fprintf(stderr,"\t-p password - implies crypt flag\n");
	fprintf(stderr,"\t-r = remove source file after processing\n");
	fprintf(stderr,"\t-t = test (crypted)/compressed file\n");
	fprintf(stderr,"\t-v = set verbose flag\n");
	exit(1);
}
