#include <QtGui/QColor>
#include <QtGui/QColorDialog>
#include "data.h"
#include "palette.h"

Palette::Palette(int p_index, Data *p_data, QWidget *parent, Qt::WindowFlags f) : QWidget(parent, f) {

  index = p_index;
  data = p_data;
  setAutoFillBackground(true);
  setMinimumSize(2 * data->getMaxIterations(index) + 2 * MARGIN, 120);
  curIndex = 0;
}

Palette::~Palette() {

}

void Palette::paintEvent(QPaintEvent *) {

  QPainter p;
  int i, pos;
  double scale;
  
  scale = data->getPaletteScale(index);
  p.begin(this);
  for (i = 0; i < width() - 2 * MARGIN; i++) {
    p.setPen(data->getPalette(index, (int)((double)i * scale)));
    p.drawLine(i + MARGIN, 20, i + MARGIN , height() - 21);
  }  
  for (i = 0; i < data->getColorCount(index); i++) {
    pos = data->getColorPos(index, i);
    if (i == curIndex) {
      p.setPen(QPen(QColor(150, 50, 50), 3));
    } else {
      p.setPen(QPen(QColor(100, 100, 100), 3));
    }  
    p.drawLine((int)((double)pos/scale) + MARGIN, 0, (int)((double)pos/scale) + MARGIN, 18);
    p.drawLine((int)((double)pos/scale) + MARGIN, height() - 19, (int)((double)pos/scale) + MARGIN, height() - 1);
  }
  p.end();
}

void Palette::mousePressEvent (QMouseEvent *ev) {

  QColor color;
  int i, nearest, dist, minDist;
  double scale;

  if (ev->button() == Qt::LeftButton) {
    mousePos = ev->pos();
    mousePressed = true;
    scale = data->getPaletteScale(index);
    if (data->getColorCount(index) > 1) {
      nearest = 0;
      dist = abs(mousePos.x() - MARGIN - (int)((double)data->getColorPos(index, 0) / scale));
      minDist = dist;
      for (i = 1; i < data->getColorCount(index); i++) {
        dist = abs(mousePos.x() - MARGIN - (int)((double)data->getColorPos(index, i) / scale));
        if (dist < minDist) {
          minDist = dist;
          nearest = i;
        }
      }
      if (minDist < 5) {
        curIndex = nearest;
        repaint();
      }
    }  
  }  
}

void Palette::mouseReleaseEvent (QMouseEvent *ev) {

  if (ev->button() == Qt::LeftButton) {
    mousePressed = false;
  }  
  if (ev->button() == Qt::RightButton) {
    QColor color = QColorDialog::getColor();
    if (color.isValid()) {
      data->addColor(index, (int)((double)(ev->x() - MARGIN) * data->getPaletteScale(index)), color, curIndex);
      repaint();
      emit paletteChanged();
    }  
  }
}

void Palette::mouseMoveEvent (QMouseEvent *ev) {

  int newPos;
  double scale;
  
  if (mousePressed) {
    mousePos = ev->pos();
    if (curIndex && curIndex < (data->getColorCount(index) - 1)) {
      scale = data->getPaletteScale(index);
      newPos = (int)((double)(mousePos.x() - MARGIN) * scale);
      if ((newPos > 0) && (newPos < data->getMaxIterations(index))) {
        data->setColorPos(index, curIndex, newPos);
        repaint();
        emit paletteChanged();
      }
    }  
  }
}

void Palette::editColor() {

  int pos;
  QColor initial;
  
  initial = data->getColor(index, curIndex, pos);
  QColor color = QColorDialog::getColor(initial);
  if (color.isValid()) {
    data->setColor(index, curIndex, color);
    repaint();
    emit paletteChanged();
  }  
}

void Palette::deleteColor() {

  if (curIndex && curIndex < (data->getColorCount(index) - 1)) {
    data->delColor(index, curIndex);
    curIndex--;
    repaint();
    emit paletteChanged();
  }  
}
