﻿using System;
using System.Collections.Generic;
using System.Linq;
using WWWings_GO;

namespace WWWings_DZS
{

 /// <summary>
 /// Datenmanager für Passagiere
 /// </summary>
 public class PassagierDataManager : IDisposable
 {

  // Eine Instanz des Framework-Kontextes pro Manager-Instanz
  WWWingsModellContainer modell = new WWWingsModellContainer();

  /// <summary>
  /// Konstruktor
  /// </summary>
  public PassagierDataManager(bool LazyLoading = false)
  {
   modell.ContextOptions.LazyLoadingEnabled = LazyLoading;
  }

  /// <summary>
  /// Objekt vernichten
  /// </summary>
  public void Dispose()
  {
   modell.Dispose();
  }

  /// <summary>
  /// Holt einen Passagier
  /// </summary>
  public Passagier GetPassagier(int PassagierID)
  {
   // .OfType<Passagier>() notwendig wegen Vererbung
   var abfrage = from p in modell.PersonSet.OfType<Passagier>() where p.ID == PassagierID select p;
   return abfrage.SingleOrDefault();
  }


  /// <summary>
  /// Holt alle Passagiere mit einem Namensbestandteil
  /// </summary>
  public List<Passagier> GetPassagiere(string Namensbestandteil)
  {
   // .OfType<Passagier>() notwendig wegen Vererbung
   var abfrage = from p in modell.PersonSet.OfType<Passagier>() where p.Name.Contains(Namensbestandteil) || p.Vorname.Contains(Namensbestandteil) select p;
   return abfrage.ToList();
  }

  /// <summary>
  /// Füge einen Passagier zu einem Flug hinzu
  /// </summary>
  public bool AddPassagierZuFlug(int PassagierID, int FlugID)
  {
   try
   {
    Flug flug = modell.FlugSet.Where(f => f.ID == FlugID).SingleOrDefault();
    Passagier passagier = GetPassagier(PassagierID);
    flug.Passagier.Add(passagier);

    modell.SaveChanges();
    return true;
   }
   catch (Exception ex)
   {
    return false;
   }
  }

  /// <summary>
  /// Änderungen an einer Liste von Passagieren speichern
  /// Die neu hinzugefügten Passagiere muss die Routine wieder zurückgeben, da die IDs für die 
  /// neuen Passagiere erst beim Speichern von der Datenbank vergeben werden
  /// </summary>
  public List<Passagier> SavePassagierSet(List<Passagier> PassagierSet, out string Statistik)
  {

   // Änderungen für jeden einzelnen Passagier übernehmen
   foreach (Passagier p in PassagierSet)
   {
    modell.PersonSet.ApplyChanges(p);
   }

   // Statistik der Änderungen zusammenstellen
   Statistik = "";
   Statistik += "Geändert: " + modell.ObjectStateManager.GetObjectStateEntries(System.Data.EntityState.Modified).Count();
   Statistik += " Neu: " + modell.ObjectStateManager.GetObjectStateEntries(System.Data.EntityState.Added).Count();
   Statistik += " Gelöscht: " + modell.ObjectStateManager.GetObjectStateEntries(System.Data.EntityState.Deleted).Count();

   // Neue Datensätze merken, da diese nach Speichern zurückgegeben werden müssen (haben dann erst ihre IDs!)
   List<Passagier> NeuePassagiere = PassagierSet.Where(f => f.ChangeTracker.State == ObjectState.Added).ToList();
   
   // Änderungen speichern
   modell.SaveChanges();

   // Statistik der Änderungen zurückgeben
   return NeuePassagiere;
  }


 }
}
