package de.ebuchner.je;

import java.awt.Component;
import java.awt.datatransfer.*;
import java.awt.dnd.*;
import java.io.*;
import java.util.*;

/**
 * Adapter fuer Drop-Events in der GUI
 */
public class JeDTAdapter implements DropTargetListener {

  // Welche Drop-Actions unterstuetzt dieser Adapter?
  final int dropActions = DnDConstants.ACTION_COPY_OR_MOVE;

  JeDirector director;

  /**
   * Konstruktor
   * @param component GUI-Komponente, fuer den der Adapter zustaendig ist
   */
  public JeDTAdapter(Component c, JeDirector director) {
    this.director = director;
    // Anwendung als DropTarget
    new DropTarget(c, dropActions, this);
  }

  public void dragEnter(DropTargetDragEvent dtde) {
    checkDrop(dtde);
  }

  public void dragOver(DropTargetDragEvent dtde) {
    checkDrop(dtde);
  }

  public void dropActionChanged(DropTargetDragEvent dtde) {
    checkDrop(dtde);
  }

  public void dragExit(DropTargetEvent dte) {
  }

  public void drop(DropTargetDropEvent dtde) {
    int action = dtde.getDropAction();

    // GUI-Parent
    final Component parent = dtde.getDropTargetContext().getComponent();

    try {
      // Vor dem Datenzugriff
      dtde.acceptDrop(dtde.getDropAction());

      // Datenzugriff
      Transferable t = dtde.getTransferable();
      List files = (List)t.getTransferData(DataFlavor.javaFileListFlavor);

      File cwd = director.getCwd();
      if(cwd==null) {
        dtde.dropComplete(false);
        System.err.println("Bug: Current working directory is null");
        return;
      }

      List srcFiles = new ArrayList((Collection)t.getTransferData(DataFlavor.javaFileListFlavor));
      List trgFiles = new ArrayList();

      // Pruefen
      for(Iterator it=srcFiles.iterator(); it.hasNext(); ) {
        File src = (File)it.next();
        if(src.isDirectory()) {
          System.err.println("Skipping " + src + ". Cannot drag directories");
          it.remove();
          continue;
        }

        File target = new File(cwd, src.getName());
        if(target.exists()) {
          System.err.println("Skipping " + src + ". A file with that name exists");
          it.remove();
          continue;
        }

        if(target.equals(src)) {
          System.err.println("Skipping " + src + ". Source and target are the same");
          it.remove();
          continue;
        }
        trgFiles.add(target);
      }

      boolean ok;

      if ((action&DnDConstants.ACTION_COPY)>0)
        ok = copyFiles(srcFiles.iterator(), trgFiles.iterator());
      else
        ok = moveFiles(srcFiles.iterator(), trgFiles.iterator());

      dtde.dropComplete(ok);
      director.refresh();

    } catch(Exception e) {
      e.printStackTrace();
      dtde.dropComplete(false);
    }
  }

  private boolean copyFiles(Iterator itSrc, Iterator itTrg) {
    boolean ok = true;
    while(ok && itSrc.hasNext() && itTrg.hasNext())
      ok = copy((File)itSrc.next(), (File)itTrg.next());
    return ok;
  }

  private boolean moveFiles(Iterator itSrc, Iterator itTrg) {
    boolean ok = true;
    while(ok && itSrc.hasNext() && itTrg.hasNext())
      ok = move((File)itSrc.next(), (File)itTrg.next());
    return ok;
  }

  public boolean copy(File src, File target) {
    // Muss man das selbst machen?
    FileInputStream is = null;
    FileOutputStream os = null;
    try {
      is = new FileInputStream(src);
      os = new FileOutputStream(target);
      byte [] buf = new byte[1024];
      int len;
      while((len=is.read(buf))>0)
        os.write(buf, 0, len);
      return true;
    } catch(IOException ioe) {
      ioe.printStackTrace();
      return false;
    } finally {
      try {
        if(is!=null) is.close();
        if(os!=null) os.close();
      } catch(IOException ioe) {
        ioe.printStackTrace();
        return false;
      }
    }
  }

  public boolean move(File src, File target) {
    try {
      src.renameTo(target);
      return true;
    } catch (SecurityException se) {
      se.printStackTrace();
      return false;
    }
  }

  /**
   * Hilfsroutine: Prueft, ob der Baum das Datenformat und die Aktion
   * unterstuetzt
   */
  private void checkDrop(DropTargetDragEvent dtde) {
    // Dateiliste?
    boolean accept = dtde.isDataFlavorSupported(DataFlavor.javaFileListFlavor);

    // Copy oder Move?
    int dropAction = dtde.getDropAction();

    if(accept && (dropAction & dropActions)>0)
      dtde.acceptDrag(dropAction);
    else
      dtde.rejectDrag();
  }
}
