// 
// Exercise 1 in part 1 of the C#/.NET tutorial
// Example library DLL written in C# by Michael Stal in 2001
// struct Point: trivial implementation of geometric 2D points 
// class Polygon: implementation of Polygons consisting of a closed polygon line where no point is appearing twice
//

using System;
using System.Collections;

namespace exercise1_shapeslib
{
	/// <summary>
	/// exercise1_shapeslib.Point struct
	/// </summary>
	public struct Point {
		/// <summary>
		/// long m_instances denotes number of instances already created
		/// </summary>
		static long m_instances;
		/// <summary>
		/// class constructor 
		/// </summary>
		static Point () 
		{
			m_instances = 0;
			Console.WriteLine("Class \"{0}\" instantiated", typeof(Point).Name);
		}

		/// <summary>
		/// Enumeration type Coordinate defines kind of coordinate
		/// </summary>
		public enum Coordinate : short { X, Y };
		/// <summary>
		/// x coordinate: type double
		/// </summary>
		private double m_x;
		/// <summary>
		/// y coordinate: type double
		/// </summary>
		private double m_y;

		/// <summary>
		/// constructor
		/// </summary>
		/// <param name="x"> x coordinate </param>
		/// <param name="y"> y coordinate </param>
		public Point (double x, double y) 
		{
			// deleted sample exception
			m_x = x;
			m_y = y;
		}

		/// <summary>
		/// indexer to retrieve a particular coordinate
		/// </summary>
		public double this[Coordinate which] 
		{
			get 
			{
				if (which == Coordinate.X) return m_x; else return m_y;
			}
			set 
			{
				if (which == Coordinate.X)	m_x = value; else m_y = value;
			}
		}

		/// <summary>
		/// property double x represents x coordinate
		/// </summary>
		public double x 
		{
			set { m_x = value; }
			get { return m_x;  }
		}

		/// <summary>
		/// property double y represents y coordinate
		/// </summary>
		public double y 
		{
			set { m_y = value;  }
			get { return m_y; }
		}	

		/// <summary>
		/// Add two points resulting in vector addition
		/// </summary>
		/// <param name="op1"> first point to add </param>
		/// <param name="op2"> second point to add </param>
		/// <returns> result of vector addition</returns>
		public static Point operator+(Point op1, Point op2) 
		{
			return new Point(op1.x+op2.x,op1.y+op2.y);
		}

		/// <summary>
		/// Display point as coordinate (x,y)
		/// </summary>
		/// <returns> string representation </returns>
		public override string ToString() 
		{
			return "(" + m_x.ToString() + "," + m_y.ToString() + ")";
		}

		/// <summary>
		/// Create hash code: must be implemented because Equals is implemented
		/// </summary>
		/// <returns> hash code</returns>
		// Values 5 and 3 are prime constants. This is just one possible example how to implement hashes.
		public override int GetHashCode() {
			double tmp = 5;
			tmp = 3*tmp + x;
			tmp = 3*tmp + y;
			return (int) tmp;
		}

		/// <summary>
		/// Compares two polygons for logical equality
		/// </summary>
		/// <param name="o"> second polygon</param>
		/// <returns> true if polygons are equal, wrong otherwise </returns>
		public override bool Equals(object o) {
			if (!(o is Point))
				return false; // wrong type
			else {
				Point rhs = (Point)o;
				return ((this.x == rhs.x) && (this.y == rhs.y)) ? true : false;
			}
		}

		/// <summary>
		/// Swap x and y coordinate
		/// </summary>
		/// <param name="o"> ref Point o: point to swap</param>
		public static void swapXY(ref Point o) 
		{
			double tmp = o.m_x;
			o.m_x = o.m_y;
			o.m_y = tmp;
		}	
	}

	/// <summary>
	/// exercise1_shapeslib.Polygon: a polygon is represented by a polygon line with each point only appearing once.
	/// </summary>
	public class Polygon : IEnumerable {

		/// <summary>
		/// m_Points contains list of points
		/// </summary>
		// unsorted list of points
		private ArrayList m_Points;

		/// <summary>
		/// constructor takes one point: no empty polygons
		/// </summary>
		/// <param name="p"> start point </param>
		// we assume that a polygon consists of at least one point
		public Polygon(Point p) {
			m_Points = new ArrayList();
			m_Points.Add(p);
		}

		/// <summary>
		/// get position of a point in a polygon
		/// </summary>
		/// <param name="q"> point to look for</param>
		/// <returns> position if point is found, -1 otherwise </returns>
		public virtual int GetPointPosition(Point q) {
			for(int pos = 0; pos < m_Points.Count;pos++) {
				if (m_Points[pos].Equals(q)) return pos;
			}
			return -1; // not found
		}

		/// <summary>
		/// indexer to retrieve a particular point
		/// </summary>
		public virtual Point this[int i] {
			get {
				if ((i < 0) || (i >= Count)) 
					throw new ArgumentOutOfRangeException();
				else
					return (Point)m_Points[i];
			}
		}

		/// <summary>
		/// get the size of the polygon, i.e., the number of points
		/// </summary>
		public virtual int Count {
			get {
				return m_Points.Count;
			}
		}

		/// <summary>
		/// insert a point into the polygon
		/// </summary>
		/// <param name="pos"> position where to insert </param>
		/// <param name="q"> point to insert </param>
		public virtual void Insert(int pos, Point q) {
			if (GetPointPosition(q) != -1) // prevent duplettes
				throw new ArgumentException("Point already in polygon");
			if ((pos < 0) || (pos > Count)) // insertion not possible
				throw new ArgumentOutOfRangeException();
			m_Points.Insert(pos, q);
		}

		/// <summary>
		/// add a point at the end of the current polygon line
		/// </summary>
		/// <param name="q">point to add</param>
		public virtual void Add(Point q) 
		{
			if (GetPointPosition(q) != -1) // prevent duplettes
				throw new ArgumentException("Point already in polygon");
			m_Points.Add(q);
		}

		/// <summary>
		/// Get iterator
		/// </summary>
		/// <returns>object of type IEnumerator</returns>
		public IEnumerator GetEnumerator() {
			return new PolygonEnumerator(m_Points);
		}

		/// <summary>
		/// exercise1_shapeslib.Polygon.PolygonEnumerator: nested class for iterating over the points in a polygon
		/// </summary>
		public class PolygonEnumerator : IEnumerator {
			/// <summary>
			/// position variable of type int m_Pos
			/// </summary>
			private int m_Pos = -1;
			
			/// <summary>
			/// list of points
			/// </summary>
			private ArrayList m_Points;

			/// <summary>
			/// Constructor
			/// </summary>
			/// <param name="al">the member variable of type array list is initialized with copy of al</param>
			public PolygonEnumerator(ArrayList al) {
				m_Points = (ArrayList)al.Clone();
			}

			/// <summary>
			/// return to initial position
			/// </summary>
			public void Reset() {
				m_Pos = -1;
			}

			/// <summary>
			/// retrieve current object
			/// </summary>
			public object Current {
				get {
					if ((m_Pos == -1) || (m_Pos >= m_Points.Count))
						return null; 
					else {
						return m_Points[m_Pos];
					}
				}
			}	

			/// <summary>
			/// move to next object in list
			/// </summary>
			/// <returns> true if next object exists, false otherwise</returns>
			public bool MoveNext() {
				m_Pos++;
				return (m_Pos >= m_Points.Count) ? false : true;
			}
		}

		/// <summary>
		/// representing a polygon: inefficient approach for sake of brevity
		/// </summary>
		/// <returns>string representation, e.g., [(x1,y1)(x2,y2)...(xk,yk)]</returns>
		public override string ToString() {
			String res = "[";
			foreach (Point q in this) {
				res += q.ToString();
			}
			res += "]";
			return res;
		}

		/// <summary>
		/// Calculate hash calue
		/// </summary>
		/// <returns>hash value</returns>
		public override int GetHashCode() {
			double tmp = 5;
			foreach (Point q in this)
				tmp = 3*tmp + q.GetHashCode();
			return (int) tmp;
		}

		/// <summary>
		/// search forward
		/// </summary>
		/// <param name="p">first polygon</param>
		/// <param name="i">first index</param>
		/// <param name="q">second polygon </param>
		/// <param name="j">second index</param>
		/// <returns> true, if polygons are forward-equal, false otherwise </returns>
		// search in same direction.
		// precond: p and q of same size
		//          i and j refer to existing points within p and q
		//          p[i].Equals(q[j]) == true
		private static bool parallelCompare(Polygon p, int i, Polygon q, int j) {
			int count = p.Count;
			for (int k = 1; k < count; k++) {
				Point pPoint = p[(i + k) % count];
				Point qPoint = q[(j + k) % count];
				if (!pPoint.Equals(qPoint))	
					return false;
			}
			return true;
		}

		/// <summary>
		/// search backward
		/// </summary>
		/// <param name="p">first polygon</param>
		/// <param name="i">first index</param>
		/// <param name="q">second polygon</param>
		/// <param name="j">second index</param>
		/// <returns>true, if polygons are backward-equal, false otherwise</returns>
		// search in opposite direction.
		// precond: p and q of same size
		//          i and j refer to existing points within p and q
		//          p[i].Equals(q[j]) == true
		private static bool oppositeCompare(Polygon p, int i, Polygon q, int j) 
		{
			int count = p.Count;
			for (int k = 1; k < count; k++) 
			{
				Point pPoint = p[(i + k) % count];
				Point qPoint = q[(j + count- k) % count];
				if (!pPoint.Equals(qPoint))	
					return false;
			}
			return true;
		}

		/// <summary>
		/// test polygons for equality
		/// </summary>
		/// <param name="o">second polygon</param>
		/// <returns>true if polygons are equal, false otherwise</returns>
		public override bool Equals(object o) {
			if (!(o is Polygon))
				return false; // wrong type
			else {
				Polygon rhs = (Polygon)o;
				// same number of points?
				if (rhs.Count != this.Count) return false;
				// take first point of this polygon ...
				Point p1 = this[0];
				// and search for it in second polygon
				int posInRhs = rhs.GetPointPosition(p1);
				// if not found, polygons cannot be equal
				if (posInRhs == -1) return false;
				// otherwise compare in same and opposite direction
				return parallelCompare(this, 0, rhs, posInRhs) || oppositeCompare(this, 0, rhs, posInRhs);
			}
		}
	}
}
