// 
// Exercise 2 in part 2 of the C#/.NET tutorial
// Example library DLL written in C# by Michael Stal in 2001
// Contains ShareTicker server
//

using System;
using System.Collections;


namespace Shares
{
	public class NotFoundException : Exception {}
	public class ExistsException : Exception {}

	/// <summary>
	/// struct Share contains info about shares
	/// </summary>
	public struct Share {
		private readonly string m_ShortName;
		private readonly string m_CompanyName;
		private readonly string m_URL;
		private decimal m_Value;

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="sn"> short name of company </param>
		/// <param name="cn"> long name of company </param>
		/// <param name="url"> Web address </param>
		public Share(string sn, string cn, string url) {
			m_ShortName = sn;
			m_CompanyName = cn;
			m_URL = url;
			m_Value = 0;
		}

		/// <summary>
		/// Constructor 2
		/// </summary>
		/// <param name="sn"> short name of company </param>
		/// <param name="cn"> long name of company </param>
		/// <param name="url"> Web address </param>
		/// <param name="val"> share value </param>
		public Share(string sn, string cn, string url, decimal val) : this(sn,cn,url) {
			m_Value = val;
		}

		public string ShortName { get { return m_ShortName; } }
		public string CompanyName { get { return m_CompanyName; } }
		public string URL { get { return m_URL; } }
		public decimal Value 
		{
			get { return m_Value; }
			set { m_Value = value; }
		}	
	}

	/// <summary>
	/// ChangeEventArgs is returned when shares change
	/// </summary>
	sealed public class ChangeEventArgs : EventArgs {
		private readonly string m_ShortName;
		private readonly decimal m_OldValue;
		private readonly decimal m_NewValue;

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="sn"> short name of share </param>
		/// <param name="oldval"> old share value </param>
		/// <param name="newval"> new share value </param>
		public ChangeEventArgs(string sn, decimal oldval, decimal newval) {
			m_ShortName = sn;
			m_OldValue = oldval;
			m_NewValue = newval;
		}
		public string ShortName { get { return m_ShortName; } }
		public decimal OldValue { get { return m_OldValue; } }
		public decimal NewValue { get { return m_NewValue; } }
	}

	/// <summary>
	/// delegate for notification about share value changes
	/// </summary>
	public delegate void ChangeEvent(object src, ChangeEventArgs e);

	/// <summary>
	/// StockTicker is the container of shares
	/// </summary>
	public class StockTicker {
		protected Hashtable m_Map;
		public event ChangeEvent OnChangeEvent;

		/// <summary>
		/// Constructor initializes a hash map
		/// </summary>
		public StockTicker() {
			m_Map = new Hashtable(1013);
		}

		/// <summary>
		/// indexer let's you access shares
		/// </summary>
		public Share this[string shortName] {
			set 
			{
				lock(this) {
					if (m_Map.ContainsKey(shortName))
						throw new ExistsException();
					else
						m_Map[shortName] = value;
				}
			}
			get {
				lock(this) {
					if (!m_Map.ContainsKey(shortName))
						throw new NotFoundException();
					else
						return (Share)m_Map[shortName];
				}
			}
		}

		/// <summary>
		/// ChangeValue used to change shares
		/// </summary>
		/// <param name="shortName"> short name of share </param>
		/// <param name="newval"> new value </param>
		public void ChangeValue (string shortName, decimal newval) {
			lock(this) {
				if (!m_Map.ContainsKey(shortName))
					throw new NotFoundException();
				else {
					Share s = (Share)m_Map[shortName];
					decimal oldval = s.Value;
					s.Value = newval;
					ChangeEventArgs args = new ChangeEventArgs(shortName, oldval, newval);
					if (OnChangeEvent != null) OnChangeEvent(this, args);
				}
			}
		}
	}

	/// <summary>
	/// StockTickerEventAdapter intercepts particular events from the stock ticker on behalf of a client
	/// </summary>
	public class StockTickerEventAdapter
	{
		protected ChangeEvent m_Callback ;
		protected string m_ShortName;

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="ticker"> non-null StockTicker to be used </param>
		/// <param name="shortName"> short name of share </param>
		/// <param name="callback"> non-null callback to be called when specified share is being changed </param>
		// for simplicity we only allow to match one single share by its short name:
		public StockTickerEventAdapter(StockTicker ticker, string shortName, ChangeEvent callback){
			// nulls are not allowed:
			if ((ticker == null) || (callback == null))
				throw new ArgumentException();
			// register with StockTicker instance:
			ticker.OnChangeEvent += new ChangeEvent(OnChange);
			// register method passed with own eventsystem:
			m_Callback = callback;
			// assign name
			m_ShortName = shortName;
		}

		/// <summary>
		/// OnChange is our callback we register with the StockTicker
		/// </summary>
		/// <param name="src"> originator of change </param>
		/// <param name="e"> details about event </param>
		public void OnChange(object src, ChangeEventArgs e) 
		{
			// if client has specified interest notify it:
			if (e.ShortName.Equals(m_ShortName)) m_Callback(src, e);
		}
	}
}
