/**
 * Copyright @ 2002 Lars Rwekamp (lars.roewekamp@openKnowlege.de)
 *
 * Source is only for non commercialand coaching usage.
 *
 * Not Warranty to use it.
 */

package de.ix.wstutorial.registrybrowser;



/* imports */

import de.ix.wstutorial.util.soap.SOAPRPCResponse;

import javax.servlet.http.*;
import javax.servlet.*;

import javax.xml.soap.*;
import javax.xml.soap.Name;
import javax.xml.messaging.URLEndpoint;
import javax.xml.registry.*;
import javax.xml.registry.infomodel.*;
import java.util.*;
import java.net.PasswordAuthentication;


/**
 * Einfaches Servlet zum Durchsuchen einer XML
 * Registry mittels JAXR API.
 *
 * Das Servlet ...
 *
 */
public class RegistryBrowserServlet extends HttpServlet {

    /* private */

    /* Zielseite, die nach erfolgreicher Abarbeitung angezeigt werden soll. */
    private static final String TARGET_VIEW = "/jsp/de/registrybrowser.jsp";


    /* Konstanten zur Identifizierung der verschiedenen XML Registries */
    private static final String IBM_TEST_REGISTRY = "ibmTestReg";
    private static final String IBM_REGISTRY      = "ibmReg";
    private static final String MS_TEST_REGISTRY  = "msTestReg";
    private static final String MS_REGISTRY       = "msReg";
    private static final String LOCAL_REGISTRY    = "lokalReg";

    /* ACHTUNG: Konstanten zur Authorisierung innerhalb der verschiedenen Registries
                Bei Bedarf bitte mit eigenen Angaben fuellen !
    */
    private static final String IBM_TEST_REG_USER = "";
    private static final String IBM_TEST_REG_PWD  = "";

    private static final String IBM_REG_USER = "";
    private static final String IBM_REG_PWD = "";

    private static final String MS_TEST_REG_USER = "";
    private static final String MS_TEST_REG_PWD  = "";

    private static final String MS_REG_USER = "";
    private static final String MS_REG_PWD = "";

    private static final String LOCAL_REG_USER = "";
    private static final String LOCAL_REG_PWD = "";


    /* Konstanten fuer notwendige Propertie Werte zum Erstellen einer Connection */
    private static final String PROP_QUERY_MANAGER_URL = "javax.xml.registry.queryManagerURL";
    private static final String PROP_LIFE_CYCLE_URL    = "javax.xml.registry.lifeCycleManagerURL";
    private static final String PROP_FACTORY_CLASS     = "javax.xml.registry.factoryClass";


    private static final String PROP_HTTP_HOST  = "javax.xml.registry.http.proxyHost";
    private static final String PROP_HTTP_PORT  = "javax.xml.registry.http.proxyPort";
    private static final String PROP_HTTPS_HOST = "javax.xml.registry.https.proxyHost";
    private static final String PROP_HTTPS_PORT = "javax.xml.registry.https.proxyPort";


    /* ACHTUNG: Wenn sie hinter einer Firewall agieren und ein Proxy vorgeschaltet
                ist, muessen sie diese Werte korrekt fuellen. Falls sie die
                notwendigen Angaben nicht kennen, fragen sie ihren Systemadministrator.
    */
    private static final String MY_HTTP_HOST    = "localhost";
    private static final String MY_HTTP_PORT    = "8080";
    private static final String MY_HTTPS_HOST   = "myHost.myDomain";
    private static final String MY_HTTPS_PORT   = "8080";

    /* Konstanten fuer erlaubte Request Parameter */
    private static final String PARAM_REGISTRY_ID       = "regID";
    private static final String PARAM_ORGANIZATION_NAME = "orgName";

    /* Konstanten fuer zu setzende Reqeust Attribute */
    private static final String ATTR_REGISTRY_NAME      = "regName";
    private static final String ATTR_REGISTRY_FLAG      = "registryFlag";
    private static final String ATTR_SEARCH_PATTERN     = "searchPattern";
    private static final String ATTR_MATCH_COUNT        = "anzahl";
    private static final String ATTR_ORGANIZATIONS      = "orgs";
    private static final String DEFAULT_REGISTRY_FLAG   = "TRUE";

    private static final String ATTR_REGISTRY_BROWSER_ERROR_FLAG = "registryErrorFlag";
    private static final String DEFAULT_REGISTRY_BROWSER_ERROR_FLAG = "TRUE";
    private static final String ATTR_ERROR_TEXT  = "errorText";
    private static final String ATTR_ERROR_TEXT_DETAIL  = "errorTextDetail";
    private static final String ATTR_ERROR_MESSAGE  = "errorMessage";

    private static final String ATTR_ERROR_REGISTRY_EXCEPTION = "Kein lesender Zugriff mglich.";
    private static final String ATTR_ERROR_CONNECTION_EXCEPTION = "Verbindung kann nicht aufgebaut werden.";
    private static final String ATTR_ERROR_GENERAL_EXCEPTION = "Allgemeiner Fehler.";


    /* Konstanten fuer den Zugriff auf die IBM UDDI Registry */
    private static final String IBM_REG_INQUIRY_URL = "http://www-3.ibm.com/services/uddi/v2beta/inquiryapi";
    private static final String IBM_REG_PUBLISH_URL = "http://www-3.ibm.com/services/uddi/v2beta/protect/publishapi";

    /* Konstanten fuer den Zugriff auf die IBM UDDI Registry (Test Version)*/
    private static final String IBM_TEST_REG_INQUIRY_URL = "http://www-3.ibm.com/services/uddi/testregistry/inquiryapi";;
    private static final String IBM_TEST_REG_PUBLISH_URL = "http://www-3.ibm.com/services/uddi/v2beta/protect/publishapi";

    /* Konstanten fuer den Zugriff auf die Microsoft UDDI Registry */
    private static final String MS_REG_INQUIRY_URL = "http://uddi.rte.microsoft.com/inquire";
    private static final String MS_REG_PUBLISH_URL = "http://uddi.rte.microsoft.com/publish";

    /* Konstanten fuer den Zugriff auf die Microsoft UDDI Registry (Test Version)*/
    private static final String MS_TEST_REG_INQUIRY_URL = "http://test.uddi.microsoft.com/inquire";
    private static final String MS_TEST_REG_PUBLISH_URL = "http://test.uddi.microsoft.com/publish";

    /* Konstanten fuer den Zugriff auf die Lokale UDDI Registry */
    private static final String LOCAL_REG_INQUIRY_URL = "http://localhost:8080/registry-server/RegistryServerServlet";
    private static final String LOCAL_REG_PUBLISH_URL = "http://localhost:8080/registry-server/RegistryServerServlet";

    /* Konstanten fuer die benoetigte Connetion Factory Klasse */
    private static final String UDDI_CONNECTION_FACTORY = "com.sun.xml.registry.uddi.ConnectionFactoryImpl";

    /**
     * Verbindung zur XML Registry.
     */
    Connection connection;

    /**
     * Initialisierung des Servlets.
     *
     * @param servletConfig       Aktuelle Konfiguration des Servlets
     * @throws ServletException   Servlet spezifische Exception
     */

    public void init(ServletConfig servletConfig) throws
            ServletException {
        super.init(servletConfig);
    }


    /**
     * Verabeitung eines HTTP Get Aufrufs.
     *
     * Im Verlauf der Abarbeitung werden folgende Schritte durchlaufen:
     *
     *   1. Aufbau einer Verbindung mittels ConnectionFactory.
     *   2. Zugriff auf das RegistryService Objekt als Repraesentant der XML Registry.
     *   3. Zugriff auf den BusinessQueryManaer fuer spaetere Abfragen.
     *   4. Aufbauen der Abfrage mit Hilfe der uebergebenen Request Parameter.
     *   5. Absetzen der Abfrage.
     *   6. Durchlaufen des Results und "Verpacken" in einfachen Rueckgabeobjekten.
     *   7. Ruekgabe der Objekte mittels Request Attribute
     *   8. Schliessen der Connection.
     *   9. Weiterleiten/Umleiten auf JSP
     *
     * @param request              HTTP Servlet Request Objekt
     * @param response             HTTP Servlet Response Objekt
     * @throws ServletException    Servlet spezifische Exception
     */

    public void doGet(HttpServletRequest request, HttpServletResponse response)
            throws ServletException {

        /* extrahieren der Request Parameter  */

        try {

            String regID            = request.getParameter(PARAM_REGISTRY_ID);
            String organizationName = request.getParameter(PARAM_ORGANIZATION_NAME);

            /* initialer Aufbau der XML Registry Connection */

            try {

                ConnectionFactory connectionFactory = ConnectionFactory.newInstance();
                Properties properties = getConnectionProperties(regID);
                connectionFactory.setProperties(properties);
                connection = connectionFactory.createConnection();

                /* ACHTUNG: Setzen von User Credentials - Bei Bedarf aktivieren.
                            Wichtig - vorher die notwendigen Zugangswerte in den
                            entsprechenden Konstanten

                            IBM_TEST_REG_USER = ...
                            IBM_TEST_REG_PWD  = ...

                            IBM_REG_USER = ...
                            IBM_REG_PWD = ...
                            ...

                            hinterlegen. */

                /*
                connection.setCredentials(this.getCredentials(regID));
                */

                /* Registry Service Objekt erzeugen */
                RegistryService registryService = null;
                registryService = connection.getRegistryService();

                /* Business Query Manager fuer die Abfrage der XML Registry erzeugen */
                BusinessQueryManager queryManager = registryService.getBusinessQueryManager();

                /* Name Pattern Parameter setzen */
                ArrayList namePatterns = new ArrayList();
                namePatterns.add("%" + organizationName + "%");

                BulkResponse queryResponse = queryManager.findOrganizations(null, namePatterns, null, null, null, null);
                Collection responseCollection = queryResponse.getCollection();

                request.setAttribute(ATTR_REGISTRY_NAME, this.getRegistryName(regID));
                request.setAttribute(ATTR_SEARCH_PATTERN, organizationName);

                if (responseCollection != null) {

                    request.setAttribute(ATTR_MATCH_COUNT, new Integer(responseCollection.size()));

                    Collection shortOrgs = new ArrayList();
                    Collection shortServices = null;
                    Iterator responseIterator = responseCollection.iterator();

                    String orgName = null;
                    String orgDesc = null;
                    String orgKey  = null;

                    String serviceName = null;
                    String serviceDesc = null;
                    String serviceKey  = null;

                    ShortOrganization shortOrg = null;
                    ShortService shortService = null;

                    while (responseIterator.hasNext()) {

                        shortServices = new ArrayList();

                        Organization org = (Organization)responseIterator.next();
                        orgName = org.getName().getValue(Locale.ENGLISH);
                        orgDesc = org.getDescription().getValue(Locale.ENGLISH);
                        orgKey  = org.getKey().getId();

                        Collection services = org.getServices();
                        Iterator serviceIter = services.iterator();
                        while (serviceIter.hasNext()) {
                            Service service = (Service)serviceIter.next();
                            serviceName = service.getName().getValue(Locale.ENGLISH);
                            serviceDesc = service.getDescription().getValue(Locale.ENGLISH);
                            serviceKey  = service.getKey().getId();

                            shortService = new ShortService(serviceName, serviceDesc, serviceKey);
                            shortServices.add(shortService);

                        }
                        shortOrg = new ShortOrganization(orgName,
                                                         orgDesc,
                                                         org.getKey().getId(),
                                                         shortServices);
                        shortOrgs.add(shortOrg);

                    }
                    request.setAttribute(ATTR_ORGANIZATIONS, shortOrgs);
                    request.setAttribute(ATTR_REGISTRY_FLAG, DEFAULT_REGISTRY_FLAG);

                } else {
                    request.setAttribute(ATTR_MATCH_COUNT, new Integer(0));
                    request.setAttribute(ATTR_ORGANIZATIONS, null);
                    request.setAttribute(ATTR_REGISTRY_FLAG, DEFAULT_REGISTRY_FLAG);
                }

                connection.close();

            } catch (RegistryException ex) {

                /* Fehler beim lesenden Zugriff auf die Registry */
                request.setAttribute(ATTR_REGISTRY_BROWSER_ERROR_FLAG, DEFAULT_REGISTRY_BROWSER_ERROR_FLAG);
                request.setAttribute(ATTR_ERROR_TEXT, ATTR_ERROR_REGISTRY_EXCEPTION);
                request.setAttribute(ATTR_ERROR_TEXT_DETAIL, getRegistryName(regID));
                request.setAttribute(ATTR_ERROR_MESSAGE, ex.getMessage());


            } catch (JAXRException ex) {

                /* Genereller Fehler beim  Zugriff auf die Registry, z.B. keine Connection */
                request.setAttribute(ATTR_REGISTRY_BROWSER_ERROR_FLAG, DEFAULT_REGISTRY_BROWSER_ERROR_FLAG);
                request.setAttribute(ATTR_ERROR_TEXT, ATTR_ERROR_CONNECTION_EXCEPTION);
                request.setAttribute(ATTR_ERROR_TEXT_DETAIL, getRegistryName(regID));
                request.setAttribute(ATTR_ERROR_MESSAGE, ex.getMessage());


            } catch (Exception ex) {

                /* Genereller Fehler  */
                request.setAttribute(ATTR_REGISTRY_BROWSER_ERROR_FLAG, DEFAULT_REGISTRY_BROWSER_ERROR_FLAG);
                request.setAttribute(ATTR_ERROR_TEXT, ATTR_ERROR_GENERAL_EXCEPTION);
                request.setAttribute(ATTR_ERROR_TEXT_DETAIL, "");
                request.setAttribute(ATTR_ERROR_MESSAGE, ex.getMessage());
            }



            /* Ziel JSP zur Anzeige des Ergebnisses aufrufen */

            request.getRequestDispatcher(TARGET_VIEW).forward(request, response);


        } catch (Exception ex) {

            /* generelles, rudimentares Exception Handling */
            ex.printStackTrace();
            ex.getMessage();
        }


    }

    /**
     * Liefert den Namen der angegebenen XML Registry.
     *
     * @param registryID  Eindeutige ID der XML Registry
     * @return Namen der gesuchten XML Registry
     *
     */
    private String getRegistryName(String registryID) {
        if (IBM_TEST_REGISTRY.equals(registryID)) {
            return "IBM UDDI V2 Registry (Test Version)";
        } else if (IBM_REGISTRY.equals(registryID)) {
            return "IBM UDDI V2 Registry";
        }  else if (MS_TEST_REGISTRY.equals(registryID)) {
            return "Microsooft UDDI Registry (Test Version)";
        }  else if (MS_REGISTRY.equals(registryID)) {
            return "Microsooft UDDI Registry";
        }  else if (LOCAL_REGISTRY.equals(registryID)) {
            return "Locale WSDP XML Registry";
        }
        return "unbekannt";
    }


    /**
     * Liefert alle notwendigen Properties fuer die angegebene
     * XML Registry.
     *
     * @param registryID  Eindeutige ID der XML Registry
     * @return Properties fuer die angebene XML Registry
     *
     */
    private Properties getConnectionProperties(String registryID) {

        Properties properties = new Properties();

        if (IBM_TEST_REGISTRY.equals(registryID)) {
            properties.setProperty(PROP_QUERY_MANAGER_URL, IBM_TEST_REG_INQUIRY_URL);
            properties.setProperty(PROP_LIFE_CYCLE_URL,    IBM_TEST_REG_PUBLISH_URL);
            properties.setProperty(PROP_FACTORY_CLASS,     UDDI_CONNECTION_FACTORY);

        } else if (IBM_REGISTRY.equals(registryID)) {
            properties.setProperty(PROP_QUERY_MANAGER_URL, IBM_REG_INQUIRY_URL);
            properties.setProperty(PROP_LIFE_CYCLE_URL,    IBM_REG_PUBLISH_URL);
            properties.setProperty(PROP_FACTORY_CLASS,     UDDI_CONNECTION_FACTORY);

        }  else if (MS_TEST_REGISTRY.equals(registryID)) {
            properties.setProperty(PROP_QUERY_MANAGER_URL, MS_TEST_REG_INQUIRY_URL);
            properties.setProperty(PROP_LIFE_CYCLE_URL,    MS_TEST_REG_PUBLISH_URL);
            properties.setProperty(PROP_FACTORY_CLASS,     UDDI_CONNECTION_FACTORY);

        }  else if (MS_REGISTRY.equals(registryID)) {
            properties.setProperty(PROP_QUERY_MANAGER_URL, MS_REG_INQUIRY_URL);
            properties.setProperty(PROP_LIFE_CYCLE_URL,    MS_REG_PUBLISH_URL);
            properties.setProperty(PROP_FACTORY_CLASS,     UDDI_CONNECTION_FACTORY);

        }  else if (LOCAL_REGISTRY.equals(registryID)) {
            properties.setProperty(PROP_QUERY_MANAGER_URL, LOCAL_REG_INQUIRY_URL);
            properties.setProperty(PROP_LIFE_CYCLE_URL,    LOCAL_REG_PUBLISH_URL);
            properties.setProperty(PROP_FACTORY_CLASS,     UDDI_CONNECTION_FACTORY);
        }


        /* PROXY Settings - bei Bedarf aktivieren */
        /*
        properties.setProperty(PROP_HTTP_HOST,  MY_HTTP_HOST);
        properties.setProperty(PROP_HTTP_PORT,  MY_HTTP_PORT);
        properties.setProperty(PROP_HTTPS_HOST, MY_HTTPS_HOST);
        properties.setProperty(PROP_HTTPS_PORT, MY_HTTPS_PORT);
        */

        return properties;
    }

    /**
     * Liefert ein Set von Credentieals, welches zum Setzen
     * der Zugangsberechtigungen genutzt werden kann.
     *
     * @param registryID  Eindeutige ID der genutzten Registry
     * @return Set mit Credientials
     */
    private Set getCredentials(String registryID) {

        String userName = "";
        String pwd      = "";

        if (IBM_TEST_REGISTRY.equals(registryID)) {
            userName = IBM_TEST_REG_USER;
            pwd      = IBM_TEST_REG_PWD;

        } else if (IBM_REGISTRY.equals(registryID)) {
            userName = IBM_REG_USER;
            pwd      = IBM_REG_PWD;

        }  else if (MS_TEST_REGISTRY.equals(registryID)) {
            userName = MS_TEST_REG_USER;
            pwd      = MS_TEST_REG_PWD;

        }  else if (MS_REGISTRY.equals(registryID)) {
            userName = MS_REG_USER;
            pwd      = MS_REG_PWD;

        }  else if (LOCAL_REGISTRY.equals(registryID)) {
            userName = LOCAL_REG_USER;
            pwd      = LOCAL_REG_PWD;

        }

        PasswordAuthentication pwdAuthentication = new PasswordAuthentication(userName, pwd.toCharArray());
        Set credentials = new HashSet();
        credentials.add(pwdAuthentication);

        return credentials;

    }


}




