using System;

namespace Interfaces
{
	/// <summary>
	/// A worker, which starts working when he is requested to do so and sends notifications to registred people.
	/// </summary>
	public class CWorker:CPerson
	{
		/// <summary>
		/// A list of objects that can be notified
		/// </summary>
		private CNotifyableObjCollection m_colNotificationList;
		/// <summary>
		/// Current status of the work
		/// </summary>
		private EStatus m_enmStatus;
		/// <summary>
		/// Timer that fires each second as long as work is in progress
		/// </summary>
		private System.Timers.Timer m_timer;
		private const int TIMER_INTERVAL=2000;
		/// <summary>
		/// The current progress of the work (0..100 percent)
		/// </summary>
		private byte m_bytProgress=0;
		#region ctors
		public CWorker():this("", "")
		{}
		public CWorker(string strFirstName, string strLastName):this(strFirstName, strLastName, DateTime.Today)
		{}
		public CWorker(string strFirstName, string strLastName, DateTime dtDateOfBirth):base(strFirstName, strLastName, dtDateOfBirth)
		{
			m_colNotificationList = new CNotifyableObjCollection();
			m_enmStatus=EStatus.enmIdle; 
			m_timer=new System.Timers.Timer(TIMER_INTERVAL); //A timer that fires all second
			m_timer.AutoReset=true;
			m_timer.Elapsed+=new System.Timers.ElapsedEventHandler(OnTimerElapsed);
		}
		#endregion
		/// <summary>
		/// Allows self-registration of a boss for notifications. 
		/// Registred objects will receive notifications.
		/// </summary>
		/// <param name="obj">The receiver who wants to be notified about the status</param>
		public void RegisterNotification(INotifyable obj)
		{
			//Register object only if not already contained in the Notification List
			if (!m_colNotificationList.Contains(obj))
			{
				m_colNotificationList.Add(obj);
				Console.WriteLine("Ein {0}-Objekt wurde der Benachrichtigungsliste hinzugefgt", obj.GetType().Name);
			}
		}
		/// <summary>
		/// Allows unregistration of a boss for notification. 
		/// Unregistred objects will not receive notifications anymore.
		/// </summary>
		/// <param name="obj">The receiver who sould be unregistred</param>
		public void UnregisterNotification(INotifyable obj)
		{
			if (m_colNotificationList.Contains(obj))
			 m_colNotificationList.RemoveAt(m_colNotificationList.BinarySearch(obj));
		}
		/// <summary>
		/// Tells the worker to start working
		/// </summary>
		public void StartWork()
		{ 
			//If the old task was finished, begin from scratch, 
			//otherwise continue where you stopped previously.
			if (m_enmStatus==EStatus.enmWorkCompleted) m_bytProgress=0;
			m_enmStatus=EStatus.enmWorkStarted;
			//Tell everybody in the receiver list the about the change of status
			foreach (INotifyable objReceiver in m_colNotificationList)
			{
				objReceiver.ReportStatusChange(this, m_enmStatus);
			}
			//From now on, the progress counter should be increased
			//by using an internal timer
			m_timer.Start(); 
		}
		/// <summary>
		/// Tells the worker to stop working
		/// </summary>
		public void StopWork()
		{
			m_timer.Stop();
			m_enmStatus= m_bytProgress <100 ? EStatus.enmIdle : EStatus.enmWorkCompleted;
			//Tell everybody in the receiver list the change of status.
			foreach (INotifyable objReceiver in m_colNotificationList)
			{
				objReceiver.ReportStatusChange(this, m_enmStatus);
			}
		}
		/// <summary>
		/// A timer simulates the progress of the work. 
		/// This method implements a handler for the timer event: 
		/// Shows the progress of work by incrementing the percentage by 10%
		/// each time the timer elapses, until 100% are reached.
		/// </summary>
		/// <param name="sender">The timer that has raised the event</param>
		/// <param name="e">The event arguments provided by the sending timer </param>
		private void OnTimerElapsed(object sender, System.Timers.ElapsedEventArgs e)
		{
			m_enmStatus=EStatus.enmWorkInProgress;
			if (m_bytProgress < 100) m_bytProgress+=10; //Increase the progress percentage
			//Notify everybody in the receiver list about the new percentage 		
			foreach (INotifyable objReceiver in m_colNotificationList)
			{
				EFeedback enmFeedback =objReceiver.ReportProgress(this, m_bytProgress);
				//Write out the feedback.
				//Console.WriteLine("Der Feedback von {0}, "+
				//	"dem {1} ber den Arbeitsfortschritt war: "+ 
				//	"\"{2}\"", objReceiver.FirstName, 
				//	objReceiver.GetType().Name, enmFeedback.ToString());
				//Write out the feedback.
				//Please note that it is not really possible to identify from who the feedback was
				//A dog has a name, but a person has a firstname property and 
				//you don't know if the object that provided the feedback was a dog or a person.
				//you could check this by checking the type of objReceiver and react on it, 
				//but this would result in a switch case statement which MUST BE UPDATED EACH TIME THERE WILL BE 
				//A NEW NOTIFIABLE OBJECT
				Console.WriteLine("Der Feedback von {0} ber den Arbeitsfortschritt war: \"{1}\"", objReceiver.GetType().Name, enmFeedback.ToString());
			}
			Console.WriteLine();
			//Automatically stop working if work is done ;-)
			if (m_bytProgress>=100) this.StopWork();   
			
		}
	}
}
