package de.heise.ix.builders;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;

import de.heise.ix.builders.contributors.Contributor;
import de.heise.ix.builders.contributors.ContributorsFile;
import de.heise.ix.builders.scanner.AuthorDetector;
import de.heise.ix.builders.scanner.FileScanner;

/**
 * TODO: Provide description for "ContributorsBuilder".
 * @see IncrementalProjectBuilder
 */
public class ContributorsBuilder extends IncrementalProjectBuilder implements IResourceVisitor, IResourceDeltaVisitor {
	
	public static String BUILDER_ID = ContributorsPlugin.PLUGIN_ID + ".contributorsBuilder";
	
	private final static int NO_AUTHOR_DEFINED = 0;
	private final static String CONTRIBUTORS_FILE_NAME = "CONTRIBUTORS";

	private ContributorsFile contributorsFile;

	/**
	 * TODO: Implement the "ContributorsBuilder" constructor.
	 */
	public ContributorsBuilder() {
	}

	/**
	 * TODO: Implement "build".
	 * @see IncrementalProjectBuilder#build
	 */
	protected IProject [] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		
		readContributorsFile();
		
		switch (kind) {
			case IncrementalProjectBuilder.FULL_BUILD :
				performFullBuild(monitor);
				System.out.println("FULL BUILD");
				break;

			case IncrementalProjectBuilder.AUTO_BUILD :
				performAutoBuild(monitor);
				System.out.println("AUTO BUILD");
				break;

			case IncrementalProjectBuilder.INCREMENTAL_BUILD :
				performAutoBuild(monitor);
				System.out.println("INCREMENTAL BUILD");
				break;

			default :
				System.out.println("DEFAULT");
				break;
		}
		
		writeContributorsFile();
		
		return null;
	}
	
	private String getContributorsFileName() {
		IFile contributorsFile = this.getProject().getFile(CONTRIBUTORS_FILE_NAME);
		IPath path = contributorsFile.getRawLocation();
		return path.toOSString();
	}

	private void writeContributorsFile() throws CoreException {
		try {
			OutputStream os = new FileOutputStream(getContributorsFileName());
			contributorsFile.writeFile(os);
			os.flush();
			os.close();
			this.getProject().refreshLocal(IResource.DEPTH_ONE, null);
		}
		catch (FileNotFoundException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
		catch (IOException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
	}

	private void readContributorsFile() {
		contributorsFile = new ContributorsFile();
		try {
			contributorsFile.scanFile(new FileInputStream(getContributorsFileName()));
		}
		catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * @param monitor
	 */
	private void performAutoBuild(IProgressMonitor monitor) throws CoreException {
		IResourceDelta delta = getDelta(getProject());
		if (delta != null) {
			delta.accept(this);
		} 
	}

	/**
	 * @param monitor
	 */
	private void performFullBuild(IProgressMonitor monitor) throws CoreException {
		IProject project = this.getProject();
		project.accept(this);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.core.resources.IResourceVisitor#visit(org.eclipse.core.resources.IResource)
	 */
	public boolean visit(IResource resource) throws CoreException {
		System.out.println("Visiting " + resource.getName());
		process(resource);
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.core.resources.IResourceDeltaVisitor#visit(org.eclipse.core.resources.IResourceDelta)
	 */
	public boolean visit(IResourceDelta delta) throws CoreException {
		IResource resource = delta.getResource();
		System.out.println("Visiting " + resource.getName());
		process(resource);
		return true;
	}

	/**
	 * @param resource
	 */
	private void process(IResource resource) throws CoreException {
		// ist es eine Java-Datei?
		String extension = resource.getFileExtension();
		if (
				(extension != null) && 
				(extension.equalsIgnoreCase("java"))) {
			
			// evtl. vorhandene alte Marker entfernen
			resource.deleteMarkers(ContributorsMarker.NAME, true,
					IResource.DEPTH_INFINITE);
			
			// Datei holen
			IFile file = (IFile) resource.getAdapter(IFile.class);
			if (file != null) {
				try {
					// Inhalt der Datei lesen
					InputStream is = file.getContents();
					FileScanner scanner = new FileScanner(is);
					scanner.addDetector(new AuthorDetector());
					scanner.scan();
					// Liste der gefundenen Autoren
					Collection authors = 
						scanner.getResult(AuthorDetector.NAME);
					if (authors.size() > 0) {
						// der globalen Liste zufgen
						manageAuthors(authors);
					} 
					else {
						// Fehler, wenn kein Autor definiert
						createMarker(file, NO_AUTHOR_DEFINED);
					}
				} 
				catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	private void manageAuthors(Collection authors) {
		for (Iterator iter = authors.iterator(); iter.hasNext();) {
			String author = (String) iter.next();
			
			Contributor contributor = new Contributor();
			contributor.setName(author);
			
			contributorsFile.addContributor(contributor);
		}
	}

	/**
	 * Dateien, fr die kein Autor definiert wurde,
	 * werden mit einem Marker als fehlerhaft markiert.
	 * 
	 * Marker werden an drei Stellen angezeigt:
	 * 1) In der Taskliste
	 * 2) Am linken Editorrand als Icon
	 * 3) In der bersichtsspalte am rechten Editorrand
	 * 
	 * Durch Anklicken eines Eintrags in der Taskliste
	 * springt die Eclipse zu der markierten Stelle. 
	 * 
	 * @param file
	 * @param errorcode
	 */
	private void createMarker(IFile file, int errorcode) throws CoreException {
		IMarker marker = file.createMarker(ContributorsMarker.NAME);
				
		String msg = "";
		if (errorcode == NO_AUTHOR_DEFINED) {
			msg = "No author defined for file " + file.getName() + ".";
		}	
		marker.setAttribute(IMarker.MESSAGE, msg);
		marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_WARNING);
	}
}
