/*
 * ContributorsNature.java
 *
 * Copyright (c) 2003 Peter Friese
 * All rights reserved.
 *  
 */
package de.heise.ix.builders.nature;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.runtime.CoreException;

import de.heise.ix.builders.ContributorsBuilder;
import de.heise.ix.builders.ContributorsPlugin;

/**
 * Die Klasse <code>ContributorsNature</code> verwaltet
 * die Zuordnung des Builders 
 * {@link de.heise.ix.builders.ContributorsBuilder} zu
 * einem Projekt.
 *  
 * @author Peter Friese
 */
public class ContributorsNature
	implements IProjectNature {

	private IProject project;

	/**
	 * The identifier for the FindBugs nature
	 * (value <code>"de.tobject.findbugs.findbugsnature"</code>).
	 *
	 * @see org.eclipse.core.resources.IProject#hasNature(java.lang.String)
	 */
	public static final String NATURE_ID =
		ContributorsPlugin.PLUGIN_ID
			+ ".contributorsNature";

	/**
	 * Konfiguriert das Projekt.
	 * @throws CoreException Wenn ein Fehler auftritt. 
	 */
	public void configure() throws CoreException {
		addToBuildSpec(ContributorsBuilder.BUILDER_ID);
	}

	/**
	 * Bereinigt die Konfiguration des Projekts.
	 * @throws CoreException Wenn ein Fehler auftritt.
	 */
	public void deconfigure() throws CoreException {
		removeFromBuildSpec(
			ContributorsBuilder.BUILDER_ID);
	}

	/**
	 * Entfernt den bergebenen Builder aus der Build
	 * Specification des Projekts.
	 * 
	 * @param builderID Der zu entfernende Builder.
	 * @throws CoreException Wenn ein Fehler auftritt.
	 */
	protected void removeFromBuildSpec(String builderID)
		throws CoreException {

		IProjectDescription description =
			getProject().getDescription();
		ICommand[] commands = description.getBuildSpec();
		for (int i = 0; i < commands.length; ++i) {
			String builderName = commands[i].getBuilderName();
			if (builderName.equals(builderID)) {
				ICommand[] newCommands =
					new ICommand[commands.length - 1];
				System.arraycopy(commands, 0,
					newCommands, 0, i);
				System.arraycopy(commands, i + 1,
					newCommands, i, commands.length - i - 1);
				description.setBuildSpec(newCommands);
				getProject().setDescription(
					description, null);
				return;
			}
		}
	}

	/**
	 * Fgt den bergebenen Builder in die Build 
	 * Specification des Projekts ein.
	 * 
	 * @param builderID Der hinzuzufgende Builder.
	 * @throws CoreException Wenn ein Fehler auftritt.
	 */
	protected void addToBuildSpec(String builderID)
		throws CoreException {

		IProjectDescription description =
			getProject().getDescription();
		ICommand buildCommand =
			getBuilderCommand(description);

		// ist der Builder evtl. schon eingetragen?
		if (buildCommand == null) {
			// nein -> neues Build Command hinzufgen
			ICommand newCommand =
				description.newCommand();
			newCommand.setBuilderName(builderID);
			setBuilderCommand(description, newCommand);
		}
	}

	/**
	 * Sucht den Builder in der bergebenen Project
	 * Description.
	 * 
	 * @param description Die Project Description des
	 * 	Projekts. 
	 * @return Das Build Command des gesuchten Builders.
	 * @throws CoreException Wenn ein Fehler auftritt.
	 */
	private ICommand getBuilderCommand(IProjectDescription description)
		throws CoreException {

		ICommand[] commands = description.getBuildSpec();
		for (int i = 0; i < commands.length; ++i) {
			String builderName =
				commands[i].getBuilderName();
			if (builderName
				.equals(ContributorsBuilder.BUILDER_ID)) {
				return commands[i];
			}
		}
		return null;
	}

	/**
	 * Fgt ein Build Command in die Project Description
	 * des Projekts ein.
	 *  
	 * @param description Die Project Description.
	 * @param newCommand Das hinzuzufgende Build Command.
	 * @throws CoreException Wenn ein Fehler auftritt.
	 */
	private void setBuilderCommand(
		IProjectDescription description,
		ICommand newCommand)
		throws CoreException {

		ICommand[] oldCommands =
			description.getBuildSpec();
		ICommand oldFindBugsCommand =
			getBuilderCommand(description);
		ICommand[] newCommands;

		// Build Command bereits vorhanden?
		if (oldFindBugsCommand == null) {
			// nein -> neu hinzufgen (am Ende der Liste) 
			newCommands =
				new ICommand[oldCommands.length + 1];
			System.arraycopy(oldCommands, 0,
				newCommands, 0, oldCommands.length);
			newCommands[oldCommands.length] = newCommand;
		}
		else {
			// ja -> aktualisieren
			for (int i = 0, max = oldCommands.length;
				i < max;
				i++) {
				if (oldCommands[i] == oldFindBugsCommand) {
					oldCommands[i] = newCommand;
					break;
				}
			}
			newCommands = oldCommands;
		}

		// Project Description aktualisieren
		description.setBuildSpec(newCommands);
		getProject().setDescription(description, null);
	}

	/**
	 * Gibt das Projekt zurck, das dieser Nature
	 * zugeordnet ist.
	 * 
	 * @return Das zugeordnete Projekt. 
	 */
	public IProject getProject() {
		return this.project;
	}

	/**
	 * Setzt das Projekt, das dieser Nature zugeordnet 
	 * ist.
	 * @param project Das zuzuordnende Projekt.
	 */
	public void setProject(IProject project) {
		this.project = project;
	}

}
