package VCSSelectors;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.types.selectors.*;
import org.apache.tools.ant.types.Parameter;

import java.io.*;

/** 
 * This class acts as parent class for all 
 * VCS-Selectors. It implements the Strategy-Pattern
 * for the selection, but leaves the VCS related
 * checks open for concrete subclasses
 *
 * @author Torsten Gonther
 */
public abstract class AbstractVCSSelector 
	extends BaseExtendSelector {
	
	protected String hasTag, isVersion, isOnBranch;
	protected boolean isVCSFile = true;	
	protected boolean checkDirs = false;	

	/**
	 * Set if we should only check for a VCS File
	 */
	public void setIsVCSFile(boolean val) {
		isVCSFile = val;
	}
	/**
	 * The tag a file must have to be selected.
	 *
	 * @param tag The Tag that the file should have
	 */
	public void setHasTag(String tag) {
		hasTag  = tag;
	}
	
	/**
	 * The version a file must have to be selected.
	 *
	 * @param version The VCS-Version that the 
	 * file should have
	 */
	public void setIsVersion(String version) {
		isVersion = version;
	}
	
	/**
	 * The branch a file must be on to be selected.
	 *
	 * @param branch The VCS-Branch that the 
	 * file should reside on 
	 */
	public void setIsOnBranch(String branch) {
		isOnBranch = branch;
	}
	

	/**
	 * This method will translate each parameter from 
	 * the custom selector's param tag into the 
	 * appropriate setXXX() call. The method looks for
	 * the parameter-names "hasTag", "isVersion" or
	 * "isOnBranch". If one of these param-names is
	 * found, the selector will disable the simple 
	 * IsVCSFile-Check. If none of the stated 
	 * param-names is present, the selector will only
	 * do the IsVCSFile-Check. Unrecognized param-names
	 * will be ignored. 
	 * @param parameters the complete set of parameters 
	 * for this selector
	 */
	public void setParameters(Parameter[] parameters) {
		super.setParameters(parameters);
		if ((parameters != null) && 
			(parameters.length > 0)) {
			for (int i = 0; i < parameters.length; i++) {
				String paramName = parameters[i].getName();
				if (paramName.equalsIgnoreCase("hasTag")) {
					setHasTag(parameters[i].getValue());
					setIsVCSFile(false);
				} else if (paramName.equalsIgnoreCase
					("isVersion")) {
					setIsVersion(parameters[i].getValue());
					setIsVCSFile(false);
				} else if (paramName.equalsIgnoreCase
					("IsOnBranch")) {
					setIsOnBranch(parameters[i].getValue());
					setIsVCSFile(false);
				}
			}
		}
	}

	/** Check this selector's consistence */
	public void verifySettings() {
		if ( isVCSFile || 
			(hasTag != null) || 
			(isVersion != null) ||
			(isOnBranch != null) )
			return;	
		throw new BuildException
				("AbstractVCSSelector: verifySettings failed !");
	}
	/** The core functionality of the Selector. This is
	 * also the interface to Ant's FileSet Datatype 
	 */
	public boolean isSelected(File basedir, 
		String filename, 
		File file) 
		throws BuildException {
		
		if (file.isDirectory() && !checkDirs)
			return true;	

		validate();	

		if (isVCSFile) return checkIsVCSFile(file);	
				
		boolean ret = true;
		
		if ( (hasTag!=null) && 
			(!checkTag(hasTag, file)) ) return false;
		
		if ( (isVersion!=null) && 
			(!checkVersion(isVersion, file)) ) return false;
		
		if ( (isOnBranch!=null) && 
			(!checkBranch(isOnBranch, file)) ) return false;
		
		return ret;
	
	}
	
	public abstract boolean checkTag
		(String tag, File file);
	public abstract boolean checkVersion
		(String version, File file);
	public abstract boolean checkBranch
		(String branch, File file) ;
	public abstract boolean checkIsVCSFile
		(File file);
}
