package addressbook.service;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.io.xml.DomDriver;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

/**
 * Einfache Realisierung von {@link AddressDao} auf der Basis einer XML Datei.
 */
public class XmlAddressDao implements AddressDao {

  private static final String ENCODING = "UTF-8";

  private String storageFileName = "addresses.xml";
  private XStream xstream;

  public XmlAddressDao() {
    xstream = new XStream(new DomDriver(ENCODING));
    xstream.alias("address", Address.class);
  }

  public String getStorageFileName() {
    return storageFileName;
  }

  public void setStorageFileName(String storageFileName) {
    this.storageFileName = storageFileName;
  }

  public synchronized Address updateAddress(Address address)
      throws AddressDaoException {
    List<Address> addresses = loadAddresses();
    Address storedAddress = getAddress(address.getId(), addresses);
    if (storedAddress == null) {
      address.setId(addresses.size() + 1);
      addresses.add(address);
    } else {
      storedAddress.fill(address);
    }
    saveAddresses(addresses);
    return address;
  }

  public synchronized List<Address> findAddresses(String filter)
      throws AddressDaoException {
    List<Address> addresses = loadAddresses();
    List<Address> filtered = new ArrayList<Address>(addresses.size());
    for (Address address : addresses) {
      if (address.matches(filter)) {
        filtered.add(address);
      }
    }
    return Collections.unmodifiableList(filtered);
  }

  public synchronized void removeAddress(Address address)
      throws AddressDaoException {
    List<Address> addresses = loadAddresses();
    Address a = getAddress(address.getId(), addresses);
    if (a != null) {
      addresses.remove(a);
      saveAddresses(addresses);
    }
  }

  private List<Address> loadAddresses() throws AddressDaoException {
    InputStream stream = null;
    try {
      if (!new File(storageFileName).exists()) {
        return createDefaultAddressList();
      }
      stream = new FileInputStream(storageFileName);
      return (List<Address>) xstream.fromXML(stream);
    } catch (Exception e) {
      throw new AddressDaoException("failed to load addresses", e);
    } finally {
      close(stream);
    }
  }

  private void saveAddresses(List<Address> addresses)
      throws AddressDaoException {
    Writer writer = null;
    try {
      writer = new OutputStreamWriter(
          new FileOutputStream(storageFileName, false), ENCODING);
      String header = "<?xml version='1.0' encoding='" + ENCODING + "'?>\n";
      writer.write(header);
      xstream.toXML(addresses, writer);
    } catch (Exception e) {
      throw new AddressDaoException("failed to save addresses", e);
    } finally {
      close(writer);
    }
  }

  private Address getAddress(int id, List<Address> addresses) {
    for (Address address : addresses) {
      if (address.getId() == id) {
        return address;
      }
    }
    return null;
  }

  private void close(Closeable stream) {
    if (stream != null) {
      try {
        stream.close();
      } catch (IOException e) {
        // ignore
      }
    }
  }

  private List<Address> createDefaultAddressList() {
    ArrayList<Address> addresses = new ArrayList<Address>();
    addAddress("Lars", "Röwekamp", addresses);
    addAddress("Detlef", "Bartetzko", addresses);
    addAddress("Arvid", "Hülsebus", addresses);
    return addresses;
  }

  private void addAddress(String firstName, String lastName,
      ArrayList<Address> addresses) {
    Address address = new Address();
    address.setFirstName(firstName);
    address.setLastName(lastName);
    address.setId(addresses.size() + 1);
    address.setCountry(Locale.GERMANY);
    addresses.add(address);
  }

}
