package addressbook.web;

import addressbook.service.Address;
import addressbook.service.AddressDao;
import addressbook.service.AddressDaoException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.faces.application.Application;
import javax.faces.context.FacesContext;
import javax.faces.model.SelectItem;
import javax.faces.event.ValueChangeEvent;
import javax.faces.component.UIComponent;
import javax.faces.component.EditableValueHolder;
import javax.faces.convert.DateTimeConverter;
import javax.faces.validator.ValidatorException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Collections;

/**
 * Controller für die Adressverwaltung, der Methoden zur Ansteuerung der
 * Geschäftslogik zur Verfügung stellt.
 */
public class Controller {

  private static final Log LOG = LogFactory.getLog(Controller.class);

  private static final String OUTCOME_LIST = "list";
  private static final String OUTCOME_EDITOR = "editor";

  /**
   * Service für Adressmanipulationen.
   */
  private AddressDao addressDao;

  /**
   * Die aktuelle, zu editierende Adresse.
   */
  private Address currentAddress = new Address();

  /**
   * Modell aller Adressen für den DataTable.
   */
  private List<AddressRow> addressRows;

  /**
   * Suchkriterium zum Einschränken der Adressen in der Tabelle.
   */
  private String searchCriterion;

  /**
   * Ausgwählte Sprache.
   */
  private Locale language;

  /**
   * Zur Verfügung stehende Sprachen.
   */
  private List<SelectItem> languages = new ArrayList<SelectItem>();

  /**
   * Zur Verfügung stehende Länder.
   */
  private Countries countries;

  public Controller() {
    FacesContext facesContext = FacesContext.getCurrentInstance();
    Application application = facesContext.getApplication();
    language = application.getDefaultLocale();
    facesContext.getExternalContext().getSession(true);
    initLanguages();
  }

  /**
   * Aktualisiert die Adressliste auf Basis des Suchkriteriums.
   */
  public String search() throws AddressDaoException {
    initAddressRows();
    return OUTCOME_LIST;
  }

  /**
   * Legt eine neue Adresse an.
   */
  public String createAddress() {
    currentAddress = new Address();
    return OUTCOME_EDITOR;
  }

  /**
   * Stellt die erste selektierte Adresse im Editor dar.
   */
  public String editAddress() {
    currentAddress = null;
    for (AddressRow row : getAddressRows()) {
      if (row.isSelected()) {
        currentAddress = row.getAddress();
        break;
      }
    }
    if (currentAddress == null) {
      return createAddress();
    }
    return OUTCOME_EDITOR;
  }

  /**
   * Löscht alle selektierten Adressen.
   */
  public String deleteAddresses() throws AddressDaoException {
    for (AddressRow row : getAddressRows()) {
      if (row.isSelected()) {
        addressDao.removeAddress(row.getAddress());
      }
    }
    return search();
  }

  /**
   * Persistieren der aktuellen Adresse im Editor.
   */
  public String store() throws AddressDaoException {
    addressDao.updateAddress(currentAddress);
    searchCriterion = null;
    return search();
  }

  /**
   * Bricht das Editieren der aktuellen Adresse ab.
   */
  public String cancel() throws AddressDaoException {
    searchCriterion = null;
    return search();
  }

  /**
   * Event Handler für Sprach Select Control.
   */
  public void languageChanged(ValueChangeEvent event) {
    language = (Locale) event.getNewValue();
    FacesContext facesContext = FacesContext.getCurrentInstance();
    countries.init(language);
    initLanguages();

    // reinit date converter
    facesContext.getViewRoot().setLocale(language);
    EditableValueHolder component = (EditableValueHolder)
        event.getComponent().findComponent(":f:dayOfBirth");
    if (component != null) {
      DateTimeConverter converter = (DateTimeConverter) component.getConverter();
      converter.setPattern(MessageUtils.getLocalizedString(facesContext, "editor_date_pattern"));
    }
  }

  /**
   * Vereinfachter Validator Methode für eMail Adressen.
   */
  public void validatePhoneNumber(
      FacesContext context, UIComponent component, Object value) {
    String phoneNumber = (String) value;
    if (phoneNumber == null || phoneNumber.length() == 0) {
      return;
    }
    if (!phoneNumber.matches("\\+?[0-9 ]*(\\([0-9 ]*\\))?[0-9 ]*")) {
      throw new ValidatorException(MessageUtils.createErrorMessage(
          "validator_phone", context));
    }
  }

  public Address getCurrentAddress() {
    return currentAddress;
  }

  public void setCurrentAddress(Address currentAddress) {
    this.currentAddress = currentAddress;
  }

  public List<AddressRow> getAddressRows() {
    if (addressRows == null) {
      try {
        initAddressRows();
      } catch (AddressDaoException e) {
        LOG.error(null, e);
      }
    }
    return addressRows;
  }

  public void setAddressRows(List<AddressRow> addressRows) {
    this.addressRows = addressRows;
  }

  public int getAddressCount() {
    return getAddressRows().size();
  }

  public String getSearchCriterion() {
    return searchCriterion;
  }

  public void setSearchCriterion(String searchCriterion) {
    this.searchCriterion = searchCriterion;
  }

  public AddressDao getAddressDao() {
    return addressDao;
  }

  public void setAddressDao(AddressDao addressDao) {
    this.addressDao = addressDao;
  }

  public Locale getLanguage() {
    return language;
  }

  public void setLanguage(Locale language) {
    this.language = language;
  }

  public List<SelectItem> getLanguages() {
    return languages;
  }

  public void setLanguages(List<SelectItem> languages) {
    this.languages = languages;
  }

  public Countries getCountries() {
    return countries;
  }

  public void setCountries(Countries countries) {
    this.countries = countries;
    countries.init(language);
  }

  /**
   * Initialisiert die Adressliste in Abhängigkeit des Suchkriteriums.
   */
  private void initAddressRows() throws AddressDaoException {
    if (addressRows == null) {
      addressRows = new ArrayList<AddressRow>();
    }
    // Aufsammeln der aktuellen Selektierungen.
    List<Integer> selectedIds = new ArrayList<Integer>();
    for (AddressRow row : addressRows) {
      if (row.isSelected()) {
        selectedIds.add(row.getAddress().getId());
      }
    }
    addressRows.clear();
    List<Address> addresses = addressDao.findAddresses(searchCriterion);
    for (Address address : addresses) {
      AddressRow row = new AddressRow();
      row.setAddress(address);
      if (selectedIds.contains(address.getId())) {
        row.setSelected(true);
      }
      addressRows.add(row);
    }
  }

  private void initLanguages() {
    languages.clear();
    FacesContext facesContext = FacesContext.getCurrentInstance();
    Application application = facesContext.getApplication();
    Iterator supportedLocales = application.getSupportedLocales();
    while (supportedLocales.hasNext()) {
      Locale locale = (Locale) supportedLocales.next();
      SelectItem item = new SelectItem(locale, locale.getDisplayName(language));
      languages.add(item);
    }
    Collections.sort(languages, new SelectItemComparator());
  }

}
