package addressbook.web;

import addressbook.service.Address;
import addressbook.service.AddressDao;
import addressbook.service.AddressDaoException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.lang.WordUtils;
import org.apache.myfaces.tobago.model.SheetState;
import org.apache.myfaces.tobago.context.Theme;
import org.apache.myfaces.tobago.context.ClientProperties;
import org.apache.myfaces.tobago.config.TobagoConfig;

import javax.faces.application.Application;
import javax.faces.context.FacesContext;
import javax.faces.model.SelectItem;
import javax.faces.component.UIComponent;
import javax.faces.component.EditableValueHolder;
import javax.faces.component.UIViewRoot;
import javax.faces.convert.DateTimeConverter;
import javax.faces.validator.ValidatorException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Collections;

/**
 * Controller für die Adressverwaltung, der Methoden zur Ansteuerung der
 * Geschäftslogik zur Verfügung stellt.
 */
public class Controller {

  private static final Log LOG = LogFactory.getLog(Controller.class);

  private static final String OUTCOME_LIST = "list";
  private static final String OUTCOME_EDITOR = "editor";

  /**
   * Service für Adressmanipulationen.
   */
  private AddressDao addressDao;

  /**
   * Die aktuelle, zu editierende Adresse.
   */
  private Address currentAddress = new Address();

  /**
   * Modell aller Adressen für das Sheet.
   */
  private List<AddressRow> addressRows;

  /**
   * Statusinformationen des Sheets.
   */
  private SheetState selectedAddresses;

  /**
   * Suchkriterium zum Einschränken der Adressen in der Tabelle.
   */
  private String searchCriterion;

  /**
   * Ausgwählte Sprache.
   */
  private Locale language;

  /**
   * Zur Verfügung stehende Sprachen.
   */
  private List<SelectItem> languages = new ArrayList<SelectItem>();

  /**
   * Zur Verfügung stehende Länder.
   */
  private Countries countries;

  /**
   * Ausgewähltes Design.
   */
  private Theme theme;

  /**
   * Zur Verfügung stehende Designs.
   */
  private List<SelectItem> themeItems = new ArrayList<SelectItem>();

  /**
   * Soll die Darstellung einfach oder komplex sein?
   */
  private boolean simple;

  /**
   * Selektierter Reiter.
   */
  private int selectedTab;

  public Controller() {
    FacesContext facesContext = FacesContext.getCurrentInstance();
    Application application = facesContext.getApplication();
    language = application.getDefaultLocale();
    facesContext.getExternalContext().getSession(true);
    initLanguages();

    TobagoConfig tobagoConfig = TobagoConfig.getInstance(facesContext);
    List<Theme> themes = new ArrayList<Theme>(tobagoConfig.getSupportedThemes());
    themes.add(0, tobagoConfig.getDefaultTheme());
    themeItems = new ArrayList<SelectItem>();
    for (Theme theme : themes) {
      themeItems.add(new SelectItem(theme, theme.getDisplayName()));
    }

    ClientProperties client = ClientProperties.getInstance(facesContext);
    theme = client.getTheme();
  }

  /**
   * Aktualisiert die Adressliste auf Basis des Suchkriteriums.
   */
  public String search() throws AddressDaoException {
    initAddressRows();
    return OUTCOME_LIST;
  }

  /**
   * Legt eine neue Adresse an.
   */
  public String createAddress() {
    currentAddress = new Address();
    return OUTCOME_EDITOR;
  }

  /**
   * Stellt die erste selektierte Adresse im Editor dar.
   */
  public String editAddress() {
    currentAddress = getSelectedAddress();
    if (currentAddress == null) {
      return createAddress();
    }
    return OUTCOME_EDITOR;
  }

  private Address getSelectedAddress() {
    List<Integer> selection = selectedAddresses.getSelectedRows();
    if (selection.size() >= 1) {
      return getAddressRows().get(selection.get(0)).getAddress();
    }
    return null;
  }

  /**
   * Löscht alle selektierten Adressen.
   */
  public String deleteAddresses() throws AddressDaoException {
    List<Integer> selection = selectedAddresses.getSelectedRows();
    for (Integer index : selection) {
      Address address = getAddressRows().get(index).getAddress();
      addressDao.removeAddress(address);
    }
    return search();
  }

  /**
   * Persistieren der aktuellen Adresse im Editor.
   */
  public String store() throws AddressDaoException {
    addressDao.updateAddress(currentAddress);
    searchCriterion = null;
    return search();
  }

  /**
   * Bricht das Editieren der aktuellen Adresse ab.
   */
  public String cancel() throws AddressDaoException {
    searchCriterion = null;
    return search();
  }

  /**
   * Änderung der ausgewählten Sprache.
   */
  public String languageChanged() {
    FacesContext facesContext = FacesContext.getCurrentInstance();
    countries.init(language);
    initLanguages();

    // reinit date converter
    UIViewRoot viewRoot = facesContext.getViewRoot();
    EditableValueHolder component = (EditableValueHolder)
        viewRoot.findComponent(":page:dayOfBirth");
    if (component != null) {
      DateTimeConverter converter = (DateTimeConverter) component.getConverter();
      converter.setPattern(MessageUtils.getLocalizedString(facesContext, "editor_date_pattern"));
    }
    return null;
  }

  /**
   * Änderung des ausgewählten Designs.
   */
  public String themeChanged() {
    FacesContext facesContext = FacesContext.getCurrentInstance();
    ClientProperties client = ClientProperties.getInstance(facesContext);
    client.setTheme(theme);
    return null;
  }

  /**
   * Vereinfachter Validator Methode für eMail Adressen.
   */
  public void validatePhoneNumber(
      FacesContext context, UIComponent component, Object value) {
    String phoneNumber = (String) value;
    if (phoneNumber == null || phoneNumber.length() == 0) {
      return;
    }
    if (!phoneNumber.matches("\\+?[0-9 ]*(\\([0-9 ]*\\))?[0-9 ]*")) {
      throw new ValidatorException(MessageUtils.createErrorMessage(
          "validator_phone", context));
    }
  }

  public Address getCurrentAddress() {
    return currentAddress;
  }

  public void setCurrentAddress(Address currentAddress) {
    this.currentAddress = currentAddress;
  }

  public List<AddressRow> getAddressRows() {
    if (addressRows == null) {
      try {
        initAddressRows();
      } catch (AddressDaoException e) {
        LOG.error(null, e);
      }
    }
    return addressRows;
  }

  public void setAddressRows(List<AddressRow> addressRows) {
    this.addressRows = addressRows;
  }

  public SheetState getSelectedAddresses() {
    return selectedAddresses;
  }

  public void setSelectedAddresses(SheetState selectedAddresses) {
    this.selectedAddresses = selectedAddresses;
  }

  public int getAddressCount() {
    return getAddressRows().size();
  }

  public String getSearchCriterion() {
    return searchCriterion;
  }

  public void setSearchCriterion(String searchCriterion) {
    this.searchCriterion = searchCriterion;
  }

  public AddressDao getAddressDao() {
    return addressDao;
  }

  public void setAddressDao(AddressDao addressDao) {
    this.addressDao = addressDao;
  }

  public Locale getLanguage() {
    return language;
  }

  public String getDisplayLanguage() {
    return language.getDisplayName(language);
  }

  public void setLanguage(Locale language) {
    this.language = language;
  }

  public List<SelectItem> getLanguages() {
    return languages;
  }

  public void setLanguages(List<SelectItem> languages) {
    this.languages = languages;
  }

  public Countries getCountries() {
    return countries;
  }

  public void setCountries(Countries countries) {
    this.countries = countries;
    countries.init(language);
  }

  public List<SelectItem> getThemeItems() {
    return themeItems;
  }

  public Theme getTheme() {
    return theme;
  }

  @Deprecated
  public String getThemeName() {
    return WordUtils.capitalize(theme.getDisplayName());
  }

  public void setTheme(Theme theme) {
    this.theme = theme;
  }

  public boolean isSimple() {
    return simple;
  }

  public void setSimple(boolean simple) {
    this.simple = simple;
  }

  public int getSelectedTab() {
    if (simple) {
      selectedTab = 0;
    }
    return selectedTab;
  }

  public void setSelectedTab(int selectedTab) {
    this.selectedTab = selectedTab;
  }

  /**
   * Initialisiert die Adressliste in Abhängigkeit des Suchkriteriums.
   */
  private void initAddressRows() throws AddressDaoException {
    if (addressRows == null) {
      addressRows = new ArrayList<AddressRow>();
    }
    // Aufsammeln der aktuellen Selektierungen.
    List<Integer> selectedIds = new ArrayList<Integer>();
    for (AddressRow row : addressRows) {
      if (row.isSelected()) {
        selectedIds.add(row.getAddress().getId());
      }
    }
    addressRows.clear();
    List<Address> addresses = addressDao.findAddresses(searchCriterion);
    for (Address address : addresses) {
      AddressRow row = new AddressRow();
      row.setAddress(address);
      if (selectedIds.contains(address.getId())) {
        row.setSelected(true);
      }
      addressRows.add(row);
    }
  }

  private void initLanguages() {
    languages.clear();
    FacesContext facesContext = FacesContext.getCurrentInstance();
    Application application = facesContext.getApplication();
    Iterator supportedLocales = application.getSupportedLocales();
    while (supportedLocales.hasNext()) {
      Locale locale = (Locale) supportedLocales.next();
      SelectItem item = new SelectItem(locale, locale.getDisplayName(language));
      languages.add(item);
    }
    Collections.sort(languages, new SelectItemComparator());
  }

}
