<?php
/*
WMS Reflector for Google Earth. 

Accepts "network link" requests from Google Earth and returns a KML file that 
instructs Google Earth to obtain a map image from a single WMS.

Based on code put in the public domain by James McGill. 

2005-08- 28: Extended by Andrew Hallam to:

1. Limit the extent if the map request sent to the WMS server,
2. Calculate the correct aspect ratio of the map image based on the 
BBOX values, and
3. Allow the passing through of one or more WMS layer names. This
overrides the default layer list.

This reflector can be used in two different ways:

1. It can provide access to a WMS with a fixed list of layers. i.e. Multiple 
layers in a single map image. To specify the layers assign them to 
$WMS_LAYERS as a comma delimited list (no spaces). The KML that 
defines the location of the network link then needs to contain the URL 
to this script. e.g.
<href>http://www.yourserver.com/path/wms-reflector.php</href>

2. It can accept layer names from Google Earth. This allows you to set up 
multiple network links in Google Earth that display one, or more, data layers
from a WMS. i.e. One, or more, data layers in each map image.
<href>http://www.yourserver.com/path/wms-reflector.php?LAYERS=layer1,layer2</href>

2005-09-02: Edited by Andrew Hallam. Added check for map images that 
2006-05-31: Edited by Christian Wilk for the MapServer WMS Example published in iX computer magazine
*/

// =======================================================================

// The default map image width in pixels. This should not exceed the maximum value 
// accepted by the WMS. The height value is calulated below based on
// the extent of the request. 
$MAP_IMAGE_WIDTH = 1000;

// Maximum map image height accepted by the WMS. 
$MAP_IMAGE_MAX_HEIGHT = 1000;

// Define the maximum extent that the WMS will accept. This is used to prevent 
// Google Earth from requesting a map for the entire planet. The values were obtained from 
// the LatLonBoundingBox element from the WMS's top level Layer element in the 
// capabilities file but modified to include the part of Europe around Switzerland. 
// XPAth: /WMT_MS_Capabilities/Capability/Layer/LatLonBoundingBox
$WMS_NORTH_LIMIT = 55;
$WMS_SOUTH_LIMIT = 40;
$WMS_EAST_LIMIT = 12;
$WMS_WEST_LIMIT = -2;

// Define the WMS layers to display.
$WMS_LAYERS = "countries"; 

// Define the base URI used to form the GetMap request. Excludes the BBOX, LAYERS, 
// WIDTH and HEIGHT parameters.
$WMS_BASE_URI = "http://127.0.0.1/cgi-bin/mapserv.exe?map=/ms4w/apps/ix-artikel/welt-9.map&VERSION=1.1.1&REQUEST=GetMap&SRS=EPSG:4326&STYLES=&";

// =======================================================================

// Grab the full query string provided by Google Earth. It is sent back 
// in  the KML as a comment to assist with testing. Not used for any 
// other purpose. 
$geReqQueryString = $_SERVER['QUERY_STRING'];

// Extract the bounding box coordinates from the HTTP GET request.
$geReqBBox = $_REQUEST['BBOX'];
$coords = explode(",",$geReqBBox);

// If there was a BBOX parameter use the coordinates provided, 
// otherwiser default to the maximim extent.
if ($geReqBBox) {
	$geReqNorth = (float) $coords[3];
	$geReqSouth = (float) $coords[1];
	$geReqEast = (float) $coords[2];
	$geReqWest = (float) $coords[0];
} else {
	// No BBOX URL parameter. Default to the maximum extent.
	$geReqNorth = $WMS_NORTH_LIMIT;
	$geReqSouth = $WMS_SOUTH_LIMIT;
	$geReqEast = $WMS_EAST_LIMIT;
	$geReqWest = $WMS_WEST_LIMIT;
}

// Extract the layer name from the HTTP GET request, if there is one. 
// If not, use the full list of layers.
$geReqLayers = $_REQUEST['LAYERS'];
if ($geReqLayers) {
	$wmsUrlLayers = $geReqLayers;
} else {
	$wmsUrlLayers = $WMS_LAYERS;
}

// Ensure that the request extent does not exceed the extent limit 
// of the WMS extent. If it does then trim down the extent as required
// and re
$wmsUrlNorth = ($geReqNorth > $WMS_NORTH_LIMIT) ? $WMS_NORTH_LIMIT : $geReqNorth;
$wmsUrlSouth = ($geReqSouth < $WMS_SOUTH_LIMIT) ? $WMS_SOUTH_LIMIT : $geReqSouth; 
$wmsUrlEast = ($geReqEast > $WMS_EAST_LIMIT) ? $WMS_EAST_LIMIT : $geReqEast;
$wmsUrlWest = ($geReqWest < $WMS_WEST_LIMIT) ? $WMS_WEST_LIMIT : $geReqWest;

// Calculate the height of the map image to be requested from the WMS.
$wmsMapWidth = $wmsUrlEast - $wmsUrlWest;
$wmsMapHeight = $wmsUrlNorth - $wmsUrlSouth;
$mapImgHeight = (int) ($MAP_IMAGE_WIDTH * ($wmsMapHeight / $wmsMapWidth)); 
$mapImgWidth = $MAP_IMAGE_WIDTH;

// Do a quick check on the image to make sure that maximum dimensions 
// are not being exceeded. If they are then rescale the map image dimensions.
// This needs to be done in the situation that the extent limits have created
// a map image dimension that is taller than it is wide.
if ($mapImgHeight > $MAP_IMAGE_MAX_HEIGHT) {
	$mapImgWidth = (int) ($mapImgWidth * ($MAP_IMAGE_MAX_HEIGHT / $mapImgHeight));
	$mapImgHeight = $MAP_IMAGE_MAX_HEIGHT;
}

// Complete the WMS GetMap request URI by adding the LAYERS, 
// WIDTH, HEIGHT and BBOX parameters.
$wmsUrl = $WMS_BASE_URI . "LAYERS=" . $wmsUrlLayers . "&";
$wmsUrl .= "WIDTH=" . $mapImgWidth . "&";
$wmsUrl .= "HEIGHT=" . $mapImgHeight . "&";
$wmsUrl .= "BBOX=" . $wmsUrlWest . "," . $wmsUrlSouth . "," . $wmsUrlEast . "," . $wmsUrlNorth . "&";

// Build up the KML response document.
$kml = "<GroundOverlay>\n";
$kml .= "<Icon>\n";
$kml .= "<href><![CDATA[" . $wmsUrl . "]]></href>\n";
$kml .= "</Icon>\n";
$kml .= "<LatLonBox>\n";
$kml .= "<north>". $wmsUrlNorth . "</north>\n";
$kml .= "<south>" . $wmsUrlSouth . "</south>\n";
$kml .= "<east>" . $wmsUrlEast . "</east>\n";
$kml .= "<west>" . $wmsUrlWest . "</west>\n";
$kml .= "</LatLonBox>\n";
$kml .= "</GroundOverlay>\n";

// This is not required, but is useful for testing.
$kml .= "\n<!-- Google Earth request query string: " . $geReqQueryString . " -->\n";

// Send the response.
echo $kml;
?>