package biz.doodle.ws; // (c) Doodle AG

import java.io.*;
import java.security.*;
import java.util.*;
import net.oauth.*;

public class OAuthExampleTokensBase
{
    private static final long
        requestTokenTimeout   = 1000L * 60 * 10; // ten minutes in ms
    private static final long
        accessTokenTimeout    = 1000L * 60 * 60; // sixty minutes in ms
    private static final String
        timestamp             = "timestamp";
    private static final Map<String, OAuthAccessor>
        requestTokenAccessors = new LinkedHashMap<String, OAuthAccessor>();
    private static final Map<String, OAuthAccessor>
        accessTokenAccessors  = new LinkedHashMap<String, OAuthAccessor>();

    private final static SecureRandom
        random = new SecureRandom();
    static { random.nextBoolean(); } // force RNG to seed itself

    private static String bytesToHexString( byte[] input )
    {
        StringBuffer hexDigest = new StringBuffer();
        for ( int i = 0; i < input.length; i++ ) {
            hexDigest.append( Integer.toHexString( 0xFF & input[ i ] ) );
        }
        return new String( hexDigest );
    }

    public static void generateRequestToken( OAuthAccessor accessor )
    {
        random.setSeed( System.currentTimeMillis() );
        byte[] entropy = new byte[ 16 ];
        random.nextBytes( entropy );
        accessor.requestToken = bytesToHexString( entropy );
        random.nextBytes( entropy );
        accessor.tokenSecret = bytesToHexString( entropy );
        accessor.accessToken = null;
        accessor.setProperty( timestamp, System.currentTimeMillis() );
        requestTokenAccessors.put( accessor.requestToken, accessor );
    }

    public static void generateAccessToken( OAuthAccessor accessor )
    {
        random.setSeed( System.currentTimeMillis() );
        byte[] entropy = new byte[ 16 ];
        random.nextBytes( entropy );
        requestTokenAccessors.remove( accessor.requestToken );
        accessor.requestToken = null;
        accessor.accessToken = bytesToHexString( entropy );
        accessor.setProperty( timestamp, System.currentTimeMillis() );
        accessTokenAccessors.put( accessor.accessToken, accessor );
    }

    public static OAuthAccessor getRequestAccessor( String token )
        throws IOException,
            OAuthProblemException
    {
        OAuthAccessor accessor = requestTokenAccessors.get( token );
        if ( accessor == null ) {
            throw new OAuthProblemException( "token not found" );
        } else if ( System.currentTimeMillis() >
        requestTokenTimeout + (Long)accessor.getProperty( timestamp ) ) {
            requestTokenAccessors.remove( token );
            throw new OAuthProblemException( "token expired" );
        }
        return accessor;
    }

    public static OAuthAccessor getAccessAccessor( String token )
        throws IOException,
            OAuthProblemException
    {
        OAuthAccessor accessor = accessTokenAccessors.get( token );
        if ( accessor == null ) {
            throw new OAuthProblemException( "token not found" );
        } else if ( System.currentTimeMillis() >
        accessTokenTimeout + (Long)accessor.getProperty( timestamp ) ) {
            accessTokenAccessors.remove( token );
            throw new OAuthProblemException( "token expired" );
        }
        return accessor;
    }
}
