/********************************************************************
 * $Id: LibraryServiceImpl.java 90 2009-10-05 00:39:43Z rvandenh $
 *
 * (c) Copyright 2009 TH Wildau
 * All rights reserved
 ********************************************************************/
package de.thwildau.tm.library.service;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Random;
import java.util.Vector;

import org.osgi.service.event.Event;
import org.osgi.service.event.EventAdmin;

import de.thwildau.tm.library.api.Book;
import de.thwildau.tm.library.api.ILibraryService;
 

/**
 * Simple implementation of {@link ILibraryService} for demonstration
 * @author Ralf Vandenhouten
 * @version $Revision: 90 $
 */
public class LibraryServiceImpl implements ILibraryService {
    
    public static final String TOPIC = "service/library/BOOKOFTHEMINUTE";
    
    private List<Book> bookList = new Vector<Book>();
    private HashMap<Long,Long> lendMap = new HashMap<Long,Long>();
    private EventThread eventThread = null;

    /**
     * Constructor of {@link LibraryServiceImpl}
     */
    public LibraryServiceImpl() {
        // Add some good books...
        bookList.add(new Book(1001, "Kant, Immanuel", "Kritik der reinen Vernunft",
                "Reclam", 1781, "3150064619"));
        bookList.add(new Book(1002, "Zeh, Juli", "Spieltrieb",
                "btb", 2006, "3442733693"));
        bookList.add(new Book(1003, "Adams, Douglas", "The Hitchhiker's Guide to the Galaxy",
                "Harmony", 1980, "1400052939"));
        for (Book b: bookList)
            lendMap.put(b.getId(), null);
    }
    
    /*
     * (non-javadoc)
     * @see de.thwildau.tm.library.api.ILibraryService#getBookList()
     */
    public List<Book> getBookList() {
        return bookList;
    }

    /*
     * (non-javadoc)
     * @see de.thwildau.tm.library.api.ILibraryService#getSortedBookList()
     */
    public List<Book> getSortedBookList() {
        List<Book> copy = new Vector<Book>(bookList);
        Collections.sort(copy, new Comparator<Book>() {
            public int compare(Book a, Book b) {
                return a.getAuthor().compareTo(b.getAuthor());
            }
        });
        return copy;
    }

    /*
     * (non-javadoc)
     * @see de.thwildau.tm.library.api.ILibraryService#getBookList(long)
     */
    public List<Book> getBookList(long userId) {
        List<Book> userList = new Vector<Book>();
        for (Book b: bookList) {
            Long uId = lendMap.get(b.getId());
            if (uId!=null && uId==userId)
                userList.add(b);
        }
        return userList;
    }
    
    /*
     * (non-javadoc)
     * @see de.thwildau.tm.library.api.ILibraryService#isAvailable(long)
     */
    public synchronized boolean isAvailable(long bookId) {
        return lendMap.containsKey(bookId) && lendMap.get(bookId)==null;
    }

    /*
     * (non-javadoc)
     * @see de.thwildau.tm.library.api.ILibraryService#lendBook(long, long)
     */
    public synchronized boolean lendBook(long userId, long bookId) {
        if (lendMap.containsKey(bookId) && lendMap.get(bookId)==null) {
            lendMap.put(bookId, userId);
            return true;
        } else
            return false;
    }

    /*
     * (non-javadoc)
     * @see de.thwildau.tm.library.api.ILibraryService#returnBook(long)
     */
    public synchronized void returnBook(long bookId) {
        lendMap.put(bookId, null);
    }
    
    /**
     * Hook method for EventAdmin service
     */
    public void bindEvent(EventAdmin eventAdmin) {
        if (eventThread==null) {
            eventThread = new EventThread(eventAdmin);
            eventThread.start();
        }
    }
    
    /**
     * Hook method for EventAdmin service
     */
    public void unbindEvent(EventAdmin eventAdmin) {
        if (eventThread!=null) {
            eventThread.cancel();
            eventThread = null;
        }
    }
    
    /**
     * Thread class that sends a "book of the minute" every 60 seconds
     */
    private class EventThread extends Thread {
        
        private EventAdmin eventAdmin;
        
        private boolean cancel = false;
        
        public EventThread(EventAdmin eventAdmin) {
            super("BookOfTheMinuteThread");
            this.eventAdmin = eventAdmin;
        }
        
        public void cancel() {
            cancel = true;
        }
        
        public void run() {
            setName("BookOfTheMinuteThread");
            HashMap<String,Long> props = new HashMap<String,Long>();
            
            while (!cancel) {

                // Pick random book ID...
                int select = new Random().nextInt(bookList.size());
                long bookId = bookList.get(select).getId();
                props.put("bookId", bookId);
                
                // ...create Event from it...
                final Event event = new Event(TOPIC, props);
                
                // ...and send it as "book of the minute" event
                System.out.println("\nSENDING EVENT " + event);
                eventAdmin.postEvent(event);
                
                try {
                    Thread.sleep(60000);
                } catch (InterruptedException e) {
                    break;
                }
            }
        }
    }
}
